
package.path = package.path .. ";data/scripts/entity/merchants/?.lua;"
package.path = package.path .. ";data/scripts/lib/?.lua;"

include ("galaxy")
include ("utility")
include ("faction")
include ("player")
include ("randomext")
include ("stringutility")
include ("callable")
local SellableInventoryItem = include ("sellableinventoryitem")
local SectorTurretGenerator = include ("sectorturretgenerator")
local Dialog = include("dialogutility")
include("weapontypeutility")

-- Don't remove or alter the following comment, it tells the game the namespace this script lives in. If you remove it, the script will break.
-- namespace ResearchStation
ResearchStation = {}

local button
ResearchStation.interactionThreshold = -30000

function ResearchStation.initialize()
    if onClient() and EntityIcon().icon == "" then
        EntityIcon().icon = "data/textures/icons/pixel/research.png"
        InteractionText().text = Dialog.generateStationInteractionText(Entity(), random())
    end
end

function ResearchStation.interactionPossible(playerIndex, option)
    return CheckFactionInteraction(playerIndex, ResearchStation.interactionThreshold)
end

function ResearchStation.initUI()

    local res = getResolution()
    local size = vec2(800, 600)

    local menu = ScriptUI()
    local window = menu:createWindow(Rect(res * 0.5 - size * 0.5, res * 0.5 + size * 0.5))

    window.caption = "Research /* station title */"%_t
    window.showCloseButton = 1
    window.moveable = 1
    menu:registerWindow(window, "Research"%_t);

    local hsplit = UIHorizontalSplitter(Rect(window.size), 10, 10, 0.4)

    inventory = window:createInventorySelection(hsplit.bottom, 11)

    local vsplit = UIVerticalSplitter(hsplit.top, 10, 10, 0.4)

    local hsplitleft = UIHorizontalSplitter(vsplit.left, 10, 10, 0.5)

    hsplitleft.padding = 6
    local rect = hsplitleft.top
    rect.width = 220
    required = window:createSelection(rect, 3)

    local rect = hsplitleft.bottom
    rect.width = 150
    optional = window:createSelection(rect, 2)

    for _, sel in pairs({required, optional}) do
        sel.dropIntoEnabled = 1
        sel.entriesSelectable = 0
        sel.onReceivedFunction = "onRequiredReceived"
        sel.onDroppedFunction = "onRequiredDropped"
        sel.onClickedFunction = "onRequiredClicked"
    end

    inventory.dragFromEnabled = 1
    inventory.onClickedFunction = "onInventoryClicked"


    vsplit.padding = 30
    local rect = vsplit.right
    rect.width = 70
    rect.height = 70
    results = window:createSelection(rect, 1)
    results.entriesSelectable = 0
    results.dropIntoEnabled = 0
    results.dragFromEnabled = 0

    vsplit.padding = 10
    local organizer = UIOrganizer(vsplit.right)
    organizer.marginBottom = 5

    button = window:createButton(Rect(), "Research"%_t, "onClickResearch")
    button.width = 200
    button.height = 40
    organizer:placeElementBottom(button)

end

function ResearchStation.getUpdateInterval()
    return 30
end

function ResearchStation.updateServer(timeStep)

    ResearchStation.newsBroadcastInterval = 250
    ResearchStation.newsBroadcastCounter = (ResearchStation.newsBroadcastCounter or ResearchStation.newsBroadcastInterval - 20) + timeStep

    if ResearchStation.newsBroadcastCounter >= ResearchStation.newsBroadcastInterval then
        local texts =
        {
            "Completely random AI-supported research, it's basically a gamble!"%_t,
            "Feed objects to the research AI and see what crazy things it creates from them!"%_t,
            "Our research AI eats up objects and creates new ones based on the old ones - often better, sometimes worse!"%_t,
        }

        Sector():broadcastChatMessage(Entity(), ChatMessageType.Chatter, randomEntry(texts))
        ResearchStation.newsBroadcastCounter = 0
    end
end

function ResearchStation.initializationFinished()
    -- use the initilizationFinished() function on the client since in initialize() we may not be able to access Sector scripts on the client
    if onClient() then
        local ok, r = Sector():invokeFunction("radiochatter", "addSpecificLines", Entity().id.string,
        {
            "We remind all researchers to turn on the ventilation after finishing experiments."%_t,
            "Research for everyone."%_t,
            "Good news everyone!"%_t,
            "We will destroy everything you have if you want us to."%_t,
            "Get rid of your superfluous items and build new ones!"%_t,
            "Our researchers will offer you all capacities they have. They're being paid for that."%_t,
            "Make the best of your old things!"%_t,
            "Doing what we must because we can."%_t,
            "Completely random AI-supported research, it's basically a gamble!"%_t,
            "Feed objects to the research AI and see what crazy things it creates from them!"%_t,
            "Our research AI eats up objects and creates new ones - Often better, sometimes worse!"%_t,
        })
    end
end

function ResearchStation.removeItemFromMainSelection(key)
    local item = inventory:getItem(key)
    if not item then return end

    if item.amount then
        item.amount = item.amount - 1
        if item.amount == 0 then item.amount = nil end
    end

    inventory:remove(key)

    if item.amount then
        inventory:add(item, key)
    end

end

function ResearchStation.addItemToMainSelection(item)
    if not item then return end
    if not item.item then return end

    if item.item.stackable then
        -- find the item and increase the amount
        for k, v in pairs(inventory:getItems()) do
            if v.item and v.item == item.item then
                v.amount = v.amount + 1

                inventory:remove(k)
                inventory:add(v, k)
                return
            end
        end

        item.amount = 1
    end

    -- when not found or not stackable, add it
    inventory:add(item)

end

function ResearchStation.moveItem(item, from, to, fkey, tkey)
    if not item then return end

    if from.index == inventory.index then -- move from inventory to a selection
        -- first, move the item that might be in place back to the inventory
        if tkey then
            ResearchStation.addItemToMainSelection(to:getItem(tkey))
            to:remove(tkey)
        end

        ResearchStation.removeItemFromMainSelection(fkey)

        -- fix item amount, we don't want numbers in the upper selections
        item.amount = nil
        to:add(item, tkey)

    elseif to.index == inventory.index then
        -- move from selection to inventory
        ResearchStation.addItemToMainSelection(item)
        from:remove(fkey)
    end
end

function ResearchStation.onRequiredReceived(selectionIndex, fkx, fky, item, fromIndex, toIndex, tkx, tky)
    if not item then return end

    -- don't allow dragging from/into the left hand selections
    if fromIndex == optional.index or fromIndex == required.index then
        return
    end

    ResearchStation.moveItem(item, inventory, Selection(selectionIndex), ivec2(fkx, fky), ivec2(tkx, tky))

    ResearchStation.refreshButton()
    results:clear()
    results:addEmpty()
end

function ResearchStation.onRequiredClicked(selectionIndex, fkx, fky, item, button)
    if button == 3 or button == 2 then
        ResearchStation.moveItem(item, Selection(selectionIndex), inventory, ivec2(fkx, fky), nil)
        ResearchStation.refreshButton()
    end
end

function ResearchStation.onRequiredDropped(selectionIndex, kx, ky)
    local selection = Selection(selectionIndex)
    local key = ivec2(kx, ky)
    ResearchStation.moveItem(selection:getItem(key), Selection(selectionIndex), inventory, key, nil)
    ResearchStation.refreshButton()
end

function ResearchStation.onInventoryClicked(selectionIndex, kx, ky, item, button)

    if button == 2 or button == 3 then
        -- fill required first, then, once it's full, fill optional
        local items = required:getItems()
        if tablelength(items) < 3 then
            ResearchStation.moveItem(item, inventory, required, ivec2(kx, ky), nil)

            ResearchStation.refreshButton()
            results:clear()
            results:addEmpty()
            return
        end

        local items = optional:getItems()
        if tablelength(items) < 2 then
            ResearchStation.moveItem(item, inventory, optional, ivec2(kx, ky), nil)

            ResearchStation.refreshButton()
            results:clear()
            results:addEmpty()
            return
        end
    end
end

function ResearchStation.refreshButton()
    local requiredItems = required:getItems()
    button.active = (tablelength(requiredItems) == 3)

    if tablelength(requiredItems) ~= 3 then
        button.tooltip = "Place at least 3 items for research!"%_t
    else
        button.tooltip = "Feed to Research AI"%_t
    end

    for _, items in pairs({requiredItems, optional:getItems()}) do
        for _, item in pairs(items) do
            if item.item
                and item.item.itemType ~= InventoryItemType.TurretTemplate
                and item.item.itemType ~= InventoryItemType.SystemUpgrade
                and item.item.itemType ~= InventoryItemType.Turret then

                button.active = false
                button.tooltip = "Invalid items in ingredients."%_t
            end
        end
    end

end

function ResearchStation.onShowWindow()

    inventory:clear()
    required:clear()
    optional:clear()

    required:addEmpty()
    required:addEmpty()
    required:addEmpty()

    optional:addEmpty()
    optional:addEmpty()

    results:addEmpty()

    ResearchStation.refreshButton()

    for i = 1, 50 do
        inventory:addEmpty()
    end

    local player = Player()
    local ship = player.craft
    local alliance = player.alliance

    if alliance and ship.factionIndex == player.allianceIndex then
        inventory:fill(alliance.index)
    else
        inventory:fill(player.index)
    end

end

function ResearchStation.checkRarities(items) -- items must not be more than 1 rarity apart
    local min = math.huge
    local max = -math.huge

    for _, item in pairs(items) do
        if item.rarity.value < min then min = item.rarity.value end
        if item.rarity.value > max then max = item.rarity.value end
    end

    if max - min <= 1 then
        return true
    end

    return false
end

function ResearchStation.getRarityProbabilities(items)

    local probabilities = {}

    -- for each item there is a 20% chance that the researched item has a rarity 1 better
    for _, item in pairs(items) do
        -- next rarity cannot exceed legendary
        local nextRarity = math.min(RarityType.Legendary, item.rarity.value + 1)

        local p = probabilities[nextRarity] or 0
        p = p + 0.2
        probabilities[nextRarity] = p
    end

    -- if the amount of items is < 5 then add their own rarities as a result as well
    if #items < 5 then
        local left = (1.0 - #items * 0.2)
        local perItem = left / #items

        for _, item in pairs(items) do
            local p = probabilities[item.rarity.value] or 0
            p = p + perItem
            probabilities[item.rarity.value] = p
        end
    end

    local sum = 0
    for _, p in pairs(probabilities) do
        sum = sum + p
    end

    return probabilities
end

function ResearchStation.getTypeProbabilities(items)
    local probabilities = {}

    for _, item in pairs(items) do
        local p = probabilities[item.itemType] or 0
        p = p + 1
        probabilities[item.itemType] = p
    end

    return probabilities
end

function ResearchStation.getWeaponProbabilities(items)
    local probabilities = {}
    local typesByIcons = getWeaponTypesByIcon()

    for _, item in pairs(items) do
        if item.itemType == InventoryItemType.Turret
            or item.itemType == InventoryItemType.TurretTemplate then

            local weaponType = WeaponTypes.getTypeOfItem(item)

            local p = probabilities[weaponType] or 0
            p = p + 1
            probabilities[weaponType] = p
        end
    end

    return probabilities
end

function ResearchStation.getWeaponMaterials(items)
    local probabilities = {}

    for _, item in pairs(items) do
        if item.itemType == InventoryItemType.Turret
            or item.itemType == InventoryItemType.TurretTemplate then

            local p = probabilities[item.material.value] or 0
            p = p + 1
            probabilities[item.material.value] = p
        end
    end

    return probabilities
end

function ResearchStation.getAutoFires(items)
    local probabilities = {}

    for _, item in pairs(items) do
        if item.itemType == InventoryItemType.Turret
            or item.itemType == InventoryItemType.TurretTemplate then

            local p = probabilities[item.automatic] or 0
            p = p + 1
            probabilities[item.automatic] = p
        end
    end

    return probabilities
end

function ResearchStation.getSystemProbabilities(items)
    local probabilities = {}

    for _, item in pairs(items) do
        if item.itemType == InventoryItemType.SystemUpgrade then
            local p = probabilities[item.script] or 0
            p = p + 1
            probabilities[item.script] = p
        end
    end

    return probabilities
end





function ResearchStation.onClickResearch()

    local items = {}
    local itemIndices = {}

    for _, item in pairs(required:getItems()) do
        if item.item then
            table.insert(items, item.item)

            local amount = itemIndices[item.index] or 0
            amount = amount + 1
            itemIndices[item.index] = amount
        end
    end
    for _, item in pairs(optional:getItems()) do
        if item.item then
            table.insert(items, item.item)

            local amount = itemIndices[item.index] or 0
            amount = amount + 1
            itemIndices[item.index] = amount
        end
    end

    if not ResearchStation.checkRarities(items) then
        displayChatMessage("Your items cannot be more than one rarity apart!"%_t, Entity().title, 1)
        return
    end

    invokeServerFunction("research", itemIndices)
end

function ResearchStation.research(itemIndices)
    if not itemIndices then return end

    if not CheckFactionInteraction(callingPlayer, ResearchStation.interactionThreshold) then return end

    local buyer, ship, player = getInteractingFaction(callingPlayer, AlliancePrivilege.SpendResources)
    if not buyer then return end

    -- check if the player has enough of the items
    local items = {}

    for index, amount in pairs(itemIndices) do
        local item = buyer:getInventory():find(index)
        local has = buyer:getInventory():amount(index)

        if not item or has < amount then
            player:sendChatMessage(Entity(), 1, "You don't have enough items!"%_t)
            return
        end

        for i = 1, amount do
            table.insert(items, item)
        end
    end

    if #items < 3 then
        player:sendChatMessage(Entity(), 1, "You need at least 3 items to do research!"%_t)
        return
    end

    if not ResearchStation.checkRarities(items) then
        player:sendChatMessage(Entity(), 1, "Your items cannot be more than one rarity apart!"%_t)
        return
    end

    local station = Entity()

    local errors = {}
    errors[EntityType.Station] = "You must be docked to the station to research items."%_T
    errors[EntityType.Ship] = "You must be closer to the ship to research items."%_T
    if not CheckPlayerDocked(player, station, errors) then
        return
    end

    local result = ResearchStation.transform(items)

    if result then
        for index, amount in pairs(itemIndices) do
            for i = 1, amount do
                buyer:getInventory():take(index)
            end
        end

        local inventory = buyer:getInventory()
        if not inventory:hasSlot(result) then
            buyer:sendChatMessage(station, ChatMessageType.Warning, "Your inventory is full (%1%/%2%). Your researched item was dropped."%_T, inventory.occupiedSlots, inventory.maxSlots)
        end

        inventory:addOrDrop(result)

        invokeClientFunction(player, "receiveResult", result)
    else
        print ("no result")
    end
end
callable(ResearchStation, "research")

function ResearchStation.researchTest(...)
    local indices = {}

    for _, index in pairs({...}) do
        local amount = indices[index] or 0
        indices[index] = amount + 1
    end

    ResearchStation.research(indices)
end

function ResearchStation.receiveResult(result)
    results:clear();

    local item = InventorySelectionItem()
    item.item = result

    results:add(item)
    ResearchStation.onShowWindow()
end

function ResearchStation.transform(items)

    local transformToKey

    if items[1].itemType == InventoryItemType.SystemUpgrade
        and items[2].itemType == InventoryItemType.SystemUpgrade
        and items[3].itemType == InventoryItemType.SystemUpgrade
        and items[1].rarity.value == RarityType.Legendary
        and items[2].rarity.value == RarityType.Legendary
        and items[3].rarity.value == RarityType.Legendary then

        local inputKeys = 0
        for _, item in pairs(items) do
            if item.itemType == InventoryItemType.SystemUpgrade then
                if string.match(item.script, "systems/teleporterkey") then
                    inputKeys = inputKeys + 1
                end
            end
        end

        if inputKeys <= 1 then
            transformToKey = true
        end
    end

    local result

    if transformToKey then
        result = SystemUpgradeTemplate("data/scripts/systems/teleporterkey2.lua", Rarity(RarityType.Legendary), random():createSeed())
    else
        local rarities = ResearchStation.getRarityProbabilities(items)
        local types = ResearchStation.getTypeProbabilities(items, "type")

        local itemType = selectByWeight(random(), types)
        local rarity = Rarity(selectByWeight(random(), rarities))


        if itemType == InventoryItemType.Turret
            or itemType == InventoryItemType.TurretTemplate then

            local weaponTypes = ResearchStation.getWeaponProbabilities(items)
            local materials = ResearchStation.getWeaponMaterials(items)
            local autoFires = ResearchStation.getAutoFires(items)

            local weaponType = selectByWeight(random(), weaponTypes)
            local material = Material(selectByWeight(random(), materials))
            local autoFire = selectByWeight(random(), autoFires)

            local x, y = Sector():getCoordinates()
            local generator = SectorTurretGenerator()
            generator.maxVariations = 10
            result = generator:generate(x, y, -5, rarity, weaponType, material)

            if itemType == InventoryItemType.Turret then
                result = InventoryTurret(result)
            end

            result.automatic = autoFire or false

        elseif itemType == InventoryItemType.SystemUpgrade then
            local scripts = ResearchStation.getSystemProbabilities(items)

            local script = selectByWeight(random(), scripts)

            result = SystemUpgradeTemplate(script, rarity, random():createSeed())
        end
    end

    return result
end
