package.path = package.path .. ";data/scripts/lib/?.lua"
include ("galaxy")
include ("utility")
include ("faction")
include ("randomext")
include ("stringutility")
include ("callable")
include ("merchantutility")
include ("reconstructiontoken")
include ("relations")
local Dialog = include("dialogutility")

-- Don't remove or alter the following comment, it tells the game the namespace this script lives in. If you remove it, the script will break.
-- namespace RepairDock
RepairDock = {}
RepairDock.tax = 0.2
RepairDock.scroll = 1
RepairDock.interactionThreshold = -30000

local window = nil
local tabbedWindow = nil
local repairTab = nil
local tokensTab = nil
local reconstructionTab = nil
local planDisplayer = nil
local repairButton = nil

local uiMoneyCost
local uiResourceCost

local priceDescriptionLabel
local reconstructionPriceLabel
local reconstructionButton

local buyTokenNameLabel
local buyTokenPriceLabel
local buyTokenAmountLabel
local buyTokenButton

local reconstructionLines = {}

-- if this function returns false, the script will not be listed in the interaction window on the client,
-- even though its UI may be registered
function RepairDock.interactionPossible(playerIndex, option)
    return CheckFactionInteraction(playerIndex, RepairDock.interactionThreshold)
end

-- this function gets called on creation of the entity the script is attached to, on client and server
function RepairDock.initialize()
    local station = Entity()

    if station.title == "" then
        station.title = "Repair Dock /* Station Title*/"%_t
    end

    if onClient() and EntityIcon().icon == "" then
        EntityIcon().icon = "data/textures/icons/pixel/repair.png"
        InteractionText(station.index).text = Dialog.generateStationInteractionText(station, random())
    end

end

-- this function gets called on creation of the entity the script is attached to, on client only
-- AFTER initialize above
-- create all required UI elements for the client side
function RepairDock.initUI()

    local res = getResolution()
    local size = vec2(600, 600)

    local menu = ScriptUI()
    window = menu:createWindow(Rect(res * 0.5 - size * 0.5, res * 0.5 + size * 0.5));
    menu:registerWindow(window, "Repair Dock /* Interaction Title*/"%_t);

    window.caption = "Repair Dock /* Station Title*/"%_t
    window.showCloseButton = 1
    window.moveable = 1

    tabbedWindow = window:createTabbedWindow(Rect(vec2(10, 10), size - vec2(10, 10)))
    local tab = tabbedWindow:createTab("", "data/textures/icons/repair.png", "Repair your ship"%_t)
    repairTab = tab

    local hsplit = UIHorizontalSplitter(Rect(tab.size), 10, 0, 1.0)
    hsplit.bottomSize = 40

    planDisplayer = tab:createPlanDisplayer(hsplit.top);
    planDisplayer.showStats = 0

    repairButton = tab:createButton(hsplit.bottom, "Repair /* Action */"%_t, "onRepairButtonPressed")

    -- setting of Reconstruction Site
    local tab = tabbedWindow:createTab("", "data/textures/icons/reconstruction-token.png", "")
    tokensTab = tab

    local hsplit = UIHorizontalSplitter(Rect(tab.size), 10, 0, 0.75)

    local lister = UIVerticalLister(hsplit.top, 5, 0)

    for i = 0, 1 do
        local rect = lister:nextRect(30)

        local vsplit = UIArbitraryVerticalSplitter(rect, 5, 0, 200, 350, 400)
        local frame = tab:createFrame(Rect(vsplit:partition(0).topLeft, vsplit:partition(2).bottomRight))

        local button = tab:createButton(vsplit:partition(3), "Buy Token"%_t, "onBuyTokenButtonPressed")
        button.textSize = 14

        vsplit:setMargin(10, 10, 3, 3)
        vsplit:setPadding(10, 10, 10, 10)

        local nameLabel = tab:createLabel(vsplit:partition(0), "Ship 123", 13)
        local priceLabel = tab:createLabel(vsplit:partition(1), "10.000.000.000", 13)
        local amountLabel = tab:createLabel(vsplit:partition(2), "1", 13)

        nameLabel:setLeftAligned()
        priceLabel:setLeftAligned()
        amountLabel:setLeftAligned()

        if i > 0 then
            buyTokenNameLabel = nameLabel
            buyTokenPriceLabel = priceLabel
            buyTokenAmountLabel = amountLabel
            buyTokenButton = button
        else
            nameLabel.caption = "Ship"%_t
            priceLabel.caption = "¢"
            amountLabel.caption = "Tk"%_t
            button:hide()
            frame:hide()
        end
    end

    -- explanation for Reconstruction Tokens
    local rect = lister:nextRect(10)
    local rect = lister:nextRect(10)
    tab:createLine(rect.topLeft, rect.topRight)

    local rect = lister:nextRect(40)
    local label = tab:createLabel(rect.topLeft, "UNIQUE OFFER: Free Repair when buying a Reconstruction Token!"%_t, 12)
    label.fontSize = 12
    label.color = ColorRGB(0.4, 1.0, 0.4)
    label.wordBreak = true

    local rect = lister:nextRect(40)
    priceDescriptionLabel = tab:createLabel(rect.topLeft, "Reconstruction Token: Costs ~30% of your ship's value (20% at Reconstruction Site).\n\nAllows instant reconstruction of destroyed ships at any repair dock. Must be bought in advance, before a ship is destroyed."%_t, 12)
    priceDescriptionLabel.fontSize = 12
    priceDescriptionLabel.wordBreak = true


    -- setting of reconstruction site
    local splitter = UIHorizontalSplitter(hsplit.bottom, 10, 0, 0.5)
    splitter.bottomSize = 40
    tab:createLine(hsplit.top.bottomLeft, hsplit.top.bottomRight)

    setReconstructionSiteButton = tab:createButton(splitter.bottom, "Use as Reconstruction Site /* Action */"%_t, "onUseReconstructionSiteButtonPressed")

    local lister = UIVerticalLister(splitter.top, 10, 0)

    local rect = lister:nextRect(40)
    local label = tab:createLabel(rect.topLeft, "Reconstruction Site: if you choose this station as your reconstruction site, your drone will be reconstructed at this station and you will be placed in this sector when you die."%_t, 12)
    label.fontSize = 12
    label.wordBreak = true

    local rect = lister:nextRect(15)

    reconstructionPriceLabel = tab:createLabel(rect.topLeft, "Price: "%_t, 12)


    -- reconstruction of ships
    local tab = tabbedWindow:createTab("", "data/textures/icons/reconstruct-ship.png", "")
    reconstructionTab = tab

    local lister = UIVerticalLister(Rect(tab.size), 5, 0)

    for i = 0, 14 do
        local rect = lister:nextRect(30)

        local vsplit = UIArbitraryVerticalSplitter(rect, 5, 0, 220, 400)
        local frame = tab:createFrame(Rect(vsplit:partition(0).topLeft, vsplit:partition(1).bottomRight))

        local button = tab:createButton(vsplit:partition(2), "Reconstruct"%_t, "onReconstructButtonPressed")
        button.textSize = 14

        vsplit:setMargin(10, 10, 3, 3)
        vsplit:setPadding(10, 10, 10, 10)

        local nameLabel = tab:createLabel(vsplit:partition(0), "Ship 123", 12)
        local priceLabel = tab:createLabel(vsplit:partition(1), "10.000", 12)

        nameLabel:setLeftAligned()
        priceLabel:setLeftAligned()

        if i > 0 then
            local line = {nameLabel = nameLabel, priceLabel = priceLabel, frame = frame, button = button}
            table.insert(reconstructionLines, line)

            line.hide = function(self)
                self.nameLabel:hide()
                self.priceLabel:hide()
                self.frame:hide()
                self.button:hide()
            end

            line.show = function(self)
                self.nameLabel:show()
                self.priceLabel:show()
                self.frame:show()
                self.button:show()
            end
        else
            nameLabel.caption = "Ship"%_t
            priceLabel.caption = "Cost"%_t
            button:hide()
            frame:hide()
        end
    end

end

function RepairDock.initializationFinished()

    local specificLinesRepairDock = {}
    specificLinesRepairDock = {
        "Problems with docking? Scratches on the hull? We'll fix that for you."%_t,
        "Asteroid-fall damage on the windshield? We'll fix that for you."%_t,
        "Almost got destroyed by pirates? We'll fix that for you."%_t,
        "The Xsotan are really good for business. Just like pirates. But don't tell anyone I said that."%_t,
        "We're lucky that shields don't protect against asteroids. We'd be out of business if they did."%_t,
    }

    -- use the initilizationFinished() function on the client since in initialize() we may not be able to access Sector scripts on the client
    if onClient() then
        local ok, r = Sector():invokeFunction("radiochatter", "addSpecificLines", Entity().id.string, specificLinesRepairDock)
    end
end

-- this function gets called every time the window is shown on the client, ie. when a player presses F and if interactionPossible() returned 1
function RepairDock.onShowWindow(option)
    -- this could get called by the server at seemingly random times, so we must check that the UI was initialized
    if not window then return end

    -- repairing
    RepairDock.refreshRepairUI()

    if not GameSettings().reconstructionAllowed then
        tabbedWindow:deactivateTab(tokensTab)
        tabbedWindow:deactivateTab(reconstructionTab)
    else
        -- reconstruction site & tokens
        RepairDock.refreshReconstructionTokens()

        -- reconstructing ships
        RepairDock.refreshReconstructionLines()
    end

end

function RepairDock.refreshRepairUI()
    -- repair ship
    local buyer = Player()
    local ship = buyer.craft
    if ship.factionIndex == buyer.allianceIndex then
        buyer = buyer.alliance
        DebugInfo():log("RepairDock.onShowWindow Alliance")
    else
        DebugInfo():log("RepairDock.onShowWindow Player")
    end

    -- get the plan of the player (or alliance)'s ship template
    local intact = buyer:getShipPlan(ship.name)

    if intact then
        -- get the plan of the player's ship
        local broken = ship:getFullPlanCopy()

        if not broken then
            DebugInfo():log("RepairDock.onShowWindow broken == nil")
        end

        -- set to display
        planDisplayer:setPlans(broken, intact)

        uiMoneyCost = RepairDock.getRepairMoneyCostAndTax(buyer, ship, intact, broken, ship.durability / ship.maxDurability)
        uiResourceCost = RepairDock.getRepairResourcesCost(buyer, ship, intact, broken, ship.durability / ship.maxDurability)

        local damaged = false

        if uiMoneyCost > 0 then
            damaged = true
        end

        for _, cost in pairs(uiResourceCost) do
            if cost > 0 then
                damaged = true
            end
        end

        if damaged then
            repairButton.active = true
            repairButton.tooltip = "Repair ship"%_t
        else
            repairButton.active = false
            repairButton.tooltip = "Your ship is not damaged."%_t
        end
    else
        repairButton.active = false
        repairButton.tooltip = nil
    end
end

function RepairDock.refreshReconstructionTokens()

    if RepairDock.isShipyardRepairDock() then
        tabbedWindow:deactivateTab(tokensTab)
        tokensTab.description = "Only available at Repair Docks, not Shipyards!"%_t
        return
    else
        tabbedWindow:activateTab(tokensTab)
        tokensTab.description = "Buy Reconstruction Tokens"%_t
    end

    local player = Player()
    local buyer = Galaxy():getPlayerCraftFaction()
    local ship = player.craft

    reconstructionPriceLabel.caption = "Price: ¢${money}"%_t % {money = createMonetaryString(RepairDock.getReconstructionSiteChangePrice())}

    if buyer.isAlliance then
        setReconstructionSiteButton.active = false
        setReconstructionSiteButton.tooltip = "Alliances don't have reconstruction sites."%_t
    elseif RepairDock.isReconstructionSite() then
        setReconstructionSiteButton.active = false
        setReconstructionSiteButton.tooltip = "This sector is already your reconstruction site."%_t
    else
        setReconstructionSiteButton.active = true
        setReconstructionSiteButton.tooltip = nil
    end

    local tokens = countReconstructionTokens(player, ship.name, buyer.index)
    local atokens = 0

    local alliance = player.alliance
    if alliance then
        atokens = countReconstructionTokens(alliance, ship.name, buyer.index)
    end

    local price = RepairDock.getReconstructionTokenPrice(buyer, ship)

    if price and price > 0 then
        buyTokenAmountLabel.caption = tostring(tokens)
        buyTokenNameLabel.caption = ship.name
        buyTokenPriceLabel.caption = createMonetaryString(price)

        if atokens > 0 then
            buyTokenButton.active = false
            buyTokenButton.tooltip = "Your alliance already owns a Reconstruction Token for this ship."%_t
            buyTokenAmountLabel.active = false
        elseif tokens > 0 then
            buyTokenButton.active = false
            buyTokenButton.tooltip = "You already own a Reconstruction Token for this ship."%_t
            buyTokenAmountLabel.active = false
        else
            buyTokenButton.active = true
            buyTokenAmountLabel.active = true
            buyTokenButton.tooltip = "Buy a token and get your ship repaired FOR FREE!"%_t
        end

        if buyer.isPlayer and RepairDock.isReconstructionSite() then
            buyTokenPriceLabel.color = ColorRGB(0, 1, 0)
            buyTokenPriceLabel.tooltip = "You get a 30% discount for tokens at your Reconstruction Site!"%_t
        else
            buyTokenPriceLabel.color = ColorRGB(1, 1, 1)
            buyTokenPriceLabel.tooltip = nil
        end
    else
        buyTokenAmountLabel.caption = ""
        buyTokenNameLabel.caption = ""
        buyTokenPriceLabel.caption = ""

        buyTokenButton.active = false
        buyTokenButton.tooltip = nil
    end

    local malus, reason = ship:getMalusFactor()
    if reason and reason == MalusReason.Boarding then
        priceDescriptionLabel:hide()
    else
        priceDescriptionLabel:show()
    end

end

function RepairDock.refreshReconstructionLines()

    if RepairDock.isShipyardRepairDock() then
        tabbedWindow:deactivateTab(reconstructionTab)
        reconstructionTab.description = "Only available at Repair Docks, not Shipyards!"%_t
        return
    else
        tabbedWindow:activateTab(reconstructionTab)
        reconstructionTab.description = "Reconstruct a ship"%_t
    end

    local buyer = Galaxy():getPlayerCraftFaction()
    local shipNames = {buyer:getShipNames()}
    local scroll = RepairDock.scroll
    local s = 1
    local l = 1

    for _, name in pairs(shipNames) do

        if not buyer:getShipDestroyed(name) then goto continue end
        local line = reconstructionLines[l]

        local tokens = countReconstructionTokens(buyer, name)
        if tokens > 0 then
            line.priceLabel.caption = "1 Token"%_t
            line.priceLabel.color = ColorRGB(0, 1, 0)
            line.button.tooltip = "Reconstruct using a Reconstruction Token"%_t
            line.priceLabel.tooltip = "You have a Reconstruction Token for this ship!"%_t
        else
            local price = RepairDock.getReconstructionPrice(buyer, name)
            if not price then goto continue end

            line.priceLabel.caption = "¢${price}"%_t % {price = createMonetaryString(price)}
            line.priceLabel.color = ColorRGB(1, 1, 0.3)
            line.button.tooltip = "Reconstruct using money"%_t
            line.priceLabel.tooltip = "No Reconstruction Token!"%_t
        end

        line:show()
        line.shipName = name
        line.nameLabel.caption = name

        l = l + 1
        if l > #reconstructionLines then break end

        ::continue::
    end

    for i = l, #reconstructionLines do
        reconstructionLines[i]:hide()
    end
end

-- this function gets called every time the window is closed on the client
function RepairDock.onCloseWindow()
end

-- this function gets called whenever the ui window gets rendered, AFTER the window was rendered (client only)
function RepairDock.renderUI()
    if tabbedWindow:getActiveTab().index == repairTab.index then
        renderPrices(repairTab.lower + 5, "Repair Costs:"%_t, uiMoneyCost, uiResourceCost)
    end
end

function RepairDock.onBuyTokenButtonPressed()
    invokeServerFunction("buyToken")
end

function RepairDock.onReconstructButtonPressed(arg)
    -- find the matching button and thus name of the ship that should be reconstructed
    local name
    for _, line in pairs(reconstructionLines) do
        if line.button.index == arg.index then
            name = line.shipName
            break
        end
    end

    if name then
        invokeServerFunction("reconstruct", name)
    end
end

function RepairDock.clientReconstruct(shipName)
    invokeServerFunction("reconstruct", shipName)
end

function RepairDock.onRepairButtonPressed()
    invokeServerFunction("repairCraft")
end

function RepairDock.onUseReconstructionSiteButtonPressed()
    invokeServerFunction("setAsReconstructionSite")
end

function RepairDock.transactionComplete()
    ScriptUI():stopInteraction()
end

function RepairDock.repairCraft()

    if not CheckFactionInteraction(callingPlayer, RepairDock.interactionThreshold) then return end

    local buyer, ship, player = getInteractingFaction(callingPlayer, AlliancePrivilege.SpendResources)
    if not buyer then return end

    local station = Entity()
    local seller = Faction()

    local dist = station:getNearestDistance(ship)
    if dist > 100 then
        player:sendChatMessage(station, 1, "You can't be more than 1km away to repair your ship."%_t)
        return
    end

    -- this function is executed on the server
    local perfectPlan = buyer:getShipPlan(ship.name)
    if not perfectPlan then return end

    local damagedPlan = ship:getFullPlanCopy()
    if not damagedPlan then return end

    local requiredMoney, tax = RepairDock.getRepairMoneyCostAndTax(buyer, ship, perfectPlan, damagedPlan, ship.durability / ship.maxDurability)
    local requiredResources = RepairDock.getRepairResourcesCost(buyer, ship, perfectPlan, damagedPlan, ship.durability / ship.maxDurability)

    local canPay, msg, args = buyer:canPay(requiredMoney, unpack(requiredResources))

    if not canPay then
        player:sendChatMessage(station, 1, msg, unpack(args))
        return
    end

    receiveTransactionTax(station, tax)

    buyer:pay(requiredMoney, unpack(requiredResources))

    perfectPlan:resetDurability()
    ship:setMalusFactor(1.0, MalusReason.None)
    ship:setMovePlan(perfectPlan)
    ship.durability = ship.maxDurability

    -- relations of the player to the faction owning the repair dock get better
    local relationsChange = requiredMoney / 40

    for i = 1, NumMaterials() do
        relationsChange = relationsChange + requiredResources[i] / 4
    end

    changeRelations(buyer, Faction(), relationsChange, RelationChangeType.ServiceUsage)

    invokeClientFunction(player, "onShowWindow", 0)
    invokeClientFunction(player, "transactionComplete")

end
callable(RepairDock, "repairCraft")

function RepairDock.reconstruct(shipName)

    if not CheckFactionInteraction(callingPlayer, RepairDock.interactionThreshold) then return end

    local buyer, ship, player = getInteractingFaction(callingPlayer, AlliancePrivilege.ManageShips, AlliancePrivilege.SpendItems)
    if not buyer then return end

    if RepairDock.isShipyardRepairDock() then
        player:sendChatMessage(Entity(), ChatMessageType.Error, "Shipyards don't offer these kinds of services."%_T)
        return
    end

    -- reconstructing stations is not possible
    if buyer:getShipType(shipName) ~= EntityType.Ship then
        player:sendChatMessage(Entity(), ChatMessageType.Error, "Can only reconstruct ships."%_T)
        return
    end

    if not GameSettings().reconstructionAllowed then
        player:sendChatMessage(Entity(), ChatMessageType.Error, "Reconstruction impossible."%_T)
        return
    end

    -- reconstructing non-destroyed ships is impossible
    if not buyer:getShipDestroyed(shipName) then
        player:sendChatMessage(Entity(), ChatMessageType.Error, "Ship wasn't destroyed."%_T)
        return
    end

    -- check if we can pay with tokens
    local paidWithToken = false
    local tokens, item, idx = countReconstructionTokens(buyer, shipName)
    if tokens > 0 then
        local taken = buyer:getInventory():take(idx)
        if not taken then
            player:sendChatMessage(Entity(), ChatMessageType.Error, "Token for this ship not found."%_T)
        else
            paidWithToken = true
            player:sendChatMessage(Entity(), ChatMessageType.Information, "Used a Reconstruction Token to reconstruct '%s'."%_T, shipName)
        end
    end

    if not paidWithToken then
        -- if we can't, use the (higher) reconstruction price
        local price = RepairDock.getReconstructionPrice(buyer, shipName)
        local canPay, msg, args = buyer:canPay(price)

        if not canPay then
            player:sendChatMessage(Entity(), ChatMessageType.Error, msg, unpack(args))
            return
        end

        buyer:pay("Paid %1% Credits to reconstruct a ship."%_T, price)
    end

    -- find a position to put the craft
    local position = Matrix()
    local station = Entity()
    local box = buyer:getShipBoundingBox(shipName)

    -- try putting the ship at a dock
    local docks = DockingPositions(station)
    local dockIndex = docks:getFreeDock()
    if dockIndex then
        local pos, dir = docks:getDockingPosition(dockIndex)

        pos = station.position:transformCoord(pos)
        dir = station.position:transformNormal(dir)

        pos = pos + dir * (box.size.z / 2 + 10)

        local up = station.position.up

        position = MatrixLookUpPosition(-dir, up, pos)
    else
        -- if all docks are occupied, place it near the station
        -- use the same orientation as the station
        position = station.orientation

        local sphere = station:getBoundingSphere()
        position.translation = sphere.center + random():getDirection() * (sphere.radius + length(box.size) / 2 + 50);
    end


    local craft = buyer:restoreCraft(shipName, position, true)
    CargoBay(craft):clear()

    if not craft then
        player:sendChatMessage(Entity(), ChatMessageType.Error, "Error reconstructing craft."%_t)
        return
    end

    if ship.isDrone then
        player.craft = craft
        invokeClientFunction(player, "transactionComplete")
    end

    Sector():broadcastChatMessage(Entity(), ChatMessageType.Normal, "Insta-Reconstruction complete! Your ship may have suffered some minor structural damages due to the reconstruction process."%_t)
    Sector():broadcastChatMessage(Entity(), ChatMessageType.Normal, "If you buy a new Reconstruction Token, we'll fix her up for free!"%_t)

    invokeClientFunction(player, "onShowWindow", 0)
end
callable(RepairDock, "reconstruct")

function RepairDock.buyToken()

    if not CheckFactionInteraction(callingPlayer, RepairDock.interactionThreshold) then return end

    local buyer, ship, player = getInteractingFaction(callingPlayer, AlliancePrivilege.SpendResources, AlliancePrivilege.AddItems)
    if not buyer then return end

    if RepairDock.isShipyardRepairDock() then
        player:sendChatMessage(Entity(), ChatMessageType.Error, "Shipyards don't offer these kinds of services."%_t)
        return
    end

    local tokens = countReconstructionTokens(player, ship.name, buyer.index)
    if tokens > 0 then
        player:sendChatMessage(Entity(), ChatMessageType.Error, "You already own a Reconstruction Token for this ship."%_t)
        return
    end

    local alliance = player.alliance
    if alliance then
        local tokens = countReconstructionTokens(alliance, ship.name, buyer.index)
        if tokens > 0 then
            player:sendChatMessage(Entity(), ChatMessageType.Error, "Your alliance already owns a Reconstruction Token for this ship."%_t)
            return
        end
    end

    local token = createReconstructionToken(ship)
    local inventory = buyer:getInventory()
    if not inventory:hasSlot(token) then
        player:sendChatMessage(Entity(), ChatMessageType.Error, "Your inventory is full (%1%/%2%)."%_T, inventory.occupiedSlots, inventory.maxSlots)
        return
    end

    local price = RepairDock.getReconstructionTokenPrice(buyer, ship)
    local canPay, msg, args = buyer:canPay(price)

    if not canPay then
        player:sendChatMessage(Entity(), ChatMessageType.Error, msg, unpack(args))
        return
    end

    buyer:pay(price)
    inventory:addOrDrop(token)

    -- on top of the token, ship gets repaired
    -- this function is executed on the server
    local perfectPlan = buyer:getShipPlan(ship.name)
    if perfectPlan then
        perfectPlan:resetDurability()
        ship:setMalusFactor(1.0, MalusReason.None)
        ship:setMovePlan(perfectPlan)
        ship.durability = ship.maxDurability
    end

    invokeClientFunction(player, "onShowWindow", 0)
    invokeClientFunction(player, "transactionComplete")
end
callable(RepairDock, "buyToken")

function RepairDock.setAsReconstructionSite()   
    local player = Player(callingPlayer)

    if RepairDock.isShipyardRepairDock() then
        player:sendChatMessage(Entity(), ChatMessageType.Error, "Shipyards don't offer these kinds of services."%_t)
        return
    end

    if RepairDock.isReconstructionSite(player) then
        player:sendChatMessage(Entity(), ChatMessageType.Error, "This sector is already your reconstruction sector."%_t)
        return
    end

    if not CheckFactionInteraction(callingPlayer, 60000) then
        player:sendChatMessage(Entity(), ChatMessageType.Normal, "We only offer these kinds of services to people we have Excellent or better relations with."%_t)
        player:sendChatMessage("", ChatMessageType.Error, "Your relations with that faction aren't good enough."%_t)
        return
    end

    local requiredMoney = RepairDock.getReconstructionSiteChangePrice()

    local ok, msg, args = player:canPay(requiredMoney)
    if not ok then
        player:sendChatMessage(Entity(), ChatMessageType.Error, msg, unpack(args))
        return
    end

    player:pay("Paid %1% Credits to set a new Reconstruction Site."%_T, requiredMoney)

    local x, y = Sector():getCoordinates()
    player:setRespawnSectorCoordinates(x, y)

    invokeClientFunction(player, "onShowWindow", 1)
end
callable(RepairDock, "setAsReconstructionSite")

function RepairDock.getRepairResourcesCost(orderingFaction, ship, perfectPlan, damagedPlan, durabilityPercentage)

    -- value of blockplan template
    local templateValue = {perfectPlan:getUndamagedResourceValue()}
    -- value of player's craft blockplan
    local craftValue = {damagedPlan:getResourceValue()}
    local diff = {}

    local malus, reason = ship:getMalusFactor()

    -- calculate difference
    for i = 1, NumMaterials() do
        local value = 0
        local fee = 1

        if not reason or reason ~= MalusReason.Boarding then
            value = templateValue[i] - craftValue[i]
            value = value + templateValue[i] * (1.0 - durabilityPercentage)
            value = value / 2

            fee = RepairDock.getRepairFactor()
        else
            value = templateValue[i] - craftValue[i]

            fee = 1
        end

        table.insert(diff, i, value * fee)
    end

    return diff
end

function RepairDock.getRepairMoneyCostAndTax(orderingFaction, ship, perfectPlan, damagedPlan, durabilityPercentage)

    local malus, reason = ship:getMalusFactor()

    local value = 0

    if not reason or reason ~= MalusReason.Boarding then
        value = perfectPlan:getUndamagedMoneyValue() - damagedPlan:getMoneyValue();
        value = value + perfectPlan:getMoneyValue() * (1.0 - durabilityPercentage)
        value = value / 2
        fee = RepairDock.getRepairFactor() + GetFee(Faction(), orderingFaction)
    else
        value = perfectPlan:getUndamagedMoneyValue() - damagedPlan:getMoneyValue()
        fee = 1 + GetFee(Faction(), orderingFaction)
    end

    local price = value * fee
    local tax = round(price * RepairDock.tax)

    if Faction().index == orderingFaction.index then
        price = price - tax
        -- don't pay out for the second time
        tax = 0
    end

    return price, tax
end

function RepairDock.getUpdateInterval()
    return 30
end

function RepairDock.updateServer(timeStep)
    if RepairDock.isShipyardRepairDock() then return end

    RepairDock.newsBroadcastInterval = 250
    RepairDock.newsBroadcastCounter = (RepairDock.newsBroadcastCounter or RepairDock.newsBroadcastInterval - 20) + timeStep

    if RepairDock.newsBroadcastCounter >= RepairDock.newsBroadcastInterval then
        if GameSettings().reconstructionAllowed then
            Sector():broadcastChatMessage(Entity(), ChatMessageType.Chatter, "Buy a Reconstruction Token and get your ship repaired FOR FREE!"%_t)
        end
        RepairDock.newsBroadcastCounter = 0
    end
end

function RepairDock.isReconstructionSite(player)
    player = player or Player()
    local x, y = Sector():getCoordinates()
    local rx, ry = player:getRespawnSectorCoordinates()

    return x == rx and y == ry
end

function RepairDock.getReconstructionPrice(faction, name, factor)
    factor = factor or 1.2
    local type = faction:getShipType(name)

    if not type or type ~= EntityType.Ship then
        return
    end

    local price = faction:getShipReconstructionValue(name)
    price = price * factor

    if faction.isPlayer then
        if RepairDock.isReconstructionSite(faction) then
            price = price * 2 / 3
        end
    elseif faction.isAIFaction then
        return
    end

    price = math.max(1000, round(price / 1000) * 1000)

    return price
end

function RepairDock.getReconstructionTokenPrice(faction, craft)

    local name
    local factor = 0.3

    if is_type(craft, "Entity") then
        name = craft.name

        local malus, reason = craft:getMalusFactor()
        if reason and reason == MalusReason.Boarding then
            factor = 1 - malus
        end
    else
        name = tostring(craft)
    end

    return RepairDock.getReconstructionPrice(faction, name, factor)
end

function RepairDock.getReconstructionSiteChangePrice()
    local x, y = Sector():getCoordinates()

    local d = length(vec2(x, y))

    local factor = 1.0 + (1.0 - math.min(1, d / 450)) * 125

    local price = factor * 100000

    -- round to 1000's
    price = round(price / 1000) * 1000

    return price
end

function RepairDock.getRepairFactor()
    return 0.75 -- Completely repairing a ship would cost 0.75x the ship's value
end

function RepairDock.isShipyardRepairDock()
    return Entity():hasScript("shipyard.lua")
end

function RepairDock.reconstructionPossible(faction, shipName)

    if faction and shipName then
        if RepairDock.isShipyardRepairDock() then
            return false, 0, 0
        end

        if faction:getShipType(shipName) ~= EntityType.Ship then
            return false, 0, 0
        end

        local tokens = countReconstructionTokens(faction, shipName)
        local price = RepairDock.getReconstructionPrice(faction, shipName)

        if tokens > 0 or faction.money >= price then
            return true, tokens, price
        end

        return false, tokens, price
    else
        if RepairDock.isShipyardRepairDock() then
            return false
        end

        return true
    end
end


-- functions required for internal unit tests, can be ignored
function RepairDock.getRepairMoneyCostAndTaxTest()
    local ship = Player(callingPlayer).craft
    local buyer = Faction(ship.factionIndex)

    if buyer.isPlayer then
        buyer = Player(buyer.index)
    elseif buyer.isAlliance then
        buyer = Alliance(buyer.index)
    end

    local perfectPlan = buyer:getShipPlan(ship.name)
    local damagedPlan = ship:getFullPlanCopy()

    return RepairDock.getRepairMoneyCostAndTax(buyer, ship, perfectPlan, damagedPlan, ship.durability / ship.maxDurability)
end

function RepairDock.getRepairResourcesCostTest()
    local ship = Player(callingPlayer).craft
    local buyer = Faction(ship.factionIndex)

    if buyer.isPlayer then
        buyer = Player(buyer.index)
    elseif buyer.isAlliance then
        buyer = Alliance(buyer.index)
    end

    local perfectPlan = buyer:getShipPlan(ship.name)
    local damagedPlan = ship:getFullPlanCopy()

    local resources = RepairDock.getRepairResourcesCost(buyer, ship, perfectPlan, damagedPlan, ship.durability / ship.maxDurability)

    for i = 1, NumMaterials() do
        resources[i] = resources[i] or 0
    end

    return unpack(resources)
end
