local tutorials = {}

require("files")
local json = require("json")

-------------------------------------------------------------------------------------
-- Loads a tutorial from a json file and creates it according to the data.
-- Returns a table containing the tutorial id and data from the json file.
-------------------------------------------------------------------------------------
tutorials.StartTutorial = function(json_file_path)
    local tutorialtable = {}
    tutorialtable.data = json.decode(readFile(json_file_path))
    
    LogInfo("Starting tutorial '" .. tutorialtable.data.Title .. "'!")
    LogInfo("Setting tutorial script: '" .. tutorialtable.data.ScriptName .. "'.")

    tutorialtable.id = challengeservice:CreateTutorial(tutorialtable.data.ScriptName)
    
    table.foreach(tutorialtable.data.Goals, function(goalIndex, goal)
      challengeservice:AddGoal(tutorialtable.id, goal.Target, 0)
    end)
    
    LogInfo("### tutorial created: " .. tostring(tutorialtable.id)..". ".. tostring(#tutorialtable.data.Goals).." goals added.")
    
    return tutorialtable
end

-------------------------------------------------------------------------------------
-- Loads a tutorial from a json file and tries to find it in the existing components.
-- Returns a table containing the tutorial id and data from the json file.
-------------------------------------------------------------------------------------
tutorials.LoadTutorial = function(json_file_path)
    local tutorialtable = {}
    tutorialtable.data = json.decode(readFile(json_file_path))
    tutorialtable.id = challengeservice:FindTutorial(tutorialtable.data.ScriptName)
    if tutorialtable.id == 0 then
      LogError("### tutorial "..tutorialtable.data.ScriptName.." not found!")
    end
    return tutorialtable
end

tutorials.AreAllGoalsReached = function(goalsComponent)
  for i = 1,#goalsComponent.Goals do
    local goal = goalsComponent.Goals[i]
    if goal.Current < goal.Target then
      return false
    elseif goal.Current > goal.Target then
      -- goals should never get above the target, might as well set that here...
      goal.Current = goal.Target
    end
  end
  return true
end

-------------------------------------------------------------------------------------
-- Skipping company creation                                                       --
-------------------------------------------------------------------------------------

tutorials.SetTutorialCompany = function()
  local pcompany = entities:GetCompanyComponent(world.PlayerCompanyID)
    pcompany.Name = "Good Start Inc."
    pcompany.Logo = 2
    pcompany.LogoBGShape = 5
    pcompany.LogoColor = 14
    pcompany.LogoBGColor = 11
    pcompany.TextColor = 0
    pcompany.TextAlign = 0
  entities:UpdateCompanyComponent(world.PlayerCompanyID)
end


-------------------------------------------------------------------------------------
-- Specific Characters                                                             --
-------------------------------------------------------------------------------------

tutorials.CharacterMaleDefault = function(eid)
  entities:GetOrAssignNameComponent(eid).Name = "Mister Selfmade"
  entities:UpdateNameComponent(eid)
  
  local character = entities:GetOrAssignCharacterLookComponent(eid)   
  character.CharacterType = 0  
  character.CharacterLookParts:add(4)
  character.CharacterLookParts:add(9)
  character.CharacterLookParts:add(22)
  character.CharacterLookParts:add(12)
  character.CharacterLookParts:add(15)
  character.CharacterLookParts:add(26)
  character.CharacterLookParts:add(30)
  character.CharacterLookParts:add(37)  
  character.SkinColor = math.random(0, 7)
  character.HairColor = math.random(0, 12)
  character.ColorPalette = 8
  entities:UpdateCharacterLookComponent(eid)
end

tutorials.CharacterFemaleDefault = function(eid)
  entities:GetOrAssignNameComponent(eid).Name = "Miss Selfmade"
  entities:UpdateNameComponent(eid)
  
  local character = entities:GetOrAssignCharacterLookComponent(eid)   
  character.CharacterType = 1  
  character.CharacterLookParts:add(41)
  character.CharacterLookParts:add(41)
  character.CharacterLookParts:add(52)
  character.CharacterLookParts:add(44)
  character.CharacterLookParts:add(46)
  character.CharacterLookParts:add(53)
  character.CharacterLookParts:add(60)
  character.CharacterLookParts:add(69)  
  character.SkinColor = math.random(0, 7)
  character.HairColor = math.random(0, 12)
  character.ColorPalette = 8
  entities:UpdateCharacterLookComponent(eid)
end


-------------------------------------------------------------------------------------
-- Skipping character creation                                                     --
-- Unpauses the game on start and join                                             --
-------------------------------------------------------------------------------------

tutorials.SetPlayerData = function(eid)
  entities:AddPetComponentUpdatedCallback(tutorials.RemovePet)
  if entities:HasPlayerComponent(eid) == false then
    entities:GetOrAssignPlayerComponent(eid).Active = true
    entities:UpdatePlayerComponent(eid)
    entities:GetOrAssignPlayerHandComponent(eid)      
    
    math.randomseed(os.time())
    if math.random(0, 1) == 0 then
      tutorials.CharacterMaleDefault(eid)
    else
      tutorials.CharacterFemaleDefault(eid)
    end    
  end
  
  
  world:SetPause(false)
end

tutorials.RemovePet = function(eid)
  local c = entities:GetPetComponent(eid)  
  if c.Visible then
    c.Visible = false
    entities:UpdatePetComponent(eid)
  end
end

return tutorials