local session = {}
session.data = {}
session.daily_functions = {}
require("files")
local json = require("json")
local gamedatahelper = require("base.utils.gamedatahelper")
local profile = require("base.utils.profile")


-------------------------------------------------------------------------------------
-- initialize the data and set not-company related settings for the session        --
-------------------------------------------------------------------------------------
session.init_game = function(script_path, isload)
  LogInfo("session setup initiatet")
  LogInfo("any session setup warning can be ignored if you did not wanted to set it via json")
  
  if type(isload) == "nil" or type(isload) ~= "boolean" then
    LogError("session setup: isload is invalid/undefined you need to set it to true or false")
    return nil
  end

  if isload then
    LogInfo("--- THIS SESSION WAS LOADED ---")
  else
    LogInfo("--- THIS SESSION WAS NEWLY STARTED ---")
  end

 
  math.randomseed(world.GlobalSeed + 83057231) 
  local setup = json.decode(readFile(script_path.."game_setup.json"))
  
  session.data.default_map = session.set_default_map(setup.default_map, isload)

  if setup.mod_folder ~= nil and (session.check_path(scriptPath()..setup.mod_folder.."/")) then
    session.data.mod_folder = setup.mod_folder
  end

  session.data.use_profile_data = setup.use_profile_data

  session.data.discovered_markets = session.discover_markets(setup.discovered_markets, isload)
  session.data.locked_uis = session.lock_uis(setup.lock_uis, isload)
  session.data.buy_price_factor = session.set_buy_prices(setup.buy_price_factor, isload)

  session.data.money = session.validate_start_money(setup.money)
  session.data.unlocked_buildables = session.validate_buildable_data(setup.unlocked_buildables)
  session.data.unlock_projects = session.validate_project_data(setup.unlocked_researches, "unlocked_researches")
  session.data.whitelist_researches = session.validate_project_data(setup.whitelist_researches, "whitelist_researches")
  session.data.blacklist_researches = session.validate_project_data(setup.blacklist_researches, "blacklist_researches")
  
  session.data.logisticsIDs = {}
  session.data.buyzones = session.set_company_equipment(setup.buyzones, isload, "buyzone")
  session.data.sellzones = session.set_company_equipment(setup.sellzones, isload, "sellzone")
  session.data.equipment = session.set_company_equipment(setup.equipment, isload)
  session.data.logistics_connections = session.set_logistics_connections(setup.logistics_connections, isload)

  session.data.market_path = session.check_market_path(setup.market_path)
  session.data.market_handler = session.check_market_handler_path(setup.market_handler, session.data.market_path)

  if session.check_milestone_paths(setup.milestones_script, setup.milestones_data) then
    session.data.milestone_script = setup.milestones_script
    session.data.milestone_data = setup.milestones_data
  end

  if session.check_challenge_paths(setup.challenge_script, setup.challenge_data) then
    session.data.challenge_script = setup.challenge_script
    session.data.challenge_data = setup.challenge_data
  end

  if setup.handbook_pages ~= nil then
    session.unlockHandbookPages(setup.handbook_pages)
  end

  -- Some things need to be set after the initial player setup, this happens here
  -- This may require a change in the future
  world:ExecuteFunctionInTime(0, function()  
    session.data.discovery_points = session.set_discovery_points(setup.discovery_points, isload)
  end)
  
  return session.data
end


-------------------------------------------------------------------------------------
-- call this to add a company                                                      --
-- before we have multiplayer we can simply call it with the world.PlayerCompanyID --
-------------------------------------------------------------------------------------
session.add_company = function(company, data)
  if company == nil then
    LogError("session add company: no company id given")
    return false
  elseif type(company) ~= "number" and company == 0 then
    LogError("session add company: company id must be an number of 0 or bigger, given was: "..tostring(company))
    return false
  end
  if data == nil then
    data = session.data
    LogWarning("session add company: no data given, using the saved session data")
  end
  
  session.set_start_money(company, data.money)
  session.unlock_buildables(company, data.unlocked_buildables)
  session.unlock_projects(company, data.unlock_projects)
  -- unlocked research projects should also be visible!
  session.whitelist_projects(company, data.whitelist_researches, data.unlock_projects)
  session.blacklist_projects(company, data.blacklist_researches)

  if data.use_profile_data then
    world:ExecuteFunctionInTime(0, function()
    session.apply_profile_buildables(company)
    end)
  end

  return true
end

session.load_company = function(company, data)
  if company == nil then
    LogError("session add company: no company id given")
    return false
  elseif type(company) ~= "number" and company == 0 then
    LogError("session add company: company id must be an number of 0 or bigger, given was: "..tostring(company))
    return false
  end
  if data == nil then
    data = session.data
    LogWarning("session add company: no data given, using the saved session data")
  end

  if data.use_profile_data then
    session.apply_profile_buildables(company)
  end
end


-------------------------------------------------------------------------------------
-- use profile data and set default decorations to be unlocked
-------------------------------------------------------------------------------------
session.apply_profile_buildables = function(company)
  profile.use_profile_decorations(company)
  profile.unlock_decoration(company, "bld_plant_dracaena", true)
  profile.unlock_decoration(company, "bld_painting_cubic", true)
end


-------------------------------------------------------------------------------------
-- setting the default map via the game setup json                                 --
-- will validate by variable type and for nil                                      --
-- can not detect if the map has been already set                                  --
-- if a table has been given it will select a default map by chance                --
-------------------------------------------------------------------------------------
session.set_default_map = function(map, isload)
  if map == nil then
    LogWarning("session setup: no [default_map] data")
    return nil
  elseif type(map) ~= "string" and type(map) ~= "table" then
    LogError("session setup: [default_map] variable has an invalid type, must be a string or table of strings")
    return nil    
  elseif type(map) == "string" then
    if map == "" then
      LogWarning("session setup: no [default_map] data")
      return nil
    end
    if isload == false then
      world:BuildDefaultWorld(map)
    end
    LogInfo("session setup: [default_map] loaded - "..tostring(map))
    return map
  elseif type(map) == "table" and type(map[1]) == "string" then
    local mapname = map[math.random(#map)]
    if isload == false then
      world:BuildDefaultWorld(mapname)
    end
    LogInfo("session setup: [default_map] loaded - "..tostring(mapname))
    return mapname
  end
end


-------------------------------------------------------------------------------------
-- checks if a folder to the market data exists                                    --
-- returns the string if path is valid                                             --
-------------------------------------------------------------------------------------
session.check_market_path = function(market)
  if market == nil then
    LogWarning("session setup: no [market] data")
    return nil
  elseif type(market) == "string" then
    if market == "" then
      LogWarning("session setup: no [market] data")
      return nil
    end
    local path = scriptPath()..market
    if string.sub(path, string.len(path), string.len(path)) ~= "/" then
      path = path.."/"
    end
    if session.check_path(path) then
      LogInfo("session setup: [market] path found - "..tostring(market))
      return path
    else
      LogError("session setup: [market] path could not be found")
      return nil
    end      
  else
    LogError("session setup: [market] variable has an invalid type, must be a string")
    return nil
  end
end


-------------------------------------------------------------------------------------
-- checks if the market handler file exists                                        --
-- returns the string if path is valid                                             --
-------------------------------------------------------------------------------------
session.check_market_handler_path = function(market_handler, folder_path)
  if market_handler == nil then
    LogWarning("session setup: no [market_handler] data")
    return nil
  elseif type(market_handler) == "string" then
    if market_handler == "" then
      LogWarning("session setup: no [market_handler] data")
      return nil
    end
    if string.sub(market_handler, string.len(market_handler)-4, string.len(market_handler)) ~= ".json" then
      market_handler = market_handler..".json"
    end
    if session.check_path(tostring(folder_path)..market_handler) then
      LogInfo("session setup: [market_handler] file found - "..tostring(market_handler))
      return market_handler
    else
      LogError("session setup: [market_handler] file could not be found")
      return nil
    end      
  else
    LogError("session setup: [market_handler] variable has an invalid type, must be a string")
    return nil
  end
end

-------------------------------------------------------------------------------------
-- checking if the milestone files are available and returns true if so            --
-------------------------------------------------------------------------------------
session.check_milestone_paths = function(lua_file, json_file)
  if lua_file == nil or json_file == nil then
    LogWarning("session setup: missing [milestones] data")
    return false
  elseif type(lua_file) == "string" and type(json_file) == "string" then
    if lua_file == "" or json_file == "" then
      LogWarning("session setup: no [milestones] data")
      return false
    end
    local lua_path = scriptPath()..lua_file.gsub(lua_file, "[.]", "/")..".lua"
    if not session.check_path(lua_path) then
      LogError("session setup: [milestones] lua file could not be found")
      return false
    elseif not session.check_path(scriptPath()..json_file) then
      LogError("session setup: [milestones] json file could not be found")
      return false
    else
      LogInfo("session setup: [milestones] path verified")
      return true
    end
  else
    LogError("session setup: [milestones] variable has an invalid type, must be a string")
    return false
  end
end


-------------------------------------------------------------------------------------
-- checking if the challenge files are available and returns true if so            --
-------------------------------------------------------------------------------------
session.check_challenge_paths = function(lua_file, json_file)
  if lua_file == nil or json_file == nil then
    LogWarning("session setup: missing [challenge] data")
    return false
  elseif type(lua_file) == "string" and type(json_file) == "string" then
    if lua_file == "" or json_file == "" then
      LogWarning("session setup: no [challenge] data")
      return false
    end
    local lua_path = scriptPath()..lua_file.gsub(lua_file, "[.]", "/")..".lua"
    if not session.check_path(lua_path) then
      LogError("session setup: [challenge] lua file could not be found")
      return false
    elseif not session.check_path(scriptPath()..json_file) then
      LogError("session setup: [challenge] json file could not be found")
      return false
    else
      LogInfo("session setup: [challenge] path verified")
      return true
    end
  else
    LogError("session setup: [challenge] variable has an invalid type, must be a string")
    return false
  end
end

-------------------------------------------------------------------------------------
-- tries to add milestones if a path has been given                                --
-- path needs to be to the lua file                                                --
-- the json must be in the same folder with the same name                          --
-------------------------------------------------------------------------------------
session.set_challenge = function(cpath, isload)
  if cpath == nil then
    LogWarning("session setup: no [challenge] data")
    return nil
  elseif type(cpath) == "string" then
    if cpath == "" then
      LogWarning("session setup: no [challenge] data")
      return nil
    end
    local status, challenge = pcall(require, cpath)
    if status then
      local json_path = scriptPath()..cpath.gsub(cpath, "[.]", "/")..".json"
      challenge.on_init(json_path, isload)
      LogInfo("session setup: [challenge] loaded - "..tostring(cpath))
      return challenge
    else
      LogError("session setup: [challenge] path not found - "..tostring(cpath))
    end
  else
    LogError("session setup: [challenge] variable has an invalid type, must be a string")
    return nil
  end
end


-------------------------------------------------------------------------------------
-- checks for the type and validates that markets exist                            --
-- afterwards activates given marketes                                             --
-- returns table with correct market ids, will skip invalid ones                   --
-------------------------------------------------------------------------------------
session.discover_markets = function(markets, isload)
  if markets == nil then
    LogWarning("session setup: no [discovered_markets] data")
    return nil
  elseif type(markets) == "number" or type(markets) == "string" then
    markets = gamedatahelper.get_product_type_id_by_name(markets)
    if markets == 0 then
      LogWarning("session setup: no [discovered_markets] data")
      return nil
    end
    if gamedata:GetProductType(markets) == nil then
      LogError("session setup: [discovered_markets] is an invalid id - "..tostring(markets))
      return nil
    else
      if isload == false then
        marketdiscoveryservice:UnlockMarket(markets)
      end
      LogInfo("session setup: [discovered_markets] set - "..tostring(markets))
      return {markets}
    end
  elseif type(markets) == "table" then
    local list = {}
    for i=1, #markets do
      markets[i] = gamedatahelper.get_product_type_id_by_name(markets[i])
      if type(markets[i]) ~= "number" or markets[i] == 0 or gamedata:GetProductType(markets[i]) == nil then
        LogError("session setup: invalid id in [discovered_markets] - "..tostring(markets[i]))
      else
        if isload == false then
          marketdiscoveryservice:UnlockMarket(markets[i])
        end
        table.insert(list, markets[i])
        LogInfo("session setup: [discovered_market] set - "..tostring(markets[i]))
      end
    end 
    return list
  else
    LogError("session setup: [discovered_markets] variable has an invalid type, must be a number or array of numbers")
    return nil
  end    
end


-------------------------------------------------------------------------------------
-- settings the start amount of discovery points                                   -- 
-------------------------------------------------------------------------------------
session.set_discovery_points = function(points, isload)
  if type(points) == "nil" then
    LogWarning("session setup: no [discovery_points] data")
    return nil
  elseif type(points) ~= "number" then
    LogError("session setup: [discovery_points] variable has an invalid type, must be a number")
    return nil
  else
    if isload == false then
      marketdiscoveryservice:IncreaseDiscoveryPoints(points - entities:GetMarketDiscoveryComponent(1).DiscoveryPoints)
      LogInfo("session setup: [discovery_points] set - "..tostring(points))
    end
    return points
  end
end


-------------------------------------------------------------------------------------
-- on a new game, it will set the buildable price factor
-------------------------------------------------------------------------------------
session.set_buy_prices = function(factor, isload)
  if type(factor) == "nil" then
    LogWarning("session setup: no [buy_price_factor] data")
    return nil
  elseif type(factor) ~= "number" then
    LogError("session setup: [buy_price_factor] variable has an invalid type, must be a number")
    return nil
  else
    if isload == false then
      local tuning_values = entities:GetTuningFactorsComponent(1)
      tuning_values.BuildablePriceFactor = factor
      entities:UpdateTuningFactorsComponent(1)
      LogInfo("session setup: [buy_price_factor] set - "..tostring(factor))
    end
    return factor
  end
end

-------------------------------------------------------------------------------------
-- validating the money information                                                --
-------------------------------------------------------------------------------------
session.validate_start_money = function(value)
  if type(value) == "nil" then
    LogWarning("session setup: no [start money] data")
    return nil
  elseif type(value) ~= "number" then
    LogError("session setup: invalid type for [start money] - should be number")
    return nil
  elseif type(value) == "number" then
    LogInfo("session setup: [start money] validated - "..tostring(value))
    return value
  end
  return nil
end


-------------------------------------------------------------------------------------
-- trying to set start money                                                       --
-- this is a special case where we do not use a transferservice                    --
-------------------------------------------------------------------------------------
session.set_start_money = function(company, value)
  if type(value) ~= "nil" then
    local component = entities:GetMoneyComponent(company)
    component.Balance = value
    entities:UpdateMoneyComponent(company)
    return value
  end
  return nil
end


-------------------------------------------------------------------------------------
-- will lock all specified uis at the beginning of a new session                   --
-------------------------------------------------------------------------------------
session.lock_uis = function(uis, isload)
  if type(uis) == "nil" then
    LogWarning("session setup: no [ui lock] data")
    return nil
  elseif type(uis) == "table" then
    if isload == false then
      for i=1, #uis do
        if type(uis[i]) == "number" then
          uigatingservice:MakeUIElementAvailable(uis[i], false)
          LogInfo("session setup: [ui] locked - "..tostring(uis[i]))
        elseif type(uis[i]) == "string" then
          uigatingservice:MakeUIElementAvailable(UIElement[uis[i]], false)
          LogInfo("session setup: [ui] locked - "..tostring(uis[i]))
        else
          LogInfo("session setup: [ui] element had invalid type - "..tostring(uis[i]))
        end      
      end
    end
    return uis
  elseif type(uis) == "number" then
    if isload == false then
      uigatingservice:MakeUIElementAvailable(uis, false)
      LogInfo("session setup: [ui] locked - "..tostring(uis))
    end
    return uis
  elseif type(uis) == "string" then
    if isload == false then
      uigatingservice:MakeUIElementAvailable(UIElement[uis], false)
      LogInfo("session setup: [ui] locked - "..tostring(uis))
    end
    return uis
  else
    LogError("session setup: [ui lock] data has invalid type")
    return nil
  end
end


-------------------------------------------------------------------------------------
-- imports and validates the buildable data to unlock                              --
-------------------------------------------------------------------------------------
session.validate_buildable_data = function(list)
  if list == nil then
    LogWarning("session setup: no [unlocked buildable] data")
    return nil
  elseif type(list) ~= "number" and type(list) ~= "table" and type(list) ~= "string" then
    LogError("session setup: invalid type for [unlocked buildable] - should be number, string or table")
    return nil
  elseif type(list) == "number" then
    list = gamedatahelper.get_buildable_id_by_name(list)
    if list == 0 then
      LogWarning("session setup: no [unlocked buildable] data")
      return nil
    else
      LogInfo("session setup: [unlocked buildable] validated - "..tostring(list))
      return {list}
    end
  elseif type(list) == "table" then
    if #list == 0 then
      LogWarning("session setup: no [unlocked buildable] data")
      return nil
    else
      local unlock_list = {}
      for i=1, #list do
        list[i] = gamedatahelper.get_buildable_id_by_name(list[i])
        if type(list[i]) == "number" and list[i] ~= 0 then
          table.insert(unlock_list, list[i])
          LogInfo("session setup: [unlocked buildable] validated - "..tostring(list[i]))
        else
          LogError("session setup: invalid input for [unlocked buildable] element, please validate - "..tostring(i)..": "..tostring(list[i]))
        end
      end
      return unlock_list
    end
  end
  return nil
end


-------------------------------------------------------------------------------------
-- used to add a new company to the game                                           --
-------------------------------------------------------------------------------------
session.unlock_buildables = function(company, list)
  if list ~= nil then
    local component = entities:GetConstructionPossibilitiesComponent(company)
    for i=1, #list do
      component.PossibleBuildables:add(list[i])
    end
    entities:UpdateConstructionPossibilitiesComponent(company)
    return true
  end
  return false
end


-------------------------------------------------------------------------------------
-- imports and validates the research data to unlock                              --
-------------------------------------------------------------------------------------
session.validate_project_data = function(list, checktype)
  if list == nil then
    LogWarning("session setup: no ["..checktype.."] data")
    return nil
  elseif type(list) ~= "number" and type(list) ~= "string" and type(list) ~= "table" then
    LogError("session setup: invalid type for ["..checktype.."] - should be number, string or array")
    return nil
  elseif type(list) == "number" then
    list = gamedatahelper.get_research_id_by_name(list)
    if list == 0 then
      LogWarning("session setup: no ["..checktype.."] data")
      return nil
    else
      LogInfo("session setup: ["..checktype.."] validated - "..tostring(list))
      return {list}
    end
  elseif type(list) == "table" then
    if #list == 0 then
      LogWarning("session setup: no ["..checktype.."] data")
      return nil
    else
      local validated = {}
      for i=1, #list do
        list[i] = gamedatahelper.get_research_id_by_name(list[i])
        if type(list[i]) ~= "number" or list[i] == 0 then
          LogError("session setup: invalid input for ["..checktype.."] element, please validate- "..tostring(i)..": "..tostring(list[i]))
        elseif gamedata:GetResearchProject(list[i]) == nil then
          LogError("session setup: ["..checktype.."] id not found - "..tostring(list[i]))
        else
          table.insert(validated, list[i])
          LogInfo("session setup: ["..checktype.."] validated - "..tostring(list[i]))
        end
      end
      return validated
    end
  end
  return nil
end


-------------------------------------------------------------------------------------
-- used to add research projects to a new company                                  --
-------------------------------------------------------------------------------------
session.unlock_projects = function(company, list)
  if list ~= nil then
    local notification = false
    for i=1, #list do
      researchservice:UnlockResearchProject(company, list[i], false, notification)
    end
    return true
  end
  return false
end


-------------------------------------------------------------------------------------
-- used to whitelist research projects for a new company                           --
-------------------------------------------------------------------------------------
session.whitelist_projects = function(company, whitelist_table, unlock_table)
  local list = {}

  if whitelist_table ~= nil then
    for _, v in pairs(whitelist_table) do table.insert(list, v) end
  end

  if unlock_table ~= nil then
    for _, v in pairs(unlock_table) do table.insert(list, v) end
  end

  if #list > 0 then
    researchservice:HideAllProjectsExcept(company, list)
    return true
  end
  return false
end


-------------------------------------------------------------------------------------
-- used to blacklist research projects for a new company                           --
-------------------------------------------------------------------------------------
session.blacklist_projects = function(company, list)
  if list ~= nil then
    researchservice:HideProjects(company, list)
    return true
  end
  return false
end


-------------------------------------------------------------------------------------
-- sets equipment for the setup, special case, might need to be revised when       --
-- multiplayer gets added, ass equipment needs to placed for each company!         --
-------------------------------------------------------------------------------------
session.set_company_equipment = function(data, isload, equiptype)  
  if data == nil then
    LogWarning("session setup: no [equipment] data")
    return nil
  elseif type(data) ~= "table" then
    LogError("session setup: [equipment] data is expected to be in a table")
    return nil
  elseif isload == false then

    local fixed_id = nil

    if equiptype ~= nil and equiptype == "sellzone" then
      entities:ForEachSellPlaceComponent(function(e, c)
        entities:destroyEntity(e)
      end)
      fixed_id = 12
    elseif equiptype ~= nil and equiptype == "buyzone" then
      entities:ForEachBuyPlaceComponent(function(e, c)
        entities:destroyEntity(e)
      end)
      fixed_id = 11
    end

    for i=1, #data do
      if type(data[i].X) ~= "number" then
        LogError("session setup: [equipment.X] data is missing/invalid, expected a number")
      elseif type(data[i].Y) ~= "number" then
        LogError("session setup: [equipment.Y] data is missing/invalid, expected a number")
      elseif type(data[i].Rotation) ~= "number" or data[i].Rotation < 0 or data[i].Rotation > 3 then
        LogError("session setup: [equipment.Rotation] data is missing/invalid, expected a number in the range of 0 - 3")
      else
        local position = Coordinate2.new(data[i].X, data[i].Y)
        if fixed_id ~= nil then
          data[i].ID = fixed_id
        end

        data[i].ID = gamedatahelper.get_buildable_id_by_name(data[i].ID)
        if buildableservice:CanBeBuiltAtWorldPos(data[i].ID, position, data[i].Rotation, 0) == 0 then
          local placedEntity = buildableservice:PlaceBuildable(data[i].ID, position, data[i].Rotation, world.PlayerCompanyID)
          if fixed_id ~= nil then
            -- buy and sellzones should not have a ConstructComponent
            -- (which they get by being placed by the buildableservice)
            entities:RemoveConstructComponent(placedEntity)
          end
          local additionalInfo = ""
          if type(data[i].SwitchModule) ~= "nil" then
            if entities:HasCrafterComponent(placedEntity) then
              if type(data[i].SwitchModule) ~= "number" then
                LogError("session setup: [equipment.SwitchModule] has to be a number for crafters")
              else
                local switched = craftingservice:SwitchCurrentModuleWithoutWorker(placedEntity, data[i].SwitchModule)
                additionalInfo = additionalInfo.." switchModule("..tostring(data[i].SwitchModule).."):"..tostring(switched)
              end
            elseif entities:HasSamplingStationComponent(placedEntity) then
              if type(data[i].SwitchModule) ~= "table" then
                LogError("session setup: [equipment.SwitchModule] has to be an array for analysis tables")
              else
                additionalInfo = additionalInfo.." switchModule("
                for _,m in pairs(data[i].SwitchModule) do
                  local switched = samplingservice:AllowModuleSampling(placedEntity, m)
                  additionalInfo = additionalInfo..tostring(m)..":"..tostring(switched)
                end
                additionalInfo = additionalInfo..")"
              end
            else
              LogError("session setup: [equipment.SwitchModule] is not implemented for this equipment type")
            end
          end
          if type(data[i].LogisticsID) == "string" then
            session.data.logisticsIDs[data[i].LogisticsID] = placedEntity
            additionalInfo = additionalInfo.." logisticsID: "..data[i].LogisticsID
          end
          if type(data[i].CreateEmployeeWithWage) == "number" then
            local employeeID = employeeservice:GenerateNewApplicant(math.random(999999), 0, data[i].CreateEmployeeWithWage, 1.0)
            employeeservice:HireApplicant(employeeID, world.PlayerCompanyID, placedEntity)
            positionservice:TeleportEntityToEntity(employeeID, placedEntity)
            aiservice:SetWorkplaceForAI(employeeID, placedEntity)
            additionalInfo = additionalInfo.." hiredEmployeeID: "..tostring(employeeID)
          end
          if type(data[i].Name) == "string" then
            local nameComp = entities:GetOrAssignNameComponent(placedEntity)
            nameComp.Name = data[i].Name
          end
          LogInfo("session setup: [equipment] set - entityID: "..tostring(placedEntity).." objectID:" .. tostring(data[i].ID) .. " - position: " .. tostring(data[i].X) .."/".. tostring(data[i].Y).." "..additionalInfo)
        else
          LogError("session setup: [equipment] position blocked - objectID:" .. tostring(data[i].ID) .. " - position: " .. tostring(data[i].X) .."/".. tostring(data[i].Y))
        end
      end
    end
    return data
  else
    return data
  end
end

-------------------------------------------------------------------------------------
-- adds logistics connections according to the json data (the order matters!)      --
-- note: the logistics ids used in the json data have to be defined in the         --
--       equipment json objects as "LogisticsID"!                                  --
-------------------------------------------------------------------------------------
session.set_logistics_connections = function(data, isload)  
  if data == nil then
    LogWarning("session setup: no [logistics_connections] data")
    return nil
  elseif type(data) ~= "table" then
    LogError("session setup: [logistics_connections] data is expected to be in an array")
    return nil
  elseif isload == false then
    for _,connectiondata in ipairs(data) do
      if type(connectiondata.From) ~= "string" then
        LogError("session setup: [logistics_connection.From] data is missing/invalid, expected a string")
      elseif type(connectiondata.To) ~= "string" then
        LogError("session setup: [logistics_connection.To] data is missing/invalid, expected a string")
      else
        local fromID = session.data.logisticsIDs[connectiondata.From]
        local toID = session.data.logisticsIDs[connectiondata.To]
        local fromFound = type(fromID) == "number"
        local toFound = type(toID) == "number"
        if not fromFound or not toFound then
          LogError("session setup: [logistics_connection] from or to id wasn't defined in the equipment! from found:"..tostring(fromFound).." to found:"..tostring(toFound))
          return
        end

        local result = logisticsservice:CreateBatchedConnection(ConnectionEntity.new(fromID, -2), ConnectionEntity.new(toID, -2), 1)
        if result == BatchedConnectionError.None then
          LogInfo("session setup: [logistics_connection] connection from "..connectiondata.From.." to "..connectiondata.To.." created successfully")
        else
          LogError("session setup: [logistics_connection] create connection from "..connectiondata.From.." to "..connectiondata.To.." returned an error: "..tostring(result))
        end
      end
    end
    return data
  else
    return data
  end
end


-------------------------------------------------------------------------------------
-- Simple automisation for the the on day pass callbacks                           --
-- Can add one by calling AddOnDayPassed(function)                                 --
-- When calling the callback function it will also pass the daily routine data     --
-------------------------------------------------------------------------------------
session.OnDayPassed = function()
  for i=1, #session.daily_functions do
    if session.daily_functions[i].challenge ~= nil and entities:GetProgressComponent(session.daily_functions[i].challenge).CurrentState ~= 0 then
      session.daily_functions[i] = nil
    else
      world:ExecuteFunctionInTime(session.daily_functions[i].wait, function()
          session.daily_functions[i].func(session.daily_functions[i])
      end)
    end    
  end
  if #session.daily_functions > 0 then
    world:ExecuteFunctionOnNextDay(session.OnDayPassed)
  end
end


-------------------------------------------------------------------------------------
-- WAITTIME: adds a delay after a day has passed, default is 1 second              --
-- ISCHALLENGE: set this true if you want the daily routine to be deleted          --
-- when challenge is over. Does ignore this when no challenge is active            --
-- ARGUMENTS: you may add whatever when the day passes you get all that            --
-- data given back -> This data is not getting saved between sessions              --
-------------------------------------------------------------------------------------
session.AddOnDayPassed = function(func, waitTime, challengeID, arguments)
  local dailyCount = #session.daily_functions
  if dailyCount == 0 then
    world:ExecuteFunctionOnNextDay(session.OnDayPassed)
  end
  dailyCount = dailyCount + 1
  session.daily_functions[dailyCount] = {}
  session.daily_functions[dailyCount].func = func
  
  if waitTime == nil or waitTime < 0 then
    waitTime = 1  
  end
  session.daily_functions[dailyCount].wait = waitTime  
  session.daily_functions[dailyCount].challenge = challengeID
  
  session.daily_functions[dailyCount].arguments = arguments
  LogInfo("Daily task has been added: "..tostring(func))
end


-------------------------------------------------------------------------------------
-- Helper script to check weather a file/path exists                               --
-- Adding / at the end of the path checks for a folder instead of a file           --
-------------------------------------------------------------------------------------
session.check_path = function (file)
  local ok, err, code = os.rename(file, file)
  if not ok then
     if code == 13 then
        -- Permission denied, but it exists
        return true
     end
  end
  return ok, err
end

-------------------------------------------------------------------------------------
-- Helper script to unlock a page in the handbook                                  --
-- INDICES: an array of byte-sized indices of the pages to be unlocked             --
-------------------------------------------------------------------------------------
session.unlockHandbookPages = function(indices)
  LogInfo("session setup: unlocking handbook pages #"..tostring(#indices))
  local handbook = entities:GetHandbookComponent(1)
  handbook.UnlockedPages = indices
  entities:UpdateHandbookComponent(1)
end

return session