local research = {} -- holds temporary data, does not get saved

--- sets the progress of a possible research project between 0 to 1
---@param company number
---@param research_id number
---@param progress_amount number
---@return boolean
research.set_research_progress = function(company, research_id, progress_amount)
  if company == nil then
    LogError("utils > research > set_research_progress: invalid company id : "..tostring(company))
    return false
  end

  if type(research_id) ~= "number" then
    LogError("utils > research > set_research_progress: research id not given as number : "..tostring(research_id))
    return false
  end

  if type(progress_amount) ~= "number" or progress_amount < 0 or progress_amount > 1 then
    LogError("utils > research > set_research_progress: progress_amount not given as number between 0 and 1 : "..tostring(progress_amount))
    return false
  end

  local research_data = gamedata:GetResearchProject(research_id)
  if research_data == nil then
    LogError("utils > research > set_research_progress: research data not found : "..tostring(research_id))
    return false
  end

  local research_projects = entities:GetResearchProjectsComponent(company)
  if research_data == nil then
    LogError("utils > research > set_research_progress: company data not found : "..tostring(company))
    return false
  end

  local project = nil

  for i=1, #research_projects.Projects do
    if research_projects.Projects[i].ProjectID == research_id then
      project = research_projects.Projects[i]
    end
  end

  if project == nil then
    LogWarning("utils > research > set_research_progress: project currently unavailable for research: "..tostring(research_id))
    return false
  end

  for remaining_index=1, #project.RemainingData do
    local remaining_data = project.RemainingData[remaining_index]
    for required_index=1, #research_data.RequiredData do
      local required_data = research_data.RequiredData[required_index]
      if required_data.DataTypeID == remaining_data.DataTypeID then
        remaining_data.Amount = math.floor(required_data.Amount * (1-progress_amount))
        LogInfo("utils > research > set_research_progress: set progress of research: "..tostring(research_id))
      end
    end
  end
  entities:UpdateResearchProjectsComponent(company)
  return true
end

-------------------------------------------------------------------------------------
-- Blocks that the player can build a specific component / module                  --
-------------------------------------------------------------------------------------
research.RemovePossibleModule = function(modules)
  local component = entities:GetProductionPossibilitiesComponent(world.PlayerCompanyID)
  
  if type(modules) == "table" then
    for i=1, #modules do
      component:RemoveModuleType(modules[i])
    end
  else
    component:RemoveModuleType(modules)
  end
  entities:UpdateProductionPossibilitiesComponent(world.PlayerCompanyID)
end

-------------------------------------------------------------------------------------
-- Function to track changes on research data, used in tutorial 05                 --
-- Plesae be aware, it's not cheat proof, research tables allows                   --
-- to start and stop spending points triggering increase / decrease                --
-------------------------------------------------------------------------------------

research.data = {}

research.SetDataUpdateCheck = function(dataId, callback)
  research.ChangeResearchData(world.PlayerCompanyID, dataId, 0)
  if research.dataStorageComponent == nil then
    entities:ForEachSamplingDataStorageComponent(function (eid) 
      research.dataStorageComponent = entities:GetSamplingDataStorageComponent(eid)
    end)
    entities:AddSamplingDataStorageComponentUpdatedCallback(research.CheckDataUpdate)
  end
    
  local id = #research.data + 1
  research.data[id] = {}
    
  for i=1, #research.dataStorageComponent.SamplingData do
    if research.dataStorageComponent.SamplingData:at(i).DataTypeID == dataId then
      research.data[id].index = i
      break
    end
  end  
  
  research.data[id].dataId = dataId
  research.data[id].currentValue = research.dataStorageComponent.SamplingData:at(research.data[id].index).Amount
  research.data[id].callback = callback
end


-------------------------------------------------------------------------------------
-- Doesn't need to be called manually, just use SetDataUpdateCheck                 --
-------------------------------------------------------------------------------------

research.CheckDataUpdate = function(eid)
  if research.dataStorageComponent == nil then
    research.dataStorageComponent = entities:GetSamplingDataStorageComponent(eid)
  end
  for _, data in pairs(research.data) do
    local dif = research.dataStorageComponent.SamplingData:at(data.index).Amount - data.currentValue
    if dif ~= 0 then
      data.callback(data.dataId, dif)
      data.currentValue = research.dataStorageComponent.SamplingData:at(data.index).Amount
    end    
  end
end


-------------------------------------------------------------------------------------
-- Add Research Data                --
-------------------------------------------------------------------------------------
research.AddToResearchData = function(companyID, dataID, amount)
  local c = entities:GetSamplingDataStorageComponent(companyID)
  if c == nil then return end
  
  local dataAmount = SamplingDataAmount.new()
  dataAmount.Amount = amount
  dataAmount.DataTypeID = dataID
  c:AddAmount(dataAmount)
  
  entities:UpdateSamplingDataStorageComponent(companyID)
end

research.UnlockTypes = function(typeID)
  entities:ForEachProductionPossibilitiesComponent(function(e, c)
    if c:MakeProductTypePossible(typeID) then
      entities:UpdateProductionPossibilitiesComponent(e)
    end
  end)
end


research.UnlockModule = function(moduleID)
  entities:ForEachProductionPossibilitiesComponent(function(e, c)
    if c:MakeModuleTypePossible(moduleID) then
      entities:UpdateProductionPossibilitiesComponent(e)
    end
  end)
end


research.UnlockMaterial = function(materialID)
  entities:ForEachProductionPossibilitiesComponent(function(e, c)
    if c:MakeMaterialTypePossible(materialID) then
      entities:UpdateProductionPossibilitiesComponent(e)
    end
  end)
end

research.UnlockProject = function(
    companyId,
    projectID,
    trigger_listeners,
    display_notification)
  researchservice:UnlockResearchProject(
    companyId,
    projectID,
    trigger_listeners,
    display_notification)
end

return research