local script = {}

local utils = {}
utils.gamedatahelper = require("base.utils.gamedatahelper")

--- returns the product id of the selected blueprint
---@param blueprint number|BlueprintDefinitionComponent
---@return number
script.get_product_id = function(blueprint)
  if type(blueprint) == "number" then
    blueprint = entities:GetBlueprintDefinitionComponent(blueprint)
  end
  if blueprint ~= nil then
    return blueprint.ProductType
  else
    return 0
  end
end


--- returns the market appeal of a blueprint
---@param blueprint number|BlueprintDefinitionComponent
---@return number
script.get_product_appeal = function(blueprint)
  if type(blueprint) == "number" then
    blueprint = entities:GetBlueprintDefinitionComponent(blueprint)
  end
  if blueprint ~= nil then
    return blueprint.MarketAppeal
  else
    return 0
  end
end


--- checks if blueprint contains at least one module of module_list
---@param blueprint number|BlueprintDefinitionComponent
---@param module_list number|table
---@return boolean
script.contains_module = function(blueprint, module_list)
  if type(blueprint) == "number" then
    blueprint = entities:GetBlueprintDefinitionComponent(blueprint)
  end

  if blueprint == nil then
    LogWarning("utils.products.contains_module -> blueprint not found")
    return false
  end

  if type(module_list) == "number" then
    module_list = {module_list}
  end

  for target_module_index=1, #module_list do
    if blueprint.CaseModule == module_list[target_module_index] then
      return true
    end
    for grid_module_index=1, #blueprint.GridModules do
      if blueprint.GridModules[grid_module_index].ModuleID == module_list[target_module_index] then
        return true
      end
    end
  end
  return false
end


--- counts module(s) of module_list occurence in the blueprint and returns it
---@param blueprint number|BlueprintDefinitionComponent
---@param module_list number|table
---@return number
script.count_modules = function(blueprint, module_list)
  if type(blueprint) == "number" then
    blueprint = entities:GetBlueprintDefinitionComponent(blueprint)
  end

  if blueprint == nil then
    LogWarning("utils.products.count_modules -> blueprint not found")
    return 0
  end

  if type(module_list) == "number" then
    module_list = {module_list}
  end

  local n = 0
  for target_module_index=1, #module_list do
    if blueprint.CaseModule == module_list[target_module_index] then
      n = n + 1
      return
    end
    for grid_module_index=1, #blueprint.GridModules do
      if blueprint.GridModules[grid_module_index].ModuleID == module_list[target_module_index] then
        n = n + blueprint.GridModules[grid_module_index].Amount
        return
      end
    end
  end
  return n
end


--- returns the raw feature_sum of feature_list occurence in the blueprint and returns it (ignores if substration happened)
---@param blueprint number|BlueprintDefinitionComponent
---@param feature_list number|table
---@return number
script.get_raw_feature_value = function(blueprint, feature_list)
  if type(blueprint) == "number" then
    blueprint = entities:GetBlueprintDefinitionComponent(blueprint)
  end

  if blueprint == nil then
    LogWarning("utils.products.sum_feature_value -> blueprint not found")
    return 0
  end

  if type(feature_list) == "number" then
    feature_list = {feature_list}
  end

  local n = 0
  local module_amount = 0
  local module_data
  for grid_module_index=0, #blueprint.GridModules do
    if grid_module_index == 0 then
      module_data = gamedata:GetModule(blueprint.CaseModule)
      module_amount = 1
    else
      module_data = gamedata:GetModule(blueprint.GridModules[grid_module_index].ModuleID)
      module_amount = blueprint.GridModules[grid_module_index].Amount
    end
      
    for module_feature_index=1, #module_data.Features do
      for target_feature_index=1, #feature_list do
        if module_data.Features[module_feature_index].FeatureID == feature_list[target_feature_index] then
          n = n + (module_data.Features[module_feature_index].Value * module_amount)
        end
      end
    end
  end
  return n
end


--- counts feature_sum of feature_list occurence in the blueprint and returns it
---@param blueprintID number
---@param feature_list number|string|table
---@return number
script.get_feature_value = function(blueprintID, feature_list)
  local feature_component = entities:GetFeaturesComponent(blueprintID)
  if feature_component == nil then
    LogError("get_feature_value -> blueprintId did not had a featuresComponent")
    return 0
  end

  if type(feature_list) == "number" or type(feature_list) == "string" then
    feature_list = {feature_list}
  end

  local n = 0
  for target_feature_index=1, #feature_list do
    local feature = utils.gamedatahelper.get_feature_id_by_name(feature_list[target_feature_index])
    n = n + feature_component:GetFeatureValue(feature).Value
  end
  return n
end

--- returns the current market rating of a product
---@param market_price_component number|ProductMarketPriceComponent
---@return number
script.get_market_rating = function(market_price_component)
  if type(market_price_component) == "number" then
    market_price_component = entities:GetProductMarketPriceComponent(market_price_component)
  end
  if market_price_component ~= nil then
    return market_price_component.MarketRating
  else
    LogWarning("utils.products.get_market_rating -> market_price_component not found")
    return 0
  end
end

return script


