local script = {}

local utils = {}
utils.gamedata = require("base.utils.gamedata")
utils.gdh = require("base.utils.gamedatahelper")


--- adds a custom message by a default template
---@param string_identifier string
---@param icon_asset_id string
---@param notification_id string
---@return number
script.custom_notification = function(string_identifier, icon_asset_id, notification_id)
  if type(string_identifier) ~= "string" then
    string_identifier = "not_missing"
    LogWarning("utils.notifications > custom_notification - invalid string_identifier: "..tostring(string_identifier))
  end

  if type(icon_asset_id) ~= "string" then
    icon_asset_id = ""
    LogWarning("utils.notifications > custom_notification - invalid icon_asset_id: "..tostring(icon_asset_id))
  end

  if notification_id == nil then
    notification_id = "gcm_custom"..tostring(string_identifier)
  end

  notificationservice:DismissNotification(tostring(notification_id), true)
  local notification_entity = notificationservice:CreateOrGetNotification(tostring(notification_id))
  local notification = entities:GetNotificationComponent(notification_entity)

  notification.Flags = NotificationFlags.CanBeDismissed
  notification.Type = NotificationType.Custom
  notification.Icon = tostring(icon_asset_id)

  notification.StringIdentifier = string_identifier
  notification.ReplacementParameters:clear()
  entities:UpdateNotificationComponent(notification_entity)

  return notification_entity
end


--- adds an unlock notification for decorations
---@param buildable_id string
---@return number
script.unlocked_decoration = function(buildable_id)
  local id = utils.gdh.get_buildable_id_by_name(buildable_id)
  if id == 0 then
    LogError("utils.notifications > unlocked_decoration - could not find buildable: "..tostring(buildable_id))
    return 0
  end

  notificationservice:DismissNotification("gcm_unlocked_decoration_"..tostring(id), true)
  local notification_entity = notificationservice:CreateOrGetNotification("gcm_unlocked_decoration_"..tostring(id))
  local notification = entities:GetNotificationComponent(notification_entity)

  notification.Flags = NotificationFlags.CanBeDismissed
  notification.Type = NotificationType.Custom
  notification.Icon = "icons_buildables/"..buildable_id

  notification.StringIdentifier = "not_unlocked_decoration"
  notification.ReplacementParameters:clear()
  notification.ReplacementParameters:add("decoration_name:T!"..tostring(buildable_id))
  entities:UpdateNotificationComponent(notification_entity)

  return notification_entity
end


--- adds an unlock notification for buildables
---@param buildable_id string
---@return number
script.unlocked_buildable = function(buildable_id)
  local id = utils.gdh.get_buildable_id_by_name(buildable_id)
  if id == 0 then
    LogError("utils.notifications > unlocked_buildable - could not find buildable: "..tostring(buildable_id))
    return 0
  end

  notificationservice:DismissNotification("gcm_unlocked_buildable_"..tostring(id), true)
  local notification_entity = notificationservice:CreateOrGetNotification("gcm_unlocked_buildable_"..tostring(id))
  local notification = entities:GetNotificationComponent(notification_entity)

  notification.Flags = NotificationFlags.CanBeDismissed
  notification.Type = NotificationType.Custom
  notification.Icon = "icons_buildables/"..buildable_id
  LogError(notification.Icon)

  notification.StringIdentifier = "not_unlocked_buildable"
  notification.ReplacementParameters:clear()
  notification.ReplacementParameters:add("buildable_name:T!"..tostring(buildable_id))
  entities:UpdateNotificationComponent(notification_entity)

  return notification_entity
end


--- adds an unlock notification for markets
---@param product_id string|number
---@return number
script.unlocked_market = function(product_id)
  local id = utils.gdh.get_product_type_id_by_name(product_id)
  if id == 0 then
    LogError("utils.notifications > unlocked_market - could not find buildable: "..tostring(product_id))
    return 0
  end

  notificationservice:DismissNotification("gcm_market_"..tostring(id), true)
  local notification_entity = notificationservice:CreateOrGetNotification("gcm_market_"..tostring(id))
  local notification = entities:GetNotificationComponent(notification_entity)
  local product_data = gamedata:GetProductType(utils.gdh.get_product_type_id_by_name(id))

  notification.Flags = NotificationFlags.CanBeDismissed + NotificationFlags.PopUp
  notification.Type = NotificationType.Custom
  notification.Icon = tostring(product_data.IconAssetID)

  notification.StringIdentifier = "not_unlocked_market"
  notification.ReplacementParameters:clear()
  notification.ReplacementParameters:add("product_name:T!"..tostring(product_data.Name))
  entities:UpdateNotificationComponent(notification_entity)

  return notification_entity
end


--- adds a market progressed notification for markets
---@param product_id string|number
---@param phase_index number
---@return number
script.new_market_phase = function(product_id, phase_index)
  local id = utils.gdh.get_product_type_id_by_name(product_id)
  if id == 0 then
    LogError("utils.notifications > unlocked_market - could not find buildable: "..tostring(product_id))
    return 0
  end

  notificationservice:DismissNotification("gcm_market_"..tostring(id), true)
  local notification_entity = notificationservice:CreateOrGetNotification("gcm_market_"..tostring(id))
  local notification = entities:GetNotificationComponent(notification_entity)
  local product_data = gamedata:GetProductType(utils.gdh.get_product_type_id_by_name(id))

  notification.Flags = NotificationFlags.CanBeDismissed + NotificationFlags.PopUp
  notification.Type = NotificationType.Custom
  notification.Icon = tostring(product_data.IconAssetID)

  notification.StringIdentifier = "not_new_market_phase"
  notification.ReplacementParameters:clear()
  notification.ReplacementParameters:add("product_name:T!"..tostring(product_data.Name))
  notification.ReplacementParameters:add("phase_index:"..tostring(phase_index))
  entities:UpdateNotificationComponent(notification_entity)

  return notification_entity
end


--- adds a message that money was added
---@param string_identifier string|number
---@param money_amount number
---@param param1 any
---@param param2 any
---@return number
script.money_added = function(string_identifier, money_amount, param1, param2)
  if type(string_identifier) ~= "string" then
    string_identifier = "not_missing"
    LogWarning("utils.notifications > money_added - invalid string_identifier: "..tostring(string_identifier))
  end

  if type(money_amount) ~= "number" then
    money_amount = 0
    LogWarning("utils.notifications > money_added - invalid money_amount: "..tostring(money_amount))
  end

  notificationservice:DismissNotification("gcm_money_added_"..tostring(string_identifier), true)
  local notification_entity = notificationservice:CreateOrGetNotification("gcm_money_added_"..tostring(string_identifier))
  local notification = entities:GetNotificationComponent(notification_entity)

  notification.Flags = NotificationFlags.CanBeDismissed
  notification.Type = NotificationType.Custom
  notification.Icon = "icons_common/greditC"

  notification.StringIdentifier = string_identifier
  notification.ReplacementParameters:clear()
  notification.ReplacementParameters:add("money_amount:"..tostring(money_amount))
  notification.ReplacementParameters:add("param1:T!"..tostring(param1))
  notification.ReplacementParameters:add("param2:T!"..tostring(param2))
  entities:UpdateNotificationComponent(notification_entity)

  return notification_entity
end


--- adds a message that money was added
---@param string_identifier string|number
---@param data_type string|number
---@param data_amount number
---@param param1 any
---@return number
script.data_added = function(string_identifier, data_type, data_amount, param1)
  if type(string_identifier) ~= "string" then
    string_identifier = "not_missing"
    LogWarning("utils.notifications > data_added - invalid string_identifier: "..tostring(string_identifier))
  end

  if type(data_amount) ~= "number" then
    data_amount = 0
    LogWarning("utils.notifications > data_added - invalid data_added: "..tostring(data_amount))
  end

  local id = utils.gdh.get_data_type_id_by_name(data_type)
  if id == 0 then
    LogWarning("utils.notifications > data_added - could not find data_type: "..tostring(data_type))
    return 0
  end

  notificationservice:DismissNotification("gcm_data_added_"..tostring(string_identifier), true)
  local notification_entity = notificationservice:CreateOrGetNotification("gcm_data_added_"..tostring(string_identifier))
  local notification = entities:GetNotificationComponent(notification_entity)
  local sampling_type_data = gamedata:GetSamplingDataType(utils.gdh.get_data_type_id_by_name(id))

  notification.Flags = NotificationFlags.CanBeDismissed
  notification.Type = NotificationType.Custom
  notification.Icon = tostring(sampling_type_data.IconAssetID)

  notification.StringIdentifier = string_identifier
  notification.ReplacementParameters:clear()
  notification.ReplacementParameters:add("data_type:T!"..tostring(sampling_type_data.Name))
  notification.ReplacementParameters:add("data_amount:"..tostring(data_amount))
  notification.ReplacementParameters:add("data_icon:[s:"..tostring(sampling_type_data.IconAssetID)..":t]")
  notification.ReplacementParameters:add("param1:T!"..tostring(param1))
  entities:UpdateNotificationComponent(notification_entity)

  return notification_entity
end




script.current_sold_data = {}
script.last_sold_data = {}

script.activate_sold_notifications = function(target_company)
  script.current_sold_data[target_company] = {}
  script.current_sold_data[target_company].modules = {}
  script.current_sold_data[target_company].modules.money_received = 0
  script.current_sold_data[target_company].modules.amount = 0
  script.current_sold_data[target_company].components = {}
  script.current_sold_data[target_company].components.money_received = 0
  script.current_sold_data[target_company].components.amount = 0
  script.current_sold_data[target_company].products = {}
  script.current_sold_data[target_company].products.money_received = 0
  script.current_sold_data[target_company].products.amount = 0

  marketsegmentservice:AddModulesSoldCallback(function(company, module_amount, money_received)
    if company == target_company then
      local data = script.current_sold_data[target_company]
      local modules = data.modules
      local components = data.components
      data.new_entry = true
      
      local module_data = gamedata:GetModule(module_amount.ModuleID)
      
      if #module_data.Features > 0  then
        modules.money_received = modules.money_received + money_received
        modules.amount = modules.amount + module_amount.Amount

        if modules[module_amount.ModuleID] == nil then
          modules[module_amount.ModuleID] = {}
          modules[module_amount.ModuleID].amount = 0
          modules[module_amount.ModuleID].money_received = 0
        end

        modules[module_amount.ModuleID].amount = modules[module_amount.ModuleID].amount + module_amount.Amount
        modules[module_amount.ModuleID].money_received = modules[module_amount.ModuleID].money_received + money_received
      else
        components.money_received = components.money_received + money_received
        components.amount = components.amount + module_amount.Amount

        if components[module_amount.ModuleID] == nil then
          components[module_amount.ModuleID] = {}
          components[module_amount.ModuleID].amount = 0
          components[module_amount.ModuleID].money_received = 0
        end

        components[module_amount.ModuleID].amount = components[module_amount.ModuleID].amount + module_amount.Amount
        components[module_amount.ModuleID].money_received = components[module_amount.ModuleID].money_received + money_received
      end
    end
  end)

  marketsegmentservice:AddProductsSoldCallback(function(company, blueprint_id, amount, money_received)
    if company == target_company then
      local data = script.current_sold_data[target_company]
      local products = data.products
      local blueprint_data = entities:GetBlueprintDefinitionComponent(blueprint_id)

      data.new_entry = true
      products.money_received = products.money_received + money_received
      products.amount = products.amount + amount

      if products[blueprint_data.ProductType] == nil then
        products[blueprint_data.ProductType] = {}
        products[blueprint_data.ProductType].amount = 0
        products[blueprint_data.ProductType].money_received = 0
      end
      local ptype = products[blueprint_data.ProductType]
      ptype.amount = ptype.amount + amount
      ptype.money_received = ptype.money_received + money_received

      if ptype[blueprint_id] == nil then
        ptype[blueprint_id] = {}
        ptype[blueprint_id].amount = 0
        ptype[blueprint_id].money_received = 0
      end
      local blueprint = ptype[blueprint_id]      
      blueprint.amount = blueprint.amount + amount
      blueprint.money_received = blueprint.money_received + money_received
    end
  end)

  script.update_sold_notification(target_company)
end

script.update_sold_notification = function(target_company)
  world:ExecuteFunctionInTime(0.1, function()
    local data = script.current_sold_data[target_company]
    if data.new_entry then
      local products = data.products
      local modules = data.modules
      local components = data.components

      notificationservice:DismissNotification("gcm_items_sold", true)
      local notification_entity = notificationservice:CreateOrGetNotification("gcm_items_sold")
      local notification = entities:GetNotificationComponent(notification_entity)

      notification.Flags = NotificationFlags.CanBeDismissed
      notification.Type = NotificationType.Custom
      notification.Icon = "icons_common/barsC"

      local s = ""
      for index, content in pairs(products) do
        if type(index) == "number" then
          local product_data = gamedata:GetProductType(index)
          s = s .. "\n"..content.amount.." [t:"..product_data.Name.."_pl] > *"..content.money_received.."* [s:icons_common/gredit:t]"
        end
      end


      notification.StringIdentifier = "not_sold_message"
      notification.ReplacementParameters:clear()
      notification.ReplacementParameters:add("component_amount:"..tostring(components.amount))
      notification.ReplacementParameters:add("component_money:"..tostring(components.money_received))
      notification.ReplacementParameters:add("module_amount:"..tostring(modules.amount))
      notification.ReplacementParameters:add("module_money:"..tostring(modules.money_received))
      notification.ReplacementParameters:add("product_amount:"..tostring(products.amount))
      notification.ReplacementParameters:add("product_money:"..tostring(products.money_received))
      notification.ReplacementParameters:add("product_list:"..tostring(s))
      entities:UpdateNotificationComponent(notification_entity)

      script.current_sold_data[target_company].modules = {}
      script.current_sold_data[target_company].modules.money_received = 0
      script.current_sold_data[target_company].modules.amount = 0
      script.current_sold_data[target_company].products = {}
      script.current_sold_data[target_company].products.money_received = 0
      script.current_sold_data[target_company].products.amount = 0
      script.current_sold_data[target_company].components = {}
      script.current_sold_data[target_company].components.money_received = 0
      script.current_sold_data[target_company].components.amount = 0
      data.new_entry = false
    end
    script.update_sold_notification(target_company)
  end)
end

return script