require("files")
local json = require("json")
local markets = {}

local utils = {}
utils.gamedatahelper = require("base.utils.gamedatahelper")
utils.notifications = require("base.utils.notifications")
utils.cv = require("base.utils.customvalues")

markets.Tendencies = {["StaysTheSame"]=0, ["RisesALittle"]=1, ["RisesMuch"]=2, ["DeclinesALittle"]=3, ["DiclinesMuch"]=4}
markets.TendencyLimits = {["DiclinesMuch"]=0.7, ["DeclinesALittle"]=0.9, ["StaysTheSame"]=1.1, ["RisesALittle"]=1.3, ["RisesMuch"]=math.maxinteger}

markets.on_init = function(market_path, file_name)
  markets.products = {}
  markets.products = json.decode(readFile(market_path..file_name))
  markets.additional_demands_entity = utils.cv.find_or_create_tagged_entity(8394830257)

  for i=1, #markets.products.product_types do
    markets.products.product_types[i].data = {}
    markets.products.product_types[i].data = json.decode(readFile(market_path..markets.products.product_types[i].name..".json"))
    if markets.products.product_types[i].data.is_extended then
      markets.extended_setup(markets.products.product_types[i].data)
    else
      markets.set_phases(markets.products.product_types[i].data)
    end
  end
end

markets.unlock_market = function(product_type_id, notification)
  local unlocked = marketdiscoveryservice:UnlockMarket(product_type_id)
  if unlocked == false then
    return
  end
  if notification == true then
    utils.notifications.unlocked_market(product_type_id)
  end
end

markets.phase_notification = function(market_segment_id, phase)
  local segment = entities:GetMarketSegmentComponent(market_segment_id)
  if phase == 1 then
    utils.notifications.unlocked_market(segment.MarketTargetID)
  else
    utils.notifications.new_market_phase(segment.MarketTargetID, phase)
  end
end

markets.set_phases = function(data)
  for i=1, #data.phases do
    marketprogression:AddMarketPhase(data.product_id, data.phases[i].duration, data.phases[i].discovery_points, function(market_segment_id)
        local marketExpectations = entities:GetMarketSegmentExpectationsComponent(market_segment_id)
        marketExpectations.Features:clear()
        marketExpectations.Drawbacks:clear()
        
        markets.update_price(market_segment_id, data.phases[i].price_factor * data.base_price)
        markets.update_demand(market_segment_id, data.phases[i].demand_factor * data.base_demand)
        markets.update_min_features(market_segment_id, data.phases[i].min_features)
        markets.add_features(market_segment_id, data.phases[i].features)
        markets.add_drawbacks(market_segment_id, data.phases[i].drawbacks)

        if data.phases[i+1] ~= nil then
          markets.update_price_tendency(market_segment_id, data.phases[i].price_factor * data.base_price, data.phases[i+1].price_factor * data.base_price)
          markets.update_demand_tendency(market_segment_id, data.phases[i].demand_factor * data.base_demand, data.phases[i+1].demand_factor * data.base_demand)
          markets.update_feature_prognosis(market_segment_id, data.phases[i].features, data.phases[i+1].features, data.phases[i].drawbacks, data.phases[i+1].drawbacks)
        end

        if data.phases[i].display_notification == true then
          markets.phase_notification(market_segment_id, i)
        end
    end)
  end    
end


markets.extended_setup = function(data)
  local market_entity = marketsegmentservice:GetMarketSegmentEntityID(1, data.product_id)
  local current_phase = 0
  if market_entity ~= 0 then
    local expectations = entities:GetMarketSegmentExpectationsComponent(market_entity)
    current_phase = expectations.CurrentMarketPhase
    for i = 1, expectations.CurrentMarketPhase do
      markets.add_extended_phase(data, i, false)
      --marketprogression:AddMarketPhase(data.product_id, 0, 0, function() end)
    end
  end
  markets.add_extended_phase(data, current_phase+1, true, true)
end


markets.add_extended_phase = function(data, phase_index, add_phase, add_buffer)
  local current_phase
  if #data.phases > phase_index then
    current_phase = data.phases[phase_index]
  else
    current_phase = data.phases[#data.phases]
    if not data.is_endless then
      add_phase = false
    end
  end

  local next_phase
  if #data.phases > phase_index+1 then
    next_phase = data.phases[phase_index+1]
  else
    next_phase = data.phases[#data.phases]
  end

  local current_noise = markets.get_noise(data, phase_index)
  local next_noise = markets.get_noise(data, phase_index + 1)      

  local duration = markets.get_duration_for_phase (data, current_phase, phase_index, current_noise)

  marketprogression:AddMarketPhase(data.product_id, duration, current_phase.discovery_points, function(market_segment_id)
    local expectations = entities:GetMarketSegmentExpectationsComponent(market_segment_id)
    expectations.Features:clear()
    expectations.Drawbacks:clear()

    local current_noise = markets.get_noise(data, phase_index)
    local next_noise = markets.get_noise(data, phase_index + 1)

    local techrate = markets.get_techrate(data, current_phase)
    if techrate > 0 then
      techrate = math.sqrt(1+techrate)-1
    end
    
    if techrate > 0.5 then
      techrate = 0.5
    end
    
    --local duration = markets.get_duration_for_phase (data, current_phase, phase_index, current_noise, techrate)
    --expectations.CurrentMarketPhaseDuration = duration
    --entities:UpdateMarketSegmentExpectationsComponent(market_segment_id)

    local current_price = markets.get_price_for_phase (data, current_phase, phase_index, current_noise, techrate)
    local next_price = markets.get_price_for_phase (data, next_phase, phase_index, next_noise, techrate)
    markets.update_price(market_segment_id, current_price)

    local current_demand = markets.get_demand_for_phase (data, current_phase, phase_index, current_noise)
    local next_demand = markets.get_demand_for_phase (data, next_phase, phase_index, next_noise)
    markets.update_demand(market_segment_id, current_demand)

    local current_features = markets.get_features_for_phase (data, current_phase.features, phase_index, current_noise, techrate)
    local next_features = markets.get_features_for_phase (data, next_phase.features, phase_index, next_noise, techrate)
    markets.add_features(market_segment_id, current_features)

    local minfeatures = markets.get_minfeatures_for_phase (data, current_features, current_phase, phase_index, current_noise, techrate)
    markets.update_min_features(market_segment_id, minfeatures)

    local current_drawbacks = markets.get_features_for_phase (data, current_phase.drawbacks, phase_index, current_noise, techrate)
    local next_drawbacks = markets.get_features_for_phase (data, next_phase.drawbacks, phase_index, next_noise, techrate)
    markets.add_drawbacks(market_segment_id, current_drawbacks)

    if add_phase then
      markets.update_price_tendency(market_segment_id, current_price, next_price)
      markets.update_demand_tendency(market_segment_id, current_demand, next_demand)
      markets.update_feature_prognosis(market_segment_id, current_features, next_features, current_drawbacks, next_drawbacks)
      world:ExecuteFunctionInTime(0, function()
        markets.add_extended_phase(data, phase_index+2, add_phase)
      end)
    end

    if current_phase.display_notification then
      markets.phase_notification(market_segment_id, phase_index)
    end
  end)

  if add_phase and add_buffer then
    markets.add_extended_phase(data, phase_index+1, add_phase)
  end
end


markets.get_noise = function(data, phase_index)
  local noise = 0
  local multiplier = 0
  if phase_index >= data.noise_start then
    multiplier = phase_index - data.noise_start + 1
  end
  if data.use_noise then
    noise = data.noise_per_phase * multiplier
  end

  if noise > data.noise_cap then
    noise = data.noise_cap
  end
  return noise
end


markets.get_techrate = function(data, phase)
  local tech_count = 0
  local finished_projects = entities:GetFinishedResearchProjectsComponent(world.PlayerCompanyID)
  --print(finished_projects:ContainsProject(94))
  for p=1, #finished_projects.Projects do
    for i=1, #data.tech_list do
      if utils.gamedatahelper.get_research_id_by_name(data.tech_list[i]) == finished_projects.Projects[p] then
        tech_count = tech_count + 1
      end
    end
  end
  return ((data.tech_offset + tech_count) / phase.expected_techs) - 1
end


markets.get_duration_for_phase = function(data, phase, phase_index, noise, techrate)
  math.randomseed(world.GlobalSeed + data.product_id + phase_index + 1000)
  local duration = phase.duration
  local noise_effect = ((math.random()*2)-1) * noise

  --duration = duration + (duration * techrate * data.time_handicap) + (duration * noise_effect)
  duration = duration + (duration * noise_effect)
  return math.ceil(duration)
end


markets.get_price_for_phase = function(data, phase, phase_index, noise, techrate)
  math.randomseed(world.GlobalSeed + data.product_id + phase_index + 2000)
  local price = phase.price_factor * data.base_price
  local noise_effect = ((math.random()*2)-1) * noise
  price = price + (price * techrate * data.price_handicap) + (price * noise_effect)
  return math.ceil(price)
end


markets.get_demand_for_phase = function(data, phase, phase_index, noise)
  math.randomseed(world.GlobalSeed + data.product_id + phase_index + 3000)
  local demand = phase.demand_factor * data.base_demand
  local noise_effect = ((math.random()*2)-1) * noise
  demand = demand + (demand * noise_effect)
  return math.ceil(demand)
end



markets.get_minfeatures_for_phase = function(data, features, phase, phase_index, noise, techrate)
  --math.randomseed(world.GlobalSeed + data.product_id + phase_index + 4000)
  local min = 0
  if phase.feature_range then
    local feature_sum = 0
    for i=1, #features do
      feature_sum = feature_sum + features[i].value
    end
    min = feature_sum - phase.feature_range
  elseif phase.min_features then
    min = phase.min_features
    min = min + (min * techrate * data.minfeat_handicap)
  end
  --local noise_effect = ((math.random()*2)-1) * noise
  --min = min + (min * techrate * data.minfeat_handicap)
  return math.floor(min)
end


markets.get_features_for_phase = function(data, feature_list, phase_index, noise, techrate)
  local noise_effect

  local features = {}
  for i=1, #feature_list do
    math.randomseed(world.GlobalSeed + data.product_id + phase_index + 4000 + feature_list[i].id)
    noise_effect = ((math.random()*2)-1) * noise
    features[i] = {}
    features[i].id = feature_list[i].id
    features[i].value = math.floor(
      (feature_list[i].value) + 
      (feature_list[i].value * techrate * data.feat_handicap) + 
      (feature_list[i].value * noise_effect) +
      0.3 -- balance value
    )
  end
  return features
end
--------------------------------


markets.update_price = function(market_segment_id, newPrice)
  if newPrice ==  nil then
    newPrice = 0
    LogError("Market Utilities: update_price -> newPrice was nil, default to 0")
  end
  local marketExpectations = entities:GetMarketSegmentExpectationsComponent(market_segment_id)
  marketExpectations.ExpectedProductPrice = math.floor(newPrice)
  entities:UpdateMarketSegmentExpectationsComponent(market_segment_id)  
end


markets.update_demand = function(market_segment_id, newDemand)
  if newDemand ==  nil then
    newDemand = 0
    LogError("Market Utilities: update_demand -> newDemand was nil, default to 0")
  end

  local marketSegment = entities:GetMarketSegmentComponent(market_segment_id)

  local custom_values = entities:GetCustomValuesComponent(markets.additional_demands_entity)
  while custom_values.CustomNumbers[marketSegment.MarketTargetID+1] == nil do
    custom_values.CustomNumbers:add(0)
  end
  entities:UpdateCustomValuesComponent(markets.additional_demands_entity)

  marketSegment.MarketDemand = math.floor(newDemand + custom_values.CustomNumbers[marketSegment.MarketTargetID+1])
  entities:UpdateMarketSegmentComponent(market_segment_id)
end


markets.update_min_features = function (market_segment_id, new_min)
  if new_min ==  nil then
    new_min = 0
    LogError("Market Utilities: update_min_features -> new_min was nil, default to 0")
  end
  local marketExpectations = entities:GetMarketSegmentExpectationsComponent(market_segment_id)
  marketExpectations.MinFeatureValue = math.floor(new_min)
  entities:UpdateMarketSegmentExpectationsComponent(market_segment_id)  
end


markets.add_features = function(market_segment_id, features)
  if features ==  nil then
    LogError("Market Utilities: add_features -> features was nil, function aborted")
    return
  end
  local marketExpectations = entities:GetMarketSegmentExpectationsComponent(market_segment_id)
  for i=1, #features do
    local feature = FeatureValue.new()
    feature.FeatureID = features[i].id
    feature.Value = features[i].value
    marketExpectations.Features:add(feature)
  end
  entities:UpdateMarketSegmentExpectationsComponent(market_segment_id)  
end


markets.add_drawbacks = function(market_segment_id, drawbacks)
  if drawbacks ==  nil then
    LogError("Market Utilities: add_drawbacks -> drawbacks was nil, function aborted")
    return
  end
  local marketExpectations = entities:GetMarketSegmentExpectationsComponent(market_segment_id)
  for i=1, #drawbacks do
    local drawback = FeatureValue.new()
    drawback.FeatureID = drawbacks[i].id
    drawback.Value = drawbacks[i].value
    marketExpectations.Drawbacks:add(drawback) 
  end
  entities:UpdateMarketSegmentExpectationsComponent(market_segment_id)  
end


markets.update_price_tendency = function(market_segment_id, currentPrice, nextPrice)
  if nextPrice == nil then
    return
  end
  
  local marketExpectations = entities:GetMarketSegmentExpectationsComponent(market_segment_id)
  local rate = nextPrice/currentPrice
  
  if rate > markets.TendencyLimits["RisesALittle"] then
    marketExpectations.PrognosisPriceTendency = markets.Tendencies["RisesMuch"]
  elseif rate > markets.TendencyLimits["StaysTheSame"] then
    marketExpectations.PrognosisPriceTendency = markets.Tendencies["RisesALittle"]
  elseif rate > markets.TendencyLimits["DeclinesALittle"] then
    marketExpectations.PrognosisPriceTendency = markets.Tendencies["StaysTheSame"]
  elseif rate > markets.TendencyLimits["DiclinesMuch"] then
    marketExpectations.PrognosisPriceTendency = markets.Tendencies["DeclinesALittle"]
  else
    marketExpectations.PrognosisPriceTendency = markets.Tendencies["DiclinesMuch"]
  end
  
  entities:UpdateMarketSegmentExpectationsComponent(market_segment_id)  
end


markets.update_demand_tendency = function(market_segment_id, currentDemand, nextDemand)
  if nextDemand == nil then
    return
  end
  
  local marketExpectations = entities:GetMarketSegmentExpectationsComponent(market_segment_id)
  local rate = nextDemand/currentDemand
  
  if rate > markets.TendencyLimits["RisesALittle"] then
    marketExpectations.PrognosisDemandTendency = markets.Tendencies["RisesMuch"]
  elseif rate > markets.TendencyLimits["StaysTheSame"] then
    marketExpectations.PrognosisDemandTendency = markets.Tendencies["RisesALittle"]
  elseif rate > markets.TendencyLimits["DeclinesALittle"] then
    marketExpectations.PrognosisDemandTendency = markets.Tendencies["StaysTheSame"]
  elseif rate > markets.TendencyLimits["DiclinesMuch"] then
    marketExpectations.PrognosisDemandTendency = markets.Tendencies["DeclinesALittle"]
  else
    marketExpectations.PrognosisDemandTendency = markets.Tendencies["DiclinesMuch"]
  end

  entities:UpdateMarketSegmentExpectationsComponent(market_segment_id)  
end


markets.update_feature_prognosis = function(market_segment_id, currentFeatures, nextFeatures, currentDrawbacks, nextDrawbacks)
  local marketExpectations = entities:GetMarketSegmentExpectationsComponent(market_segment_id)
  local newFeature = false
  local newDrawback = false
  local featureIncrease = false
  local drawbackIncrease = false
  
  if nextFeatures == nil or nextDrawbacks == nil then
    marketExpectations:SetPrognosisEvents(newFeature, newDrawback, featureIncrease, drawbackIncrease)
    return
  end

  if #nextFeatures > #currentFeatures then
    newFeature = true
  end

  if #nextDrawbacks > #currentDrawbacks then
    newDrawback = true
  end

  for kc,vc in pairs(currentFeatures) do
    for kn,vn in pairs (nextFeatures) do
      if vc.id == vn.id then
        if vn.value > vc.value then
          featureIncrease = true
        end
      end
    end
  end

  for kc,vc in pairs(currentDrawbacks) do
    for kn,vn in pairs (nextDrawbacks) do
      if vc.id == vn.id then
        if vn.value > vc.value then
          drawbackIncrease = true
        end
      end
    end
  end

  marketExpectations:SetPrognosisEvents (newFeature, newDrawback, featureIncrease, drawbackIncrease)
end


--- function adds materials to be purchased with G
--- it requires a list were the index is the materialID and the value is the sellprice!
--- materials[241] = 800 would set the materialID 241 to the sell price 800 (the complete stack)
--- you can use helper.lua -> table_to_array to parse json data to be used with this function
---@param materials table
---@return boolean
markets.add_sold_materials = function(materials)
  if type(materials) ~= "table" or #materials <= 0 then
    LogWarning("markets > add_sold_materials: table did not contain data")
    return false
  end

  entities:ForEachMarketplaceComponent(function(entity)
    local marketplace = entities:GetMarketplaceComponent(entity)
    for material_id,sellprice in pairs(materials) do
      marketplace.MaterialStackPrices[material_id] = sellprice
    end
    entities:UpdateMarketplaceComponent(entity)
  end)
end


--- function overwrites materials to be purchased with G
--- it requires a list were the index is the materialID and the value is the sellprice!
--- materials[241] = 800 would set the materialID 241 to the sell price 800 (the complete stack)
--- you can use helper.lua -> table_to_array to parse json data to be used with this function
---@param materials table
---@return boolean
markets.set_sold_materials = function(materials)
  if type(materials) ~= "table" or #materials <= 0 then
    LogWarning("markets > set_sold_materials: table did not contain data")
    return false
  end

  entities:ForEachMarketplaceComponent(function(entity)
    local marketplace = entities:GetMarketplaceComponent(entity)
    marketplace.MaterialStackPrices:clear()
    for material_id,sellprice in pairs(materials) do
      marketplace.MaterialStackPrices[material_id] = sellprice
    end
    entities:UpdateMarketplaceComponent(entity)
  end)
end


--- function removes materials to be purchased with G
--- table can be an unsorted list of IDs
---@param materials table
---@return boolean
markets.remove_sold_materials = function(materials)
  if type(materials) ~= "table" or #materials <= 0 then
    LogWarning("markets > remove_sold_materials: table did not contain data")
    return false
  end

  entities:ForEachMarketplaceComponent(function(entity)
    local marketplace = entities:GetMarketplaceComponent(entity)
    for index,material_id in pairs(materials) do
      marketplace.MaterialStackPrices:erase(material_id)
    end
    entities:UpdateMarketplaceComponent(entity)
  end)
end


--- does modify the value for icnreased demand for a market
---@param product_id number
---@param increase_amount number
---@return boolean
markets.increase_additional_demand = function(product_id, increase_amount)
  if type(product_id) ~= "number" then
    LogError("utils.markets > increase_additional_demand - product_id must be a number "..tostring(product_id))
    return false
  end
  if type(increase_amount) ~= "number" then
    LogError("utils.markets > increase_additional_demand - increase_amount must be a number "..tostring(product_id))
    return false
  end
  if gamedata:GetProductType(product_id) == nil then
    LogError("utils.markets > increase_additional_demand - product_id does not exist "..tostring(product_id))
    return false
  end

  increase_amount = math.floor(increase_amount)

  local component = entities:GetCustomValuesComponent(markets.additional_demands_entity)
  while component.CustomNumbers[product_id+1] == nil do
    component.CustomNumbers:add(0)
  end

  component.CustomNumbers[product_id+1] = component.CustomNumbers[product_id+1] + increase_amount
  entities:UpdateCustomValuesComponent(markets.additional_demands_entity)

  local market_segment_id = marketsegmentservice:GetMarketSegmentEntityID(1, product_id)
  if market_segment_id ~= 0 then
    local marketSegment = entities:GetMarketSegmentComponent(market_segment_id)
    marketSegment.MarketDemand = marketSegment.MarketDemand + increase_amount
    entities:UpdateMarketSegmentComponent(market_segment_id)
  end
  return true
end


return markets
