/*
 * Decompiled with CFR 0.152.
 */
package com.webobjects.foundation.plist.impl;

import com.webobjects.foundation.plist.impl.Preconditions;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.EOFException;
import java.io.IOException;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.SeekableByteChannel;
import java.nio.charset.Charset;
import java.util.AbstractList;
import java.util.AbstractMap;
import java.util.AbstractSet;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.Deque;
import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class BinaryPListUtils {
    private static final int DOUBLE_SIZE_EXP = 3;
    private static final int FLOAT_SIZE_EXP = 2;
    private static final int LONG_SIZE_EXP = 3;
    static final int BYTE_MASK = 255;
    static final int SHORT_MASK = 65535;
    private static final Charset UTF16BE = Charset.forName("UTF-16BE");
    private static final Charset ASCII = Charset.forName("US-ASCII");
    private static final Logger log = LoggerFactory.getLogger(BinaryPListUtils.class);
    public static final long EPOCH_OFFSET = 978307200L;
    public static final int UUID_LENGTH = 16;
    public static final String NOT_A_BPLIST = "Not a binary property list";
    public static final int UTF8_MASK = 192;
    public static final int TYPE_MASK = 240;
    public static final int COUNT_MASK = 15;
    public static final int INT128_SIZE = 128;
    public static final int HEADER_SIZE = 8;

    BinaryPListUtils() {
    }

    public static void checkHeader(long header) {
        Header.checkHeader(header);
    }

    public static Object markerString(int marker) {
        String bits = Integer.toString(marker & 0xFF, 2);
        return String.valueOf("00000000".substring(0, 8 - bits.length())) + bits;
    }

    public static Object read(byte[] bytes) {
        return BinaryPListUtils.read(ByteBuffer.wrap(bytes));
    }

    public static Object read(ByteBuffer source) {
        ByteBuffer buffer = source.slice();
        Preconditions.checkCondition(buffer.remaining() > 8, NOT_A_BPLIST, new Object[0]);
        BinaryPListUtils.checkHeader(buffer.getLong());
        try {
            Throwable throwable = null;
            Object var3_5 = null;
            try (SeekableDataChannel data = new SeekableDataChannel(buffer);){
                buffer.position(buffer.capacity() - 32);
                Trailer trailer = BinaryPListUtils.readTrailer(data);
                ReferenceTable objectTable = new ReferenceTable(new ArrayList<Object>((int)trailer.objectCount));
                buffer.position(8);
                while ((long)objectTable.size() < trailer.objectCount) {
                    BinaryPListUtils.readNextObject(data, objectTable, trailer.objectRefSize);
                }
                return objectTable.get((int)trailer.topObject);
            }
            catch (Throwable throwable2) {
                if (throwable == null) {
                    throwable = throwable2;
                } else if (throwable != throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
        }
        catch (IOException e) {
            throw new IllegalStateException(e);
        }
    }

    public static Object read(SeekableByteChannel channel) throws IOException {
        Preconditions.checkCondition(channel.size() > 8L, NOT_A_BPLIST, new Object[0]);
        Throwable throwable = null;
        Object var2_3 = null;
        try (DataInputStream data = new DataInputStream(Channels.newInputStream(channel));){
            BinaryPListUtils.checkHeader(data.readLong());
            channel.position(channel.size() - 32L);
            Trailer trailer = BinaryPListUtils.readTrailer(data);
            channel.position(8L);
            ReferenceTable objectTable = new ReferenceTable(new ArrayList<Object>((int)trailer.objectCount));
            while ((long)objectTable.size() < trailer.objectCount) {
                BinaryPListUtils.readNextObject(data, objectTable, trailer.objectRefSize);
            }
            return objectTable.get((int)trailer.topObject);
        }
        catch (Throwable throwable2) {
            if (throwable == null) {
                throwable = throwable2;
            } else if (throwable != throwable2) {
                throwable.addSuppressed(throwable2);
            }
            throw throwable;
        }
    }

    public static ReferenceArray readArray(DataInput input, ReferenceArray array, int count, RefSize refSize) throws IOException {
        log.trace("readArray: reading array of size {}", (Object)count);
        int i = 0;
        while (i < count) {
            array.setValueRef(i, refSize.read(input));
            ++i;
        }
        return array;
    }

    public static String readAsciiString(DataInput input, int count) throws IOException {
        log.trace("readAsciiString: reading string of length {}", (Object)count);
        byte[] buf = new byte[count];
        input.readFully(buf);
        return new String(buf, "ASCII");
    }

    public static int readCount(DataInput input) throws IOException {
        log.trace("readCount: reading length byte");
        byte marker = input.readByte();
        Type type = Type.forValue(marker);
        if (type != Type.INT) {
            log.trace("readCount: read type of {}", (Object)type);
            throw new IllegalStateException("Invalid type marker reading count: " + BinaryPListUtils.markerString(marker) + " found " + (Object)((Object)type));
        }
        int count = 1 << (marker & 0xF);
        return (int)RefSize.ofSize(count).read(input);
    }

    public static ByteBuffer readData(DataInput input, int count) throws IOException {
        log.trace("readData: reading data of length {}", (Object)count);
        byte[] bytes = new byte[count];
        input.readFully(bytes);
        return ByteBuffer.wrap(bytes);
    }

    public static Date readDate(DataInput input) throws IOException {
        log.trace("readDate: reading date");
        double interval = input.readDouble();
        long adjusted = TimeUnit.SECONDS.toMillis(978307200L + (long)interval);
        return new Date(adjusted);
    }

    public static ReferenceDictionary readDictionary(DataInput input, ReferenceDictionary dict, int count, RefSize refSize) throws IOException {
        long ref;
        log.trace("readDictionary: reading dictionary of length {}", (Object)count);
        int i = 0;
        while (i < count) {
            ref = refSize.read(input);
            dict.setKeyRef(i, ref);
            ++i;
        }
        i = 0;
        while (i < count) {
            ref = refSize.read(input);
            dict.setValueRef(i, ref);
            ++i;
        }
        return dict;
    }

    public static Number readInt(DataInput input, int size) throws IOException {
        BigInteger number;
        int count = 1 << size;
        log.trace("readInt: reading int with byte size {}", (Object)count);
        byte[] array = new byte[count];
        input.readFully(array);
        BigInteger bigInteger = number = count < 8 ? new BigInteger(1, array) : new BigInteger(array);
        if (number.bitLength() > 127) {
            log.warn("read a 128 bit integer value that is incompatible with CoreFoundation");
        }
        return number.bitLength() < 64 ? Long.valueOf(number.longValue()) : number;
    }

    public static Type readNextObject(DataInput data, List<Object> objectTable, RefSize objectRefSize) throws IOException {
        byte marker = data.readByte();
        if (log.isTraceEnabled()) {
            log.trace("readNextObject: reading object for marker: {}", BinaryPListUtils.markerString(marker));
        }
        Type type = Type.forValue(marker);
        int count = type.count(data, marker);
        switch (type) {
            case INT: {
                objectTable.add(BinaryPListUtils.readInt(data, count));
                break;
            }
            case REAL: {
                objectTable.add(BinaryPListUtils.readReal(data, count));
                break;
            }
            case DATA: {
                objectTable.add(BinaryPListUtils.readData(data, count));
                break;
            }
            case ASCII: {
                objectTable.add(BinaryPListUtils.readAsciiString(data, count));
                break;
            }
            case UTF16: {
                objectTable.add(BinaryPListUtils.readUTF16String(data, count));
                break;
            }
            case UTF8: {
                objectTable.add(BinaryPListUtils.readUTF8String(data, count));
                break;
            }
            case UUID: {
                objectTable.add(BinaryPListUtils.readUUID(data));
                break;
            }
            case DATE: {
                objectTable.add(BinaryPListUtils.readDate(data));
                break;
            }
            case NULL: {
                objectTable.add(null);
                break;
            }
            case FALSE: {
                objectTable.add(Boolean.FALSE);
                break;
            }
            case TRUE: {
                objectTable.add(Boolean.TRUE);
                break;
            }
            case FILL: {
                break;
            }
            case ARRAY: {
                ReferenceArray array = new ReferenceArray(objectTable, count);
                objectTable.add(BinaryPListUtils.readArray(data, array, count, objectRefSize));
                break;
            }
            case DICT: {
                ReferenceDictionary dict = new ReferenceDictionary(objectTable, count);
                objectTable.add(BinaryPListUtils.readDictionary(data, dict, count, objectRefSize));
                break;
            }
            case SET: {
                ReferenceSet set = new ReferenceSet(objectTable, count);
                objectTable.add(BinaryPListUtils.readSet(data, set, count, objectRefSize));
                break;
            }
            default: {
                throw new IllegalStateException("Read invalid marker: " + BinaryPListUtils.markerString(marker));
            }
        }
        return type;
    }

    public static Number readReal(DataInput input, int size) throws IOException {
        Number result;
        int count = 1 << size;
        log.trace("readReal: reading real with byte size {}", (Object)count);
        switch (count * 8) {
            case 32: {
                result = new Float(input.readFloat());
                break;
            }
            case 64: {
                result = new Double(input.readDouble());
                break;
            }
            default: {
                throw new IllegalStateException("Unsupported real size: " + count);
            }
        }
        return result;
    }

    public static ReferenceSet readSet(DataInput input, ReferenceSet set, int count, RefSize refSize) throws IOException {
        log.trace("readSet: reading set of length {}", (Object)count);
        int i = 0;
        while (i < count) {
            set.setValueRef(i, refSize.read(input));
            ++i;
        }
        return set;
    }

    public static Trailer readTrailer(DataInput input) throws IOException {
        log.trace("readTrailer: reading trailer");
        return new Trailer(input);
    }

    public static String readUTF16String(DataInput input, int count) throws IOException {
        log.trace("readUTF16String: reading string of length {}", (Object)count);
        char[] chars = new char[count];
        int i = 0;
        while (i < chars.length) {
            chars[i] = input.readChar();
            ++i;
        }
        return new String(chars);
    }

    public static String readUTF8String(DataInput input, int count) throws IOException {
        log.trace("readUTF8String: reading string of length {}", (Object)count);
        char[] chars = new char[count];
        int i = 0;
        while (i < count) {
            byte b = input.readByte();
            if ((b & 0xC0) == 0) {
                chars[i] = (char)b;
            } else {
                int codepoint = b << 8 | input.readByte() & 0xFF;
                chars[i] = (char)codepoint;
            }
            ++i;
        }
        return new String(chars);
    }

    public static UUID readUUID(DataInput input) throws IOException {
        log.trace("readUUID: reading 16 byte UUID");
        long mostSigBits = input.readLong();
        long leastSigBits = input.readLong();
        return new UUID(mostSigBits, leastSigBits);
    }

    public static String toHexString(ByteBuffer value, int limit) {
        ByteBuffer buffer = value.slice();
        if (limit > 0 && buffer.remaining() > limit) {
            buffer.limit(limit);
        }
        StringBuilder sb = new StringBuilder(buffer.remaining() * 2);
        while (buffer.hasRemaining()) {
            sb.append(String.format("%02x", buffer.get()));
        }
        if (buffer.limit() != buffer.capacity()) {
            sb.append("...");
        }
        return sb.toString();
    }

    public static int writeArray(DataOutput output, ReferenceArray array, RefSize refSize) throws IOException {
        log.trace("writeArray: writing array of length {}", (Object)array.size());
        return BinaryPListUtils.writeCollection(output, Type.ARRAY, array, refSize);
    }

    public static int writeBoolean(DataOutput output, boolean bool) throws IOException {
        log.trace("writeBoolean: writing boolean {}", (Object)bool);
        int value = bool ? Type.TRUE.value() : Type.FALSE.value();
        output.write(value);
        return 1;
    }

    public static int writeCollection(DataOutput output, Type type, ReferenceCollection<?> collection, RefSize refSize) throws IOException {
        int count = collection.size();
        int written = BinaryPListUtils.writeMarker(output, type, count);
        int i = 0;
        while (i < count) {
            written += refSize.write(output, collection.getValueRef(i));
            ++i;
        }
        return written;
    }

    public static int writeData(DataOutput output, ByteBuffer source) throws IOException {
        ByteBuffer buffer = source.slice();
        log.trace("writeData: writing data of length {}", (Object)buffer.remaining());
        int length = buffer.remaining();
        length += BinaryPListUtils.writeMarker(output, Type.DATA, length);
        if (buffer.hasArray()) {
            output.write(buffer.array(), buffer.arrayOffset(), buffer.remaining());
        } else {
            byte[] bytes = new byte[buffer.remaining()];
            buffer.get(bytes);
            output.write(bytes);
        }
        return length;
    }

    public static int writeDate(DataOutput output, Date value) throws IOException {
        log.trace("writeDate: writing date {}", (Object)value.getTime());
        double convertedDate = TimeUnit.MILLISECONDS.toSeconds(value.getTime()) - 978307200L;
        output.write(Type.DATE.marker(3));
        output.writeLong(Double.doubleToRawLongBits(convertedDate));
        return 9;
    }

    public static int writeDictionary(DataOutput output, ReferenceDictionary dict, RefSize refSize) throws IOException {
        int count = dict.size();
        log.trace("writeDictionary: writing dictionary of size {}", (Object)count);
        int written = BinaryPListUtils.writeMarker(output, Type.DICT, count);
        int i = 0;
        while (i < count) {
            written += refSize.write(output, dict.getKeyRef(i));
            ++i;
        }
        i = 0;
        while (i < count) {
            written += refSize.write(output, dict.getValueRef(i));
            ++i;
        }
        return written;
    }

    public static int writeHeader(DataOutput output) throws IOException {
        log.trace("writeHeader: writing header");
        return Header.write(output);
    }

    public static int writeInt(DataOutput output, Number num) throws IOException {
        int exp;
        log.trace("writeInt: writing integer: {}", (Object)num);
        BigInteger bigint = num instanceof BigInteger ? (BigInteger)num : BigInteger.valueOf(num.longValue());
        byte[] bytes = bigint.toByteArray();
        if (bigint.bitLength() >= 64) {
            log.warn("writing large integer value that is incompatible with CoreFoundation");
        }
        if (bigint.signum() == 1 && bytes[0] == 0) {
            bytes = Arrays.copyOfRange(bytes, 1, bytes.length);
        }
        if ((exp = 32 - Integer.numberOfLeadingZeros(bytes.length - 1)) < 3 && bigint.signum() == -1) {
            exp = 3;
        }
        output.write(Type.INT.marker(exp));
        int byteCount = 1 << exp;
        int length = byteCount + 1;
        byte padding = (byte)(bigint.signum() < 0 ? 255 : 0);
        while (byteCount-- > bytes.length) {
            output.write(padding);
        }
        output.write(bytes);
        return length;
    }

    public static int writeMarker(DataOutput output, Type type, long count) throws IOException {
        log.trace("writeMarker: writing marker for: Type = {}; count = {}", (Object)type, (Object)count);
        output.write(type.marker((int)count));
        return count < 15L ? 1 : BinaryPListUtils.writeInt(output, count) + 1;
    }

    public static int writeNull(DataOutput output) throws IOException {
        log.trace("writeNull: writing null");
        output.write(Type.NULL.value());
        return 1;
    }

    public static int writeNumber(DataOutput output, Number number) throws IOException {
        if (number instanceof Float || number instanceof Double || number instanceof BigDecimal) {
            return BinaryPListUtils.writeReal(output, number);
        }
        return BinaryPListUtils.writeInt(output, number);
    }

    public static int writeObject(DataOutput output, Object obj, RefSize refSize) throws IOException {
        int result;
        if (obj == null) {
            result = BinaryPListUtils.writeNull(output);
        } else if (obj instanceof ReferenceArray) {
            result = BinaryPListUtils.writeArray(output, (ReferenceArray)obj, refSize);
        } else if (obj instanceof Boolean) {
            result = BinaryPListUtils.writeBoolean(output, (Boolean)obj);
        } else if (obj instanceof ByteBuffer) {
            result = BinaryPListUtils.writeData(output, (ByteBuffer)obj);
        } else if (obj instanceof byte[]) {
            result = BinaryPListUtils.writeData(output, ByteBuffer.wrap((byte[])obj));
        } else if (obj instanceof Date) {
            result = BinaryPListUtils.writeDate(output, (Date)obj);
        } else if (obj instanceof ReferenceDictionary) {
            result = BinaryPListUtils.writeDictionary(output, (ReferenceDictionary)obj, refSize);
        } else if (obj instanceof Number) {
            result = BinaryPListUtils.writeNumber(output, (Number)obj);
        } else if (obj instanceof ReferenceSet) {
            result = BinaryPListUtils.writeSet(output, (ReferenceSet)obj, refSize);
        } else if (obj instanceof CharSequence) {
            result = BinaryPListUtils.writeString(output, obj.toString());
        } else if (obj instanceof UUID) {
            result = BinaryPListUtils.writeUUID(output, (UUID)obj);
        } else {
            throw new IllegalArgumentException("Unsupported object type " + obj.getClass());
        }
        return result;
    }

    public static int writeReal(DataOutput output, Number value) throws IOException {
        log.trace("writeReal: writing floating point: {}", (Object)value);
        if (value instanceof Float) {
            output.write(Type.REAL.marker(2));
            output.writeFloat(((Float)value).floatValue());
            return 5;
        }
        output.write(Type.REAL.marker(3));
        output.writeDouble(value.doubleValue());
        return 9;
    }

    public static int writeRef(DataOutput output, RefSize type, Long ref) throws IOException {
        log.trace("writeRef: writing object reference: Type = {}; ref = {}", (Object)type, (Object)ref);
        type.write(output, ref);
        return type.size();
    }

    public static int writeSet(DataOutput output, ReferenceSet set, RefSize refSize) throws IOException {
        log.trace("writeSet: writing set of size {}", (Object)set.size());
        return BinaryPListUtils.writeCollection(output, Type.SET, set, refSize);
    }

    public static int writeString(DataOutput output, String value) throws IOException {
        byte[] theBytes;
        Type type;
        log.trace("writeString: writing string of length {}", (Object)value.length());
        if (ASCII.newEncoder().canEncode(value)) {
            type = Type.ASCII;
            theBytes = value.getBytes(ASCII);
        } else {
            type = Type.UTF16;
            theBytes = value.getBytes(UTF16BE);
        }
        int markerLength = BinaryPListUtils.writeMarker(output, type, value.length());
        output.write(theBytes);
        return markerLength + theBytes.length;
    }

    public static int writeTrailer(DataOutput output, Trailer trailer) throws IOException {
        log.trace("writeTrailer: writing trailer {}", (Object)trailer);
        trailer.write(output);
        return 32;
    }

    public static int writeUUID(DataOutput output, UUID value) throws IOException {
        log.trace("writeUUID: writing uuid {}", (Object)value);
        long mostSigBits = value.getMostSignificantBits();
        long leastSigBits = value.getLeastSignificantBits();
        output.write(Type.UUID.marker(16));
        output.writeLong(mostSigBits);
        output.writeLong(leastSigBits);
        return 17;
    }

    public static class Header {
        protected static final long BPLIST00 = 7093288613272891440L;

        static void checkHeader(long header) {
            if (log.isTraceEnabled()) {
                log.trace("readHeader: got magic value of: {}", (Object)new String(BigInteger.valueOf(header).toByteArray()));
            }
            Preconditions.checkCondition(Header.isValid(header), BinaryPListUtils.NOT_A_BPLIST, new Object[0]);
        }

        static boolean isValid(long header) {
            return header >> 8 == 27708158645597232L;
        }

        static int write(DataOutput output) throws IOException {
            output.writeLong(7093288613272891440L);
            return 8;
        }
    }

    public static interface PListValue<T> {
        public static final PListValue<Object> NULL = new PListValue<Object>(){

            @Override
            public Object getValue() {
                return null;
            }

            public String toString() {
                return "NULL";
            }
        };

        public T getValue();
    }

    public static final class RefSize {
        private static final Logger log = LoggerFactory.getLogger(RefSize.class);
        private static final RefSize[] REF_CACHE;
        private final int size;

        static {
            RefSize[] cache = new RefSize[17];
            int i = 0;
            while (i < cache.length) {
                cache[i] = new RefSize(i);
                ++i;
            }
            REF_CACHE = cache;
        }

        private RefSize(int typeSize) {
            this.size = typeSize;
        }

        public static RefSize forValue(long value) {
            int bits = value == 0L ? 0 : 63 - Long.numberOfLeadingZeros(value);
            int refsize = 1 << (bits >>> 3);
            return RefSize.ofSize(refsize);
        }

        public static RefSize ofSize(int bytesize) {
            Preconditions.checkCondition(bytesize > 0, "Invalid type reference size %d", bytesize);
            if (bytesize < REF_CACHE.length) {
                return REF_CACHE[bytesize];
            }
            return new RefSize(bytesize);
        }

        public long read(DataInput data) throws IOException {
            long result;
            switch (this.size * 8) {
                case 8: {
                    result = data.readUnsignedByte();
                    break;
                }
                case 16: {
                    result = data.readUnsignedShort();
                    break;
                }
                case 32: {
                    result = data.readInt() & 0xFFFFFFFF;
                    break;
                }
                case 64: {
                    result = data.readLong();
                    break;
                }
                default: {
                    byte[] bytes = new byte[this.size];
                    data.readFully(bytes);
                    return new BigInteger(1, bytes).longValue();
                }
            }
            return result;
        }

        public int size() {
            return this.size;
        }

        public String toString() {
            return "RefSize [" + this.size + "]";
        }

        public int write(DataOutput output, long ref) throws IOException {
            log.trace("write: Type = {}; ref = {}", (Object)this, (Object)ref);
            switch (this.size * 8) {
                case 8: {
                    output.writeByte((int)ref);
                    break;
                }
                case 16: {
                    output.writeShort((int)ref);
                    break;
                }
                case 32: {
                    output.writeInt((int)ref);
                    break;
                }
                case 64: {
                    output.writeLong(ref);
                    break;
                }
                default: {
                    throw new IllegalArgumentException("Unsupported type size " + this.size);
                }
            }
            return this.size;
        }
    }

    public static interface Reference {
        public List<Object> referenceTable();
    }

    public static class ReferenceArray
    extends AbstractList<Object>
    implements ReferenceCollection<Object> {
        private final List<Object> referenceTable;
        int[] valueRefs;
        private int size;

        public ReferenceArray(List<Object> table) {
            this.referenceTable = table;
            this.valueRefs = new int[16];
        }

        public ReferenceArray(List<Object> refTable, int count) {
            this.referenceTable = refTable;
            this.valueRefs = new int[count];
        }

        protected void ensureCapacity(int capacity) {
            if (this.valueRefs.length < capacity) {
                this.valueRefs = Arrays.copyOf(this.valueRefs, Math.max(capacity, this.valueRefs.length * 2));
            }
        }

        protected Object get(Object object) {
            int i = 0;
            while (i < this.size) {
                Object value = this.get(i);
                if (value.equals(object)) {
                    return value;
                }
                ++i;
            }
            return null;
        }

        @Override
        public void addValueRef(long ref) {
            this.setValueRef(this.size, ref);
        }

        @Override
        public boolean contains(Object o) {
            return this.get(o) != null;
        }

        @Override
        public Object get(int i) {
            return this.referenceTable.get(this.valueRefs[i]);
        }

        @Override
        public long getValueRef(int index) {
            return this.valueRefs[index];
        }

        @Override
        public int indexOf(Object o) {
            int i = 0;
            while (i < this.size) {
                Object value = this.get(i);
                if (value.equals(o)) {
                    return i;
                }
                ++i;
            }
            return -1;
        }

        @Override
        public boolean isEmpty() {
            return this.size == 0;
        }

        @Override
        public List<Object> referenceTable() {
            return this.referenceTable;
        }

        @Override
        public void setValueRef(int index, long ref) {
            this.size = Math.max(this.size, index + 1);
            this.ensureCapacity(this.size);
            this.valueRefs[index] = (int)ref;
        }

        @Override
        public int size() {
            return this.size;
        }

        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder("( ");
            int i = 0;
            while (i < this.size) {
                Object value;
                String valueStr;
                if (i > 0) {
                    sb.append(", ");
                }
                String string = valueStr = (value = this.get(i)) == this ? "(this)" : String.valueOf(value);
                if (value instanceof CharSequence) {
                    sb.append('\"').append(valueStr).append('\"');
                } else if (value instanceof ByteBuffer) {
                    sb.append('<').append(BinaryPListUtils.toHexString((ByteBuffer)value, 64)).append('>');
                } else {
                    sb.append(valueStr);
                }
                ++i;
            }
            sb.append(" )");
            return sb.toString();
        }
    }

    public static interface ReferenceCollection<T>
    extends Reference {
        public void addValueRef(long var1);

        public T get(int var1);

        public long getValueRef(int var1);

        public void setValueRef(int var1, long var2);

        public int size();
    }

    public static class ReferenceDictionary
    extends AbstractMap<String, Object>
    implements ReferenceCollection<Map.Entry<String, Object>> {
        int[] keyRefs;
        int[] valueRefs;
        private int size;
        private final List<Object> referenceTable;
        private final Map<String, Integer> keyMapping;

        public ReferenceDictionary(List<Object> refTable) {
            this.referenceTable = refTable;
            this.keyRefs = new int[16];
            this.valueRefs = new int[16];
            this.keyMapping = new HashMap<String, Integer>();
        }

        public ReferenceDictionary(List<Object> refTable, int count) {
            this.referenceTable = refTable;
            this.keyRefs = new int[count];
            this.valueRefs = new int[count];
            this.keyMapping = new HashMap<String, Integer>(count);
        }

        protected void ensureCapacity(int capacity) {
            if (this.valueRefs.length < capacity) {
                int newCapacity = Math.max(capacity, this.valueRefs.length * 2);
                this.valueRefs = Arrays.copyOf(this.valueRefs, newCapacity);
                this.keyRefs = Arrays.copyOf(this.keyRefs, newCapacity);
            }
        }

        public void addKeyRef(long ref) {
            this.setKeyRef(this.size, ref);
        }

        @Override
        public void addValueRef(long ref) {
            this.setValueRef(this.size - 1, ref);
        }

        @Override
        public Set<Map.Entry<String, Object>> entrySet() {
            return new ReferenceEntrySet();
        }

        @Override
        public Map.Entry<String, Object> get(int i) {
            return new AbstractMap.SimpleImmutableEntry<String, Object>(this.getKey(i), this.getValue(i));
        }

        @Override
        public Object get(Object key) {
            Integer index = this.keyMapping.get(key);
            if (index != null) {
                return this.getValue(index);
            }
            int i = this.keyMapping.size();
            while (i < this.size) {
                String aKey = this.getKey(i);
                this.keyMapping.put(aKey, i);
                if (aKey.equals(key)) {
                    return this.getValue(i);
                }
                ++i;
            }
            return null;
        }

        public String getKey(int index) {
            Object key = this.referenceTable.get(this.keyRefs[index]);
            if (key instanceof String) {
                return (String)key;
            }
            throw new IllegalStateException("Got illegal key value " + this.keyRefs[index] + " : (" + key.getClass() + ") " + key);
        }

        public long getKeyRef(int index) {
            return this.keyRefs[index];
        }

        public Object getValue(int index) {
            return this.referenceTable.get(this.valueRefs[index]);
        }

        @Override
        public long getValueRef(int index) {
            return this.valueRefs[index];
        }

        public int indexOf(Object obj) {
            return this.get(obj) != null ? this.keyMapping.get(obj) : -1;
        }

        @Override
        public List<Object> referenceTable() {
            return this.referenceTable;
        }

        public void setKeyRef(int index, long ref) {
            this.size = Math.max(this.size, index + 1);
            this.ensureCapacity(this.size);
            this.keyRefs[index] = (int)ref;
        }

        @Override
        public void setValueRef(int index, long ref) {
            Preconditions.checkCondition(this.keyRefs[index] > 0, "Attempt to insert value without a key", new Object[0]);
            this.size = Math.max(this.size, index + 1);
            this.ensureCapacity(this.size);
            this.valueRefs[index] = (int)ref;
        }

        @Override
        public int size() {
            return this.size;
        }

        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder("{ ");
            int i = 0;
            while (i < this.size) {
                String key = this.getKey(i);
                sb.append('\"').append((Object)key).append('\"');
                sb.append(" = ");
                Object value = this.getValue(i);
                if (value instanceof CharSequence) {
                    sb.append('\"').append(value.toString()).append('\"');
                } else if (value instanceof ByteBuffer) {
                    sb.append('<').append(BinaryPListUtils.toHexString((ByteBuffer)value, 64)).append('>');
                } else {
                    String valueStr = value == this ? "(this)" : String.valueOf(value);
                    sb.append(valueStr);
                }
                sb.append("; ");
                ++i;
            }
            sb.append('}');
            return sb.toString();
        }

        private final class ReferenceEntrySet
        extends AbstractSet<Map.Entry<String, Object>> {
            private ReferenceEntrySet() {
            }

            @Override
            public Iterator<Map.Entry<String, Object>> iterator() {
                return new ReferenceIterator<Map.Entry<String, Object>>(ReferenceDictionary.this);
            }

            @Override
            public int size() {
                return ReferenceDictionary.this.size;
            }
        }
    }

    public static class ReferenceIterator<T>
    implements Iterator<T> {
        private int index = 0;
        private ReferenceCollection<T> refCollection;

        ReferenceIterator(ReferenceCollection<T> collection) {
            this.refCollection = collection;
        }

        @Override
        public boolean hasNext() {
            return this.index < this.refCollection.size();
        }

        @Override
        public T next() {
            if (!this.hasNext()) {
                throw new NoSuchElementException();
            }
            return this.refCollection.get(this.index++);
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException();
        }
    }

    public static class ReferenceObject
    implements Reference,
    PListValue<Object> {
        private final List<Object> referenceTable;
        private final int ref;

        public ReferenceObject(List<Object> objectList, Object v) {
            this.referenceTable = objectList;
            this.ref = objectList.size();
            objectList.add(v);
        }

        public boolean equals(Object obj) {
            if (obj instanceof ReferenceObject) {
                return ((ReferenceObject)obj).ref == this.ref;
            }
            return false;
        }

        @Override
        public Object getValue() {
            return this.referenceTable.get(this.ref);
        }

        public int hashCode() {
            return this.ref;
        }

        public int ref() {
            return this.ref;
        }

        @Override
        public List<Object> referenceTable() {
            return this.referenceTable;
        }

        public void setValue(Object obj) {
            this.referenceTable.set(this.ref, obj);
        }

        public String toString() {
            Object obj = this.getValue();
            return "{" + obj.getClass().getSimpleName() + "@" + System.identityHashCode(obj) + " : " + this.ref + " : " + obj.toString() + "}";
        }
    }

    public static class ReferenceObjectVisitor {
        final Deque<Object> objectsToVisit = new ArrayDeque<Object>();
        final Map<VisitorObject<Object>, ReferenceObject> objectMap = new HashMap<VisitorObject<Object>, ReferenceObject>(4096);
        final List<Object> objectList;
        final Set<Object> visited = Collections.newSetFromMap(new IdentityHashMap());

        ReferenceObjectVisitor(List<Object> objectTable) {
            this.objectList = objectTable;
        }

        protected Object adapt(Object value) {
            return value;
        }

        protected ReferenceObject childRef(ReferenceObject parentRef, Object v) {
            Object obj = this.adapt(v);
            ReferenceObject ref = this.visitObj(obj);
            if (ref.ref() < parentRef.ref()) {
                ref = new ReferenceObject(this.objectList, obj);
                this.objectMap.put(new VisitorObject<Object>(obj), ref);
            }
            return ref;
        }

        void visit(Object object) {
            if (object == null) {
                return;
            }
            this.objectsToVisit.push(object);
            while (!this.objectsToVisit.isEmpty()) {
                Object obj = this.objectsToVisit.pop();
                this.visitElement(obj);
            }
        }

        void visitElement(Object obj) {
            if (obj instanceof Map) {
                this.visitMap((Map)obj);
            } else if (obj instanceof List) {
                this.visitList((List)obj);
            } else if (obj instanceof Set) {
                this.visitSet((Set)obj);
            } else {
                this.visitObj(obj);
            }
        }

        void visitList(List<?> list) {
            ReferenceObject ref = this.visitObj(list);
            if (!this.visited.add(ref)) {
                return;
            }
            ReferenceArray refArray = new ReferenceArray(this.objectList, list.size());
            ref.setValue(refArray);
            int i = 0;
            for (Object value : list) {
                ReferenceObject valueRef = this.childRef(ref, value);
                this.objectsToVisit.add(valueRef.getValue());
                refArray.setValueRef(i++, valueRef.ref());
            }
        }

        void visitMap(Map<String, ?> dict) {
            ReferenceObject ref = this.visitObj(dict);
            if (!this.visited.add(ref)) {
                return;
            }
            ReferenceDictionary refDict = new ReferenceDictionary(this.objectList, dict.size());
            ref.setValue(refDict);
            Set<String> keys = dict.keySet();
            int i = 0;
            for (String key : keys) {
                ReferenceObject keyRef = this.childRef(ref, key);
                this.objectsToVisit.add(key);
                refDict.setKeyRef(i++, keyRef.ref());
            }
            i = 0;
            for (String key : keys) {
                ReferenceObject valueRef = this.childRef(ref, dict.get(key));
                this.objectsToVisit.add(valueRef.getValue());
                refDict.setValueRef(i++, valueRef.ref());
            }
        }

        ReferenceObject visitObj(Object value) {
            Object obj = this.adapt(value);
            VisitorObject<Object> v = new VisitorObject<Object>(obj);
            if (this.objectMap.containsKey(v)) {
                return this.objectMap.get(v);
            }
            ReferenceObject refObj = new ReferenceObject(this.objectList, obj);
            this.objectMap.put(v, refObj);
            return refObj;
        }

        void visitSet(Set<?> set) {
            ReferenceObject ref = this.visitObj(set);
            if (!this.visited.add(ref)) {
                return;
            }
            ReferenceSet refSet = new ReferenceSet(this.objectList, set.size());
            ref.setValue(refSet);
            int i = 0;
            for (Object value : set) {
                ReferenceObject valueRef = this.childRef(ref, value);
                this.objectsToVisit.add(valueRef.getValue());
                refSet.setValueRef(i++, valueRef.ref());
            }
        }

        static class VisitorObject<E> {
            private final E object;
            private final boolean isCollection;
            private final int hashCode;

            VisitorObject(E e) {
                this.object = e;
                this.isCollection = e instanceof Collection || e instanceof Map;
                this.hashCode = this.isCollection ? System.identityHashCode(e) : e.hashCode();
            }

            public int hashCode() {
                return this.hashCode;
            }

            public boolean equals(Object obj) {
                if (obj == this) {
                    return true;
                }
                if (!(obj instanceof VisitorObject)) {
                    return false;
                }
                E o = ((VisitorObject)obj).object;
                if (this.isCollection) {
                    return o == this.object;
                }
                return o == this.object || this.object.equals(o);
            }
        }
    }

    public static class ReferenceSet
    extends AbstractSet<Object>
    implements ReferenceCollection<Object> {
        private final List<Object> referenceTable;
        private final Map<Object, Integer> valueMap;
        int[] valueRefs;
        private int size;

        public ReferenceSet(List<Object> table) {
            this.referenceTable = table;
            this.valueRefs = new int[16];
            this.valueMap = new HashMap<Object, Integer>();
        }

        public ReferenceSet(List<Object> refTable, int count) {
            this.referenceTable = refTable;
            this.valueRefs = new int[count];
            this.valueMap = new HashMap<Object, Integer>(count);
        }

        protected void ensureCapacity(int capacity) {
            if (this.valueRefs.length < capacity) {
                this.valueRefs = Arrays.copyOf(this.valueRefs, Math.max(capacity, this.valueRefs.length * 2));
            }
        }

        @Override
        public void addValueRef(long ref) {
            this.setValueRef(this.size, ref);
        }

        @Override
        public boolean contains(Object o) {
            return this.get(o) != null;
        }

        @Override
        public Object get(int i) {
            return this.referenceTable.get(this.valueRefs[i]);
        }

        public Object get(Object object) {
            Integer index = this.valueMap.get(object);
            if (index != null) {
                return this.get(index);
            }
            int i = this.valueMap.size();
            while (i < this.size) {
                Object value = this.get(i);
                this.valueMap.put(value, i);
                if (value.equals(object)) {
                    return value;
                }
                ++i;
            }
            return null;
        }

        @Override
        public long getValueRef(int index) {
            return this.valueRefs[index];
        }

        public int indexOf(Object o) {
            return this.get(o) != null ? this.valueMap.get(o) : -1;
        }

        @Override
        public boolean isEmpty() {
            return this.size == 0;
        }

        @Override
        public Iterator<Object> iterator() {
            return new ReferenceIterator<Object>(this);
        }

        @Override
        public List<Object> referenceTable() {
            return this.referenceTable;
        }

        @Override
        public void setValueRef(int index, long ref) {
            this.size = Math.max(this.size, index + 1);
            this.ensureCapacity(this.size);
            this.valueRefs[index] = (int)ref;
        }

        @Override
        public int size() {
            return this.size;
        }
    }

    public static class ReferenceTable
    extends AbstractList<Object> {
        protected final List<Object> delegate;

        ReferenceTable() {
            this.delegate = new ArrayList<Object>();
        }

        ReferenceTable(int size) {
            this.delegate = new ArrayList<Object>(size);
        }

        ReferenceTable(List<Object> list) {
            this.delegate = list;
        }

        protected void willRead(int index) {
            Object value = this.delegate.get(index);
            if (value instanceof PListValue) {
                value = ((PListValue)value).getValue();
                this.delegate.set(index, value);
            }
        }

        @Override
        public boolean add(Object e) {
            return this.delegate.add(e);
        }

        @Override
        public boolean addAll(Collection<?> c) {
            return this.delegate.addAll(c);
        }

        @Override
        public Object get(int index) {
            this.willRead(index);
            return this.delegate.get(index);
        }

        @Override
        public Object set(int index, Object element) {
            this.willRead(index);
            return this.delegate.set(index, element);
        }

        @Override
        public int size() {
            return this.delegate.size();
        }

        @Override
        public String toString() {
            int i = 0;
            StringBuilder sb = new StringBuilder();
            for (Object o : this) {
                int ref;
                int n;
                int n2;
                int[] nArray;
                sb.append(i++).append("\t: ");
                if (o instanceof ReferenceDictionary) {
                    sb.append("<Dictionary: keys =");
                    nArray = ((ReferenceDictionary)o).keyRefs;
                    n2 = ((ReferenceDictionary)o).keyRefs.length;
                    n = 0;
                    while (n < n2) {
                        ref = nArray[n];
                        sb.append(" ").append(ref);
                        ++n;
                    }
                    sb.append("; values =");
                    nArray = ((ReferenceDictionary)o).valueRefs;
                    n2 = ((ReferenceDictionary)o).valueRefs.length;
                    n = 0;
                    while (n < n2) {
                        ref = nArray[n];
                        sb.append(" ").append(ref);
                        ++n;
                    }
                    sb.append(">");
                } else if (o instanceof ReferenceArray) {
                    sb.append("<Array:");
                    nArray = ((ReferenceArray)o).valueRefs;
                    n2 = ((ReferenceArray)o).valueRefs.length;
                    n = 0;
                    while (n < n2) {
                        ref = nArray[n];
                        sb.append(" ").append(ref);
                        ++n;
                    }
                    sb.append(">");
                } else if (o instanceof ReferenceSet) {
                    sb.append("<Set:");
                    nArray = ((ReferenceSet)o).valueRefs;
                    n2 = ((ReferenceSet)o).valueRefs.length;
                    n = 0;
                    while (n < n2) {
                        ref = nArray[n];
                        sb.append(" ").append(ref);
                        ++n;
                    }
                    sb.append(">");
                } else {
                    sb.append(o);
                }
                sb.append("\n");
            }
            return sb.toString();
        }
    }

    public static class SeekableDataChannel
    implements DataInput,
    SeekableByteChannel {
        private static final int SHORT_MASK = 65535;
        private static final int BYTE_MASK = 255;
        private ByteBuffer byteBuffer;

        public SeekableDataChannel(ByteBuffer buffer) {
            this.byteBuffer = buffer;
        }

        public SeekableDataChannel(byte[] bytes) {
            this.byteBuffer = ByteBuffer.wrap(bytes);
        }

        private void checkClosed() throws IOException {
            if (this.byteBuffer == null) {
                throw new IOException("Channel is closed");
            }
        }

        @Override
        public void close() throws IOException {
            this.byteBuffer = null;
        }

        @Override
        public boolean isOpen() {
            return this.byteBuffer != null;
        }

        @Override
        public long position() throws IOException {
            this.checkClosed();
            return this.byteBuffer.position();
        }

        @Override
        public SeekableByteChannel position(long newPosition) throws IOException {
            this.checkClosed();
            this.byteBuffer.position((int)newPosition);
            return this;
        }

        @Override
        public int read(ByteBuffer dst) throws IOException {
            this.checkClosed();
            if (dst.remaining() < this.byteBuffer.remaining()) {
                int limit = this.byteBuffer.limit();
                int toRead = dst.remaining();
                this.byteBuffer.limit(this.byteBuffer.position() + toRead);
                try {
                    dst.put(this.byteBuffer);
                }
                finally {
                    this.byteBuffer.limit(limit);
                }
                return toRead;
            }
            int read = this.byteBuffer.remaining();
            dst.put(this.byteBuffer);
            return read;
        }

        @Override
        public boolean readBoolean() throws IOException {
            this.checkClosed();
            try {
                return this.byteBuffer.get() != 0;
            }
            catch (BufferUnderflowException e) {
                throw new EOFException(e.getMessage());
            }
        }

        @Override
        public byte readByte() throws IOException {
            this.checkClosed();
            try {
                return this.byteBuffer.get();
            }
            catch (BufferUnderflowException e) {
                throw new EOFException(e.getMessage());
            }
        }

        @Override
        public char readChar() throws IOException {
            this.checkClosed();
            try {
                return this.byteBuffer.getChar();
            }
            catch (BufferUnderflowException e) {
                throw new EOFException(e.getMessage());
            }
        }

        @Override
        public double readDouble() throws IOException {
            this.checkClosed();
            try {
                return this.byteBuffer.getDouble();
            }
            catch (BufferUnderflowException e) {
                throw new EOFException(e.getMessage());
            }
        }

        @Override
        public float readFloat() throws IOException {
            this.checkClosed();
            try {
                return this.byteBuffer.getFloat();
            }
            catch (BufferUnderflowException e) {
                throw new EOFException(e.getMessage());
            }
        }

        @Override
        public void readFully(byte[] b) throws IOException {
            this.checkClosed();
            try {
                this.byteBuffer.get(b);
            }
            catch (BufferUnderflowException e) {
                throw new EOFException(e.getMessage());
            }
        }

        @Override
        public void readFully(byte[] b, int off, int len) throws IOException {
            this.checkClosed();
            try {
                this.byteBuffer.get(b, off, len);
            }
            catch (BufferUnderflowException e) {
                throw new EOFException(e.getMessage());
            }
        }

        @Override
        public int readInt() throws IOException {
            this.checkClosed();
            try {
                return this.byteBuffer.getInt();
            }
            catch (BufferUnderflowException e) {
                throw new EOFException(e.getMessage());
            }
        }

        @Override
        public String readLine() throws IOException {
            throw new UnsupportedOperationException();
        }

        @Override
        public long readLong() throws IOException {
            this.checkClosed();
            try {
                return this.byteBuffer.getLong();
            }
            catch (BufferUnderflowException e) {
                throw new EOFException(e.getMessage());
            }
        }

        @Override
        public short readShort() throws IOException {
            this.checkClosed();
            try {
                return this.byteBuffer.getShort();
            }
            catch (BufferUnderflowException e) {
                throw new EOFException(e.getMessage());
            }
        }

        @Override
        public int readUnsignedByte() throws IOException {
            this.checkClosed();
            try {
                return this.byteBuffer.get() & 0xFF;
            }
            catch (BufferUnderflowException e) {
                throw new EOFException(e.getMessage());
            }
        }

        @Override
        public int readUnsignedShort() throws IOException {
            this.checkClosed();
            try {
                return this.byteBuffer.getShort() & 0xFFFF;
            }
            catch (BufferUnderflowException e) {
                throw new EOFException(e.getMessage());
            }
        }

        @Override
        public String readUTF() throws IOException {
            return DataInputStream.readUTF(this);
        }

        @Override
        public long size() throws IOException {
            this.checkClosed();
            return this.byteBuffer.capacity();
        }

        @Override
        public int skipBytes(int n) throws IOException {
            this.checkClosed();
            if (n <= 0) {
                return 0;
            }
            int toSkip = Math.min(n, this.byteBuffer.remaining());
            this.byteBuffer.position(this.byteBuffer.position() + toSkip);
            return toSkip;
        }

        @Override
        public SeekableByteChannel truncate(long size) throws IOException {
            this.checkClosed();
            this.byteBuffer.limit((int)size);
            return this;
        }

        @Override
        public int write(ByteBuffer src) throws IOException {
            this.checkClosed();
            if (src.remaining() > this.byteBuffer.remaining()) {
                ByteBuffer newSrc = src.slice();
                int toWrite = this.byteBuffer.remaining();
                newSrc.limit(newSrc.position() + toWrite);
                this.byteBuffer.put(newSrc);
                src.position(src.position() + toWrite);
                return toWrite;
            }
            int written = src.remaining();
            this.byteBuffer.put(src);
            return written;
        }
    }

    public static class Trailer {
        private static final Logger log = LoggerFactory.getLogger(Trailer.class);
        public static final int SIZE = 32;
        private byte[] unused = new byte[5];
        public int sortVersion;
        public RefSize offsetRefSize;
        public RefSize objectRefSize;
        public long objectCount;
        public long topObject;
        public long offsetTableOffset;

        Trailer() {
        }

        Trailer(DataInput data) throws IOException {
            this.read(data);
            this.validate();
            log.trace("read: {}", (Object)this);
        }

        public Trailer(int count, long offset) {
            this.sortVersion = 0;
            this.topObject = 0L;
            this.objectCount = count;
            this.objectRefSize = RefSize.forValue(this.objectCount);
            this.offsetTableOffset = offset;
            this.offsetRefSize = RefSize.forValue(offset);
        }

        void read(DataInput input) throws IOException {
            input.readFully(this.unused);
            this.sortVersion = input.readByte() & 0xFF;
            this.offsetRefSize = RefSize.ofSize(input.readByte() & 0xFF);
            this.objectRefSize = RefSize.ofSize(input.readByte() & 0xFF);
            this.objectCount = input.readLong();
            this.topObject = input.readLong();
            this.offsetTableOffset = input.readLong();
        }

        void validate() {
            Preconditions.checkCondition(this.sortVersion == 0, "Unsupported sort version %s", this.sortVersion);
            Preconditions.checkCondition(this.objectCount < Integer.MAX_VALUE, "Invalid number of objects %d", this.objectCount);
            Preconditions.checkCondition(this.offsetTableOffset < Integer.MAX_VALUE, "Invalid offset table position %d", this.offsetTableOffset);
            Preconditions.checkCondition(this.offsetTableOffset > 8L, "Invalid offset table position %d", this.offsetTableOffset);
            Preconditions.checkCondition(this.objectCount > 0L, "Invalid number of objects %d", this.objectCount);
            Preconditions.checkCondition(this.topObject <= this.objectCount, "Invalid root object position %0x", this.topObject);
        }

        void write(DataOutput output) throws IOException {
            output.write(this.unused);
            output.write(this.sortVersion);
            output.write((byte)this.offsetRefSize.size());
            output.write((byte)this.objectRefSize.size());
            output.writeLong(this.objectCount);
            output.writeLong(this.topObject);
            output.writeLong(this.offsetTableOffset);
        }

        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append("sortVersion: ").append(this.sortVersion);
            sb.append("; offsetRefSize: ").append(this.offsetRefSize.size);
            sb.append("; objectRefSize: ").append(this.objectRefSize.size);
            sb.append("; objectCount: ").append(this.objectCount);
            sb.append("; topObject: ").append(this.topObject);
            sb.append("; offsetTableOffset: ").append(this.offsetTableOffset);
            return sb.toString();
        }
    }

    public static enum Type {
        NULL(0, "Null", true),
        FALSE(8, "False", true),
        TRUE(9, "True", true),
        URLREL(12, "URL (Relative)", true),
        URLABS(13, "URL (Absolute)", true),
        UUID(14, "UUID", true),
        FILL(15, "Fill", true),
        INT(31, "Int", true),
        REAL(47, "Real", true),
        DATE(51, "Date", true),
        DATA(79, "Data", false),
        ASCII(95, "ASCII String", false),
        UTF16(111, "UTF-16 String", false),
        UTF8(127, "UTF-8 String", false),
        UID(143, "UID", true),
        ARRAY(175, "Array", false),
        ORDSET(191, "Ordered Set", false),
        SET(207, "Set", false),
        DICT(223, "Dictionary", false);

        private static final Type[] typeMap;
        boolean fixedLength;
        int mask;
        String name;

        static {
            typeMap = new Type[256];
            Type[] typeArray = Type.values();
            int n = typeArray.length;
            int n2 = 0;
            while (n2 < n) {
                Type aType = typeArray[n2];
                if (aType.mask <= Type.FILL.mask) {
                    Type.typeMap[aType.mask] = aType;
                } else {
                    int i = aType.mask & 0xF0;
                    while (i <= aType.mask) {
                        if ((aType.mask & i) == i) {
                            Type.typeMap[i] = aType;
                        }
                        ++i;
                    }
                }
                ++n2;
            }
        }

        private Type(int valueMask, String aName, boolean length) {
            this.mask = valueMask;
            this.name = aName;
            this.fixedLength = length;
        }

        public static Type forValue(int val) {
            return typeMap[val & 0xFF];
        }

        public int count(DataInput data, int marker) throws IOException {
            int count = marker & 0xF;
            if (count == 15 && !this.isFixedLength()) {
                count = BinaryPListUtils.readCount(data);
            }
            return count;
        }

        public boolean isFixedLength() {
            return this.fixedLength;
        }

        public byte marker(int length) {
            int value = this.value();
            if (value == this.mask) {
                return (byte)value;
            }
            return (byte)(value | (length < 15 ? length : 15));
        }

        public int mask() {
            return this.mask;
        }

        public String toString() {
            return this.name;
        }

        public int value() {
            if ((this.mask & 0xF0) == 0) {
                return this.mask;
            }
            return (this.mask | 0xF) == this.mask ? this.mask & 0xF0 : this.mask;
        }
    }
}

