/*
 * Decompiled with CFR 0.152.
 */
package datastructures;

import geometry.planar.IntPoint;
import geometry.planar.Point;
import geometry.planar.Side;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.Random;
import java.util.Set;
import java.util.TreeSet;

public class PlanarDelaunayTriangulation {
    private final TriangleGraph search_graph;
    private Collection<Edge> degenerate_edges;
    private int last_edge_id_no = 0;
    private static int seed = 99;
    private static Random random_generator = new Random(seed);

    public PlanarDelaunayTriangulation(Collection<Storable> p_object_list) {
        LinkedList<Corner> corner_list = new LinkedList<Corner>();
        for (Storable curr_object : p_object_list) {
            Point[] curr_corners;
            for (Point curr_corner : curr_corners = curr_object.get_triangulation_corners()) {
                corner_list.add(new Corner(curr_object, curr_corner));
            }
        }
        random_generator.setSeed(seed);
        Collections.shuffle(corner_list, random_generator);
        int bounding_coor = 0x2000000;
        Corner[] bounding_corners = new Corner[]{new Corner(null, new IntPoint(bounding_coor, 0)), new Corner(null, new IntPoint(0, bounding_coor)), new Corner(null, new IntPoint(-bounding_coor, -bounding_coor))};
        Edge[] edge_lines = new Edge[3];
        for (int i = 0; i < 2; ++i) {
            edge_lines[i] = new Edge(bounding_corners[i], bounding_corners[i + 1]);
        }
        edge_lines[2] = new Edge(bounding_corners[2], bounding_corners[0]);
        Triangle start_triangle = new Triangle(edge_lines, null);
        for (Edge curr_edge : edge_lines) {
            curr_edge.set_left_triangle(start_triangle);
        }
        this.search_graph = new TriangleGraph(start_triangle);
        this.degenerate_edges = new LinkedList<Edge>();
        for (Corner curr_corner : corner_list) {
            Triangle triangle_to_split = this.search_graph.position_locate(curr_corner);
            this.split(triangle_to_split, curr_corner);
        }
    }

    public Collection<ResultEdge> get_edge_lines() {
        LinkedList<ResultEdge> result = new LinkedList<ResultEdge>();
        for (Edge curr_edge : this.degenerate_edges) {
            result.add(new ResultEdge(curr_edge.start_corner.coor, curr_edge.start_corner.object, curr_edge.end_corner.coor, curr_edge.end_corner.object));
        }
        if (this.search_graph.anchor != null) {
            TreeSet<Edge> result_edges = new TreeSet<Edge>();
            this.search_graph.anchor.get_leaf_edges(result_edges);
            for (Edge curr_edge : result_edges) {
                result.add(new ResultEdge(curr_edge.start_corner.coor, curr_edge.start_corner.object, curr_edge.end_corner.coor, curr_edge.end_corner.object));
            }
        }
        return result;
    }

    private boolean split(Triangle p_triangle, Corner p_corner) {
        Triangle[] containing_edge = null;
        for (int i = 0; i < 3; ++i) {
            Triangle[] curr_edge = p_triangle.edge_lines[i];
            Side curr_side = ((Edge)curr_edge).left_triangle == p_triangle ? p_corner.side_of(curr_edge.start_corner, curr_edge.end_corner) : p_corner.side_of(curr_edge.end_corner, curr_edge.start_corner);
            if (curr_side == Side.ON_THE_RIGHT) {
                System.out.println("PlanarDelaunayTriangulation.split: p_corner is outside");
                return false;
            }
            if (curr_side != Side.COLLINEAR) continue;
            if (containing_edge != null) {
                Corner common_corner = curr_edge.common_corner((Edge)containing_edge);
                if (common_corner == null) {
                    System.out.println("PlanarDelaunayTriangulation.split: common corner expected");
                    return false;
                }
                if (p_corner.object == common_corner.object) {
                    return false;
                }
                this.degenerate_edges.add(new Edge(p_corner, common_corner));
                return true;
            }
            containing_edge = curr_edge;
        }
        if (containing_edge == null) {
            Triangle[] new_triangles = p_triangle.split_at_inner_point(p_corner);
            if (new_triangles == null) {
                return false;
            }
            for (Triangle curr_triangle : new_triangles) {
                this.search_graph.insert(curr_triangle, p_triangle);
            }
            for (int i = 0; i < 3; ++i) {
                this.legalize_edge(p_corner, p_triangle.edge_lines[i]);
            }
        } else {
            Edge curr_edge;
            int i;
            Triangle neighbour_to_split = containing_edge.other_neighbour(p_triangle);
            Triangle[] new_triangles = p_triangle.split_at_border_point(p_corner, neighbour_to_split);
            if (new_triangles == null) {
                return false;
            }
            this.search_graph.insert(new_triangles[0], p_triangle);
            this.search_graph.insert(new_triangles[1], p_triangle);
            this.search_graph.insert(new_triangles[2], neighbour_to_split);
            this.search_graph.insert(new_triangles[3], neighbour_to_split);
            for (i = 0; i < 3; ++i) {
                curr_edge = p_triangle.edge_lines[i];
                if (curr_edge == containing_edge) continue;
                this.legalize_edge(p_corner, curr_edge);
            }
            for (i = 0; i < 3; ++i) {
                curr_edge = neighbour_to_split.edge_lines[i];
                if (curr_edge == containing_edge) continue;
                this.legalize_edge(p_corner, curr_edge);
            }
        }
        return true;
    }

    private boolean legalize_edge(Corner p_corner, Edge p_edge) {
        Triangle triangle_to_change;
        if (p_edge.is_legal()) {
            return false;
        }
        if (p_edge.left_triangle.opposite_corner(p_edge) == p_corner) {
            triangle_to_change = p_edge.right_triangle;
        } else if (p_edge.right_triangle.opposite_corner(p_edge) == p_corner) {
            triangle_to_change = p_edge.left_triangle;
        } else {
            System.out.println("PlanarDelaunayTriangulation.legalize_edge: edge lines inconsistant");
            return false;
        }
        Edge flipped_edge = p_edge.flip();
        this.search_graph.insert(flipped_edge.left_triangle, p_edge.left_triangle);
        this.search_graph.insert(flipped_edge.right_triangle, p_edge.left_triangle);
        this.search_graph.insert(flipped_edge.left_triangle, p_edge.right_triangle);
        this.search_graph.insert(flipped_edge.right_triangle, p_edge.right_triangle);
        for (int i = 0; i < 3; ++i) {
            Edge curr_edge = triangle_to_change.edge_lines[i];
            if (curr_edge == p_edge) continue;
            this.legalize_edge(p_corner, curr_edge);
        }
        return true;
    }

    public boolean validate() {
        boolean result = this.search_graph.anchor.validate();
        if (result) {
            System.out.println("Delauny triangulation check passed ok");
        } else {
            System.out.println("Delauny triangulation check has detected problems");
        }
        return result;
    }

    private int new_edge_id_no() {
        ++this.last_edge_id_no;
        return this.last_edge_id_no;
    }

    private static class TriangleGraph {
        private Triangle anchor = null;

        public TriangleGraph(Triangle p_triangle) {
            if (p_triangle != null) {
                this.insert(p_triangle, null);
            } else {
                this.anchor = null;
            }
        }

        public void insert(Triangle p_triangle, Triangle p_parent) {
            p_triangle.initialize_is_on_the_left_of_edge_line_array();
            if (p_parent == null) {
                this.anchor = p_triangle;
            } else {
                p_parent.children.add(p_triangle);
            }
        }

        public Triangle position_locate(Corner p_corner) {
            if (this.anchor == null) {
                return null;
            }
            if (this.anchor.children.isEmpty()) {
                return this.anchor;
            }
            for (Triangle curr_child : this.anchor.children) {
                Triangle result = this.position_locate_reku(p_corner, curr_child);
                if (result == null) continue;
                return result;
            }
            System.out.println("TriangleGraph.position_locate: containing triangle not found");
            return null;
        }

        private Triangle position_locate_reku(Corner p_corner, Triangle p_triangle) {
            if (!p_triangle.contains(p_corner)) {
                return null;
            }
            if (p_triangle.is_leaf()) {
                return p_triangle;
            }
            for (Triangle curr_child : p_triangle.children) {
                Triangle result = this.position_locate_reku(p_corner, curr_child);
                if (result == null) continue;
                return result;
            }
            System.out.println("TriangleGraph.position_locate_reku: containing triangle not found");
            return null;
        }
    }

    private class Triangle {
        private final Edge[] edge_lines;
        private boolean[] is_on_the_left_of_edge_line = null;
        private Collection<Triangle> children;
        private final Triangle first_parent;

        public Triangle(Edge[] p_edge_lines, Triangle p_first_parent) {
            this.edge_lines = p_edge_lines;
            this.children = new LinkedList<Triangle>();
            this.first_parent = p_first_parent;
        }

        public boolean is_leaf() {
            return this.children.isEmpty();
        }

        public Corner get_corner(int p_no) {
            Corner result;
            if (p_no < 0 || p_no >= 3) {
                System.out.println("Triangle.get_corner: p_no out of range");
                return null;
            }
            Edge curr_edge = this.edge_lines[p_no];
            if (curr_edge.left_triangle == this) {
                result = curr_edge.start_corner;
            } else if (curr_edge.right_triangle == this) {
                result = curr_edge.end_corner;
            } else {
                System.out.println("Triangle.get_corner: inconsistant edge lines");
                result = null;
            }
            return result;
        }

        public Corner opposite_corner(Edge p_edge_line) {
            int edge_line_no = -1;
            for (int i = 0; i < 3; ++i) {
                if (this.edge_lines[i] != p_edge_line) continue;
                edge_line_no = i;
                break;
            }
            if (edge_line_no < 0) {
                System.out.println("Triangle.opposite_corner: p_edge_line not found");
                return null;
            }
            Edge next_edge = this.edge_lines[(edge_line_no + 1) % 3];
            Corner result = next_edge.left_triangle == this ? next_edge.end_corner : next_edge.start_corner;
            return result;
        }

        public boolean contains(Corner p_corner) {
            if (this.is_on_the_left_of_edge_line == null) {
                System.out.println("Triangle.contains: array is_on_the_left_of_edge_line not initialized");
                return false;
            }
            for (int i = 0; i < 3; ++i) {
                Edge curr_edge = this.edge_lines[i];
                Side curr_side = p_corner.side_of(curr_edge.start_corner, curr_edge.end_corner);
                if (!(this.is_on_the_left_of_edge_line[i] ? curr_side == Side.ON_THE_RIGHT : curr_side == Side.ON_THE_LEFT)) continue;
                return false;
            }
            return true;
        }

        public void get_leaf_edges(Set<Edge> p_result_edges) {
            if (this.is_leaf()) {
                for (int i = 0; i < 3; ++i) {
                    Edge curr_edge = this.edge_lines[i];
                    if (curr_edge.start_corner.object == null || curr_edge.end_corner.object == null) continue;
                    p_result_edges.add(curr_edge);
                }
            } else {
                for (Triangle curr_child : this.children) {
                    if (curr_child.first_parent != this) continue;
                    curr_child.get_leaf_edges(p_result_edges);
                }
            }
        }

        public Triangle[] split_at_inner_point(Corner p_corner) {
            Triangle[] new_triangles = new Triangle[3];
            Edge[] new_edges = new Edge[3];
            for (int i = 0; i < 3; ++i) {
                new_edges[i] = new Edge(this.get_corner(i), p_corner);
            }
            Edge[] curr_edge_lines = new Edge[]{this.edge_lines[0], new Edge(this.get_corner(1), p_corner), new Edge(p_corner, this.get_corner(0))};
            new_triangles[0] = new Triangle(curr_edge_lines, this);
            curr_edge_lines = new Edge[]{this.edge_lines[1], new Edge(this.get_corner(2), p_corner), new_triangles[0].edge_lines[1]};
            new_triangles[1] = new Triangle(curr_edge_lines, this);
            curr_edge_lines = new Edge[]{this.edge_lines[2], new_triangles[0].edge_lines[2], new_triangles[1].edge_lines[1]};
            new_triangles[2] = new Triangle(curr_edge_lines, this);
            for (int i = 0; i < 3; ++i) {
                Edge curr_edge = new_triangles[i].edge_lines[0];
                if (curr_edge.get_left_triangle() == this) {
                    curr_edge.set_left_triangle(new_triangles[i]);
                    continue;
                }
                curr_edge.set_right_triangle(new_triangles[i]);
            }
            Edge curr_edge = new_triangles[0].edge_lines[1];
            curr_edge.set_left_triangle(new_triangles[0]);
            curr_edge.set_right_triangle(new_triangles[1]);
            curr_edge = new_triangles[1].edge_lines[1];
            curr_edge.set_left_triangle(new_triangles[1]);
            curr_edge.set_right_triangle(new_triangles[2]);
            curr_edge = new_triangles[2].edge_lines[1];
            curr_edge.set_left_triangle(new_triangles[0]);
            curr_edge.set_right_triangle(new_triangles[2]);
            return new_triangles;
        }

        public Triangle[] split_at_border_point(Corner p_corner, Triangle p_neighbour_to_split) {
            Edge second_common_new_edge;
            Edge first_common_new_edge;
            Triangle[] new_triangles = new Triangle[4];
            int this_touching_edge_no = -1;
            int neigbbour_touching_edge_no = -1;
            Edge touching_edge = null;
            Edge other_touching_edge = null;
            for (int i = 0; i < 3; ++i) {
                Edge curr_edge = this.edge_lines[i];
                if (p_corner.side_of(curr_edge.start_corner, curr_edge.end_corner) == Side.COLLINEAR) {
                    this_touching_edge_no = i;
                    touching_edge = curr_edge;
                }
                curr_edge = p_neighbour_to_split.edge_lines[i];
                if (p_corner.side_of(curr_edge.start_corner, curr_edge.end_corner) != Side.COLLINEAR) continue;
                neigbbour_touching_edge_no = i;
                other_touching_edge = curr_edge;
            }
            if (this_touching_edge_no < 0 || neigbbour_touching_edge_no < 0) {
                System.out.println("Triangle.split_at_border_point: touching edge not found");
                return null;
            }
            if (touching_edge != other_touching_edge) {
                System.out.println("Triangle.split_at_border_point: edges inconsistent");
                return null;
            }
            if (this == touching_edge.left_triangle) {
                first_common_new_edge = new Edge(touching_edge.start_corner, p_corner);
                second_common_new_edge = new Edge(p_corner, touching_edge.end_corner);
            } else {
                first_common_new_edge = new Edge(touching_edge.end_corner, p_corner);
                second_common_new_edge = new Edge(p_corner, touching_edge.start_corner);
            }
            Edge prev_edge = this.edge_lines[(this_touching_edge_no + 2) % 3];
            Edge this_splitting_edge = this == prev_edge.left_triangle ? new Edge(p_corner, prev_edge.start_corner) : new Edge(p_corner, prev_edge.end_corner);
            Edge[] curr_edge_lines = new Edge[]{prev_edge, first_common_new_edge, this_splitting_edge};
            new_triangles[0] = new Triangle(curr_edge_lines, this);
            if (this == prev_edge.left_triangle) {
                prev_edge.set_left_triangle(new_triangles[0]);
            } else {
                prev_edge.set_right_triangle(new_triangles[0]);
            }
            first_common_new_edge.set_left_triangle(new_triangles[0]);
            this_splitting_edge.set_left_triangle(new_triangles[0]);
            Edge next_edge = this.edge_lines[(this_touching_edge_no + 1) % 3];
            curr_edge_lines = new Edge[]{this_splitting_edge, second_common_new_edge, next_edge};
            new_triangles[1] = new Triangle(curr_edge_lines, this);
            this_splitting_edge.set_right_triangle(new_triangles[1]);
            second_common_new_edge.set_left_triangle(new_triangles[1]);
            if (this == next_edge.left_triangle) {
                next_edge.set_left_triangle(new_triangles[1]);
            } else {
                next_edge.set_right_triangle(new_triangles[1]);
            }
            next_edge = p_neighbour_to_split.edge_lines[(neigbbour_touching_edge_no + 1) % 3];
            Edge neighbour_splitting_edge = p_neighbour_to_split == next_edge.left_triangle ? new Edge(next_edge.end_corner, p_corner) : new Edge(next_edge.start_corner, p_corner);
            curr_edge_lines = new Edge[]{neighbour_splitting_edge, first_common_new_edge, next_edge};
            new_triangles[2] = new Triangle(curr_edge_lines, p_neighbour_to_split);
            neighbour_splitting_edge.set_left_triangle(new_triangles[2]);
            first_common_new_edge.set_right_triangle(new_triangles[2]);
            if (p_neighbour_to_split == next_edge.left_triangle) {
                next_edge.set_left_triangle(new_triangles[2]);
            } else {
                next_edge.set_right_triangle(new_triangles[2]);
            }
            prev_edge = p_neighbour_to_split.edge_lines[(neigbbour_touching_edge_no + 2) % 3];
            curr_edge_lines = new Edge[]{prev_edge, second_common_new_edge, neighbour_splitting_edge};
            new_triangles[3] = new Triangle(curr_edge_lines, p_neighbour_to_split);
            if (p_neighbour_to_split == prev_edge.left_triangle) {
                prev_edge.set_left_triangle(new_triangles[3]);
            } else {
                prev_edge.set_right_triangle(new_triangles[3]);
            }
            second_common_new_edge.set_right_triangle(new_triangles[3]);
            neighbour_splitting_edge.set_right_triangle(new_triangles[3]);
            return new_triangles;
        }

        public boolean validate() {
            boolean result = true;
            if (this.is_leaf()) {
                Edge prev_edge = this.edge_lines[2];
                for (int i = 0; i < 3; ++i) {
                    Corner curr_start_corner;
                    Edge curr_edge = this.edge_lines[i];
                    if (!curr_edge.validate()) {
                        result = false;
                    }
                    Corner prev_end_corner = prev_edge.left_triangle == this ? prev_edge.end_corner : prev_edge.start_corner;
                    if (curr_edge.left_triangle == this) {
                        curr_start_corner = curr_edge.start_corner;
                    } else if (curr_edge.right_triangle == this) {
                        curr_start_corner = curr_edge.end_corner;
                    } else {
                        System.out.println("Triangle.validate: edge inconsistent");
                        return false;
                    }
                    if (curr_start_corner != prev_end_corner) {
                        System.out.println("Triangle.validate: corner inconsistent");
                        result = false;
                    }
                    prev_edge = curr_edge;
                }
            } else {
                for (Triangle curr_child : this.children) {
                    if (curr_child.first_parent != this) continue;
                    curr_child.validate();
                }
            }
            return result;
        }

        private void initialize_is_on_the_left_of_edge_line_array() {
            if (this.is_on_the_left_of_edge_line != null) {
                return;
            }
            this.is_on_the_left_of_edge_line = new boolean[3];
            for (int i = 0; i < 3; ++i) {
                this.is_on_the_left_of_edge_line[i] = this.edge_lines[i].left_triangle == this;
            }
        }
    }

    private class Edge
    implements Comparable<Edge> {
        public final Corner start_corner;
        public final Corner end_corner;
        private Triangle left_triangle = null;
        private Triangle right_triangle = null;
        private final int id_no;

        public Edge(Corner p_start_corner, Corner p_end_corner) {
            this.start_corner = p_start_corner;
            this.end_corner = p_end_corner;
            this.id_no = PlanarDelaunayTriangulation.this.new_edge_id_no();
        }

        @Override
        public int compareTo(Edge p_other) {
            return this.id_no - p_other.id_no;
        }

        public void set_left_triangle(Triangle p_triangle) {
            this.left_triangle = p_triangle;
        }

        public Triangle get_left_triangle() {
            return this.left_triangle;
        }

        public void set_right_triangle(Triangle p_triangle) {
            this.right_triangle = p_triangle;
        }

        public Triangle get_right_triangle() {
            return this.right_triangle;
        }

        public Corner common_corner(Edge p_other) {
            Corner result = null;
            if (p_other.start_corner.equals(this.start_corner) || p_other.end_corner.equals(this.start_corner)) {
                result = this.start_corner;
            } else if (p_other.start_corner.equals(this.end_corner) || p_other.end_corner.equals(this.end_corner)) {
                result = this.end_corner;
            }
            return result;
        }

        public Triangle other_neighbour(Triangle p_triangle) {
            Triangle result;
            if (p_triangle == this.left_triangle) {
                result = this.right_triangle;
            } else if (p_triangle == this.right_triangle) {
                result = this.left_triangle;
            } else {
                System.out.println("Edge.other_neighbour: inconsistant neigbour triangle");
                result = null;
            }
            return result;
        }

        public boolean is_legal() {
            if (this.left_triangle == null || this.right_triangle == null) {
                return true;
            }
            Corner left_opposite_corner = this.left_triangle.opposite_corner(this);
            Corner right_opposite_corner = this.right_triangle.opposite_corner(this);
            boolean inside_circle = right_opposite_corner.coor.to_float().inside_circle(this.start_corner.coor.to_float(), left_opposite_corner.coor.to_float(), this.end_corner.coor.to_float());
            return !inside_circle;
        }

        public Edge flip() {
            Triangle new_right_triangle;
            Triangle new_left_triangle;
            Edge flipped_edge = new Edge(this.right_triangle.opposite_corner(this), this.left_triangle.opposite_corner(this));
            Triangle first_parent = this.left_triangle;
            int left_index = -1;
            int right_index = -1;
            for (int i = 0; i < 3; ++i) {
                if (this.left_triangle.edge_lines[i] == this) {
                    left_index = i;
                }
                if (this.right_triangle.edge_lines[i] != this) continue;
                right_index = i;
            }
            if (left_index < 0 || right_index < 0) {
                System.out.println("Edge.flip: edge line inconsistant");
                return null;
            }
            Edge left_prev_edge = this.left_triangle.edge_lines[(left_index + 2) % 3];
            Edge left_next_edge = this.left_triangle.edge_lines[(left_index + 1) % 3];
            Edge right_prev_edge = this.right_triangle.edge_lines[(right_index + 2) % 3];
            Edge right_next_edge = this.right_triangle.edge_lines[(right_index + 1) % 3];
            Edge[] curr_edge_lines = new Edge[]{flipped_edge, left_prev_edge, right_next_edge};
            flipped_edge.left_triangle = new_left_triangle = new Triangle(curr_edge_lines, first_parent);
            if (left_prev_edge.left_triangle == this.left_triangle) {
                left_prev_edge.left_triangle = new_left_triangle;
            } else {
                left_prev_edge.right_triangle = new_left_triangle;
            }
            if (right_next_edge.left_triangle == this.right_triangle) {
                right_next_edge.left_triangle = new_left_triangle;
            } else {
                right_next_edge.right_triangle = new_left_triangle;
            }
            curr_edge_lines = new Edge[]{flipped_edge, right_prev_edge, left_next_edge};
            flipped_edge.right_triangle = new_right_triangle = new Triangle(curr_edge_lines, first_parent);
            if (right_prev_edge.left_triangle == this.right_triangle) {
                right_prev_edge.left_triangle = new_right_triangle;
            } else {
                right_prev_edge.right_triangle = new_right_triangle;
            }
            if (left_next_edge.left_triangle == this.left_triangle) {
                left_next_edge.left_triangle = new_right_triangle;
            } else {
                left_next_edge.right_triangle = new_right_triangle;
            }
            return flipped_edge;
        }

        public boolean validate() {
            int i;
            boolean found;
            boolean result = true;
            if (this.left_triangle == null) {
                if (this.start_corner.object != null || this.end_corner.object != null) {
                    System.out.println("Edge.validate: left triangle may be null only for bounding edges");
                    result = false;
                }
            } else {
                found = false;
                for (i = 0; i < 3; ++i) {
                    if (this.left_triangle.edge_lines[i] != this) continue;
                    found = true;
                    break;
                }
                if (!found) {
                    System.out.println("Edge.validate: left triangle does not contain this edge");
                    result = false;
                }
            }
            if (this.right_triangle == null) {
                if (this.start_corner.object != null || this.end_corner.object != null) {
                    System.out.println("Edge.validate: right triangle may be null only for bounding edges");
                    result = false;
                }
            } else {
                found = false;
                for (i = 0; i < 3; ++i) {
                    if (this.right_triangle.edge_lines[i] != this) continue;
                    found = true;
                    break;
                }
                if (!found) {
                    System.out.println("Edge.validate: right triangle does not contain this edge");
                    result = false;
                }
            }
            return result;
        }
    }

    private static class Corner {
        public final Storable object;
        public final Point coor;

        public Corner(Storable p_object, Point p_coor) {
            this.object = p_object;
            this.coor = p_coor;
        }

        public Side side_of(Corner p_1, Corner p_2) {
            return this.coor.side_of(p_1.coor, p_2.coor);
        }
    }

    public static class ResultEdge {
        public final Point start_point;
        public final Storable start_object;
        public final Point end_point;
        public final Storable end_object;

        private ResultEdge(Point p_start_point, Storable p_start_object, Point p_end_point, Storable p_end_object) {
            this.start_point = p_start_point;
            this.start_object = p_start_object;
            this.end_point = p_end_point;
            this.end_object = p_end_object;
        }
    }

    public static interface Storable {
        public Point[] get_triangulation_corners();
    }
}

