﻿using System.Collections.Generic;
using System.IO;
using UnityEditor;
using UnityEngine;

public class Validate {
    [MenuItem("Automachef SDK/Validate Mod")]
    static void ValidateMod() {
        List<string> internalNames = ValidateInternalNames();
        List<string> friendlyNames = ValidateFriendlyNames();
        List<string> levels = ValidateLevels();
        List<string> machines = ValidateMachines();
        List<string> recipes = ValidateRecipes();
        List<string> icons = ValidateIcons();
        List<string> assetBundles = ValidateAssetBundles();
        List<string> specialEvents = ValidateSpecialEvents();
        List<string> limits = ValidateLimits();

        string validationResult = "VALIDATION RESULT:\n\n" +
            "Duplicate Internal Names Check: " + (internalNames.Count == 0 ? "PASS" : "FAIL!") + "\n" +
            "Friendly Names Check: " + (friendlyNames.Count == 0 ? "PASS" : "FAIL!") + "\n" +
            "Levels Check: " + (levels.Count == 0 ? "PASS" : "FAIL!") + "\n" +
            "Machines Check: " + (machines.Count == 0 ? "PASS" : "FAIL!") + "\n" +
            "Recipes Check: " + (recipes.Count == 0 ? "PASS" : "FAIL!") + "\n" +
            "Icons Check: " + (icons.Count == 0 ? "PASS" : "FAIL!") + "\n" +
            "Asset Bundles Check: " + (assetBundles.Count == 0 ? "PASS" : "FAIL!") + "\n" +
            "Special Events Check: " + (specialEvents.Count == 0 ? "PASS" : "FAIL!") + "\n" +
            "Limits Check: " + (limits.Count == 0 ? "PASS" : "FAIL!");

        List<string> allErrors = new List<string>();
        allErrors.AddRange(internalNames);
        allErrors.AddRange(friendlyNames);
        allErrors.AddRange(levels);
        allErrors.AddRange(machines);
        allErrors.AddRange(recipes);
        allErrors.AddRange(icons);
        allErrors.AddRange(assetBundles);
        allErrors.AddRange(specialEvents);
        allErrors.AddRange(limits);

        if (allErrors.Count > 0) {
            validationResult = validationResult + "\n\nERRORS:\n\n" +
                string.Join("\n", allErrors);
        }

        EditorUtility.DisplayDialog("Validation Result", validationResult, "Okay");
    }

    static List<string> ValidateInternalNames() {
        List<string> errors = new List<string>();
        string[] ids = AssetDatabase.FindAssets("t:GameObject", new [] { "Assets/Data" });
        List<string> internalNames = new List<string>();

        internalNames.AddRange(InternalNames.GetIngredients());
        internalNames.AddRange(InternalNames.GetDishes());
        internalNames.AddRange(InternalNames.GetLiquidIngredients());

        foreach (string id in ids) {
            string path = AssetDatabase.GUIDToAssetPath(id);
            GameObject gameObject = (GameObject) AssetDatabase.LoadAssetAtPath(path, typeof(GameObject));
            UGCIngredient ingredient = gameObject.GetComponent<UGCIngredient>();
            if (ingredient == null) {
                continue;
            }

            if (internalNames.Contains(ingredient.internalName)) {
                errors.Add("Duplicate internal name " + ingredient.internalName);
            }
            internalNames.Add(ingredient.internalName);
        }

        return errors;
    }

    static List<string> ValidateLevels() {
        List<string> errors = new List<string>();
        string[] ids = AssetDatabase.FindAssets("t:UGCLevel", new [] { "Assets/Data" });
        if (ids.Length == 0) {
            errors.Add("You need to create at least one level for the mod to be playable");
        }
        return errors;
    }

    static List<string> ValidateRecipes() {
        List<string> errors = new List<string>();
        string[] ingredientIds = AssetDatabase.FindAssets("t:GameObject", new [] { "Assets/Data" });
        List<string> internalNames = new List<string>();

        internalNames.AddRange(InternalNames.GetIngredients());
        internalNames.AddRange(InternalNames.GetDishes());
        internalNames.AddRange(InternalNames.GetLiquidIngredients());

        foreach (string id in ingredientIds) {
            string path = AssetDatabase.GUIDToAssetPath(id);
            GameObject gameObject = (GameObject) AssetDatabase.LoadAssetAtPath(path, typeof(GameObject));
            UGCIngredient ingredient = gameObject.GetComponent<UGCIngredient>();
            if (ingredient == null) {
                continue;
            }

            internalNames.Add(ingredient.internalName);
        }

        foreach (string id in ingredientIds) {
            string path = AssetDatabase.GUIDToAssetPath(id);
            GameObject gameObject = (GameObject) AssetDatabase.LoadAssetAtPath(path, typeof(GameObject));
            UGCIngredient ingredient = gameObject.GetComponent<UGCIngredient>();
            if (ingredient == null) {
                continue;
            }

            foreach (string recipeItem in ingredient.recipe) {
                if (!internalNames.Contains(recipeItem)) {
                    errors.Add("Recipe for " + ingredient.internalName + " references non-existing ingredient " + recipeItem);
                }
            }
            foreach (string recipeLiquidItem in ingredient.recipeLiquidIngredients) {
                if (!internalNames.Contains(recipeLiquidItem)) {
                    errors.Add("Recipe for " + ingredient.internalName + " references non-existing liquid ingredient " + recipeLiquidItem);
                }
            }

            UGCDish dish = gameObject.GetComponent<UGCDish>();
            if (dish == null) {
                continue;
            }

            if (ingredient.recipe.Count < 1) {
                errors.Add("Dish " + ingredient.internalName + " does not contain any items in its recipe");
            }

            if (ingredient.technique != UGCManufacturingTechnique.Assembling && ingredient.technique != UGCManufacturingTechnique.Packaging) {
                errors.Add("Dish " + ingredient.internalName + " has an invalid manufacturing technique, it should only be Assembling or Packaging");
            }
        }

        return errors;
    }

    static List<string> ValidateIcons() {
        List<string> errors = new List<string>();
        string[] ids = AssetDatabase.FindAssets("t:GameObject", new [] { "Assets/Data" });

        foreach (string id in ids) {
            string path = AssetDatabase.GUIDToAssetPath(id);
            GameObject gameObject = (GameObject) AssetDatabase.LoadAssetAtPath(path, typeof(GameObject));
            UGCIngredient ingredient = gameObject.GetComponent<UGCIngredient>();
            if (ingredient == null) {
                continue;
            }

            if (ingredient.icon == null) {
                errors.Add("Ingredient " + ingredient.internalName + " has no icon set");
            }
        }

        return errors;
    }

    static List<string> ValidateAssetBundles() {
        List<string> errors = new List<string>();
        string[] assetBundleContents = AssetDatabase.GetAssetPathsFromAssetBundle("mod");
        string[] gameObjectIds = AssetDatabase.FindAssets("t:GameObject", new [] { "Assets/Data" });
        string[] levelIds = AssetDatabase.FindAssets("t:UGCLevel", new [] { "Assets/Data" });
        string[] specialEventsIds = AssetDatabase.FindAssets("t:UGCSpecialEvent", new [] { "Assets/Data" });
        string[] textureIds = AssetDatabase.FindAssets("t:Texture", new [] { "Assets/Textures" });

        List<string> assetBundleContentsList = new List<string>();
        assetBundleContentsList.AddRange(assetBundleContents);

        List<string> ids = new List<string>();
        ids.AddRange(gameObjectIds);
        ids.AddRange(levelIds);
        ids.AddRange(specialEventsIds);
        ids.AddRange(textureIds);

        foreach (string id in ids) {
            string path = AssetDatabase.GUIDToAssetPath(id);
            if (!assetBundleContentsList.Contains(path)) {
                errors.Add("Asset at " + path + " might have a misconfigured Asset Bundle");
            }
        }

        return errors;
    }

    static List<string> ValidateSpecialEvents() {
        List<string> errors = new List<string>();
        string[] ingredientIds = AssetDatabase.FindAssets("t:GameObject");
        string[] specialEventsIds = AssetDatabase.FindAssets("t:UGCSpecialEvent", new [] { "Assets/Data" });
        List<string> internalNames = new List<string>();

        internalNames.AddRange(InternalNames.GetDishes());
        foreach (string id in ingredientIds) {
            string path = AssetDatabase.GUIDToAssetPath(id);
            GameObject gameObject = (GameObject) AssetDatabase.LoadAssetAtPath(path, typeof(GameObject));
            UGCDish dish = gameObject.GetComponent<UGCDish>();
            if (dish == null) {
                continue;
            }

            internalNames.Add(dish.internalName);
        }

        foreach (string id in specialEventsIds) {
            string path = AssetDatabase.GUIDToAssetPath(id);
            UGCSpecialEvent specialEvent = (UGCSpecialEvent) AssetDatabase.LoadAssetAtPath(path, typeof(UGCSpecialEvent));
            if (specialEvent.type == UGCSpecialEventType.Craze || specialEvent.type == UGCSpecialEventType.FoodCritic) {
                if (!internalNames.Contains(specialEvent.orderToSpawn)) {
                    errors.Add("Special event at " + path + " should have a valid Order To Spawn set");
                }
            }
        }

        return errors;
    }

    static List<string> ValidateMachines() {
        List<string> errors = new List<string>();
        List<string> allMachines = InternalNames.GetMachines();
        string[] levelIds = AssetDatabase.FindAssets("t:UGCLevel", new [] { "Assets/Data" });

        foreach (string id in levelIds) {
            string path = AssetDatabase.GUIDToAssetPath(id);
            UGCLevel level = (UGCLevel) AssetDatabase.LoadAssetAtPath(path, typeof(UGCLevel));
            foreach (string machineName in level.availableMachines) {
                if (!allMachines.Contains(machineName)) {
                    errors.Add("Level at " + path + " references a non-existing machine " + machineName);
                }
            }

            if (level.availableMachines.Count < 5) {
                errors.Add("Level at " + path + " has fewer than five available machines, are you sure this is what you want to do?");
            }
        }

        return errors;
    }

    static List<string> ValidateFriendlyNames() {
        List<string> errors = new List<string>();
        string[] ids = AssetDatabase.FindAssets("t:GameObject", new [] { "Assets/Data" });

        foreach (string id in ids) {
            string path = AssetDatabase.GUIDToAssetPath(id);
            GameObject gameObject = (GameObject) AssetDatabase.LoadAssetAtPath(path, typeof(GameObject));
            UGCIngredient ingredient = gameObject.GetComponent<UGCIngredient>();
            if (ingredient == null) {
                continue;
            }

            if (ingredient.friendlyName.Length == 0) {
                errors.Add("Ingredient " + ingredient.internalName + " has no friendly name set");
            }
        }

        return errors;
    }

    static List<string> ValidateLimits() {
        List<string> errors = new List<string>();
        string[] ids = AssetDatabase.FindAssets("t:GameObject", new [] { "Assets/Data" });
        int ingredientsCount = 0;

        foreach (string id in ids) {
            string path = AssetDatabase.GUIDToAssetPath(id);
            GameObject gameObject = (GameObject) AssetDatabase.LoadAssetAtPath(path, typeof(GameObject));
            UGCIngredient ingredient = gameObject.GetComponent<UGCIngredient>();
            if (ingredient == null) {
                continue;
            }

            ingredientsCount++;

            if (ingredient.friendlyName.Length > 30) {
                errors.Add("Ingredient " + ingredient.internalName + " has a friendly name over 30 characters long");
            }

            if (ingredient.recipe.Count > 10) {
                errors.Add("Ingredient " + ingredient.internalName + " has a recipe with more than 10 items");
            }
        }

        if (ingredientsCount > 50) {
            errors.Add("The mod contains more than 50 ingredients");
        }

        string[] levelIds = AssetDatabase.FindAssets("t:UGCLevel", new [] { "Assets/Data" });
        if (levelIds.Length > 10) {
            errors.Add("The mod contains more than 10 levels");
        }

        foreach (string id in levelIds) {
            string path = AssetDatabase.GUIDToAssetPath(id);
            UGCLevel level = (UGCLevel) AssetDatabase.LoadAssetAtPath(path, typeof(UGCLevel));
            if (level.friendlyName.Length > 30) {
                errors.Add("Level " + level.friendlyName + " has a friendly name over 30 characters long");
            }
        }

        return errors;
    }
}