#define HIMETRIC_INCH				2540
#define MAP_LOGHIM_TO_PIX(x,ppli)	(((ppli)*(x) + HIMETRIC_INCH/2) / HIMETRIC_INCH)

#include <SDL.h>
#include <olectl.h>
#include <png.h>

#include "xgui.h"

bool xgui_active = false; 
int xgui_alpha = 0;

SDL_Texture* xgui_overlay = 0;
SDL_Texture* xgui_font = 0;

int xgui_font_cw = 0;
int xgui_font_ch = 0;

struct xguiControl {
	int type;
	int x, y;
	bool hover;
	int click;
	bool checked;
	bool visible;
	char text[256];
	SDL_Texture* texture;
	int texture_w;
	int texture_h;
	int page;
};

#define XGUI_MAX_CONTROLS	256

xguiControl xgui_list[XGUI_MAX_CONTROLS];

int xgui_list_length = 0;

int xguiConfirm = 0;
int xgui_to_confirm = -1;

int xgui_current_page = 0;



SDL_Texture* xgui_load_jpg(const char* filename,SDL_Renderer* renderer)
{
	SDL_Surface* surface;
	IPicture *pic = NULL;
	unsigned int* ScreenData;
	OLE_XSIZE_HIMETRIC iwdt, ihgt;
	int wdt, hgt;
	int i, j, rgb;
	unsigned int *dst;
	BITMAPINFOHEADER bmi = { 0 };

	//open file

	HANDLE hFile = CreateFile(filename, GENERIC_READ, 0, NULL, OPEN_EXISTING, 0, NULL);

	if (hFile == INVALID_HANDLE_VALUE) return 0;

	DWORD dwFileSize = GetFileSize(hFile, NULL);

	if (dwFileSize == (DWORD)-1)
	{
		CloseHandle(hFile);

		return 0;
	}

	HGLOBAL hGlobal = GlobalAlloc(GMEM_MOVEABLE, dwFileSize);

	if (hGlobal == NULL)
	{
		CloseHandle(hFile);

		return 0;
	}

	LPVOID pvData = GlobalLock(hGlobal);

	if (pvData == NULL)
	{
		GlobalUnlock(hGlobal);
		CloseHandle(hFile);

		return 0;
	}

	DWORD dwBytesRead = 0;

	BOOL bRead = ReadFile(hFile, pvData, dwFileSize, &dwBytesRead, NULL);

	GlobalUnlock(hGlobal);
	CloseHandle(hFile);

	if (!bRead) return 0;

	LPSTREAM pStream = NULL;

	HRESULT hr = CreateStreamOnHGlobal(hGlobal, TRUE, &pStream);

	if (!(SUCCEEDED(hr)))
	{
		if (pStream != NULL) pStream->Release();

		return 0;
	}

	if (pStream == NULL) return 0;

	hr = OleLoadPicture(pStream, dwFileSize, FALSE, IID_IPicture, (void **)&pic);

	pStream->Release();

	pic->get_Width(&iwdt);
	pic->get_Height(&ihgt);

	HDC hScreenDC = CreateDC("DISPLAY", NULL, NULL, NULL);
	HDC hMemoryDC = CreateCompatibleDC(hScreenDC);

	wdt = MAP_LOGHIM_TO_PIX(iwdt, GetDeviceCaps(hScreenDC, LOGPIXELSX));
	hgt = MAP_LOGHIM_TO_PIX(ihgt, GetDeviceCaps(hScreenDC, LOGPIXELSX));

	HBITMAP hBitmap = CreateCompatibleBitmap(hScreenDC, wdt, hgt);

	SelectObject(hMemoryDC, hBitmap);

	pic->Render(hMemoryDC, 0, 0, wdt, hgt, 0, ihgt, iwdt, -ihgt, NULL);
	pic->Release();

	bmi.biSize = sizeof(BITMAPINFOHEADER);
	bmi.biPlanes = 1;
	bmi.biBitCount = 32;
	bmi.biWidth = wdt;
	bmi.biHeight = -hgt;
	bmi.biCompression = BI_RGB;
	bmi.biSizeImage = 0;

	ScreenData = (unsigned int*)malloc(wdt * hgt * sizeof(unsigned int));

	GetDIBits(hMemoryDC, hBitmap, 0, hgt, ScreenData, (BITMAPINFO*)&bmi, DIB_RGB_COLORS);

	surface = SDL_CreateRGBSurface(0, wdt, hgt, 32, 0, 0, 0, 0);

	SDL_LockSurface(surface);

	for (i = 0; i < hgt; ++i)
	{
		dst = ((unsigned int*)surface->pixels) + i * surface->w;

		for (j = 0; j < wdt; ++j)
		{
			rgb = ScreenData[i*wdt + j];

			*dst++ = rgb;
		}

	}

	SDL_UnlockSurface(surface);

	SDL_Texture* tx = SDL_CreateTextureFromSurface(renderer, surface);

	SDL_FreeSurface(surface);
	free(ScreenData);
	DeleteDC(hMemoryDC);
	DeleteObject(hBitmap);
	GlobalFree(hGlobal);

	return tx;
}

SDL_Texture* xgui_load_png(const char* filename, SDL_Renderer* renderer)
{
	int width, height;
	png_byte color_type;
	png_byte bit_depth;
	png_bytep *row_pointers;

	FILE *fp = fopen(filename, "rb");

	if (!fp) return NULL;

	png_structp png = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);

	if (!png)
	{
		fclose(fp);
		return NULL;
	}

	png_infop info = png_create_info_struct(png);
	if (!info)
	{
		fclose(fp);
		return NULL;
	}

	if (setjmp(png_jmpbuf(png)))
	{
		fclose(fp);
		return NULL;
	}

	png_init_io(png, fp);

	png_read_info(png, info);

	width = png_get_image_width(png, info);
	height = png_get_image_height(png, info);
	color_type = png_get_color_type(png, info);
	bit_depth = png_get_bit_depth(png, info);

	if (bit_depth == 16) png_set_strip_16(png);

	if (color_type == PNG_COLOR_TYPE_PALETTE) png_set_palette_to_rgb(png);

	if (color_type == PNG_COLOR_TYPE_GRAY && bit_depth < 8) png_set_expand_gray_1_2_4_to_8(png);

	if (png_get_valid(png, info, PNG_INFO_tRNS)) png_set_tRNS_to_alpha(png);

	if (color_type == PNG_COLOR_TYPE_RGB ||
		color_type == PNG_COLOR_TYPE_GRAY ||
		color_type == PNG_COLOR_TYPE_PALETTE)
		png_set_filler(png, 0xFF, PNG_FILLER_AFTER);

	if (color_type == PNG_COLOR_TYPE_GRAY ||
		color_type == PNG_COLOR_TYPE_GRAY_ALPHA)
		png_set_gray_to_rgb(png);

	png_read_update_info(png, info);

	row_pointers = (png_bytep*)malloc(sizeof(png_bytep) * height);
	for (int y = 0; y < height; y++) {
		row_pointers[y] = (png_byte*)malloc(png_get_rowbytes(png, info));
	}

	png_read_image(png, row_pointers);

	fclose(fp);


	Uint32 bmask = 0x00ff0000;
	Uint32 gmask = 0x0000ff00;
	Uint32 rmask = 0x000000ff;
	Uint32 amask = 0xff000000;

	SDL_Surface* surface = SDL_CreateRGBSurface(0, width, height, 32, rmask, gmask, bmask, amask);

	SDL_LockSurface(surface);

	for (int y = 0; y < height; y++) {
		png_bytep row = row_pointers[y];
		unsigned int* dst = ((unsigned int*)surface->pixels) + y * surface->w;
		for (int x = 0; x < width; x++) {
			png_bytep px = &(row[x * 4]);
			unsigned int rgba = (px[0]) + (px[1] << 8) + (px[2] << 16) + (px[3] << 24);
			*dst++ = rgba;
		}
	}

	SDL_UnlockSurface(surface);

	SDL_Texture* tx = SDL_CreateTextureFromSurface(renderer, surface);

	SDL_FreeSurface(surface);

	png_destroy_read_struct(&png, &info, NULL);

	free(row_pointers);

	return tx;
}



SDL_Texture* xgui_load_bmp(const char* filename, SDL_Renderer* renderer)
{
	SDL_Surface* image = SDL_LoadBMP(filename);
	SDL_Texture* texture = SDL_CreateTextureFromSurface(renderer, image);
	SDL_FreeSurface(image);

	return texture;
}



void xgui_init(const char* dir, SDL_Renderer* renderer)
{
	char path[1024];

	xgui_active = false;
	xgui_alpha = 0;
	xgui_current_page = 0;

	snprintf(path, sizeof(path), "%sres/overlay", dir);

	SDL_SetHint(SDL_HINT_RENDER_SCALE_QUALITY, "1");	//backdrop filtered

	xgui_overlay = xgui_load_png(path, renderer);

	snprintf(path, sizeof(path), "%sres/font", dir);

	SDL_SetHint(SDL_HINT_RENDER_SCALE_QUALITY, "0");	//font not filtered

	xgui_font = xgui_load_png(path, renderer);

	SDL_QueryTexture(xgui_font, 0, 0, &xgui_font_cw, &xgui_font_ch);

	xgui_font_cw /= 16;
	xgui_font_ch /= 8;
}



void xgui_done(void)
{
	if (xgui_overlay) SDL_DestroyTexture(xgui_overlay);

	if (xgui_font) SDL_DestroyTexture(xgui_font);
}



void xgui_enable(bool e)
{
	xgui_active = e;

	if (!e) xgui_visible(xguiConfirm, false);
	if (e) xgui_select_page(0);
}


bool xgui_is_enabled(void)
{
	return xgui_active;
}



void xgui_update(int w, int h, int mouse_x, int mouse_y, int mouse_button)
{
	const int delta = 8;

	if (xgui_active)
	{
		xgui_alpha += delta;

		if (xgui_alpha > 255) xgui_alpha = 255;
	}
	else
	{
		xgui_alpha -= delta;

		if (xgui_alpha < 0) xgui_alpha = 0;
	}

	for (int id = 0; id < xgui_list_length; ++id)
	{
		if (xgui_list[id].type == XGUI_NONE) continue;
		if (xgui_list[id].type == XGUI_STATIC) continue;
		if (xgui_list[id].type == XGUI_PICTURE) continue;

		if (xgui_list[id].page >= 0 && xgui_list[id].page != xgui_current_page) continue;

		if (!xgui_list[id].visible) continue;

		SDL_Rect r = xgui_put_str(0, w, h, xgui_list[id].x, xgui_list[id].y, xgui_list[id].text);

		xgui_list[id].hover = false;

		if (!(mouse_button & 1)) xgui_list[id].click = 0;

		r.x -= xgui_font_cw / 4;
		r.w += xgui_font_cw / 2;
		r.y -= xgui_font_ch / 8;
		r.h += xgui_font_ch / 4;

		if (mouse_x >= r.x&&mouse_x < r.x + r.w)
		{
			if (mouse_y >= r.y&&mouse_y < r.y + r.h)
			{
				xgui_list[id].hover = true;

				if (mouse_button & 1)
				{
					if (id == xguiConfirm)
					{
						if (xgui_to_confirm >= 0)
						{
							if (!xgui_list[xgui_to_confirm].click) xgui_list[xgui_to_confirm].click = 1;
						}
					}
					else
					{
						if (xgui_list[id].type != XGUI_BUTTON_CONFIRM)
						{
							if (!xgui_list[id].click) xgui_list[id].click = 1;
						}
						else
						{
							xgui_to_confirm = id;

							int cx = xgui_list[id].x;

							if (cx < 1920 / 2) cx += r.w + xgui_font_cw/2; else cx -= 200;

							xgui_pos(xguiConfirm, cx, xgui_list[id].y);

							xgui_visible(xguiConfirm, true);

							mouse_button = 0;
						}
					}
				}
			}
		}

		if (xgui_list[id].type == XGUI_CHECKBOX)
		{
			xgui_list[id].text[0] = xgui_list[id].checked ? 5 : 32;
		}
	}

	if (mouse_button & 1) xgui_visible(xguiConfirm, false);
}



void xgui_text_color(unsigned int color)
{
	SDL_SetTextureColorMod(xgui_font, color & 0xff, (color >> 8) & 0xff, (color >> 16) & 0xff);
}



SDL_Rect xgui_put_char(SDL_Renderer* renderer, int w, int h, int sx, int sy, char c)
{
	SDL_Rect srcr, dstr;

	srcr.w = xgui_font_cw;
	srcr.h = xgui_font_ch;
	srcr.x = c % 16 * srcr.w;
	srcr.y = c / 16 * srcr.h;

	dstr.w = 1920 * h / 1080;
	dstr.h = h;
	dstr.x = (w - dstr.w) / 2;
	dstr.y = 0;

	dstr.x += sx * dstr.w / 1920;
	dstr.y += sy * dstr.h / 1080;
	dstr.w = srcr.w * dstr.w / 1920;
	dstr.h = srcr.h * dstr.h / 1080;

	if(renderer) SDL_RenderCopy(renderer, xgui_font, &srcr, &dstr);

	return dstr;
}



SDL_Rect xgui_put_str(SDL_Renderer* renderer, int w, int h, int sx, int sy, const char* str)
{
	SDL_Rect dstr;

	dstr.x = sx;
	dstr.y = sy;
	dstr.w = 0;

	while (1)
	{
		int c = *str++;

		if (!c) break;

		SDL_Rect r = xgui_put_char(renderer, w, h, sx, sy, c);

		dstr.h = r.h;
		
		sx += 30;
	}

	dstr.w = sx - dstr.x;

	return dstr;
}



void xgui_render(SDL_Renderer* renderer, int w, int h)
{
	SDL_Rect srcr, dstr;

	if (xgui_alpha == 0) return;

	SDL_SetTextureAlphaMod(xgui_overlay, xgui_alpha);
	SDL_SetTextureAlphaMod(xgui_font, xgui_alpha);

	SDL_SetTextureBlendMode(xgui_overlay, SDL_BLENDMODE_BLEND);
	SDL_SetTextureBlendMode(xgui_font, SDL_BLENDMODE_BLEND);

	dstr.w = 1920 * h / 1080;	//overlay rect
	dstr.h = h;
	dstr.x = (w - dstr.w) / 2;
	dstr.y = 0;

	srcr.x = 0;
	srcr.y = 0;
	srcr.w = 1920;
	srcr.h = 1080;
	
	SDL_RenderCopy(renderer, xgui_overlay, &srcr, &dstr);

	for (int id = 0; id < xgui_list_length; ++id)
	{
		if (xgui_list[id].type == XGUI_NONE) continue;

		if (xgui_list[id].page >= 0 && xgui_list[id].page != xgui_current_page) continue;

		if (!xgui_list[id].visible) continue;

		if (xgui_list[id].type == XGUI_PICTURE)
		{
			SDL_Rect dstr;

			dstr.w = 1920 * h / 1080;
			dstr.h = h;
			dstr.x = (w - dstr.w) / 2;
			dstr.y = 0;

			dstr.x += (xgui_list[id].x - xgui_list[id].texture_w / 2) * dstr.w / 1920;
			dstr.y += (xgui_list[id].y - xgui_list[id].texture_h / 2) * dstr.h / 1080;
			dstr.w = xgui_list[id].texture_w * dstr.w / 1920;
			dstr.h = xgui_list[id].texture_h * dstr.h / 1080;

			if (renderer)
			{
				SDL_SetTextureAlphaMod(xgui_list[id].texture, xgui_alpha);

				SDL_RenderCopy(renderer, xgui_list[id].texture, 0, &dstr);
			}

			continue;
		}

		xgui_text_color(xgui_list[id].hover ? RGB(255, 255, 0) : RGB(255, 255, 255));

		xgui_put_str(renderer, w, h, xgui_list[id].x, xgui_list[id].y, xgui_list[id].text);
	}
}



void xgui_clear(void)
{
	xgui_list_length = 0;

	xguiConfirm = xgui_add(XGUI_BUTTON, -1, 0, 0, "Sure?");

	xgui_visible(xguiConfirm, false);

	xgui_to_confirm = -1;
}



void xgui_picture_load(int id, SDL_Renderer* renderer, const char* filename)
{
	if (xgui_list_length >= XGUI_MAX_CONTROLS) return;

	if (xgui_list[id].texture)
	{
		SDL_DestroyTexture(xgui_list[id].texture);
	}

	xgui_list[id].texture = xgui_load_bmp(filename, renderer);

	if (xgui_list[id].texture)
	{
		SDL_SetTextureBlendMode(xgui_list[id].texture, SDL_BLENDMODE_BLEND);

		SDL_QueryTexture(xgui_list[id].texture, 0, 0, &xgui_list[id].texture_w, &xgui_list[id].texture_h);
	}
}



int xgui_add(int type, int page, int x, int y, const char* text)
{
	if (xgui_list_length >= XGUI_MAX_CONTROLS) return -1;

	memset(&xgui_list[xgui_list_length], 0, sizeof(xguiControl));

	xgui_list[xgui_list_length].type = type;
	xgui_list[xgui_list_length].page = page;
	xgui_list[xgui_list_length].x = x;
	xgui_list[xgui_list_length].y = y;
	xgui_list[xgui_list_length].visible = true;

	if (type != XGUI_PICTURE)
	{
		if (type == XGUI_CHECKBOX) xgui_list[xgui_list_length].text[0] = ' ';

		strncat(xgui_list[xgui_list_length].text, text, sizeof(xgui_list[xgui_list_length].text));
	}

	++xgui_list_length;

	return xgui_list_length - 1;
}



void xgui_visible(int id, bool visible)
{
	if (id < 0 || id >= XGUI_MAX_CONTROLS) return;

	xgui_list[id].visible = visible;
}



void xgui_pos(int id, int x, int y)
{
	if (id < 0 || id >= XGUI_MAX_CONTROLS) return;

	xgui_list[id].x = x;
	xgui_list[id].y = y;
}




bool xgui_is_down(int id)
{
	if (id < 0 || id >= XGUI_MAX_CONTROLS) return false;

	if (xgui_list[id].click == 1)
	{
		xgui_list[id].click = 2;
		return true;
	}

	return false;
}



void xgui_set_check(int id, bool check)
{
	if (id < 0 || id >= XGUI_MAX_CONTROLS) return;

	xgui_list[id].checked = check;
}



int xgui_is_checked(int id)
{
	if (id < 0 || id >= XGUI_MAX_CONTROLS) return 0;

	return xgui_list[id].checked;
}



void xgui_set_text(int id, const char* text)
{
	if (id < 0 || id >= XGUI_MAX_CONTROLS) return;

	strncpy(xgui_list[id].text, text, sizeof(xgui_list[id].text));
}




void xgui_select_page(int page)
{
	xgui_current_page = page;
}