/*------------------------------------------------------------------------------*
 *	File Name: 	ocmsp.h															*
 *	Purpose: shared header between OC and VC, all exported functions should be	*
 *  listed here
 *  Creation: CPY																*
 *  	Copyright (c) OriginLab Corp.	2007									*
 *	All Rights Reserved															*
 *					                                                        	*
 *	Modification log                                                        	*
 *	Cloud 11/12/2007 FIND_PEAKS_2ND_DERIVATIVE_MOVED_FROM_OCMATH				*
 *	Jack 11/18/2008 EXPOSE_FFT_SMOOTH_CUTOFF_FREQUENCY_FOR_PA					*
 *  Fisher 04/24/09 QA80-13487  ADV_DIGITAL_FILTERING                           *
 *	Fisher 7/16/09	REORGANIZE_PRO_ONLY_OCMATH_FUNCTION							*
 *	Sophy 8/14/2009 ADD_FLEXIBLE_PARAM_FOR_MORE_USER_OPTIONS					*
 *	Sophy 11/5/2009 QA80-14598-S4 OUTPUT_MORE_RESULT								*
 *	Sophy 2/24/2011 ORG-2317 RISETIME_SUPPORT_FINDING_MIN_MAX_SEPARATELY			*
 *	Rex 2012/2/17 ORG-4117 IIR_DIGITAL_FILTER_DESIGN								*
 *	Rex 2012/3/28 ORG-5371-S1 READ_WRITE_MATLAB_FCF_FILE							*
 *	Rex 2012/4/19 ORG-4117 OCMSP_ERROR_CODES										*
 *	Rex 2012/5/14 ORG-5575 IIR_SOS_FILTER_DESIGN_AND_ANALYSIS_FUNCS				*
 *	Rex 2012/6/26 ORG-5846-P5 FIX_ERROR_WHEN_ORDER_TOO_LARGE						*
 *	Sophy 6/29/2012 ORG-4782-P2 MOVE_2ND_DERIVATIVE_FROM_OCMSP_TO_OCMATH_FOR_REGULAR_BUILD
 *------------------------------------------------------------------------------*/

#ifndef _OCMSP_H
#define _OCMSP_H

#ifdef _MSC_VER
	#ifdef ocmathsp_DLL
		#define OC_API __declspec(dllexport)
		#pragma message("$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$we are in ocmathsp dll")
	#else
		#define OC_API __declspec(dllimport)
		#pragma message("$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$we are NOT in ocmathsp dll, but some other DLL")
	#endif
	
	#define uint UINT

	#ifdef __cplusplus
		extern "C" {
			#pragma message("$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$ extern C in ocm.h")
	#endif
#else
	#define OC_API
	#pragma dll(ocmathsp)	// Associate all functions below to omath2.dll which must be in the Origin EXE folder
#endif	//_MSC_VER

///Rex 2012/4/19 ORG-4117 OCMSP_ERROR_CODES	
enum{
	OCMSP_NOERROR			= OE_NOERROR,
	OCMSP_NULL_POINTER		= OE_NULL_POINTER,			//"null pointer"
	OCMSP_IMPROPER_SYSTEM	= OE_IMPROPER_SYSTEM,		//"number of zeros > number of poles, it's not a proper system"
	OCMSP_INVALID_ORDER		= OE_INVALID_ORDER,			//"order must be greater than zero"
	OCMSP_INVALID_ODD_ORDER	= OE_INVALID_ODD_ORDER,		//"order should be even when bandpass and bandstop"
	OCMSP_INVALID_FREQ		= OE_INVALID_FREQ,			//"frequency should be (0 1) when digital and > 0 when analog
	OCMSP_INVALID_BANDFREQS	= OE_INVALID_BANDFREQS,		//"frequencies for pass and stop band are not valid"
	OCMSP_INVALID_RIPPLE	= OE_INVLAID_RIPPLE,		//"ripple should be > 0.0"
	OCMSP_INVALID_BANDRIPPLES= OE_INVALID_BANDRIPPLES,	//"Rs should be greater than Rp"
	OCMSP_UNSUPPORTED_TYPE	= OE_UNSUPPORTED_TYPE,		//"unsupported filter type"
	OCMSP_OPEN_FILE_FAILED	= OE_OPEN_FILE_FAILED,		//"open file failed"	
	OCMSP_READ_FILE_FAILED	= OE_READ_FILE_FAILED,		//"read file failed"	
	OCMSP_WRITE_FILE_FAILED	= OE_WRITE_FILE_FAILED,		//"write file failed"	
	OCMSP_INVALID_DEN0		= OE_INVALID_DEN0,			//"den[0] cannot be zero"
	OCMSP_ZEROS_MORETHAN_POLES = OE_ZEROS_MORETHAN_POLES,	//"the number of zeros cannot be greater than number of poles"
	OCMSP_INVALID_ORDER_TOO_LARGE =  OE_INVALID_ORDER_TOO_LARGE //"exceed maximum filter order allowed" ///----Rex 2012/6/26 ORG-5846-P5 FIX_ERROR_WHEN_ORDER_TOO_LARGE
};
///End OCMSP_ERROR_CODES	
///Sophy 6/29/2012 ORG-4782-P2 MOVE_2ND_DERIVATIVE_FROM_OCMSP_TO_OCMATH_FOR_REGULAR_BUILD
///// Cloud 11/12/2007 ADD_FFT_FILTER_SMOOTHING
//// Comment last updated by Cloud on 06/05/2008
///** >Analysis>Signal Processing
//	Remarks:
//		Smoomth data by using FFT low-pass filter
//
//	Example1:
//		void ocmsp_fft_filter_smoothing_ex1()
//		{
//			vector vx = {0.86528, 0.39649, 0.44901, 0.77041, 0.43139, 0.11343, 0.12061, 0.00387, 0.21913, 0.66058};
//			int nSize = vx.GetSize();
//			vector vxs(nSize);
//			ocmsp_fft_filter_smoothing(nSize, vx, vxs, 0.4);
//			for (int ii=0; ii<10; ii++)
//				printf ("%lf\n", vxs[ii]);
//		}
//
//	Parameters:
//		nSize = [input] size of the data
//		px = [input] pointer to the data to perform smoothing
//		pxs = [output] pointer to the buffer of result data
//		dPercent = [input] the cutoff frequency represented by percentile
//
//	Return:
//		Return OE_NOERROR if succeed, otherwise, non-zero error code is returned.
//*/
//OC_API int ocmsp_fft_filter_smoothing(uint nSize, const double* px, double* pxs, double dPercent);
///// End ADD_FFT_FILTER_SMOOTHING
///end MOVE_2ND_DERIVATIVE_FROM_OCMSP_TO_OCMATH_FOR_REGULAR_BUILD
/// Cloud 11/12/2007 FIND_PEAKS_2ND_DERIVATIVE_MOVED_FROM_OCMATH
// Comment last updated by Cloud on 06/05/2008
/** >Analysis>Peak and Baseline
    Remarks:
		Find curve's peaks by 2nd derivative. Calculate curve's smoothed 2nd derivative on every point in px, py.
		Then the point is a peak, if the smoothed 2nd derivative on it is extremum. 
		POSITIVE_DIRECTION peak, if the smoothed 2nd derivative on it is local maximum.
		NEGATIVE_DIRECTION peak, if the smoothed 2nd derivative on it is local minimum.
		lsize returns the number of peaks, pxPeaks and pyPeaks contain X and Y coordinate's datas of peaks,and 
		pnIndices contains the indices of peaks.
	Keywords:
		2nd derivative
	Example1:
		void ocmsp_find_peaks_2nd_derivative_ex1( )
		{
			GraphLayer gl = Project.ActiveLayer();
			if (!gl)
			{
				return;
			}
			
			DataPlot dp = gl.DataPlots(0);		
			DataRange dr;
			vector vxData, vyData;
	        if(dp.GetDataRange(dr))
	        {
	        	DWORD dwPlotID;
	        	if(dr.GetData(DRR_GET_DEPENDENT | DRR_NO_FACTORS, 0, &dwPlotID, NULL, &vyData, &vxData) < 0)
	        	{
	        		printf("get_plot_data failed GetData");
	        		return;
	        	}
	        }

			uint nDataSize = vxData.GetSize();
			int iSize = vxData.GetSize();

			vector vxPeaks, vyPeaks;
			vector<int> vnIndices;
	
			vxPeaks.SetSize(nDataSize);
			vyPeaks.SetSize(nDataSize);
			vnIndices.SetSize(nDataSize);

			int nRet = ocmsp_find_peaks_2nd_derivative( &nDataSize, vxData, vyData, vxPeaks, vyPeaks, vnIndices, POSITIVE_DIRECTION | NEGATIVE_DIRECTION,11);
			if( nRet < OE_NOERROR )
			{
				printf("error code: %d\n", nRet);
				return;
			}
			vxPeaks.SetSize(nDataSize);
			vyPeaks.SetSize(nDataSize);
			vnIndices.SetSize(nDataSize);
			WorksheetPage wksPage;
			wksPage.Create();
			Worksheet wksResult = wksPage.Layers(0);
			int nIndCol, nXCol, nYCol;
			nIndCol = wksResult.AddCol("Indices");
			nXCol = wksResult.AddCol("X Coordinate");
			nYCol = wksResult.AddCol("Y Coordinate");
			wksResult.Columns(nIndCol).SetType(OKDATAOBJ_DESIGNATION_X);
			wksResult.Columns(nXCol).SetType(OKDATAOBJ_DESIGNATION_X);
			wksResult.Columns(nYCol).SetType(OKDATAOBJ_DESIGNATION_Y);
			DataRange drOut;
			drOut.Add("X", wksResult, 0, nIndCol, -1, nIndCol);
			drOut.Add("Y", wksResult, 0, nXCol, -1, nXCol);
			drOut.Add("Z", wksResult, 0, nYCol, -1, nYCol);
			drOut.SetData(&vyPeaks, &vxPeaks, &vnIndices);
			XYRange plotRange;
			plotRange.Add("X", wksResult, 0, nXCol, -1, nXCol);
			plotRange.Add("Y", wksResult, 0, nYCol, -1, nYCol);
			gl.AddPlot(plotRange, IDM_PLOT_SCATTER);
		}
		// At the end of example:
	Parameters:
		lSize = [modify] on input, size of px, py, pxPeaks, pyPeaks, pnIndices; on output, return number of peaks  
		px = [input] it contains curve's X coordinate's datas 
        py = [input] it contains curve's Y coordinate's datas 
		pxPeaks = [output] Its size is equal to the size of px. it contains peaks' X coordinate's datas, 
		                   if the number of peaks is less than its size, the excrescent elements of it are filled
						   with 0.
		pyPeaks = [output] Its size is equal to the size of py. it contains peaks' Y coordinate's datas, 
						   if the number of peaks is less than its size, the excrescent elements of it are filled
						   with 0.
		pnIndices = [output] Its size is equal to the size of px. it contains peaks' indices, if peaks's number is 
							 less than its size, the excrescent elements of it are filled with 0.
		dwCtrl = [input]  combine of POSITIVE_DIRECTION, NEGATIVE_DIRECTION,  KEEP_OPPOSITE_SIDE
		nPtsSmooth = [input] smooth points number
	Return:
		Return OE_NOERROR if succeed, otherwise, non-zero error code is returned.
*/
///Jack 11/18/2008 EXPOSE_FFT_SMOOTH_CUTOFF_FREQUENCY_FOR_PA 
// sandy need it for adjusment in PA GUI
//OC_API int ocmsp_find_peaks_2nd_derivative(UINT* lSize, const double* px, const double* py, double* pxPeaks, 
//												double* pyPeaks,  int* pnIndices, DWORD dwCtrl = POSITIVE_DIRECTION, int nPtsSmooth = 11);
///Jack 11/19/2008 OUTPUT_SECOND_DERIVATIVE
//OC_API int ocmsp_find_peaks_2nd_derivative(UINT* lSize, const double* px, const double* py, double* pxPeaks, 
//												double* pyPeaks,  int* pnIndices, DWORD dwCtrl = POSITIVE_DIRECTION, int nPtsSmooth = 11, double dCutoffFre = 0.2);
///Sophy 6/29/2012 ORG-4782-P2 MOVE_2ND_DERIVATIVE_FROM_OCMSP_TO_OCMATH_FOR_REGULAR_BUILD
//OC_API int ocmsp_find_peaks_2nd_derivative(UINT* lSize, const double* px, const double* py, double* pxPeaks, 
//												double* pyPeaks,  int* pnIndices, DWORD dwCtrl = POSITIVE_DIRECTION, double dCutoffFre = 0.2, double * pSecondDeriv=NULL);
///end MOVE_2ND_DERIVATIVE_FROM_OCMSP_TO_OCMATH_FOR_REGULAR_BUILD
//End OUTPUT_SECOND_DERIVATIVE
///End EXPOSE_FFT_SMOOTH_CUTOFF_FREQUENCY_FOR_PA 
///End FIND_PEAKS_2ND_DERIVATIVE_MOVED_FROM_OCMATH

/// end EXPORT_MACRO_MISSING

/// Fisher 11/10/2008 QA80-12507 	OSCILLOSCOPE_SIGNAL_PROCESSING_FEATURES
struct RiseTimeResult
{
   double dRiseTime;
   int it1, it2;
   double t1, t2;
   double y1, y2;
   int i1y1, i1y2, i2y1, i2y2; 
   double b1, b2;
   ///Sophy 11/5/2009 QA80-14598-S4 OUTPUT_MORE_RESULT
   double velocity;
   double max_velocity;
   ///end OUTPUT_MORE_RESULT
};
///Sophy 8/27/2009 SUPPORT_CUSTOMER_SPECIFIED_LOW_HIGHT_STATE_LEVEL
enum {
	CTRL_FIX_LOWLEVEL = 0x0001,
	CTRL_FIX_HIGHLEVEL = 0x0002,
};
///end SUPPORT_CUSTOMER_SPECIFIED_LOW_HIGHT_STATE_LEVEL

/// Fisher 11/24/09 QA80-14719 PROPER_LOW_HIGN_LEVEL_FINDING
enum{
	MINMAX_AVERAGE = 0,
	MINMAX_MAXIMAL_MEAN,
};
/// End PROPER_LOW_HIGN_LEVEL_FINDING

///Sophy 8/14/2009 ADD_FLEXIBLE_PARAM_FOR_MORE_USER_OPTIONS
//later if we need more parameters to pass into this function, just add them in this struct
typedef	struct tagRiseTimeParams {
	double	dLowLevelVolt;			//user specified, if dwCtrl contains CTRL_FIX_LOWLEVEL
	double	dHighLevelVolt;			//user specified, if dwCtrl contains CTRL_FIX_HIGHLEVEL
	double	dLowLevelRefer;			//from of step height by percentage
	double	dHighLevelRefer;		//to of step height by percentage
	int		nMethod;				//method to find rise/fall area, should be one of linear/histogram/largest triangle
	double	dTolerance;				//tolerance value to find  rise/fall area
	int		nMinMaxFindingMethod;	/// Fisher 11/24/09 QA80-14719 PROPER_LOW_HIGN_LEVEL_FINDING
	int		nMinMaxFindingMethodEx;	///Sophy 2/24/2011 ORG-2317 RISETIME_SUPPORT_FINDING_MIN_MAX_SEPARATELY
	int		nSmoothPoints;
	int		nSmoothPointsEx;
	int		nBins;
	BOOL	bAveVal; //use average value of the bin that has the maximum number of hits
	DWORD	dwCtrl;			
}RiseTimeParams;
///end ADD_FLEXIBLE_PARAM_FOR_MORE_USER_OPTIONS
enum{
	RTIME_LINEAR_SEARCH = 0 ,
	RTIME_HISTOGRAM,
	RTIME_LOCAL_EXTREME,
};

/*
	The following return values are needed:

	1. OE_NOERROR = Success
	2. OE_LACKPOINTS = nSize too few points
	3. OE_NO_BASE_FOUND = No base found
	4. OE_ONLY_ONE_BASE_FOUND = Only one base found 

	bForward == TRUE		Find Risetime
	bForward == FALSE		Find Falltime
 */

///Sophy 8/14/2009 ADD_FLEXIBLE_PARAM_FOR_MORE_USER_OPTIONS
//OC_API int ocmsp_find_fall_or_fall_time( UINT nSize, const double* px, const double* py, RiseTimeResult *pstResult, double dReferenceLevelLow=0.1, double dReferenceLevelHigh=0.9, int nMethod = RTIME_LINEAR_SEARCH, double dTol=0.03, int nSmoothPoints = 100, int nSmoothPointsEx = 10, int nBins = 256, BOOL bForward = TRUE);		
OC_API int ocmsp_find_fall_or_fall_time( UINT nSize, const double* px, const double* py, RiseTimeResult *pstResult, RiseTimeParams *pParams, BOOL bForward = TRUE);		
///end ADD_FLEXIBLE_PARAM_FOR_MORE_USER_OPTIONS
/// End 	OSCILLOSCOPE_SIGNAL_PROCESSING_FEATURES



/// Fisher 04/24/09 QA80-13487  ADV_DIGITAL_FILTERING
//#ifdef __ADV_DIGITAL_FILTERING__
enum DECIMATE
{
	DEC_FILTER_NONE,
	DEC_FILTER_MOVING_AVE,
	DEC_FILTER_FIR,
	DEC_FILTER_IIR,
};

///Rex 2012/2/17 ORG-4117 IIR_DIGITAL_FILTER_DESIGN
// /*
// 	filter function
// 
// 	y = filter(b,a,X) filters the data in vector X with the filter described 
// 	by numerator coefficient vector b and denominator coefficient vector a. 
// 
// 	The filter function is implemented as a direct form II transposed structure
// 	
// */
// OC_API int	ocmsp_1d_digital_filter(const double *pIn, double *pOut, UINT nSize, double *pa, UINT na, double *pb, UINT nb);
// 
// /*
// 	Zero-phase digital filtering
// */
// OC_API int	ocmsp_1d_digital_filtfilt(const double *pIn, double *pOut, UINT nSize, double *pa, UINT na, double *pb, UINT nb);
///End IIR_DIGITAL_FILTER_DESIGN

// This function returns the new size after decimation, fail if return < 0 
OC_API int	ocmsp_decimate(const double *px, const double *py, UINT nSize, UINT d, double *pxd, double *pyd, int nFilterType=DEC_FILTER_NONE, UINT nOrder=0, UINT nBegin=0);

//#endif // __ADV_DIGITAL_FILTERING__
/// End ADV_DIGITAL_FILTERING

/// Fisher 7/16/09	REORGANIZE_PRO_ONLY_OCMATH_FUNCTION
OC_API int ocmsp_reduce_data(int nSize, const double* px, const double* py, UINT* pIndices, UINT* nIndSize, int nMin, int nMax, int nMethod, double dTol=0.0, BOOL bByTol=FALSE);
/// End	REORGANIZE_PRO_ONLY_OCMATH_FUNCTION

///Rex 2012/2/17 ORG-4117 IIR_DIGITAL_FILTER_DESIGN
enum {
	OMSP_BUTTERWORTH	= 0,
	OMSP_CHEBYSHEV1		= 1,
	OMSP_CHEBYSHEV2		= 2,
	OMSP_ELLIPTIC		= 3,
	OMSP_BESSEL			= 4,

	OMSP_LOWPASS		= 0,
	OMSP_HIGHPASS		= 1 << 16,
	OMSP_BANDSTOP		= 2 << 16,
	OMSP_BANDPASS		= 3 << 16,

	OMSP_DIGITAL		= 0,
	OMSP_ANALOG			= 4 << 16,

	OMSP_PROTOTYPE_MASK	= 7,
	OMSP_BAND_MASK		= 3 << 16,
	OMSP_ANALOG_DIGITAL_MASK = 4 << 16,
};

#define OMSP_FILTER_PROTOTYPE(type)	((type) & OMSP_PROTOTYPE_MASK)
#define OMSP_FILTER_BANDTYPE(type)	((type) & OMSP_BAND_MASK)
#define OMSP_FILTER_ADTYPE(type)	((type) & OMSP_ANALOG_DIGITAL_MASK)

// Comment last updated by Rex on 2012/7/24
// Comment last updated by Rex on 2012/3/20
/** 
    Remarks:
		Estimate the minimum order of a digital/analog IIR filter required to meet a set
		of filter design specifications: loses no more than Rp dB in passband and has at
		least Rs dB of attenuation in the stopband.
	Keywords:
		filter order estimation
	Example1:
		void ocmsp_iir_order_ex1()
		{
			int type = OMSP_BUTTERWORTH|OMSP_BANDPASS|OMSP_DIGITAL;
			double Wn[2] = {0.0, 0.0}, Wp[2] = {60.0/500, 200.0/500}, Ws[2] = {50.0/500, 250.0/500};
			double Rp = 3, Rs = 40;
			UINT N = 0;

			int nRet = 0;
			if (0 != (nRet = ocmsp_iir_order(&N, &Wn[0], Wp[0], Ws[0], Rp, Rs, type, &Wn[1], Wp[1], Ws[1])))
			{
				printf("ocmsp_iir_order failed, error code=%d\n", nRet);
				return;
			}

			printf("N = %d, Wn = [%lf, %lf]\n", N, Wn[0], Wn[1]);
		}
		// At the end of example:
	Parameters:
		pN	= [output] returned lowest order N
		pWn	= [output] natural frequency of the filter, for filter design with ocmsp_IIR_filter
		Wp	= [input] passband edge frequency, it's in radians/second if is analog type, and should be normalized
			to [0, 1] if is digital(where 1 corresponds to pi radians/sample)
		Ws	= [input] stopband edge frequency, it's in radians/second if is analog type, and should be normalized
			to [0, 1] if is digital(where 1 corresponds to pi radians/sample)
		Rp	= [input] passband ripple in dB. It's the maximum permissible passband loss in decibels
		Rs	= [input] stopband attenuation in dB. It's the number of decibels the stopband is down from the passband
		type= [input] filter type. It specifies the filter prototype(Butterworth, Chebshev1, Chebshev2,	Elliptic 
			or Bessel), bandtype(lowpass, highpass, bandstop or band pass) and digital or analog type. For example
			type = OMSP_BUTTERWORTH|OMSP_LOWPASS|OMSP_DIGITAL means a lowpass Butterworth digital filter.
		pWn2= [output] natural frequency 2, only needed when it's a bandpass or bandstop filter
		Wp2	= [input][optional] passband edge frequency, it's in radians/second if is analog type, and should be 
			normalized to [0, 1] if is digital(where 1 corresponds to pi radians/sample). Only needed when it's a 
			bandpass or bandstop filter
		Ws2	= [input][optional] stopband edge frequency, it's in radians/second if is analog type, and should be 
			normalized to [0, 1] if is digital(where 1 corresponds to pi radians/sample). Only needed when it's a 
			bandpass or bandstop filter

			The stopband and passband frequency range should follow:
			Filter type	|	Stopband/Passband conditions	|		Stopband		|	Passband
			Lowpass		|		Wp < Ws 					|		(Ws, 1)			|	(0, Wp)
			Highpass	|		Wp > Ws 					|		(0, Ws)			|	(Wp, 1)
			Bandpass	|	Ws < Wp < Wp2 < Ws2				| (0, Ws) and (Ws2, 1)	|	(Wp, Wp2)
			Bandstop	|	Wp < Ws < Ws2 < Wp2				| (0, Wp) and (Wp2, 1)	|	(Ws, Ws2)
			
	Return:
		Return OE_NOERROR if succeed, otherwise, non-zero error code is returned(OE_NULL_POINTER, OE_INVALID_FREQ,
		OE_INVALID_BANDFREQS, OE_INVLAID_RIPPLE, OE_INVALID_BANDRIPPLES or OE_INVALID_TYPE)
*/
OC_API int ocmsp_iir_order(UINT* pN, double* pWn, double Wp, double Ws, double Rp, double Rs, int type, 
						   double* pWn2=NULL, double Wp2=-1.0, double Ws2=-1.0);
#define ocmsp_IIR_order ocmsp_iir_order

// Comment last updated by Rex on 2012/7/24
// Comment last updated by Rex on 2012/3/20
/**
	Remarks:
		Design digital or analog IIR filters with given specifications. It returns the coefficients 
		of transfer function:
						 b(0) + b(1)z^(-1) + ... + b(n)z^(-n)
				 H(z) = --------------------------------------
						   1 + a(1)z^(-1) + ... + a(n)z^(-n)
	Keywords:
		IIR filter design
	Example1:
		void ocmsp_iir_filter_tf_ex1()
		{
			UINT N = 18; //even number if is bandpass or bandstop
			double Wn[2] = {300.0/500, 400.0/500};
			vector A(N+1), B(N+1); //size is N+1
			int type = OMSP_BUTTERWORTH|OMSP_BANDPASS|OMSP_DIGITAL;

			int nRet = 0;
			if (0 != (nRet = ocmsp_iir_filter_tf(N, Wn[0], B, A, type, 0.0, 0.0, Wn[1])))
			{
				printf("ocmsp_iir_filter_tf failed, error code=%d\n", nRet);
				return;
			}

			int ii;
			printf("b= ");
			for(ii = 0; ii < B.GetSize(); ++ii)
			printf("\t%lf\n", B[ii]);
			printf("a= ");
			for(ii = 0; ii < A.GetSize(); ++ii)
			printf("\t%lf\n", A[ii]);
		}
		// At the end of example:
	Parameters:
		N	= [input] filter order, should be even if type is bandpass or bandstop
		Wn	= [input] cutoff or passband/stopband edge frequency, it's in radians/second if is analog type, and 
			should be normalized to [0, 1] if is digital type(where 1 corresponds to half the sample rate)
		pB	= [output] returned filter numerator coefficients, size N+1
		pA	= [output] returned filter denominator coefficients, size N+1
		type= [intput] filter type, it specifies the filter prototype(Butterworth, Chebshev1, Chebshev2, Elliptic 
			or Bessel), bandtype(lowpass, highpass, bandstop or band pass) and digital or analog type. For example
			type = OMSP_BUTTERWORTH|OMSP_LOWPASS|OMSP_DIGITAL means a lowpass Butterworth digital filter.
		Rp	= [input][optional] passband ripple (dB), needed when it's Chebshev1 or Ellip type
		Rs	= [input][optional] stopband attenuation (dB), needed when it's Chebshev2 or Ellip type
		Wn2	= [input][optional] cutoff or passband/stopband-edge frequency.  Needed 	when it's a bandpass or 
			bandstop filter. It's in radians/second if is analog, and should be normalized to [0, 1] if is digital 
			type(where 1 corresponds to half the sample rate).
	Return:
		Return OE_NOERROR if succeed, otherwise, non-zero error code is returned(OE_NULL_POINTER, OE_INVALID_FREQ,
		OE_INVALID_BANDFREQS, OE_INVLAID_RIPPLE, OE_INVALID_BANDRIPPLES or OE_INVALID_TYPE)
*/
OC_API int ocmsp_iir_filter(UINT N, double Wn, double* pB, double* pA, int type, 
							double Rp=0.0, double Rs=0.0, double Wn2=-1.0);
#define ocmsp_IIR_filter ocmsp_iir_filter
#define ocmsp_iir_filter_tf ocmsp_iir_filter


///Rex 2012/5/14 ORG-5575 IIR_SOS_FILTER_DESIGN_AND_ANALYSIS_FUNCS
// Comment last updated by Rex on 2012/7/24
// Comment last updated by Rex on 2012/5/17
/**
	Remarks:
		Design digital or analog IIR filters with given specifications. It returns filter represented in state-space 
		form:
				 x = Ax + Bu
				 y = Cx + Du
	Keywords:
		IIR filter design
	Example1:
		void ocmsp_iir_filter_ss_ex1()
		{
			UINT N = 18; //even number if bandpass or bandstop
			double Wn[2] = {300.0/500, 400.0/500};
			matrix A(N, N); //size NxN
			vector B(N), C(N); //size N
			double d;
			int type = OMSP_BUTTERWORTH|OMSP_BANDPASS|OMSP_DIGITAL;

			int nRet = 0;
			if (0 != (nRet = ocmsp_iir_filter_ss(N, Wn[0], A, B, C, &d, type, 0.0, 0.0, Wn[1])))
			{
				printf("ocmsp_iir_filter_ss failed, error code=%d\n", nRet);
				return;
			}
		}
		// At the end of example:
	Parameters:
		N	= [input] filter order, should be even if is bandpass or bandstop type
		Wn	= [input] cutoff or passband/stopband edge frequency, it's in radians/second if is analog type, and should
			be normalized to [0, 1] if is digital type, where 1 corresponds to half the sample rate
		pA	= [output] returned matrix A, size N-by-N
		pB	= [output] returned vector B, size N
		pC	= [output] returned vector C, size N
		pd	= [output] returned scalar d
		type= [intput] filter type, it specifies the filter prototype(Butterworth, Chebshev1, Chebshev2, Elliptic 
			or Bessel), bandtype(lowpass, highpass, bandstop or band pass) and digital or analog type. For example
			type = OMSP_BUTTERWORTH|OMSP_LOWPASS|OMSP_DIGITAL means a lowpass Butterworth digital filter.
		Rp	= [input][optional] passband ripple (dB), needed when it's Chebshev1 or Ellip type
		Rs	= [input][optional] stopband attenuation (dB), needed when it's Chebshev2 or Ellip type
		Wn2	= [input][optional] cutoff or passband/stopband-edge frequency. Needed when it's a bandpass or bandstop 
			filter. It's in radians/second if is analog type, and should be normalized to [0, 1] if is digital type
			(where 1 corresponds to half the sample rate). 
	Return:
		Return OE_NOERROR if succeed, otherwise, non-zero error code is returned(OE_NULL_POINTER, OE_INVALID_FREQ,
		OE_INVALID_BANDFREQS, OE_INVLAID_RIPPLE, OE_INVALID_BANDRIPPLES or OE_INVALID_TYPE)
*/
OC_API int ocmsp_iir_filter_ss(UINT N, double Wn, double* pA, double* pB, double* pC, double* pd, int type, 
							   double Rp=0.0, double Rs=0.0, double Wn2=-1.0);


// Comment last updated by Rex on 2012/7/24
// Comment last updated by Rex on 2012/5/14
/**
	Remarks:
		Design digital or analog IIR filters with given specifications. It returns the zeros, poles and the gain:
						   (z-z0)*(z-z1)* ... *(z-z_(n-1))
				 H(z) = k ----------------------------------
						   (z-p0)*(z-p1)* ... *(z-p_(n-1))
	Keywords:
		IIR filter design
	Example1:
		void ocmsp_iir_filter_zp_ex1()
		{
			UINT N = 18; //even number if bandpass or bandstop
			double Wn[2] = {300.0/500, 400.0/500};
			vector<complex> zeros(N), poles(N); //size N
			double gain;
			int type = OMSP_BUTTERWORTH|OMSP_BANDPASS|OMSP_DIGITAL;

			int nRet = 0;
			if (0 != (nRet = ocmsp_iir_filter_zp(N, Wn[0], zeros, poles, &gain, type, 0.0, 0.0, Wn[1])))
			{
				printf("ocmsp_iir_filter_zp failed, error code=%d\n", nRet);
				return;
			}

			int ii;
			printf("zeros= ");
			for(ii = 0; ii < zeros.GetSize(); ++ii)
			printf("\t%lf %lf\n", zeros[ii].m_re, zeros[ii].m_im);
			printf("poles= ");
			for(ii = 0; ii < poles.GetSize(); ++ii)
			printf("\t%lf %lf\n", poles[ii].m_re, poles[ii].m_im);
		}
		// At the end of example:
	Parameters:
		N	= [input] filter order, should be even if is bandpass or bandstop type
		Wn	= [input] cutoff or passband/stopband-edge frequency, it's in radians/second if is analog type, and
			should be normalized to [0, 1] if is digital type(where 1 corresponds to half the sample rate)
		pZ	= [output] returned zeros, size N
		pP	= [output] returned poles, size N
		pk	= [output] returned gain, scalar
		type= [intput] filter type, it specifies the filter prototype(Butterworth, Chebshev1, Chebshev2, Elliptic 
			or Bessel), bandtype(lowpass, highpass, bandstop or band pass) and digital or analog type. For example
			type = OMSP_BUTTERWORTH|OMSP_LOWPASS|OMSP_DIGITAL means a lowpass Butterworth digital filter.
		Rp	= [input][optional] passband ripple (dB), needed when it's Chebshev1 or Ellip type
		Rs	= [input][optional] stopband attenuation (dB), needed when it's Chebshev2 or Ellip type
		Wn2	= [input][optional] cutoff or passband/stopband-edge frequency. Needed when it's a bandpass or bandstop
			filter. It's in radians/second if is analog type, and should be normalized to [0, 1] if is digital type
			(where 1 corresponds to half the sample rate). 
	Return:
		Return OE_NOERROR if succeed, otherwise, non-zero error code is returned(OE_NULL_POINTER, OE_INVALID_FREQ,
		OE_INVALID_BANDFREQS, OE_INVLAID_RIPPLE, OE_INVALID_BANDRIPPLES or OE_INVALID_TYPE)
*/
OC_API int ocmsp_iir_filter_zp(UINT N, double Wn, d_complex* pZ, d_complex* pP, double* pk, int type, 
							   double Rp=0.0, double Rs=0.0, double Wn2=-1.0);


// Comment last updated by Rex on 2012/7/24
// Comment last updated by Rex on 2012/5/14
/**
	Remarks:
		Design digital or analog IIR filters with given specifications. It returns a filter in second-order section 
		representation:

				 L-1	             L-1   b0k + b1k*z^(-1) + b2k*z^(-2)
		H(z) = g II {H_k(z)} = g II  -------------------------------
		         k=0             k=0    1 + a1k*z^(-1) + a2k*z(-2)

		The result is saved in an L-by-6 matrix sos, and a scalar gain.
			  | b01 b11 b21 1 a11 a21 |
		sos = | b02 b12 b22 1 a12 a22 |
			  |  :   :   :  :  :  :   |
			  | b0L b1L b2L 1 a1L a2L |
		each row in sos is the numerator and denominator coefficients b_ik and a_ik of the second order sections of H(z)
		
		section number L is the closest integer greater than or equal to n/2. (L = (n+1)/2)
	Keywords:
		IIR filter design
	Example1:
		void ocmsp_iir_filter_sos_ex1()
		{
			UINT N = 18; //even number if bandpass or bandstop
			double Wn[2] = {300.0/500, 400.0/500};

			UINT nsec = (N+1)/2; //size L = (n+1)/2;
			matrix sos(nsec, 6);
			double gain;
			int type = OMSP_BUTTERWORTH|OMSP_BANDPASS|OMSP_DIGITAL;

			int nRet = 0;
			if (0 != (nRet = ocmsp_iir_filter_sos(N, Wn[0], sos, &gain, type, 0.0, 0.0, Wn[1])))
			{
				printf("ocmsp_iir_filter_sos failed, error code=%d\n", nRet);
				return;
			}
		}
		// At the end of example:
	Parameters:
		N	= [input] filter order, should be even if is bandpass or bandstop type
		Wn	= [input] cutoff or passband/stopband-edge frequency, it's in radians/s if is analog type, and should
			be normalized to [0, 1] if is digital type, where 1 corresponds to half the sample rate
		pSOS= [output] coefficients matrix of second-order sections H(z), size nsec-by-6, nsec = (N+1)/2
		pg	= [output] returned gain, scalar
		type= [intput] filter type, it specifies the filter prototype(Butterworth, Chebshev1, Chebshev2, Elliptic 
			or Bessel), bandtype(lowpass, highpass, bandstop or band pass) and digital or analog type. For example
			type = OMSP_BUTTERWORTH|OMSP_LOWPASS|OMSP_DIGITAL means a lowpass Butterworth digital filter.
		Rp	= [input][optional] passband ripple (dB), needed when it's Chebshev1 or Ellip type
		Rs	= [input][optional] stopband attenuation (dB), needed when it's Chebshev2 or Ellip type
		Wn2	= [input][optional] cutoff or passband/stopband-edge frequency. Needed when it's a bandpass or bandstop
			filter. It's in radians/second if is analog type, and should be normalized to [0, 1] if is digital type
			(where 1 corresponds to half the sample rate). 
	Return:
		Return OE_NOERROR if succeed, otherwise, non-zero error code is returned(OE_NULL_POINTER, OE_INVALID_FREQ,
		OE_INVALID_BANDFREQS, OE_INVLAID_RIPPLE, OE_INVALID_BANDRIPPLES or OE_INVALID_TYPE)
*/
OC_API int ocmsp_iir_filter_sos(UINT N, double Wn, double* pSOS, double* pg, int type, 
								double Rp=0.0, double Rs=0.0, double Wn2=-1.0);
///End IIR_SOS_FILTER_DESIGN_AND_ANALYSIS_FUNCS


// Comment last updated by Rex on 2012/3/20
/**
	Remarks:
		filter data sequence using a digital filter
	Keywords:
		filter
	Example1:
		void ocmsp_filter_ex1()
		{
			Worksheet wks = Project.ActiveLayer();
			if(!wks)
				return;

			vector& vx = wks.Columns(0).GetDataObject();
			vector& vy = wks.Columns(1).GetDataObject();
			vecter b = {0.018886917952608, 0.169982261573470, 0.679929046293879,
					    1.586501108019050, 2.379751662028575, 2.379751662028575,
						1.586501108019050, 0.679929046293879, 0.169982261573470,
						0.018886917952608};
			vector a = {1.000000000000000, 1.791581352788596, 2.531899880898121,
						2.118229420341933, 1.370756294393234, 0.609038913076474,
						0.199331556962956, 0.043104731015281, 0.005804261654309,
						0.000355580604258};
			
			int nRet = 0, n = vx.GetSize(), na = a.GetSize(), nb = b.GetSize();
			if (0 != (nRet = ocmsp_filter(vx, vy, n, b, nb, a, na, NULL, NULL)))
			{
				printf("ocmsp_filter failed, error ocde=%d\n", nRet);
				return;
			}
		}
		// At the end of example:
	Parameters:
		px	= [input] input data
		py	= [output] output data
		n	= [input] size of data
		pb	= [input] filter numerator coefficients
		nb	= [input] size of numerator coefficients
		pa	= [input] filter denominator coefficients
		na	= [input] size of denominator coefficients
		pzi	= [input][optional] initial codition of delays, length is max(na, nb)-1
		pzf	= [output][optional] final condition of delays, length is max(na, nb)-1
	Return:
		Return OE_NOERROR if succeed, otherwise, non-zero error code is returned(OE_NULL_POINTER,
		OE_INVALID_SIZE or OE_BAD_PARAM)
*/
OC_API int ocmsp_filter(const double *px, double* py, UINT n, const double* pb, UINT nb, 
						const double *pa, UINT na, const double* pzi=NULL, double* pzf=NULL);


///Rex 2012/5/14 ORG-5575 IIR_SOS_FILTER_DESIGN_AND_ANALYSIS_FUNCS
// Comment last updated by Rex on 2012/5/14
/**
	Remarks:
		Second order (biquadratic) IIR digital filtering
	Keywords:
		filter
	Parameters:
		px	= [input] input data
		py	= [output] output data
		n	= [input] size of data
		psos= [input] second-order section filter matrix, size nsec-by-6
		nsec= [input] number of sections
		gain= [input] overall gain of filter
	Return:
		Return OE_NOERROR if succeed, otherwise, non-zero error code is returned(OE_NULL_POINTER,
		OE_INVALID_SIZE or OE_BAD_PARAM)
*/
OC_API int ocmsp_filter_sos(const double* px, double* py, UINT n, const double* pSOS, UINT nsec, double gain);
///End IIR_SOS_FILTER_DESIGN_AND_ANALYSIS_FUNCS


// Comment last updated by Rex on 2012/3/20
/**
	Remarks:
		Zero-phase digital filtering
	Keywords:
		zero-phase filter
	Example1:
		void ocmsp_filtfilt_ex1()
		{
			Worksheet wks = Project.ActiveLayer();
			if(!wks)
				return;

			vector& vx = wks.Columns(0).GetDataObject();
			vector& vy = wks.Columns(1).GetDataObject();
			vecter b = {0.018886917952608, 0.169982261573470, 0.679929046293879,
						1.586501108019050, 2.379751662028575, 2.379751662028575,
						1.586501108019050, 0.679929046293879, 0.169982261573470,
						0.018886917952608};
			vector a = {1.000000000000000, 1.791581352788596, 2.531899880898121,
						2.118229420341933, 1.370756294393234, 0.609038913076474,
						0.199331556962956, 0.043104731015281, 0.005804261654309,
						0.000355580604258};

			int nRet = 0, n = vx.GetSize(), na = a.GetSize(), nb = b.GetSize();
			if (0 != (nRet = ocmsp_filtfilt(vx, vy, n, b, nb, a, na)))
			{
				printf("ocmsp_filtfilt failed, error code=%d\n", nRet);
				return;
			}
		}
		// At the end of example:
	Parameters:
		px	= [input] input data
		py	= [output] output data
		n	= [input] size of data
		pb	= [input] filter numerator coefficients
		nb	= [input] size of numerator coefficients
		pa	= [input] filter denominator coefficients
		na	= [input] size of denominator coefficients
	Return:
		Return OE_NOERROR if succeed, otherwise, non-zero error code is returned(OE_NULL_POINTER or
		OE_INVALID_SIZE)
*/
OC_API int ocmsp_filtfilt(const double* px, double* py, UINT n, const double* pb, UINT nb, const double *pa, UINT na);


///Rex 2012/5/14 ORG-5575 IIR_SOS_FILTER_DESIGN_AND_ANALYSIS_FUNCS
// Comment last updated by Rex on 2012/5/14
/**
	Remarks:
		Second order(biquadratic) IIR Zero-phase digital filtering
	Keywords:
		zero-phase filter
	Parameters:
		px	= [input] input data
		py	= [output] output data
		n	= [input] size of data
		psos= [input] second-order section filter matrix, size nsec-by-6
		nsec= [input] number of sections
		gain= [input] overall gain of filter
	Return:
		Return OE_NOERROR if succeed, otherwise, non-zero error code is returned(OE_NULL_POINTER,
		OE_INVALID_SIZE or OE_BAD_PARAM)
*/
OC_API int ocmsp_filtfilt_sos(const double* px, double* py, UINT n, const double* pSOS, UINT nsec, double gain);
///End IIR_SOS_FILTER_DESIGN_AND_ANALYSIS_FUNCS


/*
	Initial conditions for transposed direct-form II filter implementation
*/
//OC_API int ocmsp_filtic(double* pz, UINT* nz, const double* pa, UINT na, const double* pb, UINT nb, const double* py, UINT ny, const double* px=NULL, UINT nx=0);

enum
{
	OMSP_TWO_SIDE		= 1,
	OMSP_MAG_DECIBEL	= 2,
	OMSP_PHA_DEGREE		= 4,
	OMSP_PHA_UNWRAP		= 8
};

// Comment last updated by Rex on 2012/7/24
// Comment last updated by Rex on 2012/3/20
/**
	Remarks:
		Frequency and phases response of filter:
					 B(e^(jw))    b(0) + b(1)e^(-jw) + .... + b(m)e^(-jw)
		H(e^(jw)) = ----------- = -----------------------------------------
					 A(e^(jw))    a(0) + a(1)e^(-jw) + .... + a(n)e^(-jw)
	Keywords:
		freqency and phase response
	Example1:
		void ocmsp_freqz_ex1()
		{
			vecter b = {0.018886917952608, 0.169982261573470, 0.679929046293879,
						1.586501108019050, 2.379751662028575, 2.379751662028575,
						1.586501108019050, 0.679929046293879, 0.169982261573470,
						0.018886917952608};
			vector a = {1.000000000000000, 1.791581352788596, 2.531899880898121,
						2.118229420341933, 1.370756294393234, 0.609038913076474,
						0.199331556962956, 0.043104731015281, 0.005804261654309,
						0.000355580604258};
		
			Worksheet wks = Project.ActiveLayer();
			if(!wks)
				return;			
				
			int nRet = 0, n = 512, na = a.GetSize(), nb = b.GetSize();	
			vector& vw	 = wks.Columns(0).GetDataObject();
			vector& vmag = wks.Columns(1).GetDataObject();
			vector& vpha = wks.Columns(2).GetDataObject();
			vw.SetSize(n);
			vmag.SetSize(n);
			vpha.SetSize(n);

			int type = OMSP_MAG_DECIBEL|OMSP_PHA_UNWRAP;
			if (0 != (nRet = ocmsp_freqz(vmag, vpha, vw, n, b, nb, a, na type)))
			{
				printf("ocmsp_freqz failed, error code=%d\n", nRet);
				return;
			}
		}
		// At the end of example:
	Parameters:
		mag	= [output] magnitude response, with size n
		pha	= [output] phase response, with size n
		w	= [output] frequencies(in radians/sample) on which the response values are calculated. size n
		n	= [input] points number, e.g. 512
		pb	= [input] filter numerator coefficients
		nb	= [input] size of numerator coefficients
		pa	= [input] filter denominator coefficients
		na	= [input] size of denominator coefficients
		flag= [input] indicates the unit of returned magnitude and phase values, the range of frequency
			(0~pi or 0~2*pi), and if the phases are unwrapped.
	Return:
		Return OE_NOERROR if succeed, otherwise, non-zero error code is returned(OE_NULL_POINTER,
		OE_INVALID_SIZE or OE_BAD_PARAM)
*/
OC_API int ocmsp_freqz(double* mag, double* pha, double* w, UINT n, const double* pb, UINT nb,
					   const double* pa, UINT na, int flag=OMSP_MAG_DECIBEL|OMSP_PHA_UNWRAP);


///Rex 2012/5/14 ORG-5575 IIR_SOS_FILTER_DESIGN_AND_ANALYSIS_FUNCS
// Comment last updated by Rex on 2012/5/17
/**
	Remarks:
		Frequency and phases response of second order section filter
	Keywords:
		frequency and phase response
	Parameters:
		mag	= [output] magnitude response, with size n
		pha	= [output] phase response, with size n
		w	= [output] frequencies(in radians/sample) on which the response values are calculated. size n
		n	= [input] points number, e.g. 512
		psos= [input] second-order section filter matrix, size nsec-by-6
		nsec= [input] number of sections
		gain= [input] overall gain of filter
		flag= [input] indicates the unit of returned magnitude and phase values, the range of frequency
			(0~pi or 0~2*pi), and the if the phases are unwrapped.
	Return:
		Return OE_NOERROR if succeed, otherwise, non-zero error code is returned(OE_NULL_POINTER,
		OE_INVALID_SIZE or OE_BAD_PARAM)
*/
OC_API int ocmsp_freqz_sos(double* mag, double* pha, double* w, UINT n, const double* pSOS, UINT nsec, 
						   double gain, int flag=OMSP_MAG_DECIBEL|OMSP_PHA_UNWRAP);
///End IIR_SOS_FILTER_DESIGN_AND_ANALYSIS_FUNCS


// Comment last updated by Rex on 2012/3/20
/**
	Remarks:
		Get the length of the impulse response for a digital filter
	Keywords:
		impulse response
	Parameters:
		pn	= [output] length of impulse response
		pb	= [input] filter numerator coefficients
		nb	= [input] size of numerator coefficients
		pa	= [input] filter denominator coefficients
		na	= [input] size of denominator coefficients
	Return:
		Return OE_NOERROR if succeed, otherwise, non-zero error code is returned(OE_NULL_POINTER,
		OE_INVALID_SIZE or OE_BAD_PARAM)
*/
OC_API int ocmsp_impzlength(UINT* pn, const double* pb, UINT nb, const double* pa, UINT na);


///Rex 2012/5/14 ORG-5575 IIR_SOS_FILTER_DESIGN_AND_ANALYSIS_FUNCS
// Comment last updated by Rex on 2012/3/20
/**
	Remarks:
		Get the length of the impulse response for a digital filter
	Keywords:
		impulse response
	Parameters:
		pn	= [output] length of impulse response
		psos= [input] second-order section filter matrix, size nsec-by-6
		nsec= [input] number of sections
	Return:
		Return OE_NOERROR if succeed, otherwise, non-zero error code is returned(OE_NULL_POINTER,
		OE_INVALID_SIZE or OE_BAD_PARAM)
*/
OC_API int ocmsp_impzlength_sos(UINT* pn, const double* pSOS, UINT nsec);
///End IIR_SOS_FILTER_DESIGN_AND_ANALYSIS_FUNCS


// Comment last updated by Rex on 2012/7/24
// Comment last updated by Rex on 2012/3/20
/**
	Remarks:
		Impulse response of digital filter
	Keywords:
		impulse response
	Example1:
		void ocmsp_impz_ex1()
		{
			vecter b = {0.018886917952608, 0.169982261573470, 0.679929046293879,
						1.586501108019050, 2.379751662028575, 2.379751662028575,
						1.586501108019050, 0.679929046293879, 0.169982261573470,
						0.018886917952608};
			vector a = {1.000000000000000, 1.791581352788596, 2.531899880898121,
						2.118229420341933, 1.370756294393234, 0.609038913076474,
						0.199331556962956, 0.043104731015281, 0.005804261654309,
						0.000355580604258};

			Worksheet wks = Project.ActiveLayer();
			if(!wks)
				return;			

			int nRet = 0, na = a.GetSize(), nb = b.GetSize();	
			vector& vt = wks.Columns(0).GetDataObject();
			vector& vh = wks.Columns(1).GetDataObject();
			
			UINT nh = 0;
			int type = 0;
			if (0 != (nRet = ocmsp_impzlength(&nh, b, nb, a, na)))
			{
				printf("ocmsp_impzlength failed, error code = %d\n", nRet);
				return;
			}

			vt.SetSize(nh); vh.SetSize(nh);
			
			if (0 != (nRet = ocmsp_impz(vh, vt, nh, b, nb, a, na)))
			{
				printf("ocmsp_impz failed, error code = %d\n", nRet);
				return;
			}
		}
		// At the end of example:
	Parameters:
		ph	= [output] impulse response, size n
		pt	= [output] sample times, size n
		n	= [input] size of impulse response
		pb	= [input] filter numerator coefficients
		nb	= [input] size of numerator coefficients
		pa	= [input] filter denominator coefficients
		na	= [input] size of denominator coefficients
		fs	= [input][optional] sample frequency, the samples are spaced 1/fs unit apart. Default 1
	Return:
		Return OE_NOERROR if succeed, otherwise, non-zero error code is returned(OE_NULL_POINTER,
		OE_INVALID_SIZE or OE_BAD_PARAM)
*/
OC_API int ocmsp_impz(double* ph, double* pt, UINT n, const double* pb, UINT nb, 
					  const double* pa, UINT na, double fs=1.0);


///Rex 2012/5/14 ORG-5575 IIR_SOS_FILTER_DESIGN_AND_ANALYSIS_FUNCS
// Comment last updated by Rex on 2012/5/17
/**
	Remarks:
		Impulse response of digital filter
	Keywords:
		impulse response
	Parameters:
		ph	= [output] impulse response, size n
		pt	= [output] sample times, size n
		n	= [input] size of impulse response		
		psos= [input] second-order section filter matrix, size nsec-by-6
		nsec= [input] number of sections
		gain= [input] overall gain of filter
		fs	= [input][optional] sample frequency, the samples are spaced 1/fs unit apart. Default 1
	Return:
		Return OE_NOERROR if succeed, otherwise, non-zero error code is returned(OE_NULL_POINTER,
		OE_INVALID_SIZE or OE_BAD_PARAM)
*/
OC_API int ocmsp_impz_sos(double* ph, double* pt, UINT n, const double* pSOS, UINT nsec,
						  double gain, double fs=1.0);
///End IIR_SOS_FILTER_DESIGN_AND_ANALYSIS_FUNCS
///End IIR_DIGITAL_FILTER_DESIGN


///Rex 2012/3/28 ORG-5371-S1 READ_WRITE_MATLAB_FCF_FILE
enum { OMSP_DEC = 0, OMSP_BIN, OMSP_HEX};
// Comment last updated by Rex on 2012/7/24
// Comment last updated by Rex on 2012/3/28
/**
	Remarks:
		Write filter coefficients to matlab fcf file
	Example1:
		void ocmsp_write_fcf_ex1()
		{
			vecter b = {0.018886917952608, 0.169982261573470, 0.679929046293879,
						1.586501108019050, 2.379751662028575, 2.379751662028575,
						1.586501108019050, 0.679929046293879, 0.169982261573470,
						0.018886917952608};
			vector a = {1.000000000000000, 1.791581352788596, 2.531899880898121,
						2.118229420341933, 1.370756294393234, 0.609038913076474,
						0.199331556962956, 0.043104731015281, 0.005804261654309,
						0.000355580604258};

			int nRet = 0, na = a.GetSize(), nb = b.GetSize();	
			const char* filename = "filter_tf.fcf";
			if (0 != (nRet = ocmsp_writefcf(filename, b, nb, a, na, OMSP_DEC)))
			{
				printf("ocmsp_writefcf failed, error code = %d\n", nRet);
				return;
			}
		}
		// At the end of example:
	Parameters:
		filename = [input] file name
		pb	= [input] filter numerator coefficients
		nb	= [input] size of numerator coefficients
		pa	= [input] filter denominator coefficients
		na	= [input] size of denominator coefficients
		fmt = [input] format of coefficients, in binary, decimal or hexadecimal
	Return:
		Return OE_NOERROR if succeed. non-zero error code is returned(OE_OPEN_FILE_FAILED,
		OE_WRITE_FILE_FAILED)
*/
OC_API int ocmsp_writefcf(const char* filename, const double* pb, UINT nb, 
						  const double* pa, UINT na, int fmt=OMSP_DEC);
#define ocmsp_writefcf_tf ocmsp_writefcf


///Rex 2012/5/14 ORG-5575 IIR_SOS_FILTER_DESIGN_AND_ANALYSIS_FUNCS	
// Comment last updated by Rex on 2012/5/17
/**
	Remarks:
		Write filter in state space form to matlab fcf file
	Parameters:
		filename = [input] file name
		pA	= [input] matrix A, size m-by-n
		m	= [input] rows of A
		n	= [input] cols of A
		pB	= [input] vector B, size m
		pC	= [input] vector C, size n
		d	= [input] scalar d
		fmt = [input] format of coefficients, in binary, decimal or hexadecimal
	Return:
		Return OE_NOERROR if succeed, otherwise, non-zero error code is returned(OE_NULL_POINTER,
		OE_INVALID_SIZE or OE_BAD_PARAM)
*/
OC_API int ocmsp_writefcf_ss(const char* filename, const double* pA, UINT m, UINT n,
							 const double* pB, const double* pC, double d, int fmt=OMSP_DEC);

// Comment last updated by Rex on 2012/5/17
/**
	Remarks:
		Write filter in second order section to matlab fcf file
	Parameters:
		filename = [input] file name
		psos= [input] second-order section filter matrix, size nsec-by-6
		nsec= [input] number of sections
		gain= [input] overall gain of filter
	Return:
		Return OE_NOERROR if succeed, otherwise, non-zero error code is returned(OE_NULL_POINTER,
		OE_INVALID_SIZE or OE_BAD_PARAM)
*/
OC_API int ocmsp_writefcf_sos(const char* filename, const double* pSOS, UINT nsec,
							  double gain, int fmt=OMSP_DEC);
///End IIR_SOS_FILTER_DESIGN_AND_ANALYSIS_FUNCS	

enum {OMSP_FCF_TF,
	  OMSP_FCF_SS,
	  OMSP_FCF_SOS,
	  OMSP_FCF_UNKNOWN
};
// Comment last updated by Rex on 2012/5/25
/**
	Remarks:
		Read fcf file header
	Parameters:
		ptype	 = [output] filter type in fcf file, OMSP_FCF_TF, OMSP_FCF_SS, OMSP_FCF_SOS or OMOSP_FCF_UNKNOW
		filename = [input] file name
	Return:
		Return OE_NOERROR if succeed. non-zero error code is returned(OE_OPEN_FILE_FAILED,
		OE_READ_FILE_FAILED)
*/
OC_API int ocmsp_readfcf_header(int *ptype, const char* filename);

// Comment last updated by Rex on 2012/7/25
// Comment last updated by Rex on 2012/3/28
/**
	Remarks:
		Read filter coefficients from fcf file
	Example1:
		void ocmsp_write_fcf_ex1()
		{
			int nRet = 0, type = OMSP_FCF_UNKNOWN;
			const char* filename = "my_filter.fcf";
			if (0 != (nRet = ocmsp_readfcf_header(&type, filename)))
			{
				printf("ocmsp_readfcf_header failed, error code = %d\n", nRet);
				return;
			}

			if (type != OMSP_FCF_TF)
			{
				printf("fcf file is not [num den] coefficient\n");
				return;
			}
			
			double *pb = NULL, *pa = NULL;
			UINT nb = 0, na = 0;
			if( 0 != (nRet = ocmsp_readfcf(&pb, &nb, &pa, &na, filename)))
			{
				printf("ocmsp_readfcf failed, error code = %d\n", nRet);
				return;
			}

			//...
			
			//release buffer
			ocmsp_release_filter(&pb, &pa);
			nb = na = 0;			
		}
		// At the end of example:
	Parameters:
		ppb	= [output] filter numerator coefficients
		pnb	= [output] size of numerator coefficients
		ppa	= [output] filter denominator coefficients
		pna	= [output] size of denominator coefficients
		filename = [input] file name
	Return:
		Return OE_NOERROR if succeed. non-zero error code is returned(OE_OPEN_FILE_FAILED,
		OE_READ_FILE_FAILED)
*/
OC_API int ocmsp_readfcf(double** ppb, UINT* pnb, double** ppa, UINT* pna, const char* filename);
#define ocmsp_readfcf_tf ocmsp_readfcf


// Comment last updated by Rex on 2012/5/25
/**
	Remarks:
		Read filter in state space form from fcf file
	Parameters:
		ppA	= [output] filter matrix A, size m-by-n
		pm	= [output] rows of A
		pn	= [output] cols of A
		ppB	= [output] filter vector B, size m
		ppC	= [output] filter vector C, size n
		pd	= [output] scalar d
		filename = [input] file name
	Return:
		Return OE_NOERROR if succeed, otherwise, non-zero error code is returned(OE_NULL_POINTER,
		OE_INVALID_SIZE or OE_BAD_PARAM)
*/
OC_API int ocmsp_readfcf_ss(double** ppA, UINT* pm, UINT* pn, double** ppB, 
							double** ppC, double* pd, const char* filename);

// Comment last updated by Rex on 2012/5/21
/**
	Remarks:
		Read filter in second-order-section from fcf file
	Keywords:
		impulse response
	Parameters:		
		ppsos= [output] second-order section filter matrix, size nsec-by-6
		pnsec= [output] number of sections
		pgain= [output] overall gain of filter
		filename = [input] file name
	Return:
		Return OE_NOERROR if succeed, otherwise, non-zero error code is returned(OE_NULL_POINTER,
		OE_INVALID_SIZE or OE_BAD_PARAM)
*/
OC_API int ocmsp_readfcf_sos(double** ppSOS, UINT* pnsec, double* pgain, const char* filename);


// Comment last updated by Rex on 2012/3/28
/**
	Remarks:
		Release filter arrays: num, den, A, B, C, D or sos matrix
	Parameters:
		pp1	= [modify] filter array
		pn2	= [modify] filter array
		pp3	= [modify] filter array
		pn4	= [modify] filter array
	Return:
		No return value;
*/
OC_API void ocmsp_release_filter(double** pp1, double** pp2=NULL, double** pp3=NULL, double** pp4=NULL);
///End READ_WRITE_MATLAB_FCF_FILE


/// Fisher	07/29/09 http://wiki.originlab.com/~originla/internal/index.php?title=Sales:Surface_Statistics
struct	 RoughnessParametersCtrls
{
	BOOL		bProfileByRow;
	double		dSamplingLength;
};

struct	ProfileParameters
{
	double		*pR_a;		// arithmetic average of absolute values 
	double		*pR_q;		// root mean squared 
	double		*pR_v;		// maximum valley depth
	double		*pR_p;		// maximum peak height 
	double		*pR_t;		// Maximum Height of the Profile 
	double		*pR_sk;		// skewness 
	double		*pR_ku;		// kurtosis
	double		*pR_din;	// average distance between the highest peak and lowest valley in each sampling length, ASME Y14.36M - 1996 Surface Texture Symbols 
	double		*pR_jis;	// Japanese Industrial Standard for Rz, based on the five highest peaks and lowest valleys over the entire sampling length.
	double		*pR_dq;		// the RMS slope of the profile within the sampling length 
};

struct	ArealParameters
{
	// Amplitude Parameters
	double		*pS_a;		// Average Roughness
	double		*pS_q;		// RMS Roughness	
	double		*pS_sk;		// Skewness		
	double		*pS_ku;		// Kurtosis
	double		*pS_p;		// Peak Height
	double		*pS_v;		// Valley Depth
	double		*pS_t;		// Peak to Valley Height
	double		*pS_z;		// Ten-Point Height
	double		*pS_PaX;	// Mean Profile Pa along the X axis
	double		*pS_PaY;	// Mean Profile Pa along the Y axis
	double		*pS_PtX;	// Mean Profile Pt along the X axis
	double		*pS_PtY;	// Mean Profile Pt along the Y axis

	// Functional Parameters
	//double		*pS_k;		// Core Roughness Depth
	//double		*pS_pk;		// Reduced Peak Height
	//double		*pS_vk;		// Reduced Valley Depth
	//double		*pS_r1;		// Upper Bearing Surface
	//double		*pS_r2;		// Lower Bearing Surface
	//double		*pS_bi;		// Surface Bearing Index
	//double		*pS_ci;		// Core Fluid Retention Index
	//double		*pS_vi;		// Valley Fluid Retention Index

	// Hybrid Parameters
	//double		*pS_sc;		// Arithmetic Mean Summit Curvature
	//double		*pS_dq;		// RMS Slope
	//double		*pS_dr;		// Developed Interfacial Area Ratio
	//double		*pS_HSC;	// High Spot Count
};

OC_API	int	ocmath_profile_roughness_parameters(const double *pX, const double *pData, const UINT nSize, const UINT nOffset, RoughnessParametersCtrls *pstCtrl, ProfileParameters *pstProfiles);
OC_API	int	ocmath_areal_parameters(const double *pxGrid, const double *pyGrid, const double *pzMat, const UINT nRows, const UINT nCols, RoughnessParametersCtrls *pstCtrl, ArealParameters *pstAreal);

OC_API	int	ocmath_3d_grid_statistics_roughness_parameters(	const double *pxGrid, const double *pyGrid, const double *pzMat, const UINT nRows, const UINT nCols, RoughnessParametersCtrls *pstCtrl, ArealParameters *pstAreal, ProfileParameters *pstProfiles);

///---------


#ifdef _MSC_VER
	#ifdef __cplusplus
	}
	#endif
#endif	//_MSC_VER

#endif //_OCMSP_H


