/*------------------------------------------------------------------------------*
 * File Name:	omzXML.h		 												*
 * Creation: 	Sophy 8/8/2010													*
 * Purpose: OriginC Header File													*
 * Copyright (c) OriginLab Corp.	2003 - 2010									*
 * All Rights Reserved															*
 * 																				*
 * Modification Log:															*
 * EJP 2011-10-18 ORG-3762-P2 LOCALIZE_XF_IMPMZXML								*
 *------------------------------------------------------------------------------*/
 
#include <Origin.h>
#include <ozlib.h>
class omzXML
{
public:
	omzXML(){}
	~omzXML(){ m_fExtern.Close(); }
	int	Open(LPCSTR lpcszFileName);
	int GetFileType(){ return m_nFileType; }	//detect XML file type
	int	GetPeaksCount(); //get the number of m/z array and intensity array pairs
	int	GetScanIndices(vector<int>& vnIndices); //get the indices of valid scan
	int	ReadPeaks(long lScanIndex, vector<double>& vMZ, vector<double>& vIntens, vector<double>& vTime);
	int	Import(Worksheet& wks, TreeNode& trSel);
	int	GetFileInfo(TreeNode& trInfo);
	
protected:
	int	InitHeader();
	
private:
	//members for mzData
	int	readOneArray(const TreeNode& trArray, vector<double>& vData, bool bGLittleEndian);
	
	//members for mzXML
	int	readOnePeak(const TreeNode& trPeak, const int nPeaksCount, vector<double>& vMZ, vector<double>& vIntens, bool bGLittleEndian);
	
	//members for mzML
	int	readOneBinary(const TreeNode& trArray, const TreeNode& trParams, vector<double>& vData, int nDefaultLength = 0);
	
	//members for imzML
	int	readOneExternBinary(const TreeNode& trArray, const TreeNode& trParams, vector<double>& vData, int nDataType, bool bCompressed);
	
	int	checkValiateIndex(long lScanIndex);
	
private:
	enum {
		XML_TYPE_UNKNOWN	= -1,
		XML_TYPE_MZXML,
		XML_TYPE_MZDATA,
		XML_TYPE_MZML,
		XML_TYPE_IMZML,
		XML_TYPE_TOTAL,
	}; //filetype
	
	enum {
		CONTENT_MZINT		= 0,
		CONTENT_MZRULER,
	}; //contenttype

	enum {
		ARRAY_TYPE_MZARRAY = 0,
		ARRAY_TYPE_INTENSARRAY,
		ARRAY_TYPE_TIMEARRAY,
		ARRAY_TYPE_OTHER,
	}; //binary array type
	
	enum {
		XML_UNKNOWN		= -1,
		XML_INT32,
		XML_FLOAT,
		XML_DOUBLE,
	}; //binary array element type
	
private:
	int		m_nFileType;
	string	m_strFileName;
	string	m_strAuxFileName;
	Tree	m_trFile;
	Tree	m_trHeader;
	file	m_fExtern;
};

//return value for omzXML members
enum
{
	OMZXML_OK	= 0,
	OMZXML_INVALID_FILE,
	OMZXML_INVALID_OBJECT,
	OMZXML_INVALID_SCAN_INDEX,
	OMZXML_INVALID_PEAKSCOUNT,
	OMZXML_INVALID_CONTENTTYPE,
	OMZXML_INVALID_COMPRESSTYPE,
	OMZXML_INVALID_DATATYPE,
	OMZXML_NO_SPEC_DATA,
	OMZXML_NO_SCAN_DATA,
	OMZXML_NO_PEAKS_DATA,
	OMZXML_MEMORY_ERROR,
	OMZXML_USER_ABORT,
};
//utilities
static	void	_b64_decode_mio(char* Dst, char* Src)
{
	int nRet = okutil_decode_base64(Src, Dst);
	ASSERT(nRet > 0);
}

//omzXML implementation

//file attributes
#define	STR_PRECISION_ATTRIB	"precision"
#define	STR_ENDIAN_ATTRIB		"endian"
#define	STR_LENGTH_ATTRIB		"length"
#define	STR_COUNT_ATTRIB		"count"
#define	STR_SCANCOUNT_ATTRIB	"scanCount"
#define	STR_VALUE_ATTRIB		"value"
#define	STR_NAME_ATTRIB			"name"
#define	STR_VERSION_ATTRIB		"version"
#define	STR_REF_ATTRIB			"ref"

#define	STR_ACCESSION_ATTRIB	"accession"
#define	STR_INDEX_ATTRIB		"index"
///Sophy 1/13/2011 ORG-S2 SUPPORT_MZML_FILE_BEFORE_VERSION_1_1
#define	STR_SCANNUMBER_ATTRIB	"scanNumber"
///end SUPPORT_MZML_FILE_BEFORE_VERSION_1_1
#define	STR_SCAN_NUM_ATTRIB		"num"
#define	STR_SCAN_ID_ATTRIB		"id"
#define	STR_SOFTWARE_ID_ATTRIB	"id"
#define	STR_PROCESSING_ID_ATTRIB	"id"
#define	STR_GROUP_ID_ATTRIB		"id"
#define	STR_MSLEVEL_ATTRIB		"msLevel"
#define	STR_PEAKSCOUNT_ATTRIB	"peaksCount"
#define	STR_POLARITY_ATTRIB		"polarity"
#define	STR_RETENTIONTIME_ATTRIB	"retentionTime"
#define	STR_COLLISIONENERGY_ATTRIB	"collisionEnergy"
#define	STR_LOWMZ_ATTRIB		"lowMz"
#define	STR_HIGHMZ_ATTRIB		"highMz"
#define	STR_BASEPEAKMZ_ATTRIB	"basePeakMz"
#define	STR_BASEPEAKINTENS_ATTRIB	"basePeakIntensity"
#define	STR_TOTIONCURRENT_ATTRIB	"totIonCurrent"
#define	STR_PRECURSORINTENS_ATTRIB	"precursorIntensity"
#define	STR_BYTEORDER_ATTRIB		"byteOrder"
#define	STR_PAIRORDER_ATTRIB		"pairOrder"
#define	STR_ENCODEDLENGTH_ATTRIB	"encodedLength"
#define	STR_DEFAULT_ARRAYLENGTH_ATTRIB	"defaultArrayLength"
#define	STR_ARRAYLENGTH_ATTRIB			"arrayLength"
#define	STR_SOFTWAREREF_ATTRIB		"softwareRef"
#define	STR_ORDER_ATTRIB			"order"

#define	LABEL(_Node, _Label)	_Node.SetAttribute(STR_LABEL_ATTRIB, _Label);

#define	SWAPBYTES(_pBytes, _Length)	{ int iByte = 0, iHalfWay = _Length / 2;	\
	byte bt;	\
	for ( iByte = 0; iByte < iHalfWay; iByte ++ )	\
	{	\
		bt = _pBytes[iByte];	\
		_pBytes[iByte] = _pBytes[_Length - 1 - iByte];	\
		_pBytes[_Length - 1 - iByte] = bt;	\
	}	\
}
int	omzXML::Open(LPCSTR lpcszFileName)
{
	BOOL bRet = m_trFile.Load(lpcszFileName);
	if ( !bRet )
		return OMZXML_INVALID_FILE;
	
	string strType = m_trFile.tagName;
	if ( strType.CompareNoCase("indexedmzML") == 0 ) //special handle
	{
		m_trFile = m_trFile.FirstNode;
		strType = m_trFile.tagName;
	}
	
	if ( strType.CompareNoCase("mzData") == 0 )
		m_nFileType = XML_TYPE_MZDATA;
	else if ( strType.CompareNoCase("mzXML") == 0 )
		m_nFileType = XML_TYPE_MZXML;
	else if ( strType.CompareNoCase("imzML") == 0 )
		m_nFileType = XML_TYPE_IMZML;
	else if ( strType.CompareNoCase("mzML") == 0 )
	{
		bool bCheckIMZML = true;
		TreeNode trSpecList = tree_get_node_by_tagname(m_trFile, "spectrumList", true);
		if ( !trSpecList )
			trSpecList = tree_get_node_by_tagname(m_trFile, "chromatogramList", true);
		if ( trSpecList && trSpecList.FirstNode )
		{
			TreeNode trSpec = trSpecList.FirstNode;
			int nDefaultArrayLen = 0;
			trSpec.GetAttribute(STR_DEFAULT_ARRAYLENGTH_ATTRIB, nDefaultArrayLen);
			if ( nDefaultArrayLen != 0 )
				bCheckIMZML = false;
			if ( bCheckIMZML )
			{
				TreeNode trBinaryDataArray = tree_get_node_by_tagname(trSpec, "binaryDataArray", true);
				if ( trBinaryDataArray )
				{
					int nEncodedLen = 0;
					trBinaryDataArray.GetAttribute(STR_ENCODEDLENGTH_ATTRIB, nEncodedLen);
					if ( nEncodedLen != 0 )
						bCheckIMZML = false;
					if ( bCheckIMZML )
					{
						TreeNode trBinary = trBinaryDataArray.binary;
						if ( trBinary )
						{
							string strValue = trBinary.strVal;
							if ( strValue.GetLength() != 0 )
								bCheckIMZML = false;
						}
					}
				}
			}
		}
		if ( bCheckIMZML )
		{
			m_nFileType = XML_TYPE_IMZML;
		}
		else
		{
			m_nFileType = XML_TYPE_MZML;
		}
	}
	else
		return OMZXML_INVALID_FILE;
	m_strFileName = lpcszFileName; //keep filename in case need to use.
	m_strAuxFileName = lpcszFileName;
	LPSTR lpBuffer = m_strAuxFileName.GetBuffer(m_strAuxFileName.GetLength() + 4);
	check_add_file_ext(lpBuffer, "ibd");
	m_strAuxFileName.ReleaseBuffer();
	
	if ( m_nFileType == XML_TYPE_IMZML )
	{
		if ( !m_fExtern.Open(m_strAuxFileName, file::modeRead | file::shareDenyWrite) )
			return OMZXML_INVALID_FILE;
	}
	//to do...
	return InitHeader();;
}

int	omzXML::InitHeader()
{
	if ( m_nFileType == XML_TYPE_MZXML )
	{
		//scan information
		int nValidScans = 0;
		TreeNode trIndex = m_trFile.GetNode("index");
		if ( trIndex )
			nValidScans = trIndex.Children.Count();
		
		TreeNode trScanCount = tree_check_get_node(m_trHeader, "ValidScanCount");
		LABEL(trScanCount, _L("Number of Scan"));
		trScanCount.nVal = nValidScans;
		
		vector<int> vnIndices(0);
		if ( trIndex )
		{
			foreach(TreeNode trOffset in trIndex.Children)
			{
				int nScanIndex = 0;
				trOffset.GetAttribute(STR_SCAN_ID_ATTRIB, nScanIndex);
				vnIndices.Add(nScanIndex);
			}
		}
		TreeNode trScanNums = tree_check_get_node(m_trHeader, "ValidScanIndex");
		LABEL(trScanNums, _L("Valid Scan Indices"));
		trScanNums.nVals = vnIndices;
		
		//instrument information
		string strValue;
		TreeNode trMSIns = tree_get_node_by_tagname(m_trFile, "msInstrument", true);
		if ( trMSIns )
		{
			TreeNode trInstrument = tree_check_get_node(m_trHeader, "Instrument");
			LABEL(trInstrument, _L("Instrument"));
			TreeNode trItem;
			
			trMSIns.msManufacturer.GetAttribute(STR_VALUE_ATTRIB, strValue);
			trItem = tree_check_get_node(trInstrument, "Manufacturer");
			LABEL(trItem, _L("Manufacturer"));
			trItem.strVal = strValue;
			
			trMSIns.msModel.GetAttribute(STR_VALUE_ATTRIB, strValue);
			trItem = tree_check_get_node(trInstrument, "Model");
			LABEL(trItem, _L("Model"));
			trItem.strVal = strValue;
			
			trMSIns.msIonisation.GetAttribute(STR_VALUE_ATTRIB, strValue);
			trItem = tree_check_get_node(trInstrument, "Ionisation");
			LABEL(trItem, _L("Ionisation"));
			trItem.strVal = strValue;
			
			trMSIns.msMassAnalyzer.GetAttribute(STR_VALUE_ATTRIB, strValue);
			trItem = tree_check_get_node(trInstrument, "MassAnalyzer");
			LABEL(trItem, _L("Mass Analyzer"));
			trItem.strVal = strValue;
			
			trMSIns.msDetector.GetAttribute(STR_VALUE_ATTRIB, strValue);
			trItem = tree_check_get_node(trInstrument, "Detector");
			LABEL(trItem, _L("Detector"));
			trItem.strVal = strValue;
			
			TreeNode trSoftware = tree_check_get_node(trInstrument, "Software");
			LABEL(trSoftware, _L("Software"));
			trMSIns.software.GetAttribute(STR_NAME_ATTRIB, strValue);
			trSoftware.Name.strVal = strValue;
			LABEL(trSoftware.Name, _L("Name"));
			trMSIns.software.GetAttribute(STR_VERSION_ATTRIB, strValue);
			LABEL(trSoftware.Version, _L("Version"));
			trSoftware.Version.strVal = strValue;
		}
		
		//data processing information
		TreeNode trProcess = tree_get_node_by_tagname(m_trFile, "dataProcessing", true);
		if ( trProcess )
		{
			TreeNode trDP = tree_check_get_node(m_trHeader, "DataProcessing");
			LABEL(trDP, _L("Processing Software Information"));
			TreeNode trSoftware = tree_check_get_node(trDP, "Software");
			LABEL(trSoftware, _L("Software"));
			trProcess.software.GetAttribute(STR_NAME_ATTRIB, strValue);
			trSoftware.Name.strVal = strValue;
			LABEL(trSoftware.Name, _L("Name"));
			trProcess.software.GetAttribute(STR_VERSION_ATTRIB, strValue);
			trSoftware.Version.strVal = strValue;	
			LABEL(trSoftware.Version, _L("Version"));
		}
	}
	else if ( m_nFileType == XML_TYPE_MZDATA )
	{
		
		//scan information
		int nValidScans = 0;
		TreeNode trSpecList = tree_get_node_by_tagname(m_trFile, "spectrumList", true);
		if ( trSpecList )
			trSpecList.GetAttribute(STR_COUNT_ATTRIB, nValidScans);
		
		TreeNode trScanCount = tree_check_get_node(m_trHeader, "ValidScanCount");
		LABEL(trScanCount, _L("Number of Scan"));
		trScanCount.nVal = nValidScans;
		
		vector<int> vnIndices(0);
		if ( trSpecList )
		{
			foreach(TreeNode trSpec in trSpecList.Children)
			{
				int nScanIndex = 0;
				trSpec.GetAttribute(STR_SCAN_ID_ATTRIB, nScanIndex);
				vnIndices.Add(nScanIndex);
			}
		}
		TreeNode trScanNums = tree_check_get_node(m_trHeader, "ValidScanIndex");
		LABEL(trScanNums, _L("Valid Scan Indices"));
		trScanNums.nVals = vnIndices;
		
		//data processing information
		TreeNode trProcess = tree_get_node_by_tagname(m_trFile, "dataProcessing", true);
		if ( trProcess )
		{
			TreeNode trDP = tree_check_get_node(m_trHeader, "DataProcessing");
			LABEL(trDP, _L("Processing Software Information"));
			TreeNode trSoftware = tree_check_get_node(trDP, "Software");
			LABEL(trSoftware, _L("Software"));
			trSoftware.Name.strVal = trProcess.software.name.strVal;
			LABEL(trSoftware.Name, _L("Name"));
			trSoftware.Version.strVal = trProcess.software.version.strVal;	
			LABEL(trSoftware.Version, _L("Version"));
		}
	}
	else if ( m_nFileType == XML_TYPE_MZML || m_nFileType == XML_TYPE_IMZML )
	{
		//scan information
		int nValidScans = 0;
		TreeNode trSpecList = tree_get_node_by_tagname(m_trFile, "spectrumList", true);
		if ( !trSpecList )
			trSpecList = tree_get_node_by_tagname(m_trFile, "chromatogramList", true);
		if ( trSpecList )
			trSpecList.GetAttribute(STR_COUNT_ATTRIB, nValidScans);
		
		TreeNode trScanCount = tree_check_get_node(m_trHeader, "ValidScanCount");
		LABEL(trScanCount, _L("Number of Scan"));
		trScanCount.nVal = nValidScans;
		
		vector<int> vnIndices(0);
		if ( trSpecList )
		{
			foreach(TreeNode trSpec in trSpecList.Children)
			{
				int nScanIndex = 0;
				///Sophy 1/13/2011 ORG-S2 SUPPORT_MZML_FILE_BEFORE_VERSION_1_1
				//trSpec.GetAttribute(STR_INDEX_ATTRIB, nScanIndex);
				if ( !trSpec.GetAttribute(STR_INDEX_ATTRIB, nScanIndex) )
					trSpec.GetAttribute(STR_SCANNUMBER_ATTRIB, nScanIndex);
				///end SUPPORT_MZML_FILE_BEFORE_VERSION_1_1
				vnIndices.Add(nScanIndex);
			}
		}
		TreeNode trScanNums = tree_check_get_node(m_trHeader, "ValidScanIndex");
		LABEL(trScanNums, _L("Valid Scan Indices"));
		trScanNums.nVals = vnIndices;
		
		TreeNode trSoftwareList = tree_get_node_by_tagname(m_trFile, "softwareList", true);
		if ( trSoftwareList )
		{
			TreeNode trSL = tree_check_get_node(m_trHeader, "SoftwareList");
			LABEL(trSL, _L("Software List"));
			foreach(TreeNode trSoftware in trSoftwareList.Children)
			{
				TreeNode trSW = trSL.AddNode("Software");
				LABEL(trSW, _L("Software"));
				string strVal;
				trSoftware.GetAttribute(STR_SOFTWARE_ID_ATTRIB, strVal);
				TreeNode trName = tree_check_get_node(trSW, "Name");
				LABEL(trName, _L("Name"));
				trName.strVal = strVal;
				trSoftware.GetAttribute(STR_VERSION_ATTRIB, strVal);
				TreeNode trVersion = tree_check_get_node(trSW, "Version");
				LABEL(trVersion, _L("Version"));
				trVersion.strVal = strVal;
			}
		}
		TreeNode trRefGroupList = m_trFile.GetNode("referenceableParamGroupList");

		TreeNode trDataProcessList = m_trFile.GetNode("dataProcessingList");
		if ( trDataProcessList )
		{
			TreeNode trPL = tree_check_get_node(m_trHeader, "DataProcessingList");
			LABEL(trPL, _L("Data Processing List")); /// EJP 2011-10-18 ORG-3762-P2 LOCALIZE_XF_IMPMZXML 
			int nCount;
			trDataProcessList.GetAttribute(STR_COUNT_ATTRIB, nCount);
			foreach(TreeNode trPro in trDataProcessList.Children)
			{
				TreeNode trDP = trPL.AddNode("DataProcessing");
				LABEL(trDP, _L("Data Processing Methods"));
				string strID;
				trPro.GetAttribute(STR_PROCESSING_ID_ATTRIB, strID);
				trDP.ProcessingID.strVal = strID;
				LABEL(trDP.ProcessingID, _L("Processing ID"));
				foreach(TreeNode trPM in trPro.Children)
				{
					TreeNode trProM = trDP.AddNode("ProcessingMethod");
					LABEL(trProM, _L("Processing Method")); /// EJP 2011-10-18 ORG-3762-P2 LOCALIZE_XF_IMPMZXML 
					int nOrder;
					trPM.GetAttribute(STR_ORDER_ATTRIB, nOrder);
					trProM.Order.nVal = nOrder;
					LABEL(trProM.Order, _L("Order"));
					string strSoftware;
					trPM.GetAttribute(STR_SOFTWAREREF_ATTRIB, strSoftware);
					trProM.Software.strVal = strSoftware;
					LABEL(trProM.Software, _L("Software"));
					string strDataTransform = "Unknown";
					
					if ( trPM.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000033") )
						strDataTransform = "deisotoping";
					
					if ( trPM.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000034") )
						strDataTransform = "charge deconvolution";
					
					if ( trPM.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000544") )
						strDataTransform = "Conversion to mzML";
					
					if ( trPM.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000545") )
						strDataTransform = "Conversion to mzXML";
					
					if ( trPM.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000546") )
						strDataTransform = "Conversion to mzData";
					
					if ( trPM.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000593") )
						strDataTransform = "baseline reduction";
					
					if ( trPM.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000594") )
						strDataTransform = "low intensity data point removal";
					
					if ( trPM.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000741") )
						strDataTransform = "Conversion to dta";
					
					if ( trPM.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000745") )
						strDataTransform = "retention time alignment";
					
					if ( trPM.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000746") )
						strDataTransform = "high intensity data point removal";
					
					if ( strDataTransform.CompareNoCase("Unknown") == 0 ) //try to check ref param
					{
						TreeNode trRef = trPM.GetNode("referenceableParamGroupRef");
						if ( trRef && trRefGroupList )
						{
							string strGroupID;
							trRef.GetAttribute(STR_REF_ATTRIB, strGroupID);
							TreeNode trParam = trRefGroupList.FindNodeByAttribute(STR_GROUP_ID_ATTRIB, strGroupID);
							if ( trParam )
							{
								if ( trParam.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000033") )
									strDataTransform = "deisotoping";
								
								if ( trParam.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000034") )
									strDataTransform = "charge deconvolution";
								
								if ( trParam.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000544") )
									strDataTransform = "Conversion to mzML";
								
								if ( trParam.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000545") )
									strDataTransform = "Conversion to mzXML";
								
								if ( trParam.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000546") )
									strDataTransform = "Conversion to mzData";
								
								if ( trParam.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000593") )
									strDataTransform = "baseline reduction";
								
								if ( trParam.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000594") )
									strDataTransform = "low intensity data point removal";
								
								if ( trParam.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000741") )
									strDataTransform = "Conversion to dta";
								
								if ( trParam.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000745") )
									strDataTransform = "retention time alignment";
								
								if ( trParam.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000746") )
									strDataTransform = "high intensity data point removal";
							}
						}
					}
					trProM.DataTrans.strVal = strDataTransform;
					LABEL(trProM.DataTrans, _L("Data Transformation"));
					
					string strParam = "None";
					if ( trPM.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000629") )
						strParam = "low intensity threshold";
					
					if ( trPM.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000631") )
						strParam = "high intensity threshold";
					
					if ( trPM.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000747") )
						strParam = "completion time";
					
					if ( trPM.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000787") )
						strParam = "inclusive low intensity threshold";
					
					if ( trPM.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000788") )
						strParam = "inclusive high intensity threshold";
					
					if ( strParam.CompareNoCase("None") == 0 )
					{
						TreeNode trRef = trPM.GetNode("referenceableParamGroupRef");
						if ( trRef && trRefGroupList )
						{
							string strGroupID;
							trRef.GetAttribute(STR_REF_ATTRIB, strGroupID);
							TreeNode trParam = trRefGroupList.FindNodeByAttribute(STR_GROUP_ID_ATTRIB, strGroupID);
							if ( trParam )
							{
								if ( trPM.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000629") )
									strParam = "low intensity threshold";
								
								if ( trPM.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000631") )
									strParam = "high intensity threshold";
								
								if ( trPM.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000747") )
									strParam = "completion time";
								
								if ( trPM.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000787") )
									strParam = "inclusive low intensity threshold";
								
								if ( trPM.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000788") )
									strParam = "inclusive high intensity threshold";
							}
						}
					}
					trProM.TransParam.strVal = strParam;
					LABEL(trProM.TransParam, _L("Transformation Parameter"));
					
				}
			}
		}
	}
	else
		return OMZXML_INVALID_FILE;
	return OMZXML_OK;
}

int	omzXML::GetPeaksCount()
{
	int nCount = 0;
	if ( m_nFileType == XML_TYPE_MZXML )
	{
		TreeNode trMSRun = m_trFile.GetNode("msRun");
		if ( trMSRun ) //get number of scan
			trMSRun.GetAttribute(STR_SCANCOUNT_ATTRIB, nCount);
	}
	else if ( m_nFileType == XML_TYPE_MZDATA || m_nFileType == XML_TYPE_MZML )
	{
		TreeNode trSpecList = tree_get_node_by_tagname(m_trFile, "spectrumList", true);
		if( trSpecList )
			trSpecList.GetAttribute(STR_COUNT_ATTRIB, nCount);
	}
	return nCount;
}

int	omzXML::ReadPeaks(long lScanIndex, vector<double>& vMZ, vector<double>& vIntens, vector<double>& vTime)
{
	if ( OMZXML_OK != checkValiateIndex(lScanIndex) )
		return OMZXML_INVALID_SCAN_INDEX;
	
	int nPeaksCount = 0;
	int nPrecision = 0;
	
	//test little-endian
	int	nOrderTest = 0x00000001;
	int	nLittleEndian = *((char*)&nOrderTest);
	
	if ( XML_TYPE_MZDATA == m_nFileType )
	{
		//intensity and mz are written in two different arrays
		TreeNode trMZArray = tree_get_node_by_tagname(m_trFile, "mzArrayBinary", true);
		TreeNode trIntensArray = tree_get_node_by_tagname(m_trFile, "intenArrayBinary", true);
		int nRetMZ = readOneArray(trMZArray, vMZ, nLittleEndian == nOrderTest);
		int nRetIntens = readOneArray(trIntensArray, vIntens, nLittleEndian == nOrderTest);
		ASSERT(OMZXML_OK == nRetMZ && OMZXML_OK == nRetIntens);
	}
	else if ( XML_TYPE_MZXML == m_nFileType )
	{
		TreeNode trMSRun = m_trFile.GetNode("msRun");
		if ( !trMSRun )
			return OMZXML_NO_SCAN_DATA;
		TreeNode trScan = trMSRun.FindNodeByAttribute(STR_SCAN_NUM_ATTRIB, lScanIndex, TRUE);
		if ( !trScan )
			return OMZXML_INVALID_SCAN_INDEX;
		
		int nPeaksCount;
		if ( !trScan.GetAttribute(STR_PEAKSCOUNT_ATTRIB, nPeaksCount) || nPeaksCount <= 0 )
			return OMZXML_NO_PEAKS_DATA;
		TreeNode trPeaks = trScan.GetNode("peaks");
		if ( !trPeaks )
			return OMZXML_NO_PEAKS_DATA;
		
		int nRet = readOnePeak(trPeaks, nPeaksCount, vMZ, vIntens, nLittleEndian == nOrderTest);
		ASSERT(OMZXML_OK == nRet);
	}
	else if ( XML_TYPE_MZML == m_nFileType )
	{
		TreeNode trRun = tree_get_node_by_tagname(m_trFile, "run", true);
		if ( !trRun )
			return OMZXML_NO_SCAN_DATA;
		TreeNode trSpecList = tree_get_node_by_tagname(trRun, "spectrumList", true);
		if ( !trSpecList )
			trSpecList = tree_get_node_by_tagname(trRun, "chromatogramList", true);
		if ( !trSpecList )
			return OMZXML_NO_SCAN_DATA;
		int nSpecCount;
		if ( !trSpecList.GetAttribute(STR_COUNT_ATTRIB, nSpecCount) || nSpecCount <= 0 )
			return OMZXML_NO_SPEC_DATA;
		
		TreeNode trRefGroupList = m_trFile.GetNode("referenceableParamGroupList");
		//for ( int iSpec = 0; iSpec < nSpecCount; iSpec++ )
		{
			TreeNode trSpec = trSpecList.FindNodeByAttribute(STR_INDEX_ATTRIB, lScanIndex);
			///Sophy 1/13/2011 ORG-S2 SUPPORT_MZML_FILE_BEFORE_VERSION_1_1
			if ( !trSpec )
				trSpec = trSpecList.FindNodeByAttribute(STR_SCANNUMBER_ATTRIB, lScanIndex);
			///end SUPPORT_MZML_FILE_BEFORE_VERSION_1_1
			if ( !trSpec )
				return OMZXML_NO_SPEC_DATA;
			TreeNode trArrayList = tree_get_node_by_tagname(trSpec, "binaryDataArrayList", true);
			///Sophy 1/13/2011 ORG-S2 SUPPORT_MZML_FILE_BEFORE_VERSION_1_1
			if ( !trArrayList && trSpec.GetNode("binaryDataArray") )
				trArrayList = trSpec;
			///end SUPPORT_MZML_FILE_BEFORE_VERSION_1_1
			if ( !trArrayList )
				return OMZXML_NO_SPEC_DATA;
			
			int nDefaultLength = 0;
			trSpec.GetAttribute(STR_DEFAULT_ARRAYLENGTH_ATTRIB, nDefaultLength);
			ASSERT(trArrayList.Children.Count() >= 2);
			foreach(TreeNode trArray in trArrayList.Children)
			{
				TreeNode trParams;
				int nArrayType = ARRAY_TYPE_OTHER;
				TreeNode trCVMZ = trArray.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000514");
				if ( trCVMZ ) //m/z array
				{
					nArrayType = ARRAY_TYPE_MZARRAY;
				}
				TreeNode trCVIntens = trArray.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000515");
				if ( trCVIntens ) //intensity array
				{
					nArrayType = ARRAY_TYPE_INTENSARRAY
				}
				TreeNode trCVCharge = trArray.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000516");
				if ( trCVCharge ) //charge array
				{
				}
				TreeNode trCVSig = trArray.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000517");
				if ( trCVSig ) //signal to noise array
				{
				}
				TreeNode trCVTime = trArray.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000595");
				if ( trCVTime ) //time array
				{
					nArrayType = ARRAY_TYPE_TIMEARRAY;
				}
				TreeNode trCVWaveLen = trArray.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000617");
				if ( trCVWaveLen ) //wave length array
				{
				}
				TreeNode trCVNoStand = trArray.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000786");
				if ( trCVNoStand ) //no standard data array
				{
				}
				TreeNode trCVFlowRate = trArray.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000820");
				if ( trCVFlowRate ) //flow rate array
				{
				}
				TreeNode trCVPressure = trArray.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000821");
				if ( trCVPressure ) //pressure array
				{
				}
				TreeNode trCVTemperature = trArray.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000822");
				if ( trCVTemperature ) //temperature array
				{
				}
				if ( ARRAY_TYPE_OTHER == nArrayType ) //try to get ref param
				{
					TreeNode trRef = trArray.GetNode("referenceableParamGroupRef");
					if ( trRef && trRefGroupList )
					{
						string strGroupID;
						trRef.GetAttribute(STR_REF_ATTRIB, strGroupID);
						trParams = trRefGroupList.FindNodeByAttribute(STR_GROUP_ID_ATTRIB, strGroupID);
						if ( trParams )
						{
							TreeNode trCVMZ = trParams.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000514");
							if ( trCVMZ ) //m/z array
							{
								nArrayType = ARRAY_TYPE_MZARRAY;
							}
							TreeNode trCVIntens = trParams.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000515");
							if ( trCVIntens ) //intensity array
							{
								nArrayType = ARRAY_TYPE_INTENSARRAY;
							}
							TreeNode trCVTime = trParams.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:100595");
							if ( trCVTime )
							{
								nArrayType = ARRAY_TYPE_TIMEARRAY;
							}
						}
					}
				}
				if ( ARRAY_TYPE_MZARRAY == nArrayType )
				{
					readOneBinary(trArray, trParams, vMZ, nDefaultLength);	
				}
				else if ( ARRAY_TYPE_INTENSARRAY == nArrayType )
				{
					readOneBinary(trArray, trParams, vIntens, nDefaultLength);
				}
				else if ( ARRAY_TYPE_TIMEARRAY == nArrayType )
				{
					readOneBinary(trArray, trParams, vTime, nDefaultLength);
				}
			}
		}
		//to do...
	}
	else if ( XML_TYPE_IMZML == m_nFileType )
	{
		TreeNode trSpecList = tree_get_node_by_tagname(m_trFile, "spectrumList", true);
		if ( !trSpecList )
			return OMZXML_NO_SCAN_DATA;
		
		TreeNode trSpec = trSpecList.FindNodeByAttribute(STR_INDEX_ATTRIB, lScanIndex);
		if ( !trSpec )
			return OMZXML_NO_SPEC_DATA;
		
		TreeNode trArrayList = tree_get_node_by_tagname(trSpec, "binaryDataArrayList", true);
		if ( !trArrayList )
			return OMZXML_NO_SPEC_DATA;
		
		TreeNode trRefGroupList = m_trFile.GetNode("referenceableParamGroupList");
		ASSERT(trArrayList.Children.Count() >= 2);
		foreach(TreeNode trArray in trArrayList.Children)
		{
			int nDataType = XML_UNKNOWN;
			bool bCompressed = false;

			int nArrayType = ARRAY_TYPE_OTHER;
			TreeNode trRef = trArray.GetNode("referenceableParamGroupRef");
			TreeNode trParams;
			if ( trRef && trRefGroupList )
			{
				string strGroupID;
				trRef.GetAttribute(STR_REF_ATTRIB, strGroupID);
				trParams = trRefGroupList.FindNodeByAttribute(STR_GROUP_ID_ATTRIB, strGroupID);
				ASSERT(trParams);
				if ( trParams )
				{
					TreeNode trCVMZ = trParams.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000514");
					if ( trCVMZ ) //m/z array
					{
						nArrayType = ARRAY_TYPE_MZARRAY;
					}
					TreeNode trCVIntens = trParams.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000515");
					if ( trCVIntens ) //intensity array
					{
						nArrayType = ARRAY_TYPE_INTENSARRAY;
					}
					TreeNode trCVTime = trParams.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000595");
					if ( trCVTime ) //time array
					{
						nArrayType = ARRAY_TYPE_TIMEARRAY;
					}
					TreeNode trCVTypeF = trParams.FindNodeByAttribute(STR_ACCESSION_ATTRIB,"MS:1000521");
					if ( trCVTypeF )
						nDataType = XML_FLOAT;
					TreeNode trCVTypeD = trParams.FindNodeByAttribute(STR_ACCESSION_ATTRIB,"MS:1000523");
					if ( trCVTypeD )
						nDataType = XML_DOUBLE;
					TreeNode trCVTypeI = trParams.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000519");
					if( trCVTypeI )
						nDataType = XML_INT32;
	
					TreeNode trNoComp = trParams.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000576");
					if ( trNoComp )
						bCompressed = false;
					TreeNode trComp = trParams.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000574");
					if ( trComp )
						bCompressed = true; //zlib compressed.

				}
			}
			{	//in case it is not refer to param group, just for safe.
				TreeNode trCVMZ = trArray.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000514");
				if ( trCVMZ ) //m/z array
				{
					nArrayType = ARRAY_TYPE_MZARRAY;
				}
				TreeNode trCVIntens = trArray.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000515");
				if ( trCVIntens ) //intensity array
				{
					nArrayType = ARRAY_TYPE_INTENSARRAY;
				}
				TreeNode trCVTime = trArray.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000595");
				if ( trCVTime ) //time array
				{
					nArrayType = ARRAY_TYPE_TIMEARRAY;
				}
				TreeNode trCVTypeF = trArray.FindNodeByAttribute(STR_ACCESSION_ATTRIB,"MS:1000521");
				if ( trCVTypeF )
					nDataType = XML_FLOAT;
				TreeNode trCVTypeD = trArray.FindNodeByAttribute(STR_ACCESSION_ATTRIB,"MS:1000523");
				if ( trCVTypeD )
					nDataType = XML_DOUBLE;
				TreeNode trCVTypeI = trArray.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000519");
				if( trCVTypeI )
					nDataType = XML_INT32;

				TreeNode trNoComp = trArray.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000576");
				if ( trNoComp )
					bCompressed = false;
				TreeNode trComp = trArray.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000574");
				if ( trComp )
					bCompressed = true; //zlib compressed.
			}
			if ( ARRAY_TYPE_MZARRAY == nArrayType )
			{
				readOneExternBinary(trArray, trParams, vMZ, nDataType, bCompressed);
			}
			else if ( ARRAY_TYPE_INTENSARRAY == nArrayType )
			{
				readOneExternBinary(trArray, trParams, vIntens, nDataType, bCompressed);
			}
			else
			{
				printf("Unknown type of array will be skipped.\n");
			}
		}
	}
	else
		return OMZXML_INVALID_FILE;
	return OMZXML_OK;
}

int	omzXML::Import(Worksheet& wks, TreeNode& trSel)
{
	if ( !wks )
		return OMZXML_INVALID_OBJECT;
	
	vector<int> vIndices;
	vIndices = m_trHeader.ValidScanIndex.nVals;
	wks.SetSize(-1, vIndices.GetSize() * 2); //a pair of intensity and m/z array
	///Sophy 1/7/2011 ORG-441-S1 SHOW_PROGRESS_BAR_ON_LARGE_DATA_IMPORT
	string strTitle;
	strTitle.Format("Importing %s" ,m_strFileName);
	progressBox pb(strTitle, PBOX_TOPMOST, false);
	pb.SetRange(1, vIndices.GetSize());
	///end SHOW_PROGRESS_BAR_ON_LARGE_DATA_IMPORT
	for ( int iSet = 0; iSet < vIndices.GetSize(); iSet++ )
	{
		///Sophy 1/7/2011 ORG-441-S1 SHOW_PROGRESS_BAR_ON_LARGE_DATA_IMPORT
		pb.Set(iSet + 1);
		if ( pb.IsAbort() )
			return OMZXML_USER_ABORT;
		///end SHOW_PROGRESS_BAR_ON_LARGE_DATA_IMPORT
		vector vIntens, vMZ, vTime;
		int nRet = ReadPeaks(vIndices[iSet], vMZ, vIntens, vTime);
		if ( OMZXML_OK == nRet )
		{
			Column colX(wks, iSet * 2);
			string strXLongName = vMZ.GetSize() > 0 ? "m/z array of scan #" : "time array of scan #";
			strXLongName += vIndices[iSet];
			colX.SetLongName(strXLongName);
			colX.SetType(OKDATAOBJ_DESIGNATION_X);
			Column colY(wks, iSet * 2 + 1);
			colY.SetLongName("intensity array of scan #" + vIndices[iSet]);
			colY.SetType(OKDATAOBJ_DESIGNATION_Y);
			
			vectorbase& vX = colX.GetDataObject();
			vX = vMZ.GetSize() > 0 ? vMZ : vTime;
			vectorbase& vY = colY.GetDataObject();
			vY = vIntens;
		}
		else
		{
			ASSERT(false);
		}
		
	}
	
	string strLabel = _L("MZXML Information");
	set_user_info(wks, strLabel, trSel);

	return OMZXML_OK;
}

int	omzXML::GetFileInfo(TreeNode& trInfo)
{
	trInfo.Replace(m_trHeader, TRUE, TRUE, TRUE);
	TreeNode trIndices = trInfo.GetNode("ValidScanIndex");
	vector<int> vnIndices;
	vnIndices = trIndices.nVals;
	vector<string> vsIndices;
	convert_int_vector_to_string_vector(vnIndices, vsIndices);
	string strIndices;
	strIndices.SetTokens(vsIndices, ',');
	trIndices.Reset();
	LABEL(trIndices, _L("Valid Scan Indices"));
	trIndices.strVal = strIndices;
	tree_set_attribute_to_all_nodes(trInfo, STR_ENABLE_ATTRIB, "2");
	return OMZXML_OK;	
}

int	omzXML::checkValiateIndex(long lScanIndex)
{
	vector<int> vIndices;
	vIndices = m_trHeader.ValidScanIndex.nVals;
	if ( find_in_list(lScanIndex, vIndices, false) < 0 )
		return OMZXML_INVALID_SCAN_INDEX;
	return OMZXML_OK;
}

int	omzXML::readOneExternBinary(const TreeNode& trArray, const TreeNode& trParams, vector<double>& vData, int nDataType, bool bCompressed)
{
	TreeNode trCVArrayLength = trArray.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "IMS:1000103");
	if ( !trCVArrayLength && trParams )
		trCVArrayLength = trParams.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "IMS:1000103");
	int nArrayLength = 0;
	if ( trCVArrayLength )
		trCVArrayLength.GetAttribute(STR_VALUE_ATTRIB, nArrayLength);
	else
	{
		ASSERT(false);
	}
	
	TreeNode trCVEncodedLength = trArray.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "IMS:1000104");
	if ( !trCVEncodedLength && trParams )
		trParams.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "IMS:1000104");
	int nEncodedLength = 0;
	if ( trCVEncodedLength )
		trCVEncodedLength.GetAttribute(STR_VALUE_ATTRIB, nEncodedLength);
	else
	{
		ASSERT(false);
	}
	
	TreeNode trCVOffset = trArray.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "IMS:1000102");
	if ( !trCVOffset && trParams )
		trParams.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "IMS:1000102");
	int nOffset = 0;
	if ( trCVOffset )
		trCVOffset.GetAttribute(STR_VALUE_ATTRIB, nOffset);
	else
	{
		ASSERT(false);
	}
	
	char* pBuffer = (char*)malloc(nEncodedLength * sizeof(char));
	if ( NULL == pBuffer )
		return OMZXML_MEMORY_ERROR;
	
	m_fExtern.Seek(nOffset, file::begin);
	int nReads = m_fExtern.Read(pBuffer, nEncodedLength);
	if ( nReads != nEncodedLength )
	{
		free(pBuffer);
		return OMZXML_INVALID_FILE;
	}

	byte* pDecoded = pBuffer;
	//zlib decompression
	byte* pUncomp = NULL;
	if( bCompressed )
	{
		uint unCompLen = nEncodedLength + 1;
		pUncomp = (byte*)calloc(unCompLen, 1);
		uncompress(pUncomp, &unCompLen, (byte*)pBuffer, nEncodedLength + 1);
		free(pDecoded);
		pDecoded = (char*)pUncomp;
	}	
	vData.SetSize(nArrayLength);
	if ( XML_FLOAT == nDataType )
	{
		float* pf = (float*)pDecoded;
		for ( int ii = 0; ii < nArrayLength; ii++ )
		{
			vData[ii] = (double)(*pf);
			pf++;
		}
	}
	else if ( XML_DOUBLE == nDataType )
	{
		double* pd = (double*)pDecoded;
		for ( int ii = 0; ii < nArrayLength; ii++ )
		{
			vData[ii] = *pd;
			pd++;
		}
	}
	else if ( XML_INT32 == nDataType )
	{
		int* pn = (int*)pDecoded;
		for ( int ii = 0; ii < nArrayLength; ii++ )
		{
			vData[ii] = (double)(*pn);
			pn++;
		}
	}
	else
	{
		ASSERT(false);
		printf("Unknown DataType in i-mzML file.\n");
		free(pDecoded);
		return OMZXML_INVALID_DATATYPE;
	}
	free(pDecoded);
	return OMZXML_OK;
}

int	omzXML::readOneBinary(const TreeNode& trArray, const TreeNode& trParams, vector<double>& vData, int nDefaultLength)
{
	if ( !trArray )
		return OMZXML_INVALID_OBJECT;
	
	int nEncodedLength = 0;
	trArray.GetAttribute(STR_ENCODEDLENGTH_ATTRIB, nEncodedLength);
	
	int nPrecision = 32;
	TreeNode trCVTypeF = trArray.FindNodeByAttribute(STR_ACCESSION_ATTRIB,"MS:1000521");
	if ( !trCVTypeF && trParams )
		trCVTypeF = trParams.FindNodeByAttribute(STR_ACCESSION_ATTRIB,"MS:1000521");
	if ( trCVTypeF )
		nPrecision = 32;
	TreeNode trCVTypeD = trArray.FindNodeByAttribute(STR_ACCESSION_ATTRIB,"MS:1000523");
	if ( !trCVTypeD && trParams )
		trCVTypeD = trParams.FindNodeByAttribute(STR_ACCESSION_ATTRIB,"MS:1000523");
	if ( trCVTypeD )
		nPrecision = 64;
	
	bool bComp = false;
	TreeNode trNoComp = trArray.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000576");
	if ( !trNoComp && trParams )
		trNoComp = trParams.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000576");
	if ( trNoComp )
		bComp = false;
	TreeNode trComp = trArray.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000574");
	if ( !trComp && trParams )
		trComp = trParams.FindNodeByAttribute(STR_ACCESSION_ATTRIB, "MS:1000574");
	if ( trComp )
		bComp = true; //zlib compressed.
	
	TreeNode trBinary = trArray.GetNode("binary");
	ASSERT(trBinary);
	
	string strData = trBinary.strVal;
	//whitespace maybe present in base64 char stream
	strData.Remove('\t');
	strData.Remove('\n');
	strData.Remove('\r');
	strData.Remove(' ');
	int nCC = strData.GetLength();
	char* pBuffer = strData.GetBuffer(strData.GetLength());
	
	int nLength = nDefaultLength;
	if ( nLength <= 0 )
		trArray.GetAttribute(STR_ARRAYLENGTH_ATTRIB, nLength);
	
	int nBytes = nLength * (nPrecision / 8);
	///Sophy 2/22/2011 ORG-441-S2 SUPPORT_MZML_FILE_BEFORE_VERSION_1_1
	int nDecodeSize = (nCC / 4 + ((nCC % 4) ? 1 : 0)) * 3;
	nBytes = max(nDecodeSize, nBytes) + 1;
	///end SUPPORT_MZML_FILE_BEFORE_VERSION_1_1

	char* pDecoded = (char*)malloc(nBytes);
	if ( pDecoded == NULL )
		return OMZXML_MEMORY_ERROR;
	
	_b64_decode_mio(pDecoded, pBuffer);
	
	//zlib decompression
	byte* pUncomp = NULL;
	if( bComp )
	{
		uint unCompLen = nBytes + 1;
		pUncomp = (byte*)calloc(unCompLen, 1);
		uncompress(pUncomp, &unCompLen, (byte*)pDecoded, nBytes + 1);
		free(pDecoded);
		pDecoded = (char*)pUncomp;
	}
	int nType = (32 == nPrecision) ? FSI_REAL : FSI_DOUBLE;
	/*
	vData.SetSize(nLength);
	if ( 32 == nPrecision ) //floats
	{
		float *pf = (float*)pDecoded;
		for ( int ii = 0; ii < nLength; ii++ )
		{
			vData[ii] = (double)(*pf);
			pf++;
		}
	}
	else //doubles
	{
		double *pd = (double*)pDecoded;
		for ( int ii = 0; ii < nLength; ii++ )
		{
			vData[ii] = *pd;
			pd++;
		}
	}
	*/
	okutil_buffer_to_vector(pDecoded, nType, nLength, &vData);
	strData.ReleaseBuffer();
	free(pDecoded);
	return OMZXML_OK;
}

int	omzXML::readOneArray(const TreeNode& trArray, vector<double>& vData, bool bGLittleEndian) //mzData
{
	if ( !trArray || !trArray.FirstNode )
		return OMZXML_INVALID_OBJECT;
	
	TreeNode trData = trArray.GetNode("data", true);
	if ( !trData )
		return OMZXML_NO_PEAKS_DATA;
	
	int nPrecision;
	if ( !trData.GetAttribute(STR_PRECISION_ATTRIB, nPrecision) )
		nPrecision = 32;
	
	int nLength;
	if ( !trData.GetAttribute(STR_LENGTH_ATTRIB, nLength) || nLength <= 0 )
		return OMZXML_INVALID_PEAKSCOUNT;
	
	string strEndian;
	bool bLittleEndian = true;
	if ( trData.GetAttribute(STR_ENDIAN_ATTRIB, strEndian) && strEndian.CompareNoCase("little") != 0 )
		bLittleEndian = false;
	
	string strData = trData.strVal;
	//whitespace maybe present in base64 char stream
	strData.Remove('\t');
	strData.Remove('\n');
	strData.Remove('\r');
	strData.Remove(' ');
	char* pBuffer = strData.GetBuffer(strData.GetLength());
	//base64 has 4:3 bloat, precision/8 bytes per value
	int nBytes = nLength * (nPrecision / 8);
	
	char* pDecoded = (char*)malloc(nBytes + nLength);
	if ( pDecoded == NULL )
		return OMZXML_MEMORY_ERROR;
	
	_b64_decode_mio(pDecoded, pBuffer);

	bool bByteOrderOK = (bGLittleEndian == bLittleEndian);
	/*
	vData.SetSize(nLength); //peaksCount
	if ( 32 == nPrecision ) //floats
	{
		if ( bByteOrderOK )
		{
			float *pf = (float*)pDecoded;
			for ( int ii = 0; ii < nLength; ii++ )
			{
				vData[ii] = (double)(*pf);
				pf++;
			}
		}
		else
		{
			byte *pb = (byte*)pDecoded;
			for ( int ii = 0; ii < nLength; ii++ )
			{
				SWAPBYTES(pb, sizeof(float));
				float *pf = (float*)pb;
				vData[ii] = (double)(*pf);
				pb += 4;//float occupies 4 bytes
			}
		}
	}
	else	//doubles
	{
		if ( bByteOrderOK )
		{
			double *pd = (double*)pDecoded;
			for ( int ii = 0; ii < nLength; ii++ )
			{
				vData[ii] = *pd;
				pd++;
			}
		}
		else
		{
			byte *pb = (byte*)pDecoded;
			for ( int ii = 0; ii < nLength; ii++ )
			{
				SWAPBYTES(pb, sizeof(double));
				double *pd = (double*)pb;
				vData[ii] = (double)(*pd);
				pd += 8;//double occupies 8 bytes
			}
		}
	}
	*/
	int nType = (32 == nPrecision) ? FSI_REAL : FSI_DOUBLE;
	okutil_buffer_to_vector(pDecoded, nType, nLength, &vData, bByteOrderOK);
	strData.ReleaseBuffer();
	free(pDecoded);
	return OMZXML_OK;
}


int	omzXML::readOnePeak(const TreeNode& trPeaks, const int nPeaksCount, vector<double>& vMZ, vector<double>& vIntens, bool bGLittleEndian)
{
	int nPrecision;
	if ( !trPeaks.GetAttribute(STR_PRECISION_ATTRIB, nPrecision) )
		nPrecision = 32;
	
	bool bLittleEndian = true;
	string strByteOrder;
	if ( trPeaks.GetAttribute(STR_BYTEORDER_ATTRIB, strByteOrder) && strByteOrder.CompareNoCase("network") == 0 )
		bLittleEndian = false;
	
	//content type attribute
	string strCntType;
	int nContentType = CONTENT_MZINT;
	if ( trPeaks.GetAttribute("contentType", strCntType) )
	{
		if ( strCntType.CompareNoCase("m/z ruler") == 0 )
			nContentType = CONTENT_MZRULER;
		else if ( strCntType.CompareNoCase("m/z-int") == 0 )
			nContentType = CONTENT_MZINT;
		else
			return OMZXML_INVALID_CONTENTTYPE;
	}
	//compression attribute
	bool bCompressed = false;
	string strCompCode;
	if ( trPeaks.GetAttribute("compressionType", strCompCode) )
	{
		if ( strCompCode.CompareNoCase("zlib") == 0 )
			bCompressed = true;
		else if ( strCompCode.CompareNoCase("none") == 0 )
			bCompressed = false;
		else
			return OMZXML_INVALID_COMPRESSTYPE;
	}
	int nCompLen;
	trPeaks.GetAttribute("compressedLen", nCompLen);
	
	//
	int nBytes = 0;
	if ( bCompressed )
		nBytes = nCompLen;
	else
		nBytes = nPeaksCount * (nPrecision / 4);
	
	//for every 3 bytes, base64 emites 4 characters - 1, 2 or 3 byte input emit 4 bytes
	int nTriplets = (nBytes / 3) + ((nBytes % 3) != 0 );
	int nPeaksLen = (4 * nTriplets ) + 1;
	
	string strData = trPeaks.strVal;
	strData.Remove('\t');
	strData.Remove('\n');
	strData.Remove('\r');
	strData.Remove(' ');
	char* pBuffer = strData.GetBuffer(strData.GetLength());
	
	//decode
	char* pDecoded = (char*)malloc(nBytes + 1);
	if ( pDecoded == NULL )
		return OMZXML_MEMORY_ERROR;
	
	//decoding
	_b64_decode_mio(pDecoded, pBuffer);
	
	byte* pUncomp = NULL;
	//zlib decompression
	if ( bCompressed )
	{
		///Sophy 1/7/2011 ORG-441-P1 IMPORT_ZLIB_COMPRESSED_MZXML_CAUSE_MEMORY_ACCESS_VIOLATION
		//uint unCompLen = nBytes + 1;
		uint unCompLen = nPeaksCount * (nPrecision / 4) + 1;
		///end IMPORT_ZLIB_COMPRESSED_MZXML_CAUSE_MEMORY_ACCESS_VIOLATION
		pUncomp = (byte*)calloc(unCompLen, 1);
		uncompress(pUncomp, &unCompLen, (byte*)pDecoded, nBytes + 1);
		free(pDecoded);
		pDecoded = (char*)pUncomp;
	}
	
	bool bByteOrderOK = (bLittleEndian == bGLittleEndian);
	/*
	vMZ.SetSize(nPeaksCount);
	vIntens.SetSize(nPeaksCount);
	if ( 32 == nPrecision ) //floats
	{
		if ( bByteOrderOK )
		{
			float *pf = (float*)pDecoded;
			for ( int ii = 0; ii < nPeaksCount; ii++ )
			{
				vMZ[ii] = (double)(*pf);
				pf++;
				vIntens[ii] = (double)(*pf);
				pf++;
			}
		}
		else
		{
			byte* pb = (byte*)pDecoded;
			for ( int ii = 0; ii < nPeaksCount; ii++ )
			{
				SWAPBYTES(pb, sizeof(float));
				float* pf = (float*)pb;
				vMZ[ii] = (double)(*pf);
				pb += sizeof(float);
				SWAPBYTES(pb, sizeof(float));
				pf = (float*)pb;
				vIntens[ii] = (double)(*pf);
				pb += sizeof(float);
			}
		}
	}
	else //doubles
	{
		if ( bByteOrderOK )
		{
			double *pd = (double*)pDecoded;
			for ( int ii = 0; ii < nPeaksCount; ii++ )
			{
				vMZ[ii] = (*pd);
				pd++;
				vIntens[ii] = (*pd);
				pd++;
			}
		}
		else
		{
			byte* pb = (byte*)pDecoded;
			for ( int ii = 0; ii < nPeaksCount; ii++ )
			{
				SWAPBYTES(pb, sizeof(double));
				double* pd = (double*)pb;
				vMZ[ii] = (*pd);
				pb += sizeof(double);
				SWAPBYTES(pb, sizeof(double));
				pd = (double*)pb;
				vIntens[ii] = (*pd);
				pb += sizeof(double);
			}
		}
	}
	
	if ( nContentType == CONTENT_MZRULER ) //convert to m/z -int pairs
	{
		vector<double> vMZDeRuled(nPeaksCount);
		vector<double> vIntensDeRuled(nPeaksCount);
		
		int nMultiplier = 0;
		double dLastMass = 0, dDeltaMass = 0;
		for ( int ii = 0; ii < nPeaksCount; ii++ )
		{
			if ( (int)vMZ[ii] == -1 )
			{
				dLastMass = vIntens[ii];
				if ( ii < nPeaksCount - 1 )
					dDeltaMass = vMZ[ii + 1];
				else
					dDeltaMass = -1;
				nMultiplier = 0;
			}
			vMZDeRuled[ii] = dLastMass + (double)nMultiplier * dDeltaMass;
			nMultiplier++;
			vIntensDeRuled[ii] = vIntens[ii];
		}
		vMZ = vMZDeRuled;
		vIntens = vIntensDeRuled;
	}
	*/
	int nType = (32 == nPrecision) ? FSI_REAL : FSI_DOUBLE;
	okutil_buffer_to_vectors(pDecoded, nType, nPeaksCount, &vMZ, &vIntens, bByteOrderOK, CONTENT_MZRULER == nContentType);
	free(pDecoded);
	return OMZXML_OK;
}

//#pragma dll(zlib1)
//
//int uncompress(byte* btSrc, uint* uSrcLen, byte* btDst, uint uDstLen);
