/*------------------------------------------------------------------------------*
 * File Name: statEx_utils.c 													*
 * Purpose: Purpose: Advanced Statistics Functions							   	*
 * Creation: 11/23/2005, Max													*
 * Copyright (c) OriginLab Corp.	2003, 2004, 2005, 2006, 2007, 2008, 2009	*
 * All Rights Reserved															*
 * 																				*
 * Modification Log:															*
 *	Bill 05/12/2011 ORG-2700 FUNCTION_TO_NORMALIZE_MATRIX						*
 *	Bill 06/14/2011 ORG-2698 COMMAND_FUNCTION_FOR_MULTIVARIATE_XF				*
 *------------------------------------------------------------------------------*/
 
#include <Origin.h>
#include <statEx_utils.h>
#include <OC_nag.h> /// Bill 06/14/2011 ORG-2698 COMMAND_FUNCTION_FOR_MULTIVARIATE_XF

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////////
//
//				Repeated Measure ANOVA Functions									
//
/////////////////////////////////////////////////////////////////////////////////

static bool _check_level_equality(const vector<int> &vData)
{
	vector<int> vcount;
	
	if (vData.GetSize() < 2)
		return false;
	int ncount = 1;
	for(int ii= 0; ii < vData.GetSize() - 1; ii++)
	{
		if(vData[ii] == vData[ii+1])
		{	
			ncount++;
			if (ii == vData.GetSize() - 2)
				vcount.Add(ncount);
		}
		else
		{	
			vcount.Add(ncount);
			if (ii == vData.GetSize() - 2)
				vcount.Add(1);
			ncount = 1;
		}

	}
	
	if (vcount.GetSize() < 2)
		return false;
	for(ii= 0; ii < vcount.GetSize() - 1; ii++)
	{
		for(int jj= ii+1; jj < vcount.GetSize() ; jj++)
			if (vcount[jj] != vcount[ii])
				return false;
	}
	
	return true;
}



/**

*/

int  stats_rm_anova_one_way(const MeanCompMethod& nIntervalType,
							const vector &vData,
				  			int nSizeSubject,
							int nGroupSize,
							double dAlpha, 
							int ncntrl,
							RMANOVADescStats* pDescStats, uint nSizeDescStat,
							RMANOVAOneWay *RMANOVATable, 
							MeanCompStats *pMeanCompStats, uint nSizeMeanComp,
							MultiTestTable *RVMultiStats,
							SphericityStats *RVSphericStats,
							EpsilonStats *RVEpsilonStats)
							
{
	// Terminate if error setting
	if (nSizeSubject < 1 || vData.GetSize() < 1)
		return STATS_ERROR_TOO_FEW_DATA_PTS;

	if (nGroupSize < 2)
		return STATS_ERROR_TOO_FEW_GROUPS;
	
	if (dAlpha <= 0 || dAlpha >= 1)
		return STATS_ERROR_INVALID_CONF_LEVEL;

	/*
	if (nIntervalType < 1 || nIntervalType > MEANS_COMPARISON_METHODS_NUMBER)
		return STATS_ERROR_BAD_PARAM;
	*/
	
	if ((ncntrl < 1 || ncntrl > nGroupSize) && nIntervalType.DUNNETT)
		return STATS_ERROR_SETTING;
	
	int nMissing = ocmath_count(NANUM, vData.GetSize(), vData);

	
	if (vData.GetSize() != nSizeSubject * nGroupSize || nMissing > 0)
		return STATS_ERROR_UNBALANCED_DESIGN;

	int nRet = ocStatEx_rm_anova_one_way(&nIntervalType,
										vData, vData.GetSize(),
				  			  			nSizeSubject,
							  			nGroupSize,
								  		dAlpha, 
									 	ncntrl,
									 	pDescStats, nSizeDescStat,
									 	RMANOVATable, 
									 	pMeanCompStats, nSizeMeanComp,
									 	RVMultiStats,
									 	RVSphericStats,
									 	RVEpsilonStats);
	
	return nRet;	
}


/**

*/						
int stats_rm_anova_two_way_one_rv(const MeanCompMethod& nIntervalType,
							int iInteraction,
							const vector &vData,
							const vector<int> &vNonRVData,
							int nSizeSubject,
							int nRVGroupSize,
							int nNonRVGroupSize,
							double dAlpha, 
							int ncntrl1,
							int ncntrl2,
							RMANOVADescStats* pRVDescStats, uint nSizeRVDescStat,
							RMANOVADescStats* pNonRVDescStats, uint nSizeNonRVDescStat,
							RMANOVADescStats* pInterDescStats, uint nSizeInterDescStat,
							RMANOVATwoWayOneRV *RMANOVATable, 
							MeanCompStats *pRVMeanCompStats, uint nSizeRVMeanComp,
							MeanCompStats *pNonRVMeanCompStats, uint nSizeNonRVMeanComp,
							MeanCompStats *pFixNonRVMeanCompStats, uint nSizeFixNonRVMeanComp,
							MeanCompStats *pFixRVMeanCompStats, uint nSizeFixRVMeanComp,
							MultiTestTable *RVMultiStats,
							MultiTestTable *InterMultiStats,
							SphericityStats *RVSphericStats,
							EpsilonStats *RVEpsilonStats)
{
	// Terminate if error setting
	if (iInteraction != 0 && iInteraction != 1)
		return STATS_ERROR_SETTING;
	
	if (nSizeSubject < 1 || vData.GetSize() <1 || vNonRVData.GetSize() <1)
		return STATS_ERROR_TOO_FEW_DATA_PTS;

	if (nRVGroupSize < 2 || nNonRVGroupSize < 2)
		return STATS_ERROR_TOO_FEW_GROUPS;
	
	if (dAlpha <= 0 || dAlpha >= 1)
		return STATS_ERROR_INVALID_CONF_LEVEL;

	/*
	if (nIntervalType < 1 || nIntervalType > MEANS_COMPARISON_METHODS_NUMBER)
		return STATS_ERROR_BAD_PARAM;
	*/
	
	if (nNonRVGroupSize > nSizeSubject)
		return STATS_ERROR_SETTING;

	if (((ncntrl1 < 1 || ncntrl1 > nRVGroupSize) || (ncntrl2 < 1 || ncntrl2 > nNonRVGroupSize)) && nIntervalType.DUNNETT)
		return STATS_ERROR_SETTING;
	
	bool bEqual = _check_level_equality(vNonRVData);
	
	int nMissing = ocmath_count(NANUM, vData.GetSize(), vData);
	
	if (vData.GetSize() != nSizeSubject * nRVGroupSize || nSizeSubject != vNonRVData.GetSize() || nMissing > 0 || bEqual == false)
		return STATS_ERROR_UNBALANCED_DESIGN;
	
	for (int ii = 0; ii< vNonRVData.GetSize(); ii++)
	{
		if (vNonRVData[ii] > nNonRVGroupSize)
			return STATS_ERROR_SETTING;
	}
	
	int nRet = ocStatEx_rm_anova_two_way_one_rv(&nIntervalType,
												iInteraction,
												vData, vData.GetSize(),
												vNonRVData, vNonRVData.GetSize(),
												nSizeSubject,
												nRVGroupSize,
												nNonRVGroupSize,
												dAlpha, 
												ncntrl1,
												ncntrl2,
												pRVDescStats, nSizeRVDescStat,
												pNonRVDescStats, nSizeNonRVDescStat,
												pInterDescStats, nSizeInterDescStat,
												RMANOVATable, 
												pRVMeanCompStats, nSizeRVMeanComp,
												pNonRVMeanCompStats, nSizeNonRVMeanComp,
												pFixNonRVMeanCompStats, nSizeFixNonRVMeanComp,
												pFixRVMeanCompStats, nSizeFixRVMeanComp,
												RVMultiStats,
												InterMultiStats,
												RVSphericStats,
												RVEpsilonStats);
	
	return nRet;	
}


/**

*/
int stats_rm_anova_two_way_two_rv(const MeanCompMethod& nIntervalType,
							int iInteraction,
							const vector &vData,
							int nSizeSubject,
							int nGroupSize,
							int nRV1Levels,
							int nRV2Levels,
							double dAlpha, 
							int ncntrl1,
							int ncntrl2,
							RMANOVADescStats* pRV1DescStats, uint nSizeRV1DescStat,
							RMANOVADescStats* pRV2DescStats, uint nSizeRV2DescStat,
							RMANOVADescStats* pInterDescStats, uint nSizeInterDescStat,
							RMANOVATwoWayTwoRV *RMANOVATable, 
							MeanCompStats *pRV1MeanCompStats, uint nSizeRV1MeanComp,
							MeanCompStats *pRV2MeanCompStats, uint nSizeRV2MeanComp,
							MeanCompStats *pFixRV2MeanCompStats, uint nSizeFixRV2MeanComp,
							MeanCompStats *pFixRV1MeanCompStats, uint nSizeFixRV1MeanComp,
							MultiTestTable *RV1MultiStats,
							MultiTestTable *RV2MultiStats,
							MultiTestTable *InterMultiStats,
							SphericityStats *RV1SphericStats,
							SphericityStats *RV2SphericStats,
							SphericityStats *InterSphericStats,
							EpsilonStats *RV1EpsilonStats,
							EpsilonStats *RV2EpsilonStats,
							EpsilonStats *InterEpsilonStats)
{
	// Terminate if error setting
	if (iInteraction != 0 && iInteraction != 1)
		return STATS_ERROR_SETTING;

	if (nSizeSubject < 1 || vData.GetSize() <1 )
		return STATS_ERROR_TOO_FEW_DATA_PTS;

	if (nGroupSize < 2 || nRV1Levels < 2 || nRV2Levels < 2)
		return STATS_ERROR_TOO_FEW_GROUPS;
	
	if (dAlpha <= 0 || dAlpha >= 1)
		return STATS_ERROR_INVALID_CONF_LEVEL;

	/*
	if (nIntervalType < 1 || nIntervalType > MEANS_COMPARISON_METHODS_NUMBER)
		return STATS_ERROR_BAD_PARAM;
	*/

	if (nGroupSize != nRV1Levels * nRV2Levels || nRV1Levels > nGroupSize || nRV2Levels > nGroupSize)
		return STATS_ERROR_SETTING;

	if (((ncntrl1 < 1 || ncntrl1 > nRV1Levels) || (ncntrl2 < 1 || ncntrl2 > nRV2Levels)) && nIntervalType.DUNNETT)
		return STATS_ERROR_SETTING;
	
	int nMissing = ocmath_count(NANUM, vData.GetSize(), vData);
	
	if (vData.GetSize() != nSizeSubject * nGroupSize || nMissing > 0)
		return STATS_ERROR_UNBALANCED_DESIGN;
	
	int nRet = ocStatEx_rm_anova_two_way_two_rv(&nIntervalType,
												iInteraction,
												vData, vData.GetSize(),
												nSizeSubject,
												nGroupSize,
												nRV1Levels,
												nRV2Levels,
												dAlpha, 
												ncntrl1,
												ncntrl2,
												pRV1DescStats, nSizeRV1DescStat,
												pRV2DescStats, nSizeRV2DescStat,
												pInterDescStats, nSizeInterDescStat,
												RMANOVATable, 
												pRV1MeanCompStats, nSizeRV1MeanComp,
												pRV2MeanCompStats, nSizeRV2MeanComp,
												pFixRV2MeanCompStats, nSizeFixRV2MeanComp,
												pFixRV1MeanCompStats, nSizeFixRV1MeanComp,
												RV1MultiStats,
												RV2MultiStats,
												InterMultiStats,
												RV1SphericStats,
												RV2SphericStats,
												InterSphericStats,
												RV1EpsilonStats,
												RV2EpsilonStats,
												InterEpsilonStats);
	
	return nRet;	
}

/// Bill 05/12/2011 ORG-2700 FUNCTION_TO_NORMALIZE_MATRIX
bool std_matrix(const matrix & m, const int nMethod, matrix & mNormalize)
{
	if(nMethod == NORMALIZE_NONE ) //Normalize: None
	{
		mNormalize = m;
	}
	else if (nMethod == NORMALIZE_STD || nMethod == NORMALIZE_STD_N)
	{
		int i, nCol, nRow;
		
		nCol = m.GetNumCols();
		nRow = m.GetNumRows();
		mNormalize.SetSize(nRow, nCol);
		
		for(i = 0; i < nCol; i++)
		{
			vector vCol;
			m.GetColumn(vCol, i);
		
			if(!_normalize_data(vCol, nMethod))
				return false;

			mNormalize.SetColumn(vCol, i);
		}		
	}
	
	return true;
}

static bool _normalize_data(vector& vv, int nMethod)
{
	if (0 == vv.GetSize() )
		return false;
	
	double dVal = 1.0, dSub = 0.0, dFactor = 1.0;
	if (!_get_normalize_val(dVal, dSub, vv, nMethod) )
		return false;
	
	dFactor = (is_equal(0.0, dVal)) ? NANUM : 1.0 / dVal;	
	vv -= dSub;
	vv *= dFactor;
	
	return true;
}

static bool _get_normalize_val(double& dNormalizeVal, double& dSub, const vector& vv, int nMethod)
{
	int nRet = 0;

	switch (nMethod )
	{
	case NORMALIZE_STD: //Normalize to N(0, 1)
		nRet = ocmath_basic_summary_stats(vv.GetSize(), vv, NULL, &dSub, &dNormalizeVal);
		break;
		
	case NORMALIZE_STD_N: //Normalize to (0, 1)
		double 		dMin, dMax;
		vv.GetMinMax(dMin, dMax);
		dSub = dMin;
		dNormalizeVal = dMax - dMin;
		break;
		
	default:
		return false;
	}
	
	return (nRet == 0) ? true : false;
}
/// End FUNCTION_TO_NORMALIZE_MATRIX

/// Bill 06/14/2011 ORG-2698 COMMAND_FUNCTION_FOR_MULTIVARIATE_XF
int solve_linear_uppertri_equation(const matrix mA, matrix &mB, const bool bTran /*= false*/)
{
	int nRowA, nColA, nRowB, nColB;
	nRowA = mA.GetNumRows();
	nColA = mA.GetNumCols();
	nRowB = mB.GetNumRows();
	nColB = mB.GetNumCols();

	
	if(nRowA < 1 || nColA < 1 || nRowB < 1 || nColB < 1)
		return ERR_MATRIX_EMPTY;

	if((!bTran && (nRowA != nRowB)) || (bTran && (nColA != nRowB)) || nRowA != nColA)
		return ERR_SIZE_NOT_MATCH;
	
	Nag_TransType trans = Nag_NoTrans;
	if(bTran)
		trans = Nag_Trans;
		
	vector<int> ipiv(nRowA);
	ipiv.Data(1, nRowA);
	
	NagError fail;
	f07aec(Nag_RowMajor, trans, nRowA, nColB, mA, nColA, ipiv, mB, nColB, &fail);
	
	if(fail.code != NE_NOERROR)
	{
		return ERR_NAG_f07aec;
	}
	
	return NO_ERROR;
				
}

int matrix_prod(const matrix mA, const matrix mB, matrix &mC, bool bTranA /*= false*/, bool bTranB /*= false*/)
{
	
	int nRowA, nRowB, nColA, nColB;
	nRowA = mA.GetNumRows();
	nColA = mA.GetNumCols();
	nRowB = mB.GetNumRows();
	nColB = mB.GetNumCols();
	
	if(!(nRowA && nColA && nRowB && nColB)) 
		return ERR_MATRIX_EMPTY;
	
	int nDimA = nColA, nDimB = nRowB, nDimA2 = nRowA, nDimB2 = nColB;
	if(bTranA)
	{
		nDimA = nRowA;
		nDimA2 = nColA;
	}

	if(bTranB)
	{
		nDimB = nColB;
		nDimB2 = nRowB;
	}

	if (nDimA != nDimB)
		return ERR_SIZE_NOT_MATCH;
		
	mC.SetSize(nDimA2, nDimB2);
	
	MatrixTranspose transA = NoTranspose, transB = NoTranspose;
	
	if(bTranA)
		transA = Transpose;

	if(bTranB)
		transB = Transpose;
    
	f06yac(transA, transB, nDimA2, nDimB2, nDimA, 1.0,
		mA, nColA, mB, nColB, 0, mC, nDimB2);
	
	return NO_ERROR;
}

bool get_upper_triangular_mat(const int nSize, const vector & vUpper, matrix &mUpper, bool bRow)
{
	if( vUpper.GetSize() != ( nSize + 1 )*nSize/2 )
		return false;
	
	if( !mUpper.SetSize( nSize, nSize ) )
		return false;
	
	int nInd = 0;
	
	if( bRow )
	{
		for( int nRow = 0; nRow < nSize; nRow++ )
		{
			for( int nCol = nRow; nCol < nSize; nCol++ )
			{
				mUpper[nRow][nCol] = vUpper[ nInd ];			
				nInd = nInd + 1;
			}
		}
	}
	else
	{
		for( int nCol = 0; nCol < nSize; nCol++ )
		{
			for( int nRow = 0; nRow <= nCol; nRow++ )
			{
				mUpper[nRow][nCol] = vUpper[ nInd ];			
				nInd = nInd + 1;
			}
		}
	}
	
	return true;
}

bool get_sign_eigenvector(const matrix & mV, vector<int> &vnSign)
{
	if(vnSign.SetSize(mV.GetNumCols()) == false)
		return false;

	vnSign = 1;
	
	for(int nCol = 0; nCol < mV.GetNumCols(); nCol++)
  	{
  		vector vCol;
  		mV.GetColumn(vCol, nCol);
  		
  		double dSum;
  		if(vCol.Sum(dSum) != 0)
  			return false;
  		
  		if(dSum < 0)
  			vnSign[nCol] = -1;
  	}
  	
  	return true;
}

bool correct_sign_column(const vector<int> & vnSign, matrix &mV)
{
	if(vnSign.GetSize() != mV.GetNumCols())
		return false;
	
	for(int nCol = 0; nCol < mV.GetNumCols(); nCol++)
	{
		if(vnSign[nCol] < 0)
		{
			vector vCol;
			mV.GetColumn(vCol, nCol);
			vCol = -vCol;
			mV.SetColumn(vCol, nCol);
		}
	}
		
	return true;

}

bool get_corr_cov(const matrix & mX, vector &vMean, matrix &mCov, matrix &mCorr)
{
	vMean.SetSize(mX.GetNumCols());
	mCov.SetSize(mX.GetNumCols(), mX.GetNumCols());
	mCorr.SetSize(mX.GetNumCols(), mX.GetNumCols());
	
	double sw;
	vector std(mX.GetNumCols());
	
	NagError fail;
	g02bxc(mX.GetNumRows(), mX.GetNumCols(), mX, mX.GetNumCols(),
		NULL, NULL, &sw, vMean, std, mCorr, mX.GetNumCols(), mCov, mX.GetNumCols(), &fail);
		
	if(fail.code != NE_NOERROR)
	{
		return false;
	}
	
	return true;
}
/// End COMMAND_FUNCTION_FOR_MULTIVARIATE_XF

/// Bill 06/20/2011 2697 COMMAND_FUNCTION_FOR_MULTIVARIATE_XF
bool get_corr_cov_pairwise(const matrix mX, matrix &mC, const bool bCorr)
{
	
	if(bCorr)
	{
		if(!mC.SetSize(mX.GetNumCols(), mX.GetNumCols()))
			return false;
		if(ocmath_corr_coeff(mX.GetNumRows(), mX.GetNumCols(), mX, mC, NULL, NULL, NULL, NULL, NULL))
			return false;
	}
	else
		if(!get_cov_mat_pairwise(mX, mC))
			return false;
	
	return true;	
}

bool get_basic_stat(const matrix mX, vector &vMean, vector &vStd)
{
	if(!vMean.SetSize(mX.GetNumCols()) || !vStd.SetSize(mX.GetNumCols()))
		return false;
	
	matrix mXt = mX;
	if(!mXt.Transpose())
		return false;
	
	if(ocmath_row_desc_stats(mXt.GetNumRows(), mXt.GetNumCols(), mXt, vMean, vStd))
		return false;
	
	return true;
}

bool solve_eigen_from_symmetric_mat(const matrix mX, vector &vR, matrix &mV)
{
	if(mX.GetNumRows() != mX.GetNumCols())
		return false;
	
	if(mX.GetNumRows() < 1)
		return false;
	
	if(!vR.SetSize(mX.GetNumRows()))
		return false;
	
	if(!mV.SetSize(mX.GetNumRows(), mX.GetNumRows()))
		return false;

	NagError fail;
	
	f02abc(mX.GetNumRows(), mX, mX.GetNumCols(),
		vR, mV, mX.GetNumRows(), &fail);
		
	if(fail.code != NE_NOERROR)
	{
		return false;
	}
	
	vector<uint> vInd;
	if(!vR.Sort(SORT_DESCENDING, true, vInd))
		return false;

	if(!mV.Transpose())
		return false;
	
	if(!mV.ReorderRowWise(vInd))
		return false;
	
	if(!mV.Transpose())
		return false;
	
	vector<int> vnSign;
	if(!get_sign_eigenvector(mV, vnSign))
		return false;
	
	if(!correct_sign_column(vnSign, mV))
		return false;
	
	return true;
}

bool normalize_matrix(matrix &mX, vector &vMean, vector &vStd)
{
	vector vMeant;
	if(vMean == NULL)
	{
		if(!vMeant.SetSize(mX.GetNumCols()))
			return false;
	}
	else
	{
		if(mX.GetNumCols() != vMean.GetSize())
			return false;
		vMeant = vMean;
	}
	
	vector vStdt;
	if(vStd == NULL)
	{
		if(!vStdt.SetSize(mX.GetNumCols()))
			return false;
		
		vStdt = 1;
	}
	else
	{
		if(vStd.GetSize() != mX.GetNumCols())
			return false;
		
		vStdt = vStd;
		
		if(!vStdt.Replace(0, 1, MATREPL_TEST_EQUAL))
			return false;
	}
		
	NagError fail;
	
	vector<int> isx(mX.GetNumCols());
	isx = 1;
	g03zac(mX.GetNumRows(), mX.GetNumCols(), mX, mX.GetNumCols(), mX.GetNumCols(), isx,
		vStdt, vMeant, mX, mX.GetNumCols(), &fail);
	
	if(fail.code != NE_NOERROR)
	{
		return false;
	}
	
	return true;	
}

bool get_remain_index(const vector<uint> vnInd, const uint nLen, vector<uint> &vnR)
{
	if(vnInd.GetSize() > nLen)
		return false;
	if(vnInd.GetSize() == nLen)
		return true;
	
	vector<uint> vRow(nLen);
	vRow = 1;
	
	if(vRow.Replace(vnInd, 2))
		return false;
	
	if(vRow.Find(MATREPL_TEST_EQUAL, 1, vnR) < 0)
		return false;
	
	return true;		
}

bool get_cov_mat_pairwise(const matrix mX, matrix &mCov)
{
	if(!mCov.SetSize(mX.GetNumCols(), mX.GetNumCols()))
		return false;
	
	for(int nCol1 = 0; nCol1 < mX.GetNumCols(); nCol1++)
		for(int nCol2 = 0; nCol2 <= nCol1; nCol2++)
		{
			vector vX, vY;
			if(!mX.GetColumn(vX, nCol1) || !mX.GetColumn(vY, nCol2))
				return false;
			
			if(!get_cov_pairwise(vX, vY, mCov[nCol1][nCol2]))
				return false;
			
			if(nCol2 != nCol1)
				mCov[nCol2][nCol1] = mCov[nCol1][nCol2];
		}
		
	return true;
}

bool subtract_col_mean(const matrix mX, matrix &mXs, vector &vMean/* = NULL*/)
{
	if(vMean && vMean.GetSize() != mX.GetNumCols())
		return false;
	
	mXs = mX;
	for(int nCol = 0; nCol < mXs.GetNumCols(); nCol++)
	{
		vector vCol;
		mXs.GetColumn(vCol, nCol);
		
		double dMean;
		if(vMean == NULL)
		{
			if(ocmath_basic_summary_stats(vCol.GetSize(), vCol, NULL, &dMean))
				return false;
		}
		else
			dMean = vMean[nCol];
		
		vCol = vCol - dMean;
		mXs.SetColumn(vCol, nCol);
	}
	
	return true;
}

bool get_cov_pairwise(const vector vX, const vector vY, double &dCov)
{
	if(vX.GetSize() != vY.GetSize())
		return false;
	
	matrix mX(vX.GetSize(), 2);
	if(mX.SetColumn(vX, 0))
		return false;

	if(mX.SetColumn(vY, 1))
		return false;
	
	mX.RemoveEmptyRows(false);
	
	if(mX.GetNumRows() < 2)
		return false;
	
	matrix mXs;
	if(!subtract_col_mean(mX, mXs))
		return false;
	
	if(mXs.CumulativeProduct(mXs, 2))
		return false;
	
	vector vS;
	if(!mXs.GetColumn(vS, 1))
		return false;
	
	if(vS.Sum(dCov))
		return false;
	
	dCov = dCov / (mXs.GetNumRows() - 1);
	
	return true;	
}

bool set_rows(const matrix mbSource, matrix & mbTarget, const vector<uint> vIndices)
{			
	if(mbTarget.GetNumCols() != mbSource.GetNumCols() || mbTarget.GetNumRows() < mbSource.GetNumRows() 
		|| mbSource.GetNumRows() != vIndices.GetSize())
		return false;
	
	for(int i = 0; i < mbSource.GetNumRows(); i++)
	{
		vector vRow;
		if(!(mbSource.GetRow(vRow, i)))
			return false;

		if(mbTarget.SetRow(vRow, vIndices[i]))
			return false;
	}
	
	return true;
}

bool get_corr_from_cov(const matrix mCov, matrix &mCorr)
{
	mCorr = mCov;
	
	for( int nRow = 0; nRow < mCov.GetNumRows(); nRow++ )
	{
		vector vRow;
		if( !mCorr.GetRow( vRow, nRow ) )
			return false;
		vRow = 1.0/sqrt( mCov[nRow][nRow] )*vRow;
		
		if( mCorr.SetRow( vRow, nRow ) )
			return false;
	}
	
	for( int nCol = 0; nCol < mCov.GetNumCols(); nCol++ )
	{
		vector vCol;
		if( !mCorr.GetColumn( vCol, nCol ) )
			return false;
		vCol = 1.0/sqrt( mCov[nCol][nCol] )*vCol;
		
		if( mCorr.SetColumn( vCol, nCol ) )
			return false;
	}
	
	return true;
}

bool is_output_to_input(string & strIn, string & strOut)
{
	int nType;
	int nOpen = strOut.Find('[');
	if (nOpen > -1)
	{
		int nClose = strOut.Find(']');
		string strTemp = strOut.Mid(nOpen + 1, nClose - nOpen - 1);
		nType = okutil_cvt_str_to_predefined_type(strTemp);
	}
	else
	{
		nType = okutil_cvt_str_to_predefined_type(strOut);
	}
	
	if (nType == PDS_INPUT)
		return true;

	Worksheet wksIn, wksOut;
	
	DataRange drInput;
	drInput.Create(strIn);
	
	if (!drInput || drInput.GetNumRanges() <= 0)
		return false;
	
	DataRange drOutput;
	drOutput.Create(strOut);
	
	if (!drOutput || drOutput.GetNumRanges() <= 0)
		return false;
	
	int dummy;	
	if (!drInput.GetRange(0, dummy, dummy, dummy, dummy, wksIn))
		return false;	
	if (!drOutput.GetRange(0, dummy, dummy, dummy, dummy, wksOut))
		return false;
	
	if(wksIn && wksOut && is_same_layer(wksIn, wksOut))
		return true;
	
	return false;
}

string report_nag_error(LPCSTR lpcszAnalysizeName)
{
	string strMessage;

	strMessage.Format(_L("An error occurs during %s, see details in Messages Log.\n"), lpcszAnalysizeName);

	return strMessage;
}
/// End COMMAND_FUNCTION_FOR_MULTIVARIATE_XF
