/*------------------------------------------------------------------------------*
 * File Name:	oIAEASPE.h		 												*
 * Creation: 	Sophy 7/20/2010													*
 * Purpose: OriginC Header File													*
 * Copyright (c) OriginLab Corp.	2003 - 2010									*
 * All Rights Reserved															*
 * 																				*
 * Modification Log:															*
 *------------------------------------------------------------------------------*/
 
////////////////////////////////////////////////////////////////////////////////////
#include <Origin.h>
#include <GetNBox.h>
////////////////////////////////////////////////////////////////////////////////////
#ifndef	_OIAEASPE_H_
#define	_OIAEASPE_H_

//The parse based on the following link http://www.gbs-elektronik.de/mca/spe_e.htm
#define	MAKE_BLOCK_NAME(_Name)	"$" #_Name ":"
#define	MAKE_BLOCK_ENUM(_Name)	BLOCK_##_Name
#define	MAKE_ROW_NAME(_Str)		_Str.TrimLeft("$");	_Str.TrimRight(":");
#define	STR_TRIM(_Str)	_Str.TrimLeft(); _Str.TrimRight();

const vector<string> vsBlockNames = {
	MAKE_BLOCK_NAME(ADC),
	MAKE_BLOCK_NAME(ADD_HV),
	MAKE_BLOCK_NAME(BT),
	MAKE_BLOCK_NAME(COUNTS),
	MAKE_BLOCK_NAME(DATA),
	MAKE_BLOCK_NAME(DATE_MEA),
	MAKE_BLOCK_NAME(DT),
	MAKE_BLOCK_NAME(DTC),
	MAKE_BLOCK_NAME(ENER_DATA),
	MAKE_BLOCK_NAME(ENER_DATA_X),
	MAKE_BLOCK_NAME(ENER_FIT),
	MAKE_BLOCK_NAME(FAST_DISCR),
	MAKE_BLOCK_NAME(GAIN_VALUE),
	MAKE_BLOCK_NAME(GPS),
	MAKE_BLOCK_NAME(GPS_COORDINATES),
	MAKE_BLOCK_NAME(HV),
	MAKE_BLOCK_NAME(INPUT),
	MAKE_BLOCK_NAME(INSP_INFO),
	MAKE_BLOCK_NAME(MCA_166_ID),
	MAKE_BLOCK_NAME(MCA_REPEAT),
	MAKE_BLOCK_NAME(MEAS_TIM),
	MAKE_BLOCK_NAME(MCS_ADD_DATA),
	MAKE_BLOCK_NAME(MCS_AMP_DATA),
	MAKE_BLOCK_NAME(MCS_AMP_ROI),
	MAKE_BLOCK_NAME(MCS_AMP_ROI_INFO),
	MAKE_BLOCK_NAME(MCS_CHANNELS),
	MAKE_BLOCK_NAME(MCS_INPUT),
	MAKE_BLOCK_NAME(MCS_SWEEPS),
	MAKE_BLOCK_NAME(MCS_TIME),
	MAKE_BLOCK_NAME(MODE),
	MAKE_BLOCK_NAME(PD_COUNTS),
	MAKE_BLOCK_NAME(POWER),
	MAKE_BLOCK_NAME(POWER_STATE),
	MAKE_BLOCK_NAME(PRESETS),
	MAKE_BLOCK_NAME(PUR),
	MAKE_BLOCK_NAME(PZC_VALUE),
	MAKE_BLOCK_NAME(REC_COUNTER),
	MAKE_BLOCK_NAME(REC_ERROR_COUNTER),
	MAKE_BLOCK_NAME(ROI),
	MAKE_BLOCK_NAME(ROI_INFO),
	MAKE_BLOCK_NAME(RT),
	MAKE_BLOCK_NAME(SCANDU),
	MAKE_BLOCK_NAME(SCANDU_RESULTS),
	MAKE_BLOCK_NAME(Sensors),
	MAKE_BLOCK_NAME(SINGLE_POINTS),
	MAKE_BLOCK_NAME(SLOW_DISCR),
	MAKE_BLOCK_NAME(SPEC_INTEGRAL),
	MAKE_BLOCK_NAME(SPEC_REM),
	MAKE_BLOCK_NAME(STAB),
	MAKE_BLOCK_NAME(STAB_COUNTER),
	MAKE_BLOCK_NAME(STAB_OFFSET),
	MAKE_BLOCK_NAME(STAB_OFFSET_MAX),
	MAKE_BLOCK_NAME(STAB_OFFSET_MIN),
	MAKE_BLOCK_NAME(STAB_PARAM),
	MAKE_BLOCK_NAME(TEMPERATURE),
	MAKE_BLOCK_NAME(TDF),
	MAKE_BLOCK_NAME(THR),
	MAKE_BLOCK_NAME(UF6_INSP_INFO),
	MAKE_BLOCK_NAME(WATCHMON_ROI_INFO),
	MAKE_BLOCK_NAME(WINSCAN_INFO),
	MAKE_BLOCK_NAME(WINSPEC_AUTOMATION),
	MAKE_BLOCK_NAME(WINSPEC_INFO),
	MAKE_BLOCK_NAME(MCA_CAL)	///Sophy 11/14/2011 ORG-567-S2 IAEASPE_IMPORT_MULTI_COLUMNS_SUPPORT
};



typedef	enum {
	MAKE_BLOCK_ENUM(UNKNOWN)		= -1,
	MAKE_BLOCK_ENUM(ADC)			= 0,
	MAKE_BLOCK_ENUM(ADD_HV),
	MAKE_BLOCK_ENUM(BT),
	MAKE_BLOCK_ENUM(COUNTS),
	MAKE_BLOCK_ENUM(DATA),
	MAKE_BLOCK_ENUM(DATE_MEA),
	MAKE_BLOCK_ENUM(DT),
	MAKE_BLOCK_ENUM(DTC),
	MAKE_BLOCK_ENUM(ENER_DATA),
	MAKE_BLOCK_ENUM(ENER_DATA_X),
	MAKE_BLOCK_ENUM(ENER_FIT),
	MAKE_BLOCK_ENUM(FAST_DISCR),
	MAKE_BLOCK_ENUM(GAIN_VALUE),
	MAKE_BLOCK_ENUM(GPS),
	MAKE_BLOCK_ENUM(GPS_COORDINATES),
	MAKE_BLOCK_ENUM(HV),
	MAKE_BLOCK_ENUM(INPUT),
	MAKE_BLOCK_ENUM(INSP_INFO),
	MAKE_BLOCK_ENUM(MCA_166_ID),
	MAKE_BLOCK_ENUM(MCA_REPEAT),
	MAKE_BLOCK_ENUM(MEAS_TIM),
	MAKE_BLOCK_ENUM(MCS_ADD_DATA),
	MAKE_BLOCK_ENUM(MCS_AMP_DATA),
	MAKE_BLOCK_ENUM(MCS_AMP_ROI),
	MAKE_BLOCK_ENUM(MCS_AMP_ROI_INFO),
	MAKE_BLOCK_ENUM(MCS_CHANNELS),
	MAKE_BLOCK_ENUM(MCS_INPUT),
	MAKE_BLOCK_ENUM(MCS_SWEEPS),
	MAKE_BLOCK_ENUM(MCS_TIME),
	MAKE_BLOCK_ENUM(MODE),
	MAKE_BLOCK_ENUM(PD_COUNTS),
	MAKE_BLOCK_ENUM(POWER),
	MAKE_BLOCK_ENUM(POWER_STATE),
	MAKE_BLOCK_ENUM(PRESETS),
	MAKE_BLOCK_ENUM(PUR),
	MAKE_BLOCK_ENUM(PZC_VALUE),
	MAKE_BLOCK_ENUM(REC_COUNTER),
	MAKE_BLOCK_ENUM(REC_ERROR_COUNTER),
	MAKE_BLOCK_ENUM(ROI),
	MAKE_BLOCK_ENUM(ROI_INFO),
	MAKE_BLOCK_ENUM(RT),
	MAKE_BLOCK_ENUM(SCANDU),
	MAKE_BLOCK_ENUM(SCANDU_RESULTS),
	MAKE_BLOCK_ENUM(Sensors),
	MAKE_BLOCK_ENUM(SINGLE_POINTS),
	MAKE_BLOCK_ENUM(SLOW_DISCR),
	MAKE_BLOCK_ENUM(SPEC_INTEGRAL),
	MAKE_BLOCK_ENUM(SPEC_REM),
	MAKE_BLOCK_ENUM(STAB),
	MAKE_BLOCK_ENUM(STAB_COUNTER),
	MAKE_BLOCK_ENUM(STAB_OFFSET),
	MAKE_BLOCK_ENUM(STAB_OFFSET_MAX),
	MAKE_BLOCK_ENUM(STAB_OFFSET_MIN),
	MAKE_BLOCK_ENUM(STAB_PARAM),
	MAKE_BLOCK_ENUM(TEMPERATURE),
	MAKE_BLOCK_ENUM(TDF),
	MAKE_BLOCK_ENUM(THR),
	MAKE_BLOCK_ENUM(UF6_INSP_INFO),
	MAKE_BLOCK_ENUM(WATCHMON_ROI_INFO),
	MAKE_BLOCK_ENUM(WINSCAN_INFO),
	MAKE_BLOCK_ENUM(WINSPEC_AUTOMATION),
	MAKE_BLOCK_ENUM(WINSPEC_INFO),
	MAKE_BLOCK_ENUM(MCA_CAL),	///Sophy 11/14/2011 ORG-567-S2 IAEASPE_IMPORT_MULTI_COLUMNS_SUPPORT
} BLOCK_TYPE;

static	BLOCK_TYPE	_get_block_type(LPCSTR lpcszName)
{
	vector<string> tt;
	tt = vsBlockNames;
	int nIndex = vsBlockNames.Find(lpcszName);
	return nIndex;
}

enum {
	IAEA_OK 				= 0,
	IAEA_INVALID_FILE,
	IAEA_NO_DATA,
	IAEA_INVALID_OBJECT,
	IAEA_USER_ABORT,
	IAEA_UNKNOWN_ERROR,
};
class	oIAEASPE : public stdioFile 
{
public:
	oIAEASPE(){}
	~oIAEASPE()
	{
		if ( IsOpen() )
			Close();
	}
	int		Open(LPCSTR lpcszFileName);
	int		GetBlocks(TreeNode& trInfo);
	int		Import(Worksheet& wks, const TreeNode& trFileInfo);
protected:
	BOOL	ReadString(string& strLine);
	int		ParseFile();
	BLOCK_TYPE	ReadBlockType();
private:
	string		m_strFileName;
	Tree		m_trFile;
	BLOCK_TYPE	m_tBlockType;
};

BOOL	oIAEASPE::Open(LPCSTR lpcszFileName)
{
	if ( !stdioFile::Open(lpcszFileName, file::modeRead | file::shareDenyWrite) )
		return FALSE;
	m_strFileName = lpcszFileName;
	return ParseFile();
}

int		oIAEASPE::GetBlocks(TreeNode& trInfo)
{
	if ( !trInfo )
		return IAEA_INVALID_OBJECT;
	trInfo.Replace(m_trFile.Clone(), TRUE, TRUE);
	tree_set_attribute_to_all_nodes(trInfo, STR_ENABLE_ATTRIB, "2");
	TreeNode trData = trInfo.FindNodeByAttribute(STR_LABEL_ATTRIB, MAKE_BLOCK_NAME(DATA));
	if ( trData )
		trData.Remove();
	return IAEA_OK;
}

int		oIAEASPE::Import(Worksheet& wks, const TreeNode& trFileInfo)
{
	if ( !wks )
		return IAEA_INVALID_OBJECT;
	wks.SetSize(-1, 1);
	
	TreeNode trData = m_trFile.FindNodeByAttribute(STR_LABEL_ATTRIB, MAKE_BLOCK_NAME(DATA));
	if ( !trData )
		return IAEA_NO_DATA;
	///Sophy 11/14/2011 ORG-567-S2 IAEASPE_IMPORT_MULTI_COLUMNS_SUPPORT
	//Column col(wks, 0);
	//col.SetType(OKDATAOBJ_DESIGNATION_Y);
	//col.SetLongName(_L("First Channel and Last Channel"));
	//col.SetComments(trData.ChannelRange.strVal);
	//vectorbase& vb = col.GetDataObject();
	//vb = trData.Data.dVals;
	vector<string> vsData;
	vsData = trData.Data.strVals;
	int nMaxCols;
	for ( int iRow = 0; iRow < vsData.GetSize(); iRow++ )
	{
		string strLine = vsData[iRow];
		vector<string> vsCells;
		strLine.GetTokens(vsCells);
		wks.SetCells(vsCells);
		if ( vsCells.GetSize() > nMaxCols )
			nMaxCols = vsCells.GetSize();
	}
	///end IAEASPE_IMPORT_MULTI_COLUMNS_SUPPORT
	///Sophy 11/14/2011 ORG-567-S4 COLUMN_LONGNAME_TO_INDICATE_CHANNEL_NUMBER
	for ( int iCol = 0; iCol < nMaxCols; iCol++ )
	{
		Column col(wks, iCol);
		if ( col )
		{
			col.SetType(OKDATAOBJ_DESIGNATION_Y);
			col.SetLongName("Channel " + (iCol + 1));
		}
	}
	///end COLUMN_LONGNAME_TO_INDICATE_CHANNEL_NUMBER
	set_user_info(wks, _L("File Information"), trFileInfo);
	return IAEA_OK;
}

BOOL	oIAEASPE::ReadString(string& strLine)
{
	if ( !ReadString(strLine, 0) )
		return FALSE;
	STR_TRIM(strLine);
	return TRUE;
}

int		oIAEASPE::ParseFile()
{
	SeekToBegin();
	GETN_USE(m_trFile);
	while ( (m_tBlockType = ReadBlockType()) != MAKE_BLOCK_ENUM(UNKNOWN) )
	{
		string strLine;
		GETN_BEGIN_BRANCH(Block, vsBlockNames[m_tBlockType])
		switch(m_tBlockType)
		{
		case MAKE_BLOCK_ENUM(ADC):
			ReadString(strLine);
			GETN_NUM(Channels, _L("Channels"), atol(strLine))
			ReadString(strLine);
			GETN_NUM(LLD, _L("Lower Level Discriminator Channel"), atol(strLine));
			ReadString(strLine);
			GETN_NUM(ULD, _L("Upper Level Discriminator Channel"), atol(strLine));
			break;
			
		case MAKE_BLOCK_ENUM(ADD_HV):
			ReadString(strLine);
			GETN_STR(DevName, _L("Device Name"), strLine)
			ReadString(strLine);
			GETN_STR(Volt, _L("Voltage Value"), strLine)
			break;
			
		case MAKE_BLOCK_ENUM(BT):
			ReadString(strLine);
			GETN_NUM(BusyTime, _L("Busy Time"), atof(strLine))
			break;
		
		case MAKE_BLOCK_ENUM(COUNTS):
			ReadString(strLine);
			GETN_NUM(Sum, _L("Sum of All Input Counts"), atol(strLine))
			break;
			
		case MAKE_BLOCK_ENUM(DATA):
			ReadString(strLine);
			GETN_STR(ChannelRange, _L("First Channel and Last Channel"), strLine)
			LONG pos = GetPosition();
			///Sophy 11/14/2011 ORG-567-S2 IAEASPE_IMPORT_MULTI_COLUMNS_SUPPORT
			//vector vData;
			vector<string> vsData;
			///end IAEASPE_IMPORT_MULTI_COLUMNS_SUPPORT
			while ( ReadString(strLine) && _get_block_type(strLine) == BLOCK_UNKNOWN )
			{
				if ( strLine.IsEmpty() )
					continue;
				///Sophy 11/14/2011 ORG-567-S2 IAEASPE_IMPORT_MULTI_COLUMNS_SUPPORT
				//vData.Add(atof(strLine));
				vsData.Add(strLine);
				///end IAEASPE_IMPORT_MULTI_COLUMNS_SUPPORT
				pos = GetPosition();
			}
			_tmpSubNode = _tmpNode.AddNode("Data");
			///Sophy 11/14/2011 ORG-567-S2 IAEASPE_IMPORT_MULTI_COLUMNS_SUPPORT
			//_tmpSubNode.dVals = vData;
			_tmpSubNode.strVals = vsData;
			///end IAEASPE_IMPORT_MULTI_COLUMNS_SUPPORT
			Seek(pos, file::begin);
			
			break;
			
		case MAKE_BLOCK_ENUM(DATE_MEA):
			ReadString(strLine);
			GETN_STR(MeasureDate, _L("Measurement Start Time"), strLine)
			break;
			
		case MAKE_BLOCK_ENUM(DT):
			ReadString(strLine);
			GETN_NUM(DeadTime, _L("Dead Time"), atof(strLine))
			break;
			
		case MAKE_BLOCK_ENUM(DTC):
			ReadString(strLine);
			GETN_NUM(ShapingTime, _L("Shaping Time"), atol(strLine))
			GETN_NUM(AShppingTime, _L("Actual Shaping Time"), atof(strLine))
			break;
			
		case MAKE_BLOCK_ENUM(ENER_DATA):
			ReadString(strLine);
			GETN_NUM(nPointCali, _L("N Point Calibration"), atol(strLine))
			ReadString(strLine);
			GETN_STR(LEnerPair, _L("Lower Channel Energy Pair"), strLine)
			ReadString(strLine);
			GETN_STR(HEnerPair, _L("Higher Channel Energy Pair"), strLine)
			break;
			
		case MAKE_BLOCK_ENUM(ENER_DATA_X):
			ReadString(strLine);
			GETN_STR(NPoints, _L("Number of Calibration Points"), strLine)
			ReadString(strLine);
			int nPoints = atol(strLine);
			for ( int ii = 0; ii < nPoints; ii++ )
			{
				ReadString(strLine);
				GETN_STR(PointPair, _L("Point Pair") + ii, strLine)
			}
			break;
			
		case MAKE_BLOCK_ENUM(ENER_FIT):
			ReadString(strLine);
			GETN_STR(EnerFit, _L("Offset and Slope"), strLine)
			break;
			
		case MAKE_BLOCK_ENUM(FAST_DISCR):
			ReadString(strLine);
			GETN_STR(FactorySetting, _L("Factor Setting for Autothreshold"), strLine)
			break;
			
		case MAKE_BLOCK_ENUM(GAIN_VALUE):
			ReadString(strLine);
			GETN_NUM(CoarseGain, _L("Coarse Gain"), atof(strLine))
			ReadString(strLine);
			GETN_NUM(FindGain, _L("Fine Gain"), atof(strLine))
			break;
			
		case MAKE_BLOCK_ENUM(GPS):
			ReadString(strLine);
			GETN_STR(Long, _L("Longitude"), strLine)
			ReadString(strLine);
			GETN_STR(Lati, _L("Latitude"), strLine)
			ReadString(strLine);
			GETN_STR(Alti, _L("Altitude"), strLine)
			ReadString(strLine);
			GETN_STR(Speed, _L("Speed"), strLine)
			ReadString(strLine);
			GETN_STR(Direct, _L("Direction to The Geographic North Pole"), strLine)
			break;
			
		case MAKE_BLOCK_ENUM(GPS_COORDINATES):
			ReadString(strLine);
			GETN_STR(Coord, _L("Latitude, Longitude, Altitude"), strLine)
			ReadString(strLine);
			GETN_STR(GPGGA, _L("String from GPPS Receiver"), strLine)
			break;
			
		case MAKE_BLOCK_ENUM(HV):
			ReadString(strLine);
			GETN_STR(Value, _L("High Voltage Value"), strLine)
			ReadString(strLine);
			GETN_STR(Mode, _L("High Voltage Inhibit Mode"), strLine)
			break;
			
		case MAKE_BLOCK_ENUM(INPUT):
			ReadString(strLine);
			GETN_STR(Amp, _L("Amplifier"), strLine)
			ReadString(strLine);
			GETN_STR(Polarity, _L("Amplifier Input Polarity"), strLine)
			break;
			
		case MAKE_BLOCK_ENUM(INSP_INFO):
			ReadString(strLine);
			GETN_STR(InspNum, _L("Inspection Number"), strLine)
			ReadString(strLine);
			GETN_STR(InspDate, _L("Inspection Date"), strLine)
			ReadString(strLine);
			GETN_STR(InspID, _L("Inspection ID"), strLine)
			ReadString(strLine);
			GETN_STR(FacilityCode, _L("Facility Code"), strLine)
			ReadString(strLine);
			GETN_STR(MBA, _L("MBA"), strLine)
			ReadString(strLine);
			GETN_STR(LOTID, _L("LOT ID"), strLine)
			ReadString(strLine);
			GETN_STR(SampleID, _L("Sample ID"), strLine)
			ReadString(strLine);
			GETN_STR(DecEnrich, _L("Declared Enrichment"), strLine)
			ReadString(strLine);
			GETN_STR(DecEnrichUncert, _L("Declared Enrichment Uncertainty"), strLine)
			ReadString(strLine);
			GETN_STR(ScaleUnits, _L("Scale Units"), strLine)
			ReadString(strLine);
			GETN_STR(WallThick, _L("Wall Thickness"), strLine)
			ReadString(strLine);
			GETN_STR(WallThickUncert, _L("Wall Thickness Uncertainty"), strLine)
			ReadString(strLine);
			GETN_STR(Attenuation, _L("Attenuation"), strLine)
			ReadString(strLine);
			GETN_STR(AttenumationUncert, _L("Attenuation Uncertainty"), strLine)
			ReadString(strLine);
			GETN_STR(MatCorrFactor, _L("Material Correction Factor"), strLine)
			ReadString(strLine);
			GETN_STR(MatCorrFactorUncert, _L("Material Correction Factor Uncertainty"), strLine)
			ReadString(strLine);
			GETN_STR(InsCode, _L("Instrument Code"), strLine)
			ReadString(strLine);
			GETN_STR(MMCAID, _L("MMCA ID"), strLine)
			ReadString(strLine);
			GETN_STR(ComputerID, _L("Computer ID"), strLine)
			ReadString(strLine);
			GETN_STR(DetectorType, _L("Detector Type"), strLine)
			ReadString(strLine);
			GETN_STR(DetectorID, _L("Detector ID"), strLine)
			ReadString(strLine);
			GETN_STR(Collimator, _L("Collimator"), strLine)
			ReadString(strLine);
			GETN_STR(MotorDriveID, _L("Motor Drive ID"), strLine)
			break;
			
		case MAKE_BLOCK_ENUM(MCA_166_ID):
			ReadString(strLine);
			GETN_STR(SerialNumber, _L("Serial Number"), strLine)
			ReadString(strLine);
			GETN_STR(HWVersion, _L("Hardware Version"), strLine)
			ReadString(strLine);
			GETN_STR(FWVersion, _L("Firmware Version"), strLine)
			break;
			
		case MAKE_BLOCK_ENUM(MCA_REPEAT):
			ReadString(strLine);
			GETN_NUM(Sweeps, _L("Number of Sweeps"), atol(strLine))
			ReadString(strLine);
			GETN_NUM(Type, _L("MCA Repeat Mode Type"), atol(strLine))
			break;
			
		case MAKE_BLOCK_ENUM(MEAS_TIM):
			ReadString(strLine);
			GETN_STR(SpectrumMeaTim, _L("Spectrum Measurement Time"), strLine)
			break;
			
		case MAKE_BLOCK_ENUM(MCS_ADD_DATA):
			ReadString(strLine);
			GETN_STR(Designation, _L("Designation of The Data"), strLine)
			ReadString(strLine);
			GETN_STR(ChannelRange, _L("First Channel and Last Channel"), strLine)
			LONG pos = GetPosition();
			vector vData(0);
			while ( ReadString(strLine) && _get_block_type(strLine) == BLOCK_UNKNOWN )
			{
				vData.Add(atof(strLine));
				pos = GetPosition();
			}
			_tmpSubNode = _tmpNode.AddNode("Data");
			_tmpSubNode.dVals = vData;
			Seek(pos, file::begin);
			break;
			
		case MAKE_BLOCK_ENUM(MCS_AMP_DATA):
			break;
			
		case MAKE_BLOCK_ENUM(MCS_AMP_ROI):
			break;
			
		case MAKE_BLOCK_ENUM(MCS_AMP_ROI_INFO):
			break;
			
		case MAKE_BLOCK_ENUM(MCS_CHANNELS):
			ReadString(strLine);
			GETN_NUM(Channels, _L("Channels"), atof(strLine))
			break;
			
		case MAKE_BLOCK_ENUM(MCS_INPUT):
			ReadString(strLine);
			GETN_STR(MSCInput, _L("MCS Input"), strLine)
			break;
			
		case MAKE_BLOCK_ENUM(MCS_SWEEPS):
			ReadString(strLine);
			GETN_NUM(Sweeps, _L("Number of Sweeps"), atol(strLine))
			ReadString(strLine);
			GETN_STR(Mode, _L("MCA Repeat Mode Type"), strLine)
			break;
			
		case MAKE_BLOCK_ENUM(MCS_TIME):
			ReadString(strLine);
			GETN_NUM(Time, _L("MCS Time Per Channel"), atof(strLine))
			break;
			
		case MAKE_BLOCK_ENUM(MODE):
			ReadString(strLine);
			GETN_STR(Mode, _L("Operation Mode"), strLine)
			break;
			
		case MAKE_BLOCK_ENUM(PD_COUNTS):
			break;
			
		case MAKE_BLOCK_ENUM(POWER):
			ReadString(strLine);
			GETN_STR(PowerSupA, _L("Preamplifier Power Supply 1"), strLine)
			ReadString(strLine);
			GETN_STR(PowerSupB, _L("Preamplifier Power Supply 2"), strLine)
			ReadString(strLine);
			GETN_STR(PowerSupC, _L("Preamplifier Power Supply 3"), strLine)
			ReadString(strLine);
			GETN_STR(PowerSupD, _L("Preamplifier Power Supply 4"), strLine)
			break;
			
		case MAKE_BLOCK_ENUM(POWER_STATE):
			ReadString(strLine);
			GETN_STR(PCurrent12, _L("Input Current for +12V preamplifier pow supply"), strLine)
			ReadString(strLine);
			GETN_STR(NCurrent12, _L("Input Current for -12V preamplifier pow supply"), strLine)
			
			ReadString(strLine);
			GETN_STR(PCurrent24, _L("Input Current for +24V preamplifier pow supply"), strLine)
			ReadString(strLine);
			GETN_STR(NCurrent24, _L("Input Current for -24V preamplifier pow supply"), strLine)
			
			ReadString(strLine);
			GETN_STR(TCurrent, _L("Total Current drawn from battery"), strLine)
			
			ReadString(strLine);
			GETN_STR(ICurrent, _L("Input Current drawn from battery"), strLine)
			
			ReadString(strLine);
			GETN_STR(ECurrent, _L("External Charger Current"), strLine)
			ReadString(strLine);
			GETN_STR(BV, _L("Battery Voltage"), strLine)
			ReadString(strLine);
			GETN_STR(CV, _L("Control Voltage of The HV Module"), strLine);
			break;
			
		case MAKE_BLOCK_ENUM(PRESETS):
			ReadString(strLine);
			GETN_STR(ROISum, _L("Live Time/Real Time"), strLine)
			ReadString(strLine);
			GETN_NUM(Value, _L("Value"), atol(strLine))
			ReadString(strLine);
			GETN_NUM(ROINum, _L("ROI Number"), atol(strLine))
			LONG pos = GetPosition();
			ReadString(strLine); //optional line, should check
			if ( _get_block_type(strLine) != BLOCK_UNKNOWN )
				Seek(pos, file::begin);
			else
			{
				GETN_STR(ROIBE, _L("ROI Begin and End"), strLine)
			}
			break;
			
		case MAKE_BLOCK_ENUM(PUR):
			ReadString(strLine);
			GETN_STR(State, _L("Pile Up Rejector State"), strLine)
			break;
			
		case MAKE_BLOCK_ENUM(PZC_VALUE):
			ReadString(strLine);
			GETN_NUM(DACAdjValue, _L("Adjustment Value for PZC"), atol(strLine))
			ReadString(strLine);
			GETN_NUM(PredTimeParaA, _L("Pre-defined time parameter 1"), atol(strLine))
			ReadString(strLine);
			GETN_NUM(PredTimeParaB, _L("Pre-defined time parameter 2"), atol(strLine))
			break;
			
		case MAKE_BLOCK_ENUM(REC_COUNTER):
			ReadString(strLine);
			GETN_NUM(Counter, _L("Counter of Received Commands"), atol(strLine))
			break;
			
		case MAKE_BLOCK_ENUM(REC_ERROR_COUNTER):
			ReadString(strLine);
			GETN_NUM(Counter, _L("Counter of Received Commands With Errors"), atol(strLine))
			break;
			
		case MAKE_BLOCK_ENUM(ROI):
			ReadString(strLine);
			GETN_NUM(ROINum, _L("Number of ROI"), atol(strLine))
			ReadString(strLine);
			GETN_STR(ROISB, _L("ROI Start/ROI Begin"), strLine);
			break;
			
		case MAKE_BLOCK_ENUM(ROI_INFO):
			ReadString(strLine);
			GETN_STR(ROIInfo, _L("ROI Information"), strLine)
			break;
			
		case MAKE_BLOCK_ENUM(RT):
			ReadString(strLine);
			GETN_NUM(RealTime, _L("Real Time"), atof(strLine))
			break;
			
		case MAKE_BLOCK_ENUM(SCANDU):
			ReadString(strLine);
			GETN_STR(Speed, _L("Speed"), strLine)
			ReadString(strLine);
			GETN_STR(Length, _L("Length"), strLine)
			ReadString(strLine);
			GETN_STR(Insp, _L("Inspector"), strLine)
			ReadString(strLine);
			GETN_STR(Inspection, _L("Inspection"), strLine)
			ReadString(strLine);
			GETN_STR(Anlage, _L("Anlage"), strLine)
			LONG pos = GetPosition();
			ReadString(strLine); //optional line, should check
			if ( _get_block_type(strLine) != BLOCK_UNKNOWN )
				Seek(pos, file::begin);
			else
			{
				GETN_STR(OriginEval, _L("Origin of The Evaluation Grid[mm]"), strLine)
				ReadString(strLine);
				GETN_STR(PeriodEval, _L("Period of The Evaluation Grid[mm]"), strLine)
				ReadString(strLine);
				GETN_STR(DChoiceA, _L("Data Choice"), strLine)
				ReadString(strLine);
				GETN_STR(DChoiceB, _L("Data Choice"), strLine)
				ReadString(strLine);
				GETN_STR(ScanDirect, _L("Scan Direction"), strLine)
			}
			break;
			
		case MAKE_BLOCK_ENUM(SCANDU_RESULTS):
			ReadString(strLine);
			GETN_STR(SpectrumCount, _L("Bundles Counted Based On Spectrum Integral"), strLine);
			ReadString(strLine);
			GETN_STR(ROI1Area, _L("Bundles Counted Based On ROI 1 Area"), strLine)
			ReadString(strLine);
			GETN_STR(ROI1AreaInteg, _L("Bundles Counted Based On ROI 1 Area Integral"), strLine)
			ReadString(strLine);
			GETN_STR(ROI2Area, _L("bundles Counted Based On ROI 2 Area"), strLine)
			ReadString(strLine);
			GETN_STR(ROI2AreaInteg, _L("Bundles Counted Based On ROI 2 Area Integral"), strLine)
			ReadString(strLine);
			GETN_STR(Neutronen, _L("Bundles Counted Based On Neutron"), strLine)
			break;
			
		case MAKE_BLOCK_ENUM(Sensors):
			ReadString(strLine);
			GETN_STR(DetectorTemp, _L("Detector Temperature[Celsius]"), strLine)
			ReadString(strLine);
			GETN_STR(OutsideTemp, _L("Outside Temperature[Celsius]"), strLine)
			ReadString(strLine);
			GETN_STR(AirPress, _L("Air Pressure[mBar]"), strLine)
			break;
			
		case MAKE_BLOCK_ENUM(SINGLE_POINTS):
			ReadString(strLine);
			GETN_STR(DChoiceA, _L("Data Choice"), strLine);
			ReadString(strLine);
			GETN_STR(DChoiceB, _L("Data Choice"), strLine);
			break;
			
		case MAKE_BLOCK_ENUM(SLOW_DISCR):
			ReadString(strLine);
			GETN_STR(FactorySetting, _L("Factor Setting for Autothreshold"), strLine)
			break;
			
		case MAKE_BLOCK_ENUM(SPEC_INTEGRAL):
			ReadString(strLine);
			GETN_NUM(Counter, _L("Counts in The Spectrum"), atol(strLine))
			break;
			
		case MAKE_BLOCK_ENUM(SPEC_REM):
			ReadString(strLine);
			GETN_STR(Remarks, _L("Remarks Before Save"), strLine)
			ReadString(strLine);
			GETN_STR(Spectrum, _L("Spectrum Notes"), strLine)
			ReadString(strLine);
			GETN_STR(Location, _L("File Location"), strLine)
			break;
			
		case MAKE_BLOCK_ENUM(STAB):
			ReadString(strLine);
			GETN_STR(State, _L("Stabilisation"), strLine)
			ReadString(strLine);
			GETN_NUM(PeakLimitL, _L("Lower ROI Limits of The Stabilisation Peak"), atof(strLine))
			ReadString(strLine);
			GETN_NUM(PeakLimitU, _L("Higher ROI Limits of The Stabilisation Peak"), atof(strLine))
			ReadString(strLine);
			GETN_NUM(TargetChannel, _L("Stabilisation Target Channel"), atol(strLine))
			break;
			
		case MAKE_BLOCK_ENUM(STAB_COUNTER):
			ReadString(strLine);
			GETN_NUM(Counter, _L("Counter"), atol(strLine))
			break;
			
		case MAKE_BLOCK_ENUM(STAB_OFFSET):
			ReadString(strLine);
			GETN_NUM(CO, _L("Current Offset"), atof(strLine))
			break;
			
		case MAKE_BLOCK_ENUM(STAB_OFFSET_MAX):
			ReadString(strLine);
			GETN_NUM(MaxO, _L("Maximal Offset"), atof(strLine))
			break;
			
		case MAKE_BLOCK_ENUM(STAB_OFFSET_MIN):
			ReadString(strLine);
			GETN_NUM(MinO, _L("Minimal Offset"), atof(strLine))
			break;
			
		case MAKE_BLOCK_ENUM(STAB_PARAM):
			ReadString(strLine);
			GETN_NUM(StabTime, _L("Stabilisation Time"), atof(strLine))
			ReadString(strLine);
			GETN_NUM(StabError, _L("Stabilisation Error"), atof(strLine))
			break;
			
		case MAKE_BLOCK_ENUM(TEMPERATURE):
			ReadString(strLine);
			GETN_STR(Temperature, _L("Detector Temperature[Celsius]"), strLine)
			break;
			
		case MAKE_BLOCK_ENUM(TDF):
			ReadString(strLine);
			GETN_NUM(Factor, _L("Dead Time Correction Factor"), atol(strLine))
			break;
			
		case MAKE_BLOCK_ENUM(THR):
			ReadString(strLine);
			GETN_NUM(ADCPercent, _L("Percent of ADC Channels"), atol(strLine))
			break;
			
		case MAKE_BLOCK_ENUM(UF6_INSP_INFO):
			ReadString(strLine);
			GETN_STR(InspNum, _L("Inspection Number"), strLine)
			ReadString(strLine);
			GETN_STR(InspDate, _L("Inspection Date"), strLine)
			ReadString(strLine);
			GETN_STR(InspID, _L("Inspection ID"), strLine)
			ReadString(strLine);
			GETN_STR(FacilityCode, _L("Facility Code"), strLine)
			ReadString(strLine);
			GETN_STR(LOTID, _L("LOT ID"), strLine)
			ReadString(strLine);
			GETN_STR(MBA, _L("MBA"), strLine)
			ReadString(strLine);
			GETN_STR(SampleID, _L("Sample ID"), strLine)
			ReadString(strLine);
			GETN_STR(DecEnrich, _L("Declared Enrichment"), strLine)
			ReadString(strLine);
			GETN_STR(DecEnrichUncert, _L("Declared Enrichment Uncertainty"), strLine)
			ReadString(strLine);
			GETN_STR(ScaleUnits, _L("Scale Units"), strLine)
			ReadString(strLine);
			GETN_STR(WallThick, _L("Wall Thickness"), strLine)
			ReadString(strLine);
			GETN_STR(WallThickUncert, _L("Wall Thickness Uncertainty"), strLine)
			ReadString(strLine);
			GETN_STR(Attenuation, _L("Attenuation"), strLine)
			ReadString(strLine);
			GETN_STR(AttenumationUncert, _L("Attenuation Uncertainty"), strLine)
			ReadString(strLine);
			GETN_STR(MatCorrFactor, _L("Material Correction Factor"), strLine)
			ReadString(strLine);
			GETN_STR(MatCorrFactorUncert, _L("Material Correction Factor Uncertainty"), strLine)
			ReadString(strLine);
			GETN_STR(InsCode, _L("Instrument Code"), strLine)
			ReadString(strLine);
			GETN_STR(MMCAID, _L("MMCA ID"), strLine)
			ReadString(strLine);
			GETN_STR(ComputerID, _L("Computer ID"), strLine)
			ReadString(strLine);
			GETN_STR(DetectorType, _L("Detector Type"), strLine)
			ReadString(strLine);
			GETN_STR(DetectorID, _L("Detector ID"), strLine)
			ReadString(strLine);
			GETN_STR(Collimator, _L("Collimator"), strLine)
			ReadString(strLine);
			GETN_STR(MotorDriveID, _L("Motor Drive ID"), strLine)
			break;
			
		case MAKE_BLOCK_ENUM(WATCHMON_ROI_INFO):
			//to do...
			break;
			
		case MAKE_BLOCK_ENUM(WINSCAN_INFO):
			ReadString(strLine);
			GETN_STR(InspNum, _L("Inspection Number"), strLine)
			ReadString(strLine);
			GETN_STR(InspDate, _L("Inspection Date"), strLine)
			ReadString(strLine);
			GETN_STR(InspNames, _L("Inspection Names"), strLine)
			ReadString(strLine);
			GETN_STR(FacilityName, _L("Facility Name"), strLine)
			ReadString(strLine);
			GETN_STR(FacilityCode, _L("Facility Code"), strLine)
			ReadString(strLine);
			GETN_STR(MBA, _L("MBA"), strLine)
			ReadString(strLine);
			GETN_STR(ItemID, _L("Item ID"), strLine)
			ReadString(strLine);
			GETN_STR(Stratum, _L("Stratum"), strLine)
			ReadString(strLine);
			GETN_STR(ItemDesc, _L("Item Description"), strLine)
			ReadString(strLine);
			GETN_STR(InsCode, _L("Instrument Code"), strLine)
			ReadString(strLine);
			GETN_STR(MMCAID, _L("MMCA ID"), strLine)
			ReadString(strLine);
			GETN_STR(ComputerID, _L("Computer ID"), strLine)
			ReadString(strLine);
			GETN_STR(DetectorType, _L("Detector Type"), strLine)
			ReadString(strLine);
			GETN_STR(DetectorID, _L("Detector ID"), strLine)
			ReadString(strLine);
			GETN_STR(Collimator, _L("Collimator"), strLine)
			ReadString(strLine);
			GETN_STR(MotorDriveID, _L("Motor Drive ID"), strLine)
			ReadString(strLine);
			GETN_STR(MotorCtrlID, _L("Motor Control ID"), strLine)
			ReadString(strLine);
			GETN_STR(SystemID, _L("System ID"), strLine)
			ReadString(strLine);
			GETN_STR(ExtAmpID, _L("External Amplifier ID"), strLine)
			GETN_STR(Gain, _L("Gain(coarse/fine)"), strLine)
			ReadString(strLine);
			GETN_STR(TimeConst, _L("Time Constant"), strLine)
			ReadString(strLine);
			GETN_STR(Output, _L("Output"), strLine)
			ReadString(strLine);
			GETN_STR(Remarks, _L("Remarks"), strLine)
			
			break;
			
		case MAKE_BLOCK_ENUM(WINSPEC_AUTOMATION):
			//to do...
			break;
		///Sophy 11/14/2011 ORG-567-S2 IAEASPE_IMPORT_MULTI_COLUMNS_SUPPORT
		case MAKE_BLOCK_ENUM(MCA_CAL):
			ReadString(strLine);	//%2
			ReadString(strLine);	//two numeric.
			break;
		///end IAEASPE_IMPORT_MULTI_COLUMNS_SUPPORT
			
		case MAKE_BLOCK_ENUM(WINSPEC_INFO):
			ReadString(strLine);
			GETN_STR(ExternApp, _L("External Analysis Application Attached With WinSPEC"), strLine)
			ReadString(strLine);
			GETN_STR(InspNum, _L("Inspection Number"), strLine)
			ReadString(strLine);
			GETN_STR(InspDate, _L("Inspection Date"), strLine)
			ReadString(strLine);
			GETN_STR(InspName, _L("Inspector Names"), strLine)
			ReadString(strLine);
			GETN_STR(FacilityName, _L("Facility Name"), strLine)
			ReadString(strLine);
			GETN_STR(FacilityCode, _L("Facility Code"), strLine)
			ReadString(strLine);
			GETN_STR(MBA, _L("MBA"), strLine)
			ReadString(strLine);
			GETN_STR(Stratum, _L("Stratum"), strLine)
			ReadString(strLine);
			GETN_STR(ItemID, _L("Item ID"), strLine)
			ReadString(strLine);
			GETN_STR(DecEnrich, _L("Declared Enrichment"), strLine)
			ReadString(strLine);
			GETN_STR(ItemDesc, _L("Item Description"), strLine)
			ReadString(strLine);
			GETN_STR(InsCode, _L("Instrument Code"), strLine)
			ReadString(strLine);
			GETN_STR(MMCAID, _L("MMCA ID"), strLine)
			ReadString(strLine);
			GETN_STR(ComputerID, _L("Computer ID"), strLine)
			ReadString(strLine);
			GETN_STR(DetectorType, _L("Detector Type"), strLine)
			ReadString(strLine);
			GETN_STR(Collimator, _L("Collimator"), strLine)
			ReadString(strLine);
			GETN_STR(Filter, _L("Filter"), strLine)
			
			LONG pos = GetPosition();
			ReadString(strLine); //optional block, should check
			if ( _get_block_type(strLine) != BLOCK_UNKNOWN )
				Seek(pos, file::begin);
			else
			{
				GETN_BEGIN_BRANCH(CsRation, "CsRation")
					GETN_STR(ExternApp, _L("Extern Analysis Application Attached With WinSPEC"), strLine)
					ReadString(strLine);
					GETN_STR(InspNum, _L("Inspection Number"), strLine)
					ReadString(strLine);
					GETN_STR(InspDate, _L("Inspection Date"), strLine)
					ReadString(strLine);
					GETN_STR(InspName, _L("Inspector Names"), strLine)
					ReadString(strLine);
					GETN_STR(FacilityName, _L("Facility Name"), strLine)
					ReadString(strLine);
					GETN_STR(FacilityCode, _L("Facility Code"), strLine)
					ReadString(strLine);
					GETN_STR(MBA, _L("MBA"), strLine)
					ReadString(strLine);
					GETN_STR(Stratum, _L("Stratum"), strLine)
					ReadString(strLine);
					GETN_STR(ItemID, _L("Item ID"), strLine)
					ReadString(strLine);
					GETN_STR(DateDischarge, _L("Date of Discharge"), strLine)
					ReadString(strLine);
					GETN_STR(Burnup, _L("Burnup"), strLine)
					ReadString(strLine);
					GETN_STR(NeutromRate, _L("Neutron Rate"), strLine)
					ReadString(strLine);
					GETN_STR(CsRation, _L("Calculated Cs Ratio At Discharge"), strLine)
					ReadString(strLine);
					GETN_STR(InitEnrich, _L("Initial Enrichment"), strLine)
					ReadString(strLine);
					GETN_STR(InsCode, _L("Instrument Code"), strLine)
					ReadString(strLine);
					GETN_STR(MMCAID, _L("MMCA ID"), strLine)
					ReadString(strLine);
					GETN_STR(ComputerID, _L("Computer ID"), strLine)
					ReadString(strLine);
					GETN_STR(DetectorType, _L("Detector Type"), strLine)
					ReadString(strLine);
					GETN_STR(Collimator, _L("Collimator"), strLine)
					ReadString(strLine);
					GETN_STR(Filter, _L("Filter"), strLine)
				GETN_END_BRANCH(CsRation)
				LONG pos = GetPosition();
				ReadString(strLine); //optional block, should check
				if ( _get_block_type(strLine) != BLOCK_UNKNOWN )
					Seek(pos, file::begin);
				else
				{
					GETN_BEGIN_BRANCH(CsRation, "MGA")
						GETN_STR(ExternApp, _L("Extern Analysis Application Attached With WinSPEC"), strLine)
						ReadString(strLine);
						GETN_STR(InspNum, _L("Inspection Number"), strLine)
						ReadString(strLine);
						GETN_STR(InspDate, _L("Inspection Date"), strLine)
						ReadString(strLine);
						GETN_STR(InspName, _L("Inspector Names"), strLine)
						ReadString(strLine);
						GETN_STR(DivInsp, _L("Division or Section of The Inspectors"), strLine)
						ReadString(strLine);
						GETN_STR(FacilityName, _L("Facility Name"), strLine)
						ReadString(strLine);
						GETN_STR(FacilityCode, _L("Facility Code"), strLine)
						ReadString(strLine);
						GETN_STR(MBA, _L("MBA"), strLine)
						ReadString(strLine);
						GETN_STR(Stratum, _L("Stratum"), strLine)
						ReadString(strLine);
						GETN_STR(ItemID, _L("Item ID"), strLine)
						ReadString(strLine);
						GETN_STR(BatchID, _L("Batch ID"), strLine)
						ReadString(strLine);
						GETN_STR(DateDec, _L("Date of Declaration"), strLine)
						ReadString(strLine);
						GETN_STR(Decl238, "Decl.Isotopic Abodance and Relative Uncertainty for 238 Pu", strLine)
						ReadString(strLine);
						GETN_STR(Decl239, "...for 239 Pu", strLine)
						ReadString(strLine);
						GETN_STR(Decl240, "...for 240 Pu", strLine)
						ReadString(strLine);
						GETN_STR(Decl241, "...for 241 Pu", strLine)
						ReadString(strLine);
						GETN_STR(Decl242, "...for 242 Pu", strLine)
						ReadString(strLine);
						GETN_STR(DeclAm, "...for 241 Am", strLine)
						ReadString(strLine);
						GETN_STR(InsCode, _L("Instrument Code"), strLine)
						ReadString(strLine);
						GETN_STR(MMCAID, _L("MMCA ID"), strLine)
						ReadString(strLine);
						GETN_STR(ComputerID, _L("Computer ID"), strLine)
						ReadString(strLine);
						GETN_STR(DetectorType, _L("Detector Type"), strLine)
						ReadString(strLine);
						GETN_STR(DetectorID, _L("Detector ID"), strLine)
						ReadString(strLine);
						GETN_STR(Collimator, _L("Collimator"), strLine)
						ReadString(strLine);
						GETN_STR(Filter, _L("Filter"), strLine)
					GETN_END_BRANCH(CsRation)
				}
			}		
			break;
			
		default:
			ASSERT(false);
			break;
		}
		GETN_END_BRANCH(Block)
	}
	LONG nCurrent = GetPosition();
	SeekToEnd();
	LONG nEnd = GetPosition();
	ASSERT(nCurrent == nEnd);
	TreeNode trData = m_trFile.FindNodeByAttribute(STR_LABEL_ATTRIB, MAKE_BLOCK_NAME(DATA));
	if ( !trData )
		return IAEA_NO_DATA;
	return IAEA_OK;
}

BLOCK_TYPE	oIAEASPE::ReadBlockType()
{
	BLOCK_TYPE	btTest = MAKE_BLOCK_ENUM(UNKNOWN);
	string strLine;
	while ( MAKE_BLOCK_ENUM(UNKNOWN) == btTest )
	{
		if ( !ReadString(strLine) )
			return btTest;
		btTest = _get_block_type(strLine);
	}
	return btTest;
}

#endif	//_OIAEASPE_H_