/*------------------------------------------------------------------------------*
 * File Name:	VideoWriter.h		 											*
 * Creation: 																	*
 * Purpose:		Origin C Header file for writing video file.					*
 * Copyright (c) 2012 OriginLab Corp. 											*
 * All Rights Reserved															*
 * 																				*
 * Modification Log:															*
 * EJP 2012-04-18 ORG-4928 TRY_CATCH_CALLS_TO_OPENCV							*
 * EJP 2012-06-15 ORG-4927 IMPROVE_VIDEOWRITER_ERROR_HANDLING					*
 * EJP 2012-06-26 ORG-4927 IMPROVE_VIDEOWRITER_ERROR_HANDLING					*
 * EJP 2012-06-26 ORG-4929-P1 CHECK_WRITE_ACCESS_TO_PREVENT_CRASH				*
 *	ML 7/26/2012 ORG-5223 FASTER_MOVIE_CREATION_FROM_3D_OPENGL					*
 * EJP 2012-07-30 ORG-6394 ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS			*
 * EJP 2012-08-01 ORG-6321 LOW_EXPORT_DPI_CAUSE_SYMBOL_SIZE_OF_1_TO_NOT_APPEAR	*
 * EJP 2012-08-06 ORG-4927 FIX_VIDEOWRITER_MEM_LEAK								*
 * EJP 2012-10-11 ORG-7113 VIDEO_BUILDER_MEMORY_LEAK							*
 *------------------------------------------------------------------------------*/

#ifndef _VIDEOWRITER_H_
#define _VIDEOWRITER_H_


#include <Origin.h>
#include <opencv.h>


#define VW_CODEC_PROMPT_USER	-1 // prompt user with dialog to choose codec
#define VW_CODEC_UNCOMPRESSED	0  // no codec, uncompressed, can create large files

// NOTE: You can not expect all codecs to be available on all computers.
#define VW_CODEC_MPEG1		CV_FOURCC('P', 'I', 'M', '1') // MPEG-1 codec
#define VW_CODEC_MJPEG		CV_FOURCC('M', 'J', 'P', 'G') // motion-jpeg codec (does not work well)
#define VW_CODEC_MPEG42		CV_FOURCC('M', 'P', '4', '2') // MPEG-4.2 codec
#define VW_CODEC_MPEG43     CV_FOURCC('D', 'I', 'V', '3') // MPEG-4.3 codec
#define VW_CODEC_MPEG4		CV_FOURCC('D', 'I', 'V', 'X') // MPEG-4 codec
#define VW_CODEC_H263		CV_FOURCC('U', '2', '6', '3') // H263 codec
#define VW_CODEC_H263I		CV_FOURCC('I', '2', '6', '3') // H263I codec
#define VW_CODEC_FLV1		CV_FOURCC('F', 'L', 'V', '1') // FLV1 codec

#define VW_CODEC_INVALID	CV_FOURCC('!', '!', '!', '!') // a value to indicate invalid codec

/// EJP 2012-06-15 ORG-4927 IMPROVE_VIDEOWRITER_ERROR_HANDLING
#define VW_MIN_FPS 1
#define VW_MAX_FPS 120

/// EJP 2012-06-26 ORG-4927 IMPROVE_VIDEOWRITER_ERROR_HANDLING
/// Moved to opencv_error.h for sharing with Visual C++.
/*
enum {
	VW_SUCCESS = 0,

	// Start the error numbers at some high value.
	// This will hopefully avoid conflict with error numbers in code that makes
	// use of the VideoWriter class such as VideoRecorderDlg.
	VW_ERR = 1000,

	// Method argument errors
	VW_ERR_FILENAME_ARG,
	VW_ERR_WIDTH_ARG,
	VW_ERR_HEIGHT_ARG,
	VW_ERR_IPLIMAGE_ARG,
	VW_ERR_HDIB_ARG,
	VW_ERR_MATRIXOBJ_ARG,
	VW_ERR_MATRIXLAYER_ARG,
	VW_ERR_GRAPHPAGE_ARG,

	VW_ERR_MATRIXOBJ_GETDIB,
	VW_ERR_GRAPHPAGE_CREATEIMAGE,

	// OpenCV call failed errors
	VW_ERR_CV_CREATE,
	VW_ERR_CV_GETIMAGE,
	VW_ERR_CV_CREATEIMAGE,
	VW_ERR_CV_WRITEFRAME,

	// Origin OpenCV call failed errors
	VW_ERR_OCV_CREATEIMAGEFROMDIB,
};
*/
/// end IMPROVE_VIDEOWRITER_ERROR_HANDLING
/// end IMPROVE_VIDEOWRITER_ERROR_HANDLING
/**+
http://ocwiki.originlab.com/index.php?title=OriginC:VideoWriter_(class)
*/
class VideoWriter
{
public:
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:VideoWriter-VideoWriter
	*/
	VideoWriter();
	
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:VideoWriter-~VideoWriter
	*/
	~VideoWriter();
	
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:VideoWriter-IsReady
	*/
	bool IsReady();
	
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:VideoWriter-Create
	*/
	int Create(LPCTSTR fileName, int codec, double fps, int width, int height, bool bColor = true);
	
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:VideoWriter-Release
	*/
	void Release();
	
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:VideoWriter-WriteFrame
	*/
	int WriteFrame(const IplImage* iplimg, UINT nNumFrames = 1);
	
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:VideoWriter-WriteFrame
	*/
	int WriteFrame(const CvMat& cvmat, UINT nNumFrames = 1);
	
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:VideoWriter-WriteFrame
	*/
	int WriteFrame(HDIB hDib, UINT nNumFrames = 1);
	
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:VideoWriter-WriteFrame
	*/
	int WriteFrame(const MatrixObject& mo, UINT nNumFrames = 1);
	
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:VideoWriter-WriteFrame
	*/
	int WriteFrame(const MatrixLayer& ml, int nObj = -1, UINT nNumFrames = 1); // -1 for active
	
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:VideoWriter-WriteFrame
	*/
	int WriteFrame(const GraphPage& gp, UINT nNumFrames = 1);
	
	/**+
	http://ocwiki.originlab.com/index.php?title=OriginC:VideoWriter-WriteFrames
	*/
	int WriteFrames(const MatrixLayer& ml, int first = 0, int last = -1);
	
protected:
	/**$
	*/
	CvVideoWriter* m_vw;	// OpenCV video writer object
	/**$
	*/
	CvSize m_size;			// video width and height in pixels
	/**$
	*/
	int m_inter;			// interpolation used if a frame needs resizing	
	
	/**$
	*/
	void Init();	
	/**$
	Parameters:
		iplimg = [Input] The image being written into the video.
		nNumFrames = [Input] The number of frames
	Return:
		0: Success
		none 0 values: Error code
	*/
	int write_frame(const IplImage* iplimg, UINT nNumFrames = 1);
	
private:	
	/**$
	Parameters:
		iplimg = [Input] The image being written into the video.
		nNumFrames = [Input] The number of frames
	Return:
		0: Success
		none 0 values: Error code
	*/
	UINT write_num_frames(const IplImage* iplimg, UINT nNumFrames = 1);
};

VideoWriter::VideoWriter()
{
	Init();
}

VideoWriter::~VideoWriter()
{
	if( IsReady() )
		Release();
}

bool VideoWriter::IsReady()
{
	return (m_vw != NULL);
}

int VideoWriter::Create(LPCTSTR fileName, int codec, double fps, int width, int height, bool bColor/* = true*/)
{
	/// EJP 2012-06-15 ORG-4927 IMPROVE_VIDEOWRITER_ERROR_HANDLING
	if( NULL == fileName || 0 == *fileName )
		return OCV_ERR_ARG_FILENAME;
	/// EJP 2012-06-26 ORG-4929-P1 CHECK_WRITE_ACCESS_TO_PREVENT_CRASH
	if( !ocvIsWriteAccess(fileName) )
		return OCV_ERR_NO_WRITE_ACCESS;
	/// end CHECK_WRITE_ACCESS_TO_PREVENT_CRASH
	if( width < 1 )
		return OCV_ERR_ARG_WIDTH;
	if( height < 1 )
		return OCV_ERR_ARG_HEIGHT;
	if( fps < VW_MIN_FPS )
		fps = VW_MIN_FPS;
	else if( fps > VW_MAX_FPS )
		fps = VW_MAX_FPS;
	/// end IMPROVE_VIDEOWRITER_ERROR_HANDLING

	m_size = cvSize(width, height);
	try
	{
		m_vw = cvCreateVideoWriter(fileName, codec, fps, m_size, bColor);
	}
	catch(int nErrCode)
	{
		m_vw = NULL;
	}

	/// EJP 2012-06-15 ORG-4927 IMPROVE_VIDEOWRITER_ERROR_HANDLING
	///return IsReady();
	if( NULL == m_vw )
		return OCV_ERR_CV_CREATEVIDEOWRITER;
	return 0;
	/// end IMPROVE_VIDEOWRITER_ERROR_HANDLING
}

void VideoWriter::Release()
{
	cvReleaseVideoWriter(&m_vw);
	m_vw = NULL; // is this needed?
}

//-------------------------------------------------------------------------
// WriteFrame methods
//-------------------------------------------------------------------------

/// EJP 2012-06-15 ORG-4927 IMPROVE_VIDEOWRITER_ERROR_HANDLING
/// Change all WriteFrame methods to return zero or a nonzero error number.
/// end IMPROVE_VIDEOWRITER_ERROR_HANDLING

/// EJP 2012-07-30 ORG-6394 ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
///int WriteFrame(const IplImage* iplimg)
int VideoWriter::WriteFrame(const IplImage* iplimg, UINT nNumFrames/* = 1*/)
/// end ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
{
	if( iplimg )
		/// EJP 2012-07-30 ORG-6394 ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
		///return write_frame(iplimg);
		return write_frame(iplimg, nNumFrames);
		/// end ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
	return OCV_ERR_ARG_IPLIMAGE;
}

/// EJP 2012-07-30 ORG-6394 ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
///int WriteFrame(const CvMat& cvmat)
int VideoWriter::WriteFrame(const CvMat& cvmat, UINT nNumFrames/* = 1*/)
/// end ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
{
	IplImage iplimgTemp;
	IplImage* iplimg = cvGetImage(&cvmat, &iplimgTemp);
	if( iplimg )
		/// EJP 2012-07-30 ORG-6394 ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
		///return write_frame(iplimg);
		return write_frame(iplimg, nNumFrames);
		/// end ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
	return OCV_ERR_CV_GETIMAGE;
}

/// EJP 2012-07-30 ORG-6394 ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
///int WriteFrame(HDIB hDib)
int VideoWriter::WriteFrame(HDIB hDib, UINT nNumFrames/* = 1*/)
/// end ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
{
	if( hDib )
	{
		IplImage* iplimg = NULL;
		int nErr = ocvCreateImageFromDib(&iplimg, hDib);
		if( 0 == nErr && iplimg )
		/// EJP 2012-08-06 ORG-4927 FIX_VIDEOWRITER_MEM_LEAK
		///	/// EJP 2012-07-30 ORG-6394 ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
		///	///return write_frame(iplimg);
		///	return write_frame(iplimg, nNumFrames);
		///	/// end ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
		///return MAKELONG(OCV_ERR_OCV_CREATEIMAGEFROMDIB, nErr);
			nErr = write_frame(iplimg, nNumFrames);
		if( nErr )
			nErr = MAKELONG(OCV_ERR_OCV_CREATEIMAGEFROMDIB, nErr);
		if( iplimg )
			cvReleaseImage(&iplimg);
		return nErr;
		/// end FIX_VIDEOWRITER_MEM_LEAK
	}
	return OCV_ERR_ARG_HDIB;
}

/// EJP 2012-07-30 ORG-6394 ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
///int WriteFrame(const MatrixObject& mo)
int VideoWriter::WriteFrame(const MatrixObject& mo, UINT nNumFrames/* = 1*/)
/// end ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
{
	if( mo.IsValid() )
	{
		HDIB hDib = mo.GetDIB();
		if( hDib )
			/// EJP 2012-07-30 ORG-6394 ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
			///return WriteFrame(hDib);
			return WriteFrame(hDib, nNumFrames);
			/// end ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS

		/* EJP 2012-03-19, I'm remvoing this code as it does not work as expected.
		   The following does not work for image matrix.
		int nType = ocvOTypeToCVType(mo.GetInternalData());
		CvMat cvmat;
		cvmat = cvMat(mo.GetNumRows(), mo.GetNumCols(), nType, mo.GetDataObject());
		return WriteFrame(cvmat);
		*/
		return OCV_ERR_MATRIXOBJ_GETDIB;
	}
	return OCV_ERR_ARG_MATRIXOBJ;
}

/// EJP 2012-07-30 ORG-6394 ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
///int WriteFrame(const MatrixLayer& ml, int nObj = -1) // -1 for active
int VideoWriter::WriteFrame(const MatrixLayer& ml, int nObj/* = -1*/, UINT nNumFrames/* = 1*/) // -1 for active
/// end ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
{
	if( ml.IsValid() )
	{
		if( nObj < 0 )
			nObj = ml.GetActive();
		MatrixObject mo = ml.MatrixObjects.Item(nObj);
		/// EJP 2012-07-30 ORG-6394 ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
		///return WriteFrame(mo);
		return WriteFrame(mo, nNumFrames);
		/// end ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
	}
	return OCV_ERR_ARG_MATRIXLAYER;
}

/// EJP 2012-07-30 ORG-6394 ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
///int WriteFrame(const GraphPage& gp)
int VideoWriter::WriteFrame(const GraphPage& gp, UINT nNumFrames/* = 1*/)
/// end ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
{
	if( gp.IsValid() )
	{
		/// EJP 2012-08-01 ORG-6321 LOW_EXPORT_DPI_CAUSE_SYMBOL_SIZE_OF_1_TO_NOT_APPEAR
		///	/// ML 7/26/2012 ORG-5223 FASTER_MOVIE_CREATION_FROM_3D_OPENGL
		///	//HDIB hDib = gp.CreateImage(CF_DIB, 96, 0);
		///	HDIB hDib = gp.CreateImage(CF_DIB, 96, GRPGCREATEIMAGE_FOR_MOVIE);
		///	/// end FASTER_MOVIE_CREATION_FROM_3D_OPENGL
		HDIB hDib = gp.CreateImage(CF_DIB, 150, GRPGCREATEIMAGE_FOR_MOVIE);
		/// end LOW_EXPORT_DPI_CAUSE_SYMBOL_SIZE_OF_1_TO_NOT_APPEAR
		if( hDib )
		{
			/// EJP 2012-07-30 ORG-6394 ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
			///int nErr = WriteFrame(hDib);
			int nErr = WriteFrame(hDib, nNumFrames);
			/// end ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
			/// EJP 2012-10-11 ORG-7113 VIDEO_BUILDER_MEMORY_LEAK
			///DeleteObject(hDib);
			GlobalFree(hDib);
			/// end VIDEO_BUILDER_MEMORY_LEAK
			return nErr;
		}
		return OCV_ERR_GRAPHPAGE_CREATEIMAGE;
	}
	return OCV_ERR_ARG_GRAPHPAGE;
}

int VideoWriter::WriteFrames(const MatrixLayer& ml, int first/* = 0*/, int last/* = -1*/)
{
	// This method supports writing objects in both forward and reverse order.
	if( first == -1 )
		first = ml.MatrixObjects.Count() - 1;
	if( last == -1 )
		last = ml.MatrixObjects.Count() - 1;

	int step = 1;
	if( last < first )
	{
		last--;
		step = -1;
	}
	else
		last++;

	int nErr = 0;
	for( int i = first; i != last; i += step)
	{
		nErr = WriteFrame(ml, i);
		if( nErr )
			return MAKELONG(nErr, i); // put MatrixObject index in high word
	}
	return 0;
}

void VideoWriter::Init()
{
	m_vw = NULL;
	m_inter = CV_INTER_LINEAR;
}

// All public WriteFrame methods should call this method
// which will check the frame size and resize it if needed.
/// EJP 2012-07-30 ORG-6394 ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
///int write_frame(const IplImage* iplimg)
int VideoWriter::write_frame(const IplImage* iplimg, UINT nNumFrames/* = 1*/)
/// end ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
{
	/// EJP 2012-07-30 ORG-6394 ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
	///int nWriteFrameRet = 0;
	UINT nFramesWritten = 0;
	/// end ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS

	if( iplimg->width != m_size.width || iplimg->height != m_size.height )
	{
		IplImage* iplimgResized = cvCreateImage(m_size, iplimg->depth, iplimg->nChannels);
		if( iplimgResized )
		{
			cvResize(iplimg, iplimgResized, m_inter);
			try
			{
				/// EJP 2012-07-30 ORG-6394 ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
				///nWriteFrameRet = cvWriteFrame(m_vw, iplimgResized);
				nFramesWritten = write_num_frames(iplimgResized, nNumFrames);
				/// end ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
			}
			catch(int nErrCode)
			{
				/// EJP 2012-07-30 ORG-6394 ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
				///nWriteFrameRet = 0;
				nFramesWritten = 0;
				/// end ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
			}
			cvReleaseImage(&iplimgResized);
		}
		else
			return OCV_ERR_CV_CREATEIMAGE;
	}
	else
		/// EJP 2012-07-30 ORG-6394 ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
		///nWriteFrameRet = cvWriteFrame(m_vw, iplimg);
		nFramesWritten = write_num_frames(iplimg, nNumFrames);
		/// end ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS

	/// EJP 2012-07-30 ORG-6394 ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
	///if( nWriteFrameRet )
	if( nFramesWritten == nNumFrames )
	/// end ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
		return 0;
	return OCV_ERR_CV_WRITEFRAME;
}

/// EJP 2012-07-30 ORG-6394 ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS
UINT VideoWriter::write_num_frames(const IplImage* iplimg, UINT nNumFrames/* = 1*/)
{
	UINT nFrame = 0;
	while( nFrame < nNumFrames )
	{
		if( 0 == cvWriteFrame(m_vw, iplimg) )
			break; // failed to write frame
		nFrame++;
	}
	return nFrame; // return number of frames written
}
/// end ADD_NUM_FRAMES_TO_OC_LT_WRITEFRAME_METHODS



#endif _VIDEOWRITER_H_
