/*------------------------------------------------------------------------------*
 * File Name: 	OrgObjInfo.h													*
 * Creation: 	Jasmine															*
 * Purpose: 	Origin C header file for Origin Objcet Info						*
 * Copyright (c) ABCD Corp.	2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010		*
 * All Rights Reserved															*
 * Modifications:																*
 * Kyle 07/23/08 ADD_FUNCTIONS_TO_GET_COLUMN_INFOS								*
 *------------------------------------------------------------------------------*/
#ifndef _ORG_OBJ_INFO_H
#define _ORG_OBJ_INFO_H

#include <okocUtils.h>

////////////////////////////////////////////////////////
//////////////////// OrgObjInfoBase ////////////////////
////////////////////////////////////////////////////////
class OrgObjInfoBase
{
public:	
	OrgObjInfoBase(OriginObject& obj = NULL);
	
	int SetInfoTypes(const vector<uint>	vnInfo);
	int GetInfoNames(vector<string>& vsNames);	
	int GetInfo(vector<string>& vsInfo, OriginObject& obj = NULL);
	void TurnOnOffset(bool bOffset){m_bOffset = bOffset;}
private:
	virtual int getInfoNames(vector<string>& vsNames){ASSERT(false); return -1;}
	virtual int getInfo(vector<string>& vsInfo, OriginObject& obj){ASSERT(false); return -1;}
protected:
	OriginObject 	m_obj;
	vector<uint>	m_vnInfo;
	bool			m_bOffset;
};


////////////////////////////////////////////////////////
////////////////////// ColumnInfo //////////////////////
////////////////////////////////////////////////////////
enum {
	COL_INFO_BEGIN		= 0,
	
	COL_INFO_INDEX		= COL_INFO_BEGIN,
	COL_INFO_SNAME,
	COL_INFO_LNAME,		
	COL_INFO_UNITS,
	COL_INFO_COMMENTS,
	COL_INFO_PARAMETERS,
	COL_INFO_USER_PARAME,
	
	COL_INFO_FORMULA,	
	
	COL_INFO_DESIGNATION,
	
	COL_INFO_FORMAT,			
	COL_INFO_DISPLAY,
	COL_INFO_NUMERIC,
	COL_INFO_DIGITS,
	COL_INFO_INTERNAL_DATA,
	
	COL_INFO_1ST_VALUE,
	COL_INFO_SAMPLING_INTERVAL,	
	COL_INFO_MIN,
	COL_INFO_MAX, 
	COL_INFO_MEDIAN,	
	
	COL_INFO_WIDTH,
	
	COL_INFO_SIZE,	
	
	COL_INFO_END = COL_INFO_SIZE,
};

class ColumnInfo : public OrgObjInfoBase
{
public:	
	ColumnInfo(const vector<uint>& vnInfo = NULL, OriginObject& obj = NULL);
	
private:
	virtual int getInfoNames(vector<string>& vsNames);
	virtual int getInfo(vector<string>& vsInfo, OriginObject& obj);
	///Kyle 07/23/08 ADD_FUNCTIONS_TO_GET_COLUMN_INFOS
	string getDesignationStr(int nDesignation)
	{
		switch(nDesignation)
		{
		case OKDATAOBJ_DESIGNATION_X:
			return "X";
		case OKDATAOBJ_DESIGNATION_Y:
			return "Y";
		case OKDATAOBJ_DESIGNATION_Z:
			return "Z";
		case OKDATAOBJ_DESIGNATION_X_ERROR:
			return "XErr";
		case OKDATAOBJ_DESIGNATION_ERROR:
			return "YErr";
		case OKDATAOBJ_DESIGNATION_L:
			return "L";
		default :	//OKDATAOBJ_DESIGNATION_NONE
			return " "; //CPY 6/15/06 changed from Disregard to make it shorter
		}
		return "";
	}
	string getFormatStr(int nFormat)
	{
		switch(nFormat)
		{
		case OKCOLTYPE_NUMERIC:
			return "Numeric";
		case OKCOLTYPE_TEXT:
			return "Text";
		case OKCOLTYPE_TIME:
			return "Time";
		case OKCOLTYPE_DATE:
			return "Date";
		case OKCOLTYPE_MONTH:
			return "Month";
		case OKCOLTYPE_WEEKDAY:
			return "Day";
		case OKCOLTYPE_TEXT_NUMERIC:
			return "T&N"; //CPY 6/15/06, changed from Text & Numeric to make it shorter
		default:
			return "";
		}
		return "";
	}
	///End ADD_FUNCTIONS_TO_GET_COLUMN_INFOS
};


////////////////////////////////////////////////////////
//////////////////// OrgObjInfoBase ////////////////////
////////////////////////////////////////////////////////
OrgObjInfoBase::OrgObjInfoBase(OriginObject& obj)//NULL, NULL
{
	if(obj.IsValid())
		m_obj = obj;
}
	
int OrgObjInfoBase::SetInfoTypes(const vector<uint>	vnInfo)
{
	if( NULL == vnInfo || 0 >= vnInfo.GetSize() )	
		return error_report("vnInfo is empty!");
	
	m_vnInfo = vnInfo;
	return m_vnInfo.GetSize();
}

int OrgObjInfoBase::GetInfoNames(vector<string>& vsNames)// = NULL
{
	return getInfoNames(vsNames);
}

int OrgObjInfoBase::GetInfo(vector<string>& vsInfo, OriginObject& obj)// = NULL
{
	if(obj.IsValid())
		m_obj = obj;
		
	if(!m_obj.IsValid())
	{
		error_report("invalid origin object");
		return -1;
	}
	
	return getInfo(vsInfo, m_obj);
}


////////////////////////////////////////////////////////
////////////////////// ColumnInfo //////////////////////
////////////////////////////////////////////////////////
ColumnInfo::ColumnInfo(const vector<uint>& vnInfo, OriginObject& obj) : OrgObjInfoBase(obj)
{
	if(NULL == vnInfo)
	{
		m_vnInfo.Data(COL_INFO_BEGIN, COL_INFO_END);
		if(m_vnInfo.GetSize() != COL_INFO_END + 1)
			error_report("ColumnInfo: size of m_vnInfo is wrong");
	}
	else
		m_vnInfo = vnInfo;
}

//virtual 
int ColumnInfo::getInfoNames(vector<string>& vsNames)
{
	if(NULL == vsNames)
		return -1;
	
	int nSize = m_vnInfo.GetSize();
	vsNames.SetSize(nSize);
	
	for(int ii = 0; ii < nSize; ii++)
	{
		int 	nInfo = m_vnInfo[ii];
		switch(nInfo)
		{
		case COL_INFO_INDEX:
			vsNames[ii] = _L("Index");
			break;
		case COL_INFO_SNAME:
			vsNames[ii] = _L("Column");
			break;
		case COL_INFO_LNAME:
			vsNames[ii] = STR_LONG_NAME_LABEL;
			break;
		case COL_INFO_UNITS:
			vsNames[ii] = STR_UNIT_LABEL;
			break;
		case COL_INFO_COMMENTS:
			vsNames[ii] = STR_COMMENTS_LABEL;
			break;
		case COL_INFO_PARAMETERS:
			vsNames[ii] = GetLocalized(E_STR_PARAMETERS);
			break;
		case COL_INFO_USER_PARAME:
			vsNames[ii] = GetLocalized(E_STR_USER_DEFINED_PARAMETERS);
			break;	
		case COL_INFO_1ST_VALUE:
			vsNames[ii] = _L("1st Value");
			break;
		case COL_INFO_SAMPLING_INTERVAL:
			vsNames[ii] = STR_SAMPLE_RATE_LABEL;
			break;
		case COL_INFO_FORMULA:
			vsNames[ii] = "";
			break;
		case COL_INFO_DESIGNATION:
			vsNames[ii] = _L("Type");
			break;	
		case COL_INFO_FORMAT:
			vsNames[ii] = "";
			break;
		case COL_INFO_DISPLAY:
			vsNames[ii] = "";
			break;
		case COL_INFO_NUMERIC:
			vsNames[ii] = "";
			break;
		case COL_INFO_DIGITS:
			vsNames[ii] = "";
			break;
		case COL_INFO_INTERNAL_DATA:
			vsNames[ii] = "";
			break;	
		case COL_INFO_MIN:
			vsNames[ii] = "";
			break;
		case COL_INFO_MAX:
			vsNames[ii] = "";
			break;
		case COL_INFO_MEDIAN:
			vsNames[ii] = "";
			break;
		case COL_INFO_WIDTH:
			vsNames[ii] = "";
			break;	
		case COL_INFO_SIZE:
			vsNames[ii] = "";
			break;
		}
	}

	ASSERT(ii == nSize);
	return ii;
}

//virtual 
int ColumnInfo::getInfo(vector<string>& vsInfo, OriginObject& obj)
{
	if(NULL == vsInfo)
		return -1;
	
	int nSize = m_vnInfo.GetSize();
	vsInfo.SetSize(nSize);
	
	Column col;
	col = obj;
	if( !col.IsValid() )
	{
		error_report("origin object is not a valid column");	
		return -1;
	}
	
	for(int ii = 0; ii < nSize; ii++)
	{
		int 	nInfo = m_vnInfo[ii];
		switch(nInfo)
		{
		case COL_INFO_INDEX:
			int nIndex = col.GetIndex();
			if(m_bOffset)nIndex += 1;
			vsInfo[ii] = (string)nIndex;
			break;
		case COL_INFO_SNAME:
			vsInfo[ii] = col.GetName(); 
			break;
		case COL_INFO_LNAME:
			vsInfo[ii] = col.GetLongName();
			break;
		case COL_INFO_UNITS:
			vsInfo[ii] = col.GetUnits(); 
			break;
		case COL_INFO_COMMENTS:
			vsInfo[ii] = col.GetComments();
			break;
		case COL_INFO_PARAMETERS:
			string strPara;
			col.GetParameter(strPara);
			vsInfo[ii] = strPara;
			break;
		case COL_INFO_USER_PARAME:
			string strUserPar;
			col.GetExtendedLabel(strUserPar, RCLT_UDL);
			vsInfo[ii] = strUserPar;
			break;	
		case COL_INFO_1ST_VALUE:
			vector<string> vsVal;
			if(col.GetStringArray( vsVal, 0, 0))
				vsInfo[ii] = vsVal[0];
			break;
		case COL_INFO_SAMPLING_INTERVAL:
			string 	strTemp;
			double dX0, dInc;
			if( col.IsEvenSampling(&dX0, &dInc) )
			{
				cvt_x_from_step_to_str(dX0, dInc, strTemp.GetBuffer(MAXLINE), MAXLINE);
				strTemp.ReleaseBuffer();
			}
			vsInfo[ii] = strTemp;
			break;
		case COL_INFO_FORMULA:
			vsInfo[ii] = col.GetFormula();
			break;
		case COL_INFO_DESIGNATION:
			vsInfo[ii] = getDesignationStr(col.GetType());
			break;	
		case COL_INFO_FORMAT:
			vsInfo[ii] = getFormatStr( col.GetFormat() );
			break;
		case COL_INFO_DISPLAY:
			vsInfo[ii] = "";
			break;
		case COL_INFO_NUMERIC:
			vsInfo[ii] = "";
			break;
		case COL_INFO_DIGITS:
			vsInfo[ii] = "";
			break;
		case COL_INFO_INTERNAL_DATA:
			vsInfo[ii] = "";
			break;	
		case COL_INFO_MIN:
			vsInfo[ii] = "";
			break;
		case COL_INFO_MAX:
			vsInfo[ii] = "";
			break;
		case COL_INFO_MEDIAN:
			vsInfo[ii] = "";
			break;
		case COL_INFO_WIDTH:
			int nWidth = col.GetWidth();
			vsInfo[ii] = (string)nWidth;
			break;	
		case COL_INFO_SIZE:
			vsInfo[ii] = ftoa(col.GetUpperBound() - col.GetLowerBound() + 1);
			break;
		}
	}

	
	ASSERT(ii == nSize);
	return ii;
}

/*testing sample code
#include <...\originlab\OrgObjInfo.h>
//please new a workbook with 3 columns
static void test_OrgObjInfo()
{	
	Worksheet wks = Project.ActiveLayer();
	if(!wks)
		return;
	
	Column col1(wks, 1);
	Column col2(wks, 2);
	if(!col1 ||!col2)
		return;
	
	vector<uint> vn = {	COL_INFO_INDEX,
						COL_INFO_SNAME,
						COL_INFO_LNAME,		
						COL_INFO_SAMPLING_INTERVAL};
	
	ColumnInfo myColInfo(vn);
	
	vector<string> vs;
	int nSize = myColInfo.GetInfoNames(vs);
	ASSERT(nSize == nSize);
	
	vector<string> vs1, vs2;	
	nSize = myColInfo.GetInfo(vs1, col1);
	ASSERT(nSize == nSize);	
	nSize = myColInfo.GetInfo(vs2, col2);
	ASSERT(nSize == nSize);
	
	for(int ii = 0; ii < nSize; ii++)
	{
		string 	strMsg;
		int 	nInfo = vn[ii];
		switch(nInfo)
		{
		case COL_INFO_INDEX:
			int nn1 = atoi(vs1[ii]);
			int nn2 = atoi(vs2[ii]);
			printf("%s = %d,\t%d\n", vs[ii], nn1, nn2);
			break;
		case COL_INFO_SNAME:
		case COL_INFO_LNAME:
		case COL_INFO_SAMPLING_INTERVAL:
			string str1 = vs1[ii];
			string str2 = vs2[ii];
			printf("%s = %s,\t%s\n", vs[ii], str1, str2);
			break;
		}
	}
}
*/
#endif//_ORG_OBJ_INFO_H