/*------------------------------------------------------------------------------*
 * File Name:	NLFTypes.h														*
 * Creation: 	Folger Lun														*
 * Purpose: OriginC Source H file												*
 * Copyright (c) ABCD Corp.	2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011*
 * , 2012
 * All Rights Reserved															*
 * 																				*
 * Modification Log:  															*
 *	Folger 12/29/2011 ORG-4454 REFACTOR_NLFITSESSION							*
 *	Sim 2012-03-20 ORG-5293 FIX_HIDDEN_WEIGHT_OF_INDEP_FOR_EXPLICIT_FITTING		*
 *	Sim 2012-05-07 ORG-5623 FIX_XYZ_FIT_REPLICAS_FIND_PEAKS						*
 *	Folger 05/09/2012 ORG-5592-S1 ADD_DATA_IDENTIFIER_FOR_FITTING_TOOLS			*
 *------------------------------------------------------------------------------*/

///------ Folger 12/29/2011 ORG-4454 REFACTOR_NLFITSESSION



#ifndef		__NLF_TYPES__
#define		__NLF_TYPES__

typedef enum tagNLFDATATYPE
{
	DATA_TYPE_INVALID = -1,
	DATA_TYPE_SINGLE_SET,
	DATA_TYPE_MULTI_SET,
	DATA_TYPE_SURFACE,
	DATA_TYPE_MATRIX,
} NLFDATATYPE;

struct  ReplicaInitParamsMore
{
	int*		pnNumPeaks;
	vector*		pvx;
	vector*		pvy;
	vector*		pvz;
};

#define	NLFIT_AREA_PARAMETER_CALCULATION_FACTOR		sqrt(PI/(4*ln(2)))

typedef	int	(*FUNC_INTT_1D_PEAK_PARAMETERS_FOR_REPLICA_FITTING)(const vector& vX, const vector& vY, int nNeededPeakNum, vector& vXCenter, vector& vYCenter, vector& vXWidth, const FindPeakCtrlInfo* pFindPeakCtrlInfo);
typedef	int	(*FUNC_INIT_2D_PEAK_PARAMETERS_FOR_REPLICA_FITTING)(const vector& vxGrid, const vector& vyGrid, const vector& vzGrid, int nReplica, vector& vXCenter, vector& vYCenter, vector& vZCenter, vector& vXWidth, vector& vYWidth, const FindPeakCtrlInfo* pFindPeakCtrlInfo);

typedef ODWP (*PFN_CREATE_PREVIEW)(ODWP pp);


#define NLFTrait_ERR_MSG(_str)


///////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////// NLFitModel ///////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////

class NLFitModelBase
{
public:
	NLFitModelBase()
	{
		NLFTrait_ERR_MSG("Create NLFitModelBase");
	}
	~NLFitModelBase()
	{
		NLFTrait_ERR_MSG("Destroy NLFitModelBase");
	}

	virtual	BOOL	IsImplicit()
	{
		O_A_FAIL;
		return FALSE;
	}

	virtual	void		SetFitMethod(int nMethod)
	{
	}

	virtual	int			GetFitMethod()
	{
		O_A_FAIL;
		return FITMETH_AUTO;
	}
};

class NLFModelitExplicit : public NLFitModelBase
{
public:
	NLFModelitExplicit()
	{
		m_nFitMethod = FITMETH_LEVENBERG_MARQUARDT;
	}

	virtual	BOOL	IsImplicit()
	{
		return FALSE;
	}

	virtual	void		SetFitMethod(int nMethod)
	{
		m_nFitMethod = nMethod;
	}

	virtual	int			GetFitMethod()
	{
		return m_nFitMethod;
	}

private:
	int		m_nFitMethod;
};

class NLFModelitImplicit : public NLFitModelBase
{
public:
	virtual	BOOL	IsImplicit()
	{
		return TRUE;
	}

	virtual	int			GetFitMethod()
	{
		return FITMETH_ORTHOGONAL_DISTANCE_REGRESSION;
	}
};

///////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////// NLFitModel ///////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////// NLFitTrait ///////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////

class NLFitTraitBase
{
public:
	NLFitTraitBase()
	{
		NLFTrait_ERR_MSG("Create NLFImplBase");
		m_pFitModel = NULL; ///---Sim 2012-03-20 ORG-5293 FIX_HIDDEN_WEIGHT_OF_INDEP_FOR_EXPLICIT_FITTING
	}
	~NLFitTraitBase()
	{
		NLFTrait_ERR_MSG("Destroy NLFImplBase");
	}

	virtual	DWORD		DataRules(BOOL bCheckMatrix = FALSE)
	{
		O_A_FAIL;
		return 0;
	}

	virtual	BOOL		CheckNumDepsIndeps(int nNumDeps, int nNumIndeps)
	{
		O_A_FAIL;
		return FALSE;
	}

	virtual	BOOL		GetSrcYMinMax(double& dyMin, double& dyMax, NLFDataBase& data)
	{
		O_A_FAIL;
		return FALSE;
	}

	virtual	BOOL		IsSupportReplica(BOOL bIs2DSupported)
	{
		O_A_FAIL;
		return FALSE;
	}

	virtual	BOOL		GetInputData(NLFDataBase& data, vector& vx, vector& vy, vector& vz = NULL, int nDatasetIndex = 0, vector* pvWeights = NULL)
	{
		O_A_FAIL;
		return FALSE;
	}

	virtual	BOOL		ReplicaAutoInitParams(NLFDataBase& data, FindPeakCtrlInfo& findPeakCtrlInfo, NLParametersManager& paramMngr, int nDataset, int nNumMultiplicity
		, const vector& vDefaultParams
		, ReplicaInitParamsMore& stMore
		)
	{
		O_A_FAIL;
		return FALSE;
	}

	virtual	BOOL		CreateFitterDataNode(TreeNode &trInput, TreeNode &trInputData, const StringArray& vstrIndepVars, const StringArray& vstrDepVars)
	{
		O_A_FAIL;
		return FALSE;
	}
	
	///---Sim 2012-03-20 ORG-5293 FIX_HIDDEN_WEIGHT_OF_INDEP_FOR_EXPLICIT_FITTING
	void				SetFitModel(NLFitModelBase *pFitModel)
	{
		m_pFitModel = pFitModel;
	}
	///---END ORG-5293 FIX_HIDDEN_WEIGHT_OF_INDEP_FOR_EXPLICIT_FITTING

	///------ Folger 05/09/2012 ORG-5592-S1 ADD_DATA_IDENTIFIER_FOR_FITTING_TOOLS
	virtual DWORD		GetUpdateDataIdentifierCntrl() { return 0; }
	///------ End ADD_DATA_IDENTIFIER_FOR_FITTING_TOOLS

protected:
	
	NLFitModelBase		*m_pFitModel; ///---Sim 2012-03-20 ORG-5293 FIX_HIDDEN_WEIGHT_OF_INDEP_FOR_EXPLICIT_FITTING
	
private:
};

class NLFitTraitGeneral : public NLFitTraitBase
{
public:
	virtual DWORD		DataRules(BOOL bCheckMatrix = FALSE)
	{
		return DRR_NO_FACTORS | DRR_GET_DEPENDENT | DRR_RAW_ERR_WEIGHT
#ifdef		__NLFIT_MULTI_DEPS_INDEPS_WEIGHTS__
			| DRR_NLFIT
#endif		// __NLFIT_MULTI_DEPS_INDEPS_WEIGHTS__
			;
	}

	virtual	BOOL		CheckNumDepsIndeps(int nNumDeps, int nNumIndeps)
	{
		return nNumDeps >= 1 && nNumIndeps >= 1;
	}

	virtual	BOOL		GetSrcYMinMax(double& dyMin, double& dyMax, NLFDataBase& data)
	{
		return data.GetDepMinMax(dyMin, dyMax);
	}

	virtual	BOOL		IsSupportReplica(BOOL bIs2DSupported)
	{
		return TRUE;
	}

	virtual	BOOL		GetInputData(NLFDataBase& data, vector& vx, vector& vy, vector& vz = NULL, int nDatasetIndex = 0, vector* pvWeights = NULL)
	{
		if ( !data.GetDependData(vy, nDatasetIndex) || !data.GetIndependData(vx, nDatasetIndex) )
			return FALSE;

		if ( pvWeights )
		{
			if ( !data.GetWeightData(*pvWeights, nDatasetIndex) )
				return FALSE;
		}

		return TRUE;
	}

	virtual	BOOL		ReplicaAutoInitParams(NLFDataBase& data, FindPeakCtrlInfo& findPeakCtrlInfo, NLParametersManager& paramMngr, int nDataset, int nNumMultiplicity
		, const vector& vDefaultParams
		, ReplicaInitParamsMore& stMore
		)
	{
		vector		vX, vY, vPeaksY, vXCenter, vYCenter, vXWidth;
		if ( !GetInputData(data, vX, vY, NULL, nDataset) )
			return error_report("Fail to get input data");

		#ifdef	__NLFIT_POOR_INIT_PARAMS_FOR_MULTIPLE_NEGATIVE_PEAKS__
		if ( vDefaultParams.GetSize() > 0 )
		{
			double	rBaseline = vDefaultParams[0];
			vY -= rBaseline;
		}
		#endif	/// __NLFIT_POOR_INIT_PARAMS_FOR_MULTIPLE_NEGATIVE_PEAKS__

		int			nNumPeaks = 0;
		FUNC_INTT_1D_PEAK_PARAMETERS_FOR_REPLICA_FITTING	pfn = Project.FindFunction("init_1d_peak_parameters_for_replica_fitting", "OriginLab\\curve_utils.c", true);
		if ( pfn )
			nNumPeaks = pfn(vX, vY, nNumMultiplicity, vXCenter, vYCenter, vXWidth, &findPeakCtrlInfo);

		if ( stMore.pnNumPeaks )
			*stMore.pnNumPeaks = nNumPeaks;

		if ( nNumPeaks < 1 )
		{
			error_report("No peak found!");
			return true;
		}

		ASSERT(vXCenter.GetSize() == vXWidth.GetSize() && vXCenter.GetSize() == vYCenter.GetSize() && vXCenter.GetSize() == vXWidth.GetSize());
		if ( vXCenter.GetSize() > nNumMultiplicity )
		{
			vXCenter.SetSize(nNumMultiplicity);
			vYCenter.SetSize(nNumMultiplicity);
			vXWidth.SetSize(nNumMultiplicity);
		}

		vector<uint> vnIndices;
		vXCenter.Sort(SORT_ASCENDING, false, vnIndices);
		vYCenter.Reorder(vnIndices);
		vXWidth.Reorder(vnIndices);

		double		dBaseline = 0.0;
		int			nBaselineParamIndex = 0; // Hong, to do, after SR2, need centralize this baseline index assumation into nlsf_utils, like get_baseline_param_index
		if ( vDefaultParams && vDefaultParams.GetSize() > nBaselineParamIndex )
			dBaseline= vDefaultParams[nBaselineParamIndex];

		static const double l_dSqrt = NLFIT_AREA_PARAMETER_CALCULATION_FACTOR;
		vector		vHeight;
		vHeight = vYCenter * vXWidth * l_dSqrt;
		if ( paramMngr.SetReplicasInitialParameters(vXCenter, vXWidth, vHeight, dBaseline, vDefaultParams) )
			return error_report("Fail to init replica paramters.");

		if ( stMore.pvy )
			*stMore.pvy = vYCenter;
		if ( stMore.pvx )
			*stMore.pvx = vXCenter;

		return TRUE;
	}

	virtual	BOOL		CreateFitterDataNode(TreeNode &trInput, TreeNode &trInputData, const StringArray& vstrIndepVars, const StringArray& vstrDepVars)
	{
		///---Sim 2012-03-20 ORG-5293 FIX_HIDDEN_WEIGHT_OF_INDEP_FOR_EXPLICIT_FITTING
		//BOOL bNeedIndepWeight = TRUE;
		BOOL bNeedIndepWeight = FALSE;
		if ( m_pFitModel )
			bNeedIndepWeight = ( FITMETH_ORTHOGONAL_DISTANCE_REGRESSION == m_pFitModel->GetFitMethod() );
		///---END ORG-5293 FIX_HIDDEN_WEIGHT_OF_INDEP_FOR_EXPLICIT_FITTING
		return okutil_create_fitter_data_node_ex(&trInputData, &trInput, STR_INPUT_RANGE_NAME, STR_INPUT_RANGE_LABEL, &vstrIndepVars, &vstrDepVars, WEIGHT_NONE, TRUE, bNeedIndepWeight);
	}

	///------ Folger 05/09/2012 ORG-5592-S1 ADD_DATA_IDENTIFIER_FOR_FITTING_TOOLS
	virtual DWORD		GetUpdateDataIdentifierCntrl() { return BUDI_USE_XY_RANGE; }
	///------ End ADD_DATA_IDENTIFIER_FOR_FITTING_TOOLS

private:
};

class NLFitTraitGeneralImplicit : public NLFitTraitBase
{
public:
	virtual DWORD		DataRules(BOOL bCheckMatrix = FALSE)
	{
		return DRR_NO_FACTORS | DRR_NLFIT;
	}

	virtual	BOOL		CheckNumDepsIndeps(int nNumDeps, int nNumIndeps)
	{
		return nNumDeps == 0 && nNumIndeps >= 2;
	}

	virtual	BOOL		GetSrcYMinMax(double& dyMin, double& dyMax, NLFDataBase& data)
	{
		return FALSE;
	}

	virtual	BOOL		IsSupportReplica(BOOL bIs2DSupported)
	{
		return FALSE;
	}

	virtual	BOOL		GetInputData(NLFDataBase& data, vector& vx, vector& vy, vector& vz = NULL, int nDatasetIndex = 0, vector* pvWeights = NULL)
	{
		return FALSE;
	}

	virtual	BOOL		ReplicaAutoInitParams(NLFDataBase& data, FindPeakCtrlInfo& findPeakCtrlInfo, NLParametersManager& paramMngr, int nDataset, int nNumMultiplicity
		, const vector& vDefaultParams
		, ReplicaInitParamsMore& stMore
		)
	{
		return FALSE;
	}

	virtual	BOOL		CreateFitterDataNode(TreeNode &trInput, TreeNode &trInputData, const StringArray& vstrIndepVars, const StringArray& vstrDepVars)
	{
		return okutil_create_implicit_fitter_data_node_ex(&trInputData, &trInput, STR_INPUT_RANGE_NAME, STR_INPUT_RANGE_LABEL, &vstrIndepVars, WEIGHT_NONE, TRUE);
	}

private:
};

class NLFitTrait2DWorker
{
public:
	DWORD		DataRules()
	{
		return DRR_NO_FACTORS | DRR_GET_Z_DEPENDENT | DRR_ONE_DATA;
	}

	BOOL		CheckNumDepsIndeps(int nNumDeps, int nNumIndeps)
	{
		return nNumDeps == 1 && nNumIndeps == 2;
	}

	BOOL		GetSrcYMinMax(double& dyMin, double& dyMax, NLFDataBase& data)
	{
		return data.GetIndepMinMax(dyMin, dyMax, 0, 1);
	}

	BOOL		IsSupportReplica(BOOL bIs2DSupported)
	{
		return bIs2DSupported;
	}

	BOOL		GetInputData(NLFDataBase& data, vector& vx, vector& vy, vector& vz = NULL, int nDatasetIndex = 0, vector* pvWeights = NULL)
	{
		if ( !data.GetIndependData(vx, nDatasetIndex, 0) || !data.GetIndependData(vy, nDatasetIndex, 1) )
			return FALSE;

		if ( vz )
		{
			if ( !data.GetDependData(vz, nDatasetIndex) )
				return FALSE;
		}

		return TRUE;
	}

	BOOL		ReplicaAutoInitParams(NLFDataBase& data, FindPeakCtrlInfo& findPeakCtrlInfo, NLParametersManager& paramMngr, int nDataset, int nNumMultiplicity
		, const vector& vDefaultParams
		, ReplicaInitParamsMore& stMore
		)
	{
		vector 		vxGrid, vyGrid, vzGrid;
		if ( !GetInputData(data, vxGrid, vyGrid, vzGrid, nDataset) )
			return error_report("Fail to get input data");
		int nRows, nCols;
		if ( data.GetDimention(nDataset, nRows, nCols) ) //matrix
		{
			double dXMin, dXMax, dYMin, dYMax;
			vxGrid.GetMinMax(dXMin, dXMax);
			vyGrid.GetMinMax(dYMin, dYMax);
			vxGrid.Data(dXMin, dXMax, (dXMax - dXMin) / (nCols - 1));
			vyGrid.Data(dYMin, dYMax, (dYMax - dYMin) / (nRows - 1));
			vxGrid.SetSize(nCols);
			vyGrid.SetSize(nRows);
		}

#ifdef	__NLFIT_POOR_INIT_PARAMS_FOR_MULTIPLE_NEGATIVE_PEAKS__
		if ( vDefaultParams.GetSize() > 0 )
		{
			double	rBaseline = vDefaultParams[0];
			vzGrid -= rBaseline;
		}
#endif	/// __NLFIT_POOR_INIT_PARAMS_FOR_MULTIPLE_NEGATIVE_PEAKS__

		int			nNumPeaks = 0;
		vector 		vXCenter, vYCenter, vZCenter, vXWidth, vYWidth;
		FUNC_INIT_2D_PEAK_PARAMETERS_FOR_REPLICA_FITTING	pfn = Project.FindFunction("init_2d_peak_parameters_for_replica_fitting", "OriginLab\\curve_utils.c", true);
		if ( pfn )
			nNumPeaks = pfn(vxGrid, vyGrid, vzGrid, nNumMultiplicity, vXCenter, vYCenter, vZCenter, vXWidth, vYWidth, &findPeakCtrlInfo);

		if ( stMore.pnNumPeaks )
			*stMore.pnNumPeaks = nNumPeaks;

		if ( nNumPeaks < 1 )
		{
			error_report("No peak found!");
			return true;
		}

		double dBaseLine = 0;
		if ( vDefaultParams && vDefaultParams.GetSize() > 0 )
			dBaseLine = vDefaultParams[0];
		if ( paramMngr.SetReplicasInitialParameters(vXCenter, vXWidth, vZCenter, dBaseLine, vDefaultParams, &vYCenter, &vYWidth) )
			return error_report("Fail to init replica parameters for 2D surface fitting.");
		return true;
	}

	///------ Folger 05/09/2012 ORG-5592-S1 ADD_DATA_IDENTIFIER_FOR_FITTING_TOOLS
	DWORD		GetUpdateDataIdentifierCntrl() { return BUDI_USE_XYZ_RANGE; }
	///------ End ADD_DATA_IDENTIFIER_FOR_FITTING_TOOLS
};



class NLFitTraitXYZ : public NLFitTraitBase
{
public:
	virtual DWORD		DataRules(BOOL bCheckMatrix = FALSE)
	{
		return m_worker.DataRules();
	}

	virtual	BOOL		CheckNumDepsIndeps(int nNumDeps, int nNumIndeps)
	{
		return m_worker.CheckNumDepsIndeps(nNumDeps, nNumIndeps);
	}

	virtual	BOOL		GetSrcYMinMax(double& dyMin, double& dyMax, NLFDataBase& data)
	{
		return m_worker.GetSrcYMinMax(dyMin, dyMax, data);
	}

	virtual	BOOL		IsSupportReplica(BOOL bIs2DSupported)
	{
		return m_worker.IsSupportReplica(bIs2DSupported);
	}

	virtual	BOOL		GetInputData(NLFDataBase& data, vector& vx, vector& vy, vector& vz = NULL, int nDatasetIndex = 0, vector* pvWeights = NULL)
	{
		return m_worker.GetInputData(data, vx, vy, vz, nDatasetIndex, pvWeights);
	}
	
	///---Sim 2012-05-07 ORG-5623 FIX_XYZ_FIT_REPLICAS_FIND_PEAKS
	BOOL		ReplicaAutoInitParams(NLFDataBase& data, FindPeakCtrlInfo& findPeakCtrlInfo, NLParametersManager& paramMngr, int nDataset, int nNumMultiplicity
		, const vector& vDefaultParams
		, ReplicaInitParamsMore& stMore
		)
	{
		return m_worker.ReplicaAutoInitParams(data, findPeakCtrlInfo, paramMngr, nDataset, nNumMultiplicity, vDefaultParams, stMore);
	}
	///---END ORG-5623 FIX_XYZ_FIT_REPLICAS_FIND_PEAKS

	///------ Folger 05/09/2012 ORG-5592-S1 ADD_DATA_IDENTIFIER_FOR_FITTING_TOOLS
	virtual DWORD		GetUpdateDataIdentifierCntrl() { return m_worker.GetUpdateDataIdentifierCntrl(); }
	///------ End ADD_DATA_IDENTIFIER_FOR_FITTING_TOOLS

private:
	NLFitTrait2DWorker		m_worker;
};

class NLFitTraitXYZImplicit : public NLFitTraitBase
{
public:
};

class NLFitTraitMatrix : public NLFitTraitBase
{
public:
	virtual DWORD		DataRules(BOOL bCheckMatrix = FALSE)
	{
		if ( bCheckMatrix )
			return DRR_NO_FACTORS | DRR_MATRIX_DATA | DRR_ONE_DATA;

		return m_worker.DataRules();
	}

	virtual	BOOL		CheckNumDepsIndeps(int nNumDeps, int nNumIndeps)
	{
		return m_worker.CheckNumDepsIndeps(nNumDeps, nNumIndeps);
	}

	virtual	BOOL		GetSrcYMinMax(double& dyMin, double& dyMax, NLFDataBase& data)
	{
		return m_worker.GetSrcYMinMax(dyMin, dyMax, data);
	}

	virtual	BOOL		IsSupportReplica(BOOL bIs2DSupported)
	{
		return m_worker.IsSupportReplica(bIs2DSupported);
	}

	virtual	BOOL		GetInputData(NLFDataBase& data, vector& vx, vector& vy, vector& vz = NULL, int nDatasetIndex = 0, vector* pvWeights = NULL)
	{
		return m_worker.GetInputData(data, vx, vy, vz, nDatasetIndex, pvWeights);
	}

	///---Sim 2012-05-07 ORG-5623 FIX_XYZ_FIT_REPLICAS_FIND_PEAKS
	BOOL		ReplicaAutoInitParams(NLFDataBase& data, FindPeakCtrlInfo& findPeakCtrlInfo, NLParametersManager& paramMngr, int nDataset, int nNumMultiplicity
		, const vector& vDefaultParams
		, ReplicaInitParamsMore& stMore
		)
	{
		return m_worker.ReplicaAutoInitParams(data, findPeakCtrlInfo, paramMngr, nDataset, nNumMultiplicity, vDefaultParams, stMore);
	}
	///---END ORG-5623 FIX_XYZ_FIT_REPLICAS_FIND_PEAKS

	///------ Folger 05/09/2012 ORG-5592-S1 ADD_DATA_IDENTIFIER_FOR_FITTING_TOOLS
	virtual DWORD		GetUpdateDataIdentifierCntrl() { return m_worker.GetUpdateDataIdentifierCntrl(); }
	///------ End ADD_DATA_IDENTIFIER_FOR_FITTING_TOOLS
	
private:
	NLFitTrait2DWorker		m_worker;
};

class NLFitTraitMatrixImplicit : public NLFitTraitBase
{
public:
};

///////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////// NLFitTrait ///////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////




///////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////// NLFitType ////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////


#define		CHECK_VALID_TRAIT_AND_RETURN(_return) \
			if ( NULL == m_pTrait ) \
			{ \
				O_A_FAIL; \
				return _return; \
			}

class NLFitTypeBase
{
public:
	NLFitTypeBase()
	{
		NLFTrait_ERR_MSG("Create NLFitTypeBase");
		m_dwCntrl = 0;
		m_pTrait = NULL;
	}
	~NLFitTypeBase()
	{
		NLFTrait_ERR_MSG("Destroy NLFitTypeBase");
		NICE_SAFE_REMOVAL(m_pTrait);
	}

	void				SetTrait(NLFitTraitBase* pTrait)
	{
		NICE_SAFE_REMOVAL(m_pTrait);
		m_pTrait = pTrait;
	}

	void				SetDataTypeCntrl(BOOL bSingleSet, BOOL bMultiData)
	{
		O_SET_BIT(m_dwCntrl, NLFTYPECNTRL_DATATYPESINGLESET, bSingleSet);
		O_SET_BIT(m_dwCntrl, NLFTYPECNTRL_DATATYPEMULTIDATA, bMultiData);
	}

	virtual	int			Type()
	{
		O_A_FAIL;
		return -1;
	}

	virtual	BOOL		Is2D()
	{
		return FALSE;
	}

	DWORD				DataRules(BOOL bCheckMatrix = FALSE)
	{
		CHECK_VALID_TRAIT_AND_RETURN(0);
		return m_pTrait->DataRules(bCheckMatrix);
	}

	NLFDATATYPE			DataType()
	{
		if ( O_QUERY_BOOL(m_dwCntrl, NLFTYPECNTRL_DATATYPESINGLESET) )
			return DATA_TYPE_SINGLE_SET;

		if ( O_QUERY_BOOL(m_dwCntrl, NLFTYPECNTRL_DATATYPEMULTIDATA) )
			return DATA_TYPE_MULTI_SET;

		return DataType2();
	}

	virtual	NLFDataBase*	CreateDataObj()
	{
		switch ( DataType() )
		{
		case DATA_TYPE_SINGLE_SET:
			return new NLFDataSingleIndep;

		case DATA_TYPE_MULTI_SET:
			return new NLFDataMultiIndeps;

		default:
			O_A_FAIL;
			break;
		}

		return NULL;
	}

	void				DestroyDataObj(NLFDataBase** ppData)
	{
		NICE_SAFE_REMOVAL(*ppData);
	}

	virtual	ODWP		CreatePreviewObj(ODWP pSession)
	{
		O_A_FAIL;
		return NULL;
	}

	BOOL				CheckNumDepsIndeps(int nNumDeps, int nNumIndeps)
	{
		CHECK_VALID_TRAIT_AND_RETURN(FALSE);
		return m_pTrait->CheckNumDepsIndeps(nNumDeps, nNumIndeps);
	}

	virtual	BOOL		CheckXYZRegularData(NLFDataBase* pData, int& nRows, int& nCols)
	{
		O_A_FAIL;
		return FALSE;
	}

	BOOL				GetSrcYMinMax(double& dyMin, double& dyMax, NLFDataBase& data)
	{
		CHECK_VALID_TRAIT_AND_RETURN(FALSE);
		return m_pTrait->GetSrcYMinMax(dyMin, dyMax, data);
	}

	BOOL				IsSupportReplica(BOOL bIs2DSupported)
	{
		CHECK_VALID_TRAIT_AND_RETURN(FALSE);
		return m_pTrait->IsSupportReplica(bIs2DSupported);
	}

	BOOL				GetRangeForDisplay(NLFRangeHolder& rngHolder, DataRange& drSub, int nDatasetIndex = 0, int iDep = 0, int iIndep = 0)
	{
		if ( rngHolder.IsConcatenateSrcData() )
		{
			drSub = rngHolder.GetSrcDataRange();
			return drSub.IsValid();
		}

		return GetRangeForDisplay2(rngHolder, drSub, nDatasetIndex, iDep, iIndep);
	}

	BOOL			GetInputData(NLFDataBase& data, vector& vx, vector& vy, vector& vz = NULL, int nDatasetIndex = 0, vector* pvWeights = NULL)
	{
		CHECK_VALID_TRAIT_AND_RETURN(FALSE);
		return m_pTrait->GetInputData(data, vx, vy, vz, nDatasetIndex, pvWeights);
	}
	
	virtual	BOOL		GetParamInitValues(NLFDataBase& data, const NumericFunction& fn, int nDataset, vector& vParamValue)
	{
		O_A_FAIL;
		return FALSE;
	}

	BOOL				ReplicaAutoInitParams(NLFDataBase& data, FindPeakCtrlInfo& findPeakCtrlInfo, NLParametersManager& paramMngr, int nDataset, int nNumMultiplicity
		, const vector& vDefaultParams
		, ReplicaInitParamsMore& stMore
		)
	{
		CHECK_VALID_TRAIT_AND_RETURN(FALSE);
		return m_pTrait->ReplicaAutoInitParams(data, findPeakCtrlInfo, paramMngr, nDataset, nNumMultiplicity, vDefaultParams, stMore);
	}

	virtual	BOOL		UpdateInputBranch(TreeNode &trInput, const StringArray& vstrIndepVars, const StringArray& vstrDepVars)
	{
		return FALSE;
	}

	virtual	bool		CheckInputErr(string& strErrMsg, TreeNode& trInput)
	{
		DataRange	dr;
		if ( !dr.Create(trInput, TRUE) )
		{
			strErrMsg = XFERR_INVALID_RANGE;
			return false;
		}

		return true;
	}

	///------ Folger 05/09/2012 ORG-5592-S1 ADD_DATA_IDENTIFIER_FOR_FITTING_TOOLS
	DWORD				GetUpdateDataIdentifierCntrl()
	{
		CHECK_VALID_TRAIT_AND_RETURN(FALSE);
		return m_pTrait->GetUpdateDataIdentifierCntrl();
	}
	///------ End ADD_DATA_IDENTIFIER_FOR_FITTING_TOOLS

protected:
	PFN_CREATE_PREVIEW	FactoryFunc(LPCSTR lpcszName)
	{
		PFN_CREATE_PREVIEW	pfn = Project.FindFunction(lpcszName, "OriginLab\\nlsf_factory.c", true);
		O_A(pfn);
		return pfn;
	}

	NLFitTraitBase*		m_pTrait;

private:

	virtual	NLFDATATYPE	DataType2()
	{
		O_A_FAIL;
		return DATA_TYPE_INVALID;
	}

	virtual	BOOL		GetRangeForDisplay2(NLFRangeHolder& rngHolder, DataRange& drSub, int nDatasetIndex = 0, int iDep = 0, int iIndep = 0)
	{
		O_A_FAIL;
		return FALSE;
	}

	DWORD				m_dwCntrl;
	enum
	{
		NLFTYPECNTRL_DATATYPESINGLESET					= 0x00000001,
		NLFTYPECNTRL_DATATYPEMULTIDATA					= 0x00000002,
	};
};





class NLFitTypeGeneral : public NLFitTypeBase
{
public:
	virtual	int			Type()
	{
		return NLFIT_GENERAL_XY_FITTING;
	}

	virtual	ODWP		CreatePreviewObj(ODWP pSession)
	{
		return FactoryFunc("create_general_fitting_preview")(pSession);
	}

	virtual	BOOL		GetParamInitValues(NLFDataBase& data, const NumericFunction& fn, int nDataset, vector& vParamValue)
	{
		if ( !data.IsDataValid(nDataset) )
			return FALSE;

		int		nErr = 0;
		return fn.ParamInit(vParamValue, data.GetIndependData(nDataset), data.GetDependData(nDataset), nErr) && 0 == nErr;
	}

	virtual	BOOL		UpdateInputBranch(TreeNode &trInput, const StringArray& vstrIndepVars, const StringArray& vstrDepVars)
	{
		CHECK_VALID_TRAIT_AND_RETURN(FALSE);

		if ( trInput.GetNode(STR_INPUT_RANGE_NAME) )
		{
			trInput.GetNode(STR_INPUT_RANGE_NAME).SetAttribute(STR_DATA_RULES_ATTRIB, DataRules());
		}

		TreeNode		trInputData;
		if ( !m_pTrait->CreateFitterDataNode(trInput, trInputData, vstrIndepVars, vstrDepVars) || !trInputData )
			return FALSE;

		return TRUE;
	}

	virtual	bool		CheckInputErr(string& strErrMsg, TreeNode& trInput)
	{
		DataRange	dr;
		if ( !nlsf_init_datarange(dr, trInput) )
		{
			strErrMsg = XFERR_INVALID_RANGE;
			return false;
		}
		
		return true;
	}

private:
	virtual	NLFDATATYPE	DataType2()
	{
		return DATA_TYPE_MULTI_SET;
	}

	virtual	BOOL		GetRangeForDisplay2(NLFRangeHolder& rngHolder, DataRange& drSub, int nDatasetIndex = 0, int iDep = 0, int iIndep = 0)
	{
		return rngHolder.GetSrcSubDataRange(drSub, nDatasetIndex, iDep, iIndep);
	}
};





class NLFitType2D : public NLFitTypeBase
{
public:
	virtual	BOOL		Is2D()
	{
		return TRUE;
	}

	virtual	BOOL		GetParamInitValues(NLFDataBase& data, const NumericFunction& fn, int nDataset, vector& vParamValue)
	{
		vector		vX, vY, vZ;
		if ( !GetInputData(data, vX, vY, vZ, nDataset) )
			return FALSE;

		UpdateInputDataForParamsInit(data, nDataset, vX, vY, vZ);
		
		int		nErr = 0;
		return fn.ParamInit(vParamValue, vX, vY, vZ, nErr) && 0 == nErr;
	}

private:
	virtual	void		UpdateInputDataForParamsInit(NLFDataBase& data, int nDataset, vector& vX, vector& vY, vector& vZ)
	{
	}

	virtual	BOOL		GetRangeForDisplay2(NLFRangeHolder& rngHolder, DataRange& drSub, int nDatasetIndex = 0, int iDep = 0, int iIndep = 0)
	{
		drSub = rngHolder.GetSrcDataRange();
		return drSub.IsValid();
	}
};





class NLFitTypeXYZ : public NLFitType2D
{
public:
	virtual	int			Type()
	{
		return NLFIT_XYZ_FITTING;
	}

	virtual	NLFDataBase*	CreateDataObj()
	{
		return new NLFDataSurface;
	}

	virtual	ODWP		CreatePreviewObj(ODWP pSession)
	{
		return FactoryFunc("create_xyz_surface_fitting_preview")(pSession);
	}

	virtual	BOOL		CheckXYZRegularData(NLFDataBase* pData, int& nRows, int& nCols)
	{
		if( NULL != nRows )
			nRows = 1;
		if( NULL != nCols )		
			nCols = 1;
		return true;
	}

private:
	virtual	NLFDATATYPE	DataType2()
	{
		return DATA_TYPE_SURFACE;
	}
};




#define LARGE_DATA_SET_NUM_ROWS							240
#define LARGE_DATA_SET_NUM_COLS							420
#define LARGE_DATA_SET_ROW_SHRINK_FACTOR				6
#define LARGE_DATA_SET_COL_SHRINK_FACTOR				8

class NLFitTypeMatrix : public NLFitType2D
{
public:
	virtual	int			Type()
	{
		return NLFIT_MATRIX_FITTING;
	}

	virtual	NLFDataBase*	CreateDataObj()
	{
		return new NLFDataMatrix;
	}

	virtual	ODWP		CreatePreviewObj(ODWP pSession)
	{
		return FactoryFunc("create_matrix_surface_fitting_preview")(pSession);
	}

	virtual	BOOL		CheckXYZRegularData(NLFDataBase* pData, int& nRows, int& nCols)
	{
		return pData ? pData->CheckXYZRegularData(nRows, nCols) : FALSE;
	}

	virtual	bool		CheckInputErr(string& strErrMsg, TreeNode& trInput)
	{
		DataRange	dr;
		if ( !nlsf_init_datarange(dr, trInput) )
		{
			strErrMsg = XFERR_INVALID_RANGE;
			return false;
		}

		int			c1, c2;
		Datasheet	ds;
		dr.GetRange(ds, c1, c2);
		MatrixLayer ml(ds);
		if ( !ml )
		{
			strErrMsg = XFERR_INVALID_RANGE;
			return false;
		}

		MatrixObject mo(ml, c1);
		if ( mo && ml.HasImage() && !mo.HasData() )
		{
			strErrMsg = NLSF_ERR_INPUT_IMAGE;
			return false;
		}

		return true;
	}

private:
	virtual	NLFDATATYPE	DataType2()
	{
		return DATA_TYPE_MATRIX;
	}

	virtual	void		UpdateInputDataForParamsInit(NLFDataBase& data, int nDataset, vector& vX, vector& vY, vector& vZ)
	{
		int		nCols, nRows;			
		if ( data.GetDimention(nDataset, nRows, nCols) && (nCols > LARGE_DATA_SET_NUM_COLS) && (nRows > LARGE_DATA_SET_NUM_ROWS) ) // for a large matrix data do a matrix reduction before put to initilization
		{
			int 	nRowShrFactor = LARGE_DATA_SET_ROW_SHRINK_FACTOR; // hard code the shrinkd factor or reduced factor for the matrix, with a ratio proper for a standard image
			int 	nColShrFactor = LARGE_DATA_SET_COL_SHRINK_FACTOR;
			int 	nRowsShr = (int)(nRows * 1.0 / nRowShrFactor) + 1; // !Note: hard code 1 is used to make sure the buffer size for reduced matrix is always larger than that calculated in (and return from) ocmath_reduce_2D_data
			int 	nColsShr = (int)(nCols * 1.0 / nColShrFactor) + 1; // Avoid run time error due to buffer size unmatch
			int		nReducedSize = nRowsShr * nColsShr;
			vector 	vecXReduced(nReducedSize), vecYReduced(nReducedSize), vecZReduced(nReducedSize);
			
			int 	nRet = ocmath_reduce_2D_data(nRows*nCols, vX, vY, vZ, nRows, nCols, vecXReduced, vecYReduced, vecZReduced, nColShrFactor, nRowShrFactor);
			if ( nRet > 0 )			
			{		
				vecXReduced.SetSize(nRet);
				vecYReduced.SetSize(nRet);
				vecZReduced.SetSize(nRet);
				
				vX = vecXReduced;
				vY = vecYReduced;
				vZ = vecZReduced;
			}
		}
	}
};

///////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////// NLFitType ////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////


#endif		/// __NLF_TYPES__

///------ End REFACTOR_NLFITSESSION

