/*------------------------------------------------------------------------------*
 * File Name: BatchProcess.cpp													*
 * Creation: Feb 28, 2011														*
 * Purpose: Analysis Batch Processing											*
 * Copyright (c) OriginLab Corp. 2011 2012										*
 * All Rights Reserved															*
 * 																				*
 * Modification Log:															*
 *	Folger 09/14/2011 ORG-3787-S1 BATCH_PROCESS_INTERMEDIATE_BOOK_NAME_LT_ACCESS*
 *	Folger 09/15/2011 ORG-3805-P1 SPECIFY_ROW_FAILED_TO_WORK_IN_BATCH_PROCESS	*
 *	Folger 10/18/2011 ORG-4110-P1 BATCH_PROCESS_FAILED_FOR_EXCEL_FILES			*
 *	Folger 10/20/2011 ORG-3805-P2 BATCH_PROCESS_WITH_XYRANGE_FAILED_TO_GET_CORRECT_UPPER_BOUND
 *	Folger 11/03/2011 ORG-2560-P1 USE_LONG_NAME_DATASET_IDENTIFIER_SHOULD_GET_SHORT_NAME_IF_LONG_NAME_EMPTY
 *	Folger 12/05/2011 ORG-4516-P1 ORIGIN_CRASH_WHEN_CANCEL_IMPORT_FILTER_DLG	*
 *	Folger 12/09/2011 ORG-4566-P1 BATCH_PROCESS_FAILED_WHEN_COPY_COL_SOURCE_SAME_AS_DEST
 *	Folger 06/08/2012 ORG-5929-P1 BATCH_PROCESS_FAILED_TO_HANDLE_INTERMEDIATE_BOOKS_IN_LOOP_SCRIPT_IF_CHOOSE_DELETE_TEMP_BOOK
 *	Folger 07/04/2012 ORG-6071-S1 BATCH_PROCESS_SUPPORT_COPY_YERR				*
 *	Folger 07/09/2012 ORG-6159-P1 BATCH_PROCESS_FAILED_TO_RUN_WKS_SCRIPT_AFTER_IMPORT
 *	Folger 07/31/2012 ORG-6398-S1 BATCH_PROCESSING_INDEX_LT_SUPPORT				*
 *	Folger 08/20/2012 ORG-6071-P1 BATCH_PROCESS_FAILED_IF_IMPORT_INTO_ACTIVE_LAYER_WITH_TWO_MORE_COLUMNS_NO_ERROR
 *------------------------------------------------------------------------------*/

#include <origin.h>
#include <Array.h>
#include "BatchProcess.h"
#include "fu_utils.h"
#include "dragNdrop.h"
#include "WksColLabels.h"
#include "theme_utils.h"

#define		LABEL_TYPE_INIT_VALUE		-1000

BatchProcessingBase::BatchProcessingBase()
{
	m_ppb = NULL;
	m_nLabelType = LABEL_TYPE_INIT_VALUE;
	///------ Folger 07/31/2012 ORG-6398-S1 BATCH_PROCESSING_INDEX_LT_SUPPORT
	m_pIndexTempChange = NULL;
	///------ End BATCH_PROCESSING_INDEX_LT_SUPPORT
}
BatchProcessingBase::~BatchProcessingBase()
{
	NICE_SAFE_REMOVAL(m_ppb);
	///------ Folger 07/31/2012 ORG-6398-S1 BATCH_PROCESSING_INDEX_LT_SUPPORT
	NICE_SAFE_REMOVAL(m_pIndexTempChange);
	///------ End BATCH_PROCESSING_INDEX_LT_SUPPORT
}
	
/// virtual
int		BatchProcessingBase::Execute()
{
	O_A_FAIL;
	return -1;
}
	
void	BatchProcessingBase::BeforeExecute()
{
	m_bUseActivePage = NULL == lpcszAnalysisTemplate;
	if ( m_bUseActivePage )
	{
		m_wksPage = Project.Pages();
		m_wksImport = m_wksPage.Layers(lpcszFillSheet);
	}

	m_bNeedAppend = NULL != lpcszAppend;
	if ( m_bNeedAppend )
	{
	}
	m_pbHide = IsWindowVisible(GetWindow()) ? false : true;
	m_ppb = new progressBox("", 0, m_pbHide);
	///------ Folger 07/31/2012 ORG-6398-S1 BATCH_PROCESSING_INDEX_LT_SUPPORT
	m_pIndexTempChange = new LTVarTempChange(ProcessingIndex(), 0);
	///------ End BATCH_PROCESSING_INDEX_LT_SUPPORT
}
	
void	BatchProcessingBase::AfterExecute()
{
	for ( int ii=0; ii<m_vsBooks.GetSize(); ++ii )
	{
		Page	pg(m_vsBooks[ii]);
		if ( pg )
			pg.SetShow();
	}
	if ( m_bNeedAppend )
	{
		if ( is_same_folder(wksSummary.GetPage().GetFolder(), Project.ActiveFolder()) )
			wksSummary.GetPage().SetShow();
	}
	if ( NULL != lpcszScriptAtTheEnd && '\0' != *lpcszScriptAtTheEnd )
		LT_execute(lpcszScriptAtTheEnd);
	
	ClearLabtalkStrVariable();
	///------ Folger 07/31/2012 ORG-6398-S1 BATCH_PROCESSING_INDEX_LT_SUPPORT
	NICE_SAFE_REMOVAL(m_pIndexTempChange);
	///------ End BATCH_PROCESSING_INDEX_LT_SUPPORT
}

/// virtual	
BOOL	BatchProcessingBase::PrepareProcessing(int ii, int nSize, LPCSTR lpcszTip)
{
	if ( !m_pbHide )
		m_ppb->CheckReopen();
	m_ppb->SetText(_L("Analysis Processing") + STR_THREE_DOTS, PBOXT_TITLE);
	m_ppb->SetRange(0, nSize);
	if ( m_ppb->Set(ii) )
		m_ppb->SetText(lpcszTip, PBOXT_MIDCENTER);
	else
		return FALSE;

	if ( !m_bUseActivePage )
	{
		m_wksPage.Create(lpcszAnalysisTemplate, CREATE_HIDDEN | CREATE_KEEP_LAYER_NAMES);
		if ( m_wksPage )
			m_wksImport = m_wksPage.Layers(lpcszFillSheet);
		///------ Folger 09/14/2011 ORG-3787-S1 BATCH_PROCESS_INTERMEDIATE_BOOK_NAME_LT_ACCESS
		LT_set_str(ProcessingLTBookName(), m_wksPage.GetName());
		///------ End BATCH_PROCESS_INTERMEDIATE_BOOK_NAME_LT_ACCESS
	}

	if ( !m_wksImport )
		return -1;
	
	m_nActiveSheet = m_wksPage.Layers().GetIndex();
	page_set_active_layer(m_wksPage, m_wksImport.GetIndex());
	return TRUE;
}
	
int		BatchProcessingBase::AppendSummary(int ii)
{
	ExecuteAllDirtyOperations();
	
	int			nLabelType = GetLabelType();
	if ( m_bNeedAppend )
	{
		DWORD	dwCntrl = AppendCtrl();						
		int		nColOffset = 0;
		if ( bAppendByRows )
		{
			dwCntrl |= CPYT_APPEND_BY_ROW;
			if ( RCLT_INVALID != nLabelType )
				nColOffset = 1;
		}

		Worksheet		wksSrc = m_wksPage.Layers(lpcszAppend);
		if ( !wksSrc )
			return -1;
		
		int		nSummaryR1 = -1, nSummaryR2 = -1;
		if ( 0 == ii )
		{
			wksSummary.GetBounds(nSummaryR1, 0, nSummaryR2, -1);
			if ( bClearSummarySheetBeforeAppend )
			{
				if ( bAppendByRows )
				{
					wksSummary.ClearData(CWOD_SHRINK_COLS, 0, -1, nRowStart);
					if ( 0 == wksSummary.GetNumCols() )
						wksSummary.SetSize(-1, wksSrc.GetNumCols());
					wksSummary.SetBounds(0, 0, nRowStart - 1, -1);
					nSummaryR2 = nRowStart - 1;
				}
				else
				{
					wksSummary.SetSize(-1, 0);
				}
			}
			if ( bAppendLabelRows )
			{
				append_label_rows(wksSummary, wksSrc);
				if ( nColOffset > 0 )
				{
					wksSummary.SetCell(nSummaryR2 + 1, 0, BATCH_STR_LABEL_DATASET_NAME);
				}
			}
			else
			{
				if ( nColOffset > 0 )
				{
					Column	col1(wksSummary, 0);
					if ( col1 )
						col1.SetLongName(BATCH_STR_LABEL_DATASET_NAME);
				}
			}
		}

		wksSrc.CopyTo(wksSummary, 0, -1, 0, -1, -1 - nColOffset, -1, dwCntrl);
		
		if ( 0 != nColOffset )
		{
			int			nR1, nR2;
			wksSrc.GetBounds(nR1, 0, nR2, -1);
			int			nSize = nR2 - nR1 + 1;
			batch_check_append_col_results(wksSummary, 0, nSize, GetDataIdentifier(ii), nSummaryR2 + 1);
		}
	}
	///------ Folger 09/14/2011 ORG-3787-S1 BATCH_PROCESS_INTERMEDIATE_BOOK_NAME_LT_ACCESS
	//if ( NULL != lpcszScriptAfterEachProcess && '\0' != *lpcszScriptAfterEachProcess )
		//LT_execute(lpcszScriptAfterEachProcess);
	///------ End BATCH_PROCESS_INTERMEDIATE_BOOK_NAME_LT_ACCESS
	///------ Folger 06/08/2012 ORG-5929-P1 BATCH_PROCESS_FAILED_TO_HANDLE_INTERMEDIATE_BOOKS_IN_LOOP_SCRIPT_IF_CHOOSE_DELETE_TEMP_BOOK
	/// No idea why the fix of ORG-3787-S1 will move running script after temp book deletion, it makes more sense to do before deletion
	if ( NULL != lpcszScriptAfterEachProcess && '\0' != *lpcszScriptAfterEachProcess )
		LT_execute(lpcszScriptAfterEachProcess);
	///------ End BATCH_PROCESS_FAILED_TO_HANDLE_INTERMEDIATE_BOOKS_IN_LOOP_SCRIPT_IF_CHOOSE_DELETE_TEMP_BOOK
	
	if ( !m_bUseActivePage )
	{
		if ( m_bNeedAppend && bDeleteTempateWindowsAfterAppend )
			m_wksPage.Destroy();
		else
		{
			page_set_active_layer(m_wksPage, m_nActiveSheet);
			m_vsBooks.Add(m_wksPage.GetName());
		}
	}
	else
	{
		page_set_active_layer(m_wksPage, m_nActiveSheet);
	}

	///------ Folger 06/08/2012 ORG-5929-P1 BATCH_PROCESS_FAILED_TO_HANDLE_INTERMEDIATE_BOOKS_IN_LOOP_SCRIPT_IF_CHOOSE_DELETE_TEMP_BOOK
	/////------ Folger 09/14/2011 ORG-3787-S1 BATCH_PROCESS_INTERMEDIATE_BOOK_NAME_LT_ACCESS
	//if ( NULL != lpcszScriptAfterEachProcess && '\0' != *lpcszScriptAfterEachProcess )
	//	LT_execute(lpcszScriptAfterEachProcess);
	/////------ End BATCH_PROCESS_INTERMEDIATE_BOOK_NAME_LT_ACCESS
	///------ End BATCH_PROCESS_FAILED_TO_HANDLE_INTERMEDIATE_BOOKS_IN_LOOP_SCRIPT_IF_CHOOSE_DELETE_TEMP_BOOK
	
	return 0;
}

int		BatchProcessingBase::GetLabelType()
{
	if ( LABEL_TYPE_INIT_VALUE == m_nLabelType )
	{
		if ( batch_is_none_identifier(lpcszLabel) )
			m_nLabelType = RCLT_INVALID;
		else
			MakeLabelType();
	}
	
	return m_nLabelType;
}

/// virtual
void	BatchProcessingBase::MakeLabelType()
{
	O_A_FAIL;
}

/// virtual
string	BatchProcessingBase::GetDataIdentifier(int ii)
{
	O_A_FAIL;
	return "";
}

void	BatchProcessingBase::ExecuteAllDirtyOperations()
{
	LT_execute("run -p au;");
}

void	BatchProcessingBase::ClearLabtalkStrVariable()
{
	LT_execute("fname$=\"\"");
}

DWORD	BatchProcessingBase::AppendCtrl()
{
	return CPYT_EXTERN_UPDATE_ORIGIN | (bAppendLabelRows ? 0 : (wksSummary.GetPage().GetType() == EXIST_EXTERN_WKS ?  0 : CPYT_COPY_COLUMN_LABELS))
		| CPYT_COPY_COLUMN_FORMAT
		| CPYT_COPY_COLUMN_DESIGNATIONS
		| CPYT_SKIP_EMPTY
	;
}

//////////////////////////	BatchProcessingOnFiles	//////////////////////////

BatchProcessingOnFiles::BatchProcessingOnFiles()
{
	m_arrSrcCols.SetAsOwner(TRUE);
}

BatchProcessingOnFiles::~BatchProcessingOnFiles()
{
	m_arrSrcCols.SetSize(0);
}

/// virtual
int		BatchProcessingOnFiles::Execute()
{
	if ( NULL == lpcszImportXF && NULL == lpcszFilter && NULL == lpcszScript || NULL == psaFiles || 0 == psaFiles->GetSize() )
	{
		O_A_FAIL;
		return -1;
	}
			
	Tree	trFilter;
	PrepareThemeAndFilter(trFilter);
	
	OC_POST_MESSAGE_BREAKER(PMBC_RUN_SCRIPT_AFTER_IMPORT | PMBC_SET_PAGE_CHANGED);
	
	BeforeExecute();
	///------ Folger 10/18/2011 ORG-4110-P1 BATCH_PROCESS_FAILED_FOR_EXCEL_FILES
	set_no_check_is_imported(true);
	int		nReturn = 0;
	///------ End BATCH_PROCESS_FAILED_FOR_EXCEL_FILES
	
	StringArray&	saFiles = *psaFiles;
	for ( int ii=0; ii<saFiles.GetSize(); ++ii )
	{
		string	strTips;
		strTips.Format("%d/%d %s", ii + 1, saFiles.GetSize(), saFiles[ii]);
		if ( !PrepareProcessing(ii, saFiles.GetSize(), strTips) )
			goto exit;

		///------ Folger 07/31/2012 ORG-6398-S1 BATCH_PROCESSING_INDEX_LT_SUPPORT
		SetProcessingIndex(ii + 1);
		///------ End BATCH_PROCESSING_INDEX_LT_SUPPORT
		
		string		str;
		str.Format("fname$=%s", saFiles[ii]);
		LT_execute(str);
		
		if ( NULL != lpcszImportXF )
		{
			string		strImportXF = lpcszImportXF;
			if ( '\0' != *lpcszTheme )
				strImportXF.Format("%s -t %s", lpcszImportXF, lpcszTheme);
				
			LT_execute(strImportXF);
		}
		else if ( NULL != lpcszFilter )
		{
			vector<string>		vsFiles;
			vsFiles.Add(saFiles[ii]);
			DWORD				dwCntrl = IMPORT_FILES_IGNORE_DRAG_AND_DROP_SUPPORT
										| IMPORT_FILES_SEL_FILTER_DLG_HIDE_IMP_WIZ_BTN
										| IMPORT_FILES_SEL_FILTER_DLG_NO_APPLY_TO_ALL
										| (bUseLastImportSettings ? IMPORT_FILES_CHECK_USE_LAST_IMPORT_SETTINGS	: 0)
										| IMPORT_FILES_FORCE_ASCII_FILTER_IF_NONE
										| (0 == ii ? 0 : IMPORT_FILES_NO_FILTER_APPLICABLE_CHECKING)
										;
			
			/// since wksPage is hidden when import, then it's impossible to put something into labels, needs to create Grid temporary
			///------ Folger 12/05/2011 ORG-4516-P1 ORIGIN_CRASH_WHEN_CANCEL_IMPORT_FILTER_DLG
			{
			/// This code is buggy since m_wksPage wil be destroyed if failed to import, m_wksImport is on m_wksPage so will be destroyed too.
			/// Since grid is attached to m_wksImport, so in destructor of Grid, m_wksImport is junk. Then Origin crash.
			/// The strange thing is that this code is not changed for a long time, it just crash in 86 but not 851, so maybe new compiler issue.
			///------ End ORIGIN_CRASH_WHEN_CANCEL_IMPORT_FILTER_DLG
				Grid	grid;
				grid.Attach(m_wksImport);
				import_files_ex(vsFiles, m_wksPage.GetName(), m_wksImport.GetIndex(), trFilter, NULL, dwCntrl, trFilter.IsEmpty() ? &trFilter : NULL);
			}		///------ Folger 12/05/2011 ORG-4516-P1 ORIGIN_CRASH_WHEN_CANCEL_IMPORT_FILTER_DLG
			if ( 0 == ii && trFilter.IsEmpty() )
			{
				if ( !m_bUseActivePage )
					m_wksPage.Destroy();

				nReturn = -1;
				goto exit;
			}
			
			///------ Folger 07/09/2012 ORG-6159-P1 BATCH_PROCESS_FAILED_TO_RUN_WKS_SCRIPT_AFTER_IMPORT
			CheckRunScriptAfterImport();
			///------ End BATCH_PROCESS_FAILED_TO_RUN_WKS_SCRIPT_AFTER_IMPORT
		}
		else if ( NULL != lpcszScript )
		{
			LT_execute(lpcszScript);
		}
		
		AppendSummary(ii);
	}
	
	AfterExecute();
	///------ Folger 10/18/2011 ORG-4110-P1 BATCH_PROCESS_FAILED_FOR_EXCEL_FILES
	//return 0;
exit:
	set_no_check_is_imported(false);
	return nReturn;
	///------ End BATCH_PROCESS_FAILED_FOR_EXCEL_FILES
}

/// virtual
BOOL	BatchProcessingOnFiles::PrepareProcessing(int ii, int nSize, LPCSTR lpcszTip)
{
	BOOL	bReturn = BatchProcessingBase::PrepareProcessing(ii, nSize, lpcszTip);
	if ( !bReturn )
		return FALSE;

	if ( m_wksImport )
	{
		string			str(lpcszCol);
		vector<string>	vsCols;
		str.GetTokens(vsCols);

		m_arrSrcCols.SetSize(0);
		for ( int ii=0; ii<vsCols.GetSize(); ++ii )
		{
			Column*		pCol = new Column;
			*pCol = m_wksImport.FindCol(vsCols[ii]);
			m_arrSrcCols.Add(*pCol);
		}
	}
	return TRUE;
}

/// virtual
void	BatchProcessingOnFiles::MakeLabelType()
{
	m_nLabelType = batch_get_label_type_by_name(m_wksImport, lpcszLabel);
}

/// virtual
string	BatchProcessingOnFiles::GetDataIdentifier(int ii)
{
	int		nLabelType = GetLabelType();
	if ( BATCH_RCLT_FILENAME == nLabelType )
	{
		ASSERT(psaFiles && ii < psaFiles->GetSize());
		vector<string>&		vs = *psaFiles;
		return GetFileName(vs[ii]);
	}

	vector<string>	vs;
	for ( int jj=0; jj<m_arrSrcCols.GetSize(); ++jj )
	{
		Column&		col = m_arrSrcCols.GetAt(jj);
		if ( col )
		{
			string	str;
			col.GetExtendedLabel(str, nLabelType);
			///------ Folger 11/03/2011 ORG-2560-P1 USE_LONG_NAME_DATASET_IDENTIFIER_SHOULD_GET_SHORT_NAME_IF_LONG_NAME_EMPTY
			if ( str.IsEmpty() && RCLT_LONG_NAME == nLabelType )
			{
				col.GetName(str);
			}
			///------ End USE_LONG_NAME_DATASET_IDENTIFIER_SHOULD_GET_SHORT_NAME_IF_LONG_NAME_EMPTY
			vs.Add(str);
		}
	}

	return str_combine(vs, ", ");
}

void	BatchProcessingOnFiles::PrepareThemeAndFilter(TreeNode& trFilter)
{
	StringArray&	saFiles = *psaFiles;
	if ( NULL != lpcszImportXF )
	{
		if ( PDS_AUTO == okutil_cvt_str_to_predefined_type(lpcszTheme) )
		{
			lpcszTheme = init_xf_theme_list(NULL, lpcszImportXF, lpcszFillSheet);
		}
	}
	else if ( NULL != lpcszFilter )
	{
		if ( PDS_AUTO == okutil_cvt_str_to_predefined_type(lpcszFilter) )
		{
			init_import_wizard_filter_list(NULL, saFiles[0]);
			lpcszFilter = _get_default_filter();
		}
		
		string	strFilterFile = _find_filter_file_by_display_name(lpcszFilter);
		if ( 0 == strFilterFile.Compare(SZ_IW_PAGE_FILTER) )
		{
			trFilter = get_template_sheets_info().trFilter;
		}
		else if ( strFilterFile.IsFile() )
		{
			fuLoad(trFilter, strFilterFile);
		}
	}
}

///------ Folger 07/09/2012 ORG-6159-P1 BATCH_PROCESS_FAILED_TO_RUN_WKS_SCRIPT_AFTER_IMPORT
void BatchProcessingOnFiles::CheckRunScriptAfterImport()
{
	string strScript;
	DWORD dwExecCntrl = 0;
	string strDummy;
	if ( m_wksImport.GetScript(strScript, dwExecCntrl, strDummy) && !strScript.IsEmpty() && (dwExecCntrl & WKSC_CNTL_IMPORT) )
		m_wksImport.LT_execute(strScript);
}
///------ End BATCH_PROCESS_FAILED_TO_RUN_WKS_SCRIPT_AFTER_IMPORT




//////////////////////////	BatchProcessingOnXYRange	//////////////////////////
enum
{
	TARGETCLOUMN_INVALID			 = 0,
	TARGETCLOUMN_XY,
	TARGETCLOUMN_Y,
	///------ Folger 07/04/2012 ORG-6071-S1 BATCH_PROCESS_SUPPORT_COPY_YERR
	TARGETCLOUMN_TYPE_MASK = 0xF,

	TARGETCLOUMN_HAS_ERROR = 0x00010000,
	///------ End BATCH_PROCESS_SUPPORT_COPY_YERR
};

///------ Folger 07/04/2012 ORG-6071-S1 BATCH_PROCESS_SUPPORT_COPY_YERR
#define TargetColumnType(dw) (dw & TARGETCLOUMN_TYPE_MASK)
///------ End BATCH_PROCESS_SUPPORT_COPY_YERR

///------ Folger 12/09/2011 ORG-4566-P1 BATCH_PROCESS_FAILED_WHEN_COPY_COL_SOURCE_SAME_AS_DEST
BatchProcessingOnXYRange::~BatchProcessingOnXYRange()
{
	NICE_SAFE_REMOVAL(m_pVX);
	NICE_SAFE_REMOVAL(m_pVY);
	///------ Folger 07/04/2012 ORG-6071-S1 BATCH_PROCESS_SUPPORT_COPY_YERR
	NICE_SAFE_REMOVAL(m_pVYErr);
	///------ End BATCH_PROCESS_SUPPORT_COPY_YERR
}
///------ End BATCH_PROCESS_FAILED_WHEN_COPY_COL_SOURCE_SAME_AS_DEST

/// virtual
int		BatchProcessingOnXYRange::Execute()
{
	BeforeExecute();
	
	int		nNumData = iy.GetNumData(DataRules());
	for ( int ii=0; ii<nNumData; ++ii )
	{
		XYRange	xySub = GetXYRange(ii);
		
		string	strTips;
		string	strRange;
		xySub.GetRangeString(strRange);
		strTips.Format("%d/%d %s", ii + 1, nNumData, strRange);

		///------ Folger 07/31/2012 ORG-6398-S1 BATCH_PROCESSING_INDEX_LT_SUPPORT
		SetProcessingIndex(ii + 1);
		///------ End BATCH_PROCESSING_INDEX_LT_SUPPORT
		
		if ( !PrepareProcessing(ii, nNumData, strTips) )
			return 0;
		
		Column		colDesX(m_wksImport, 0);
		Column		colDesY(m_wksImport, 1);
		Column		colDesYErr(m_wksImport, 2); ///------ Folger 07/04/2012 ORG-6071-S1 BATCH_PROCESS_SUPPORT_COPY_YERR
		int			nTargetType = TARGETCLOUMN_INVALID;
		///------ Folger 07/04/2012 ORG-6071-S1 BATCH_PROCESS_SUPPORT_COPY_YERR
//if ( (nTargetType = CheckTargetColumns(colDesX, colDesY)) == TARGETCLOUMN_INVALID )
		if ( (nTargetType = CheckTargetColumns(colDesX, colDesY, colDesYErr)) == TARGETCLOUMN_INVALID )
///------ End BATCH_PROCESS_SUPPORT_COPY_YERR
			return -1;
		
		Column		colSrcX, colSrcY;
		xySub.GetXColumn(colSrcX);
		///------ Folger 09/15/2011 ORG-3805-P1 SPECIFY_ROW_FAILED_TO_WORK_IN_BATCH_PROCESS
		//xySub.GetYColumn(colSrcY);
		int			nR1, nR2;
		xySub.GetYColumn(colSrcY, 0, &nR1, &nR2);
		///------ End SPECIFY_ROW_FAILED_TO_WORK_IN_BATCH_PROCESS
		///------ Folger 10/20/2011 ORG-3805-P2 BATCH_PROCESS_WITH_XYRANGE_FAILED_TO_GET_CORRECT_UPPER_BOUND
		if ( nR2 >= 0 )
			++nR2;
		///------ End BATCH_PROCESS_WITH_XYRANGE_FAILED_TO_GET_CORRECT_UPPER_BOUND
		if ( TARGETCLOUMN_Y == TargetColumnType(nTargetType) && colSrcX )
			continue;

		///------ Folger 07/04/2012 ORG-6071-S1 BATCH_PROCESS_SUPPORT_COPY_YERR
		Column colSrcYErr;
		xySub.GetErrColumn(colSrcYErr);
		///------ End BATCH_PROCESS_SUPPORT_COPY_YERR
		
		Worksheet	wks;
		colSrcY.GetParent(wks);
		
		m_wksImport.SetName(wks.GetName());
		m_wksImport.GetPage().SetLongName(wks.GetPage().GetLongName());		
		///------ Folger 09/15/2011 ORG-3805-P1 SPECIFY_ROW_FAILED_TO_WORK_IN_BATCH_PROCESS
		//CopyXYColumns(colDesX, colDesY, colSrcX, colSrcY, nTargetType);
		CopyXYColumns(colDesX, colDesY, colSrcX, colSrcY, nTargetType, nR1, nR2
		///------ Folger 07/04/2012 ORG-6071-S1 BATCH_PROCESS_SUPPORT_COPY_YERR
		, colDesYErr
		, colSrcYErr
		///------ End BATCH_PROCESS_SUPPORT_COPY_YERR
		);
		///------ End SPECIFY_ROW_FAILED_TO_WORK_IN_BATCH_PROCESS
		
		AppendSummary(ii);
		
		///------ Folger 12/09/2011 ORG-4566-P1 BATCH_PROCESS_FAILED_WHEN_COPY_COL_SOURCE_SAME_AS_DEST
		if ( ii != nNumData - 1 )
			CheckRestoreXYData(colDesX, colDesY);
		///------ End BATCH_PROCESS_FAILED_WHEN_COPY_COL_SOURCE_SAME_AS_DEST
	}
	
	AfterExecute();		
	return 0;
}
	
XYRange	BatchProcessingOnXYRange::GetXYRange(int ii)
{
	XYRange	xySub;
	iy.GetSubRange(xySub, DataRules(), ii);
	return xySub;
}

/// virtual
void	BatchProcessingOnXYRange::MakeLabelType()
{
	XYRange	drSub = GetXYRange(0);
	if ( !drSub )
		return;
	
	Worksheet	wks;
	int			c1, c2;
	drSub.GetRange(wks, c1, c2);
	if ( wks )
		m_nLabelType = batch_get_label_type_by_name(wks, lpcszLabel);
}

/// virtual
string	BatchProcessingOnXYRange::GetDataIdentifier(int ii)
{
	return batch_get_data_identifier_string(GetXYRange(ii), GetLabelType());
}

DWORD	BatchProcessingOnXYRange::DataRules()
{
	return DRR_NO_FACTORS | DRR_NO_WEIGHTS | DRR_GET_DEPENDENT;
}

///------ Folger 07/04/2012 ORG-6071-S1 BATCH_PROCESS_SUPPORT_COPY_YERR
//BOOL	BatchProcessingOnXYRange::CheckTargetColumns(Column& col1, Column& col2)
BOOL	BatchProcessingOnXYRange::CheckTargetColumns(Column& col1, Column& col2, Column& col3)
///------ End BATCH_PROCESS_SUPPORT_COPY_YERR
{
	if ( !col1 )
		return TARGETCLOUMN_INVALID;
	
	if ( !col2 )
	{
		if ( col1.GetType() != OKDATAOBJ_DESIGNATION_Y )
			return TARGETCLOUMN_INVALID;
		
		col2 = col1;
		return TARGETCLOUMN_Y;
	}

	///------ Folger 07/04/2012 ORG-6071-S1 BATCH_PROCESS_SUPPORT_COPY_YERR
	//return col1.GetType() == OKDATAOBJ_DESIGNATION_X && col2.GetType() == OKDATAOBJ_DESIGNATION_Y ? TARGETCLOUMN_XY : TARGETCLOUMN_INVALID;
	if ( !col3 )
	{
		if ( col2.GetType() != OKDATAOBJ_DESIGNATION_ERROR )
			return col1.GetType() == OKDATAOBJ_DESIGNATION_X && col2.GetType() == OKDATAOBJ_DESIGNATION_Y ? TARGETCLOUMN_XY : TARGETCLOUMN_INVALID;

		if ( col1.GetType() != OKDATAOBJ_DESIGNATION_Y )
			return TARGETCLOUMN_INVALID;

		col3 = col2;
		col2 = col1;
		return TARGETCLOUMN_Y | TARGETCLOUMN_HAS_ERROR;
	}

	///------ Folger 08/20/2012 ORG-6071-P1 BATCH_PROCESS_FAILED_IF_IMPORT_INTO_ACTIVE_LAYER_WITH_TWO_MORE_COLUMNS_NO_ERROR
	//return col1.GetType() == OKDATAOBJ_DESIGNATION_X && col2.GetType() == OKDATAOBJ_DESIGNATION_Y && col3.GetType() == OKDATAOBJ_DESIGNATION_ERROR ? (TARGETCLOUMN_XY | TARGETCLOUMN_HAS_ERROR) : TARGETCLOUMN_INVALID;
	if ( col1.GetType() == OKDATAOBJ_DESIGNATION_X && col2.GetType() == OKDATAOBJ_DESIGNATION_Y )
	{
		return TARGETCLOUMN_XY | (col3.GetType() == OKDATAOBJ_DESIGNATION_ERROR ? TARGETCLOUMN_HAS_ERROR : 0);
	}
	return TARGETCLOUMN_INVALID;
	///------ End BATCH_PROCESS_FAILED_IF_IMPORT_INTO_ACTIVE_LAYER_WITH_TWO_MORE_COLUMNS_NO_ERROR
	///------ End BATCH_PROCESS_SUPPORT_COPY_YERR
}

void	BatchProcessingOnXYRange::CopyXYColumns(Column& colDesX, Column& colDesY, Column& colSrcX, Column& colSrcY, int nTargetType
												///------ Folger 09/15/2011 ORG-3805-P1 SPECIFY_ROW_FAILED_TO_WORK_IN_BATCH_PROCESS
												, int nR1, int nR2
												///------ End SPECIFY_ROW_FAILED_TO_WORK_IN_BATCH_PROCESS
												///------ Folger 07/04/2012 ORG-6071-S1 BATCH_PROCESS_SUPPORT_COPY_YERR
												, Column& colDesYErr
												, Column& colSrcYErr
												///------ End BATCH_PROCESS_SUPPORT_COPY_YERR
												)
{
	///------ Folger 12/09/2011 ORG-4566-P1 BATCH_PROCESS_FAILED_WHEN_COPY_COL_SOURCE_SAME_AS_DEST
	//DWORD		dwOptions = COPY_COL_FORMAT
							//| COPY_COL_LONGNAME
							//| COPY_COL_UNITS
							//| COPY_COL_COMMENTS
							//| COPY_COL_PARAMETERS
							//| COPY_COL_UDL
							/////------ Folger 09/15/2011 ORG-3805-P1 SPECIFY_ROW_FAILED_TO_WORK_IN_BATCH_PROCESS
							//| COPY_COL_DATA
							/////------ End SPECIFY_ROW_FAILED_TO_WORK_IN_BATCH_PROCESS
							//;
	//
	/////------ Folger 09/15/2011 ORG-3805-P1 SPECIFY_ROW_FAILED_TO_WORK_IN_BATCH_PROCESS
	////copy_col(colSrcY, colDesY, dwOptions);
	////vectorbase&		vbDesY = colDesY.GetDataObject();
	////vbDesY = colSrcY.GetDataObject();
	//colDesY.SetNumRows(0);
	//copy_col(colSrcY, colDesY, dwOptions, nR1, nR2);
	/////------ End SPECIFY_ROW_FAILED_TO_WORK_IN_BATCH_PROCESS
	CopyCol(colSrcY, colDesY, nR1, nR2, &m_pVY);
	///------ End BATCH_PROCESS_FAILED_WHEN_COPY_COL_SOURCE_SAME_AS_DEST
	
	double dX0 = NANUM;
	double dXInc = NANUM;
	string strUnits;
	string strLongName;
	
	if ( TARGETCLOUMN_Y == TargetColumnType(nTargetType) )
	{
		if ( colSrcY.IsEvenSampling(&dX0, &dXInc, &strUnits, &strLongName) )
		{
			colDesY.SetEvenSampling(dX0, dXInc, strUnits, strLongName);
		}
	}
	else
	{
		O_A(TARGETCLOUMN_XY == TargetColumnType(nTargetType));
		
		///------ Folger 09/15/2011 ORG-3805-P1 SPECIFY_ROW_FAILED_TO_WORK_IN_BATCH_PROCESS
		//vectorbase&		vbDesX = colDesX.GetDataObject();
		///------ End SPECIFY_ROW_FAILED_TO_WORK_IN_BATCH_PROCESS
		
		if ( colSrcX )
		{
			///------ Folger 09/15/2011 ORG-3805-P1 SPECIFY_ROW_FAILED_TO_WORK_IN_BATCH_PROCESS
			//copy_col(colSrcX, colDesX, dwOptions);
			//vbDesX = colSrcX.GetDataObject();
			///------ Folger 12/09/2011 ORG-4566-P1 BATCH_PROCESS_FAILED_WHEN_COPY_COL_SOURCE_SAME_AS_DEST
			//colDesX.SetNumRows(0);
			//copy_col(colSrcX, colDesX, dwOptions, nR1, nR2);
			CopyCol(colSrcX, colDesX, nR1, nR2, &m_pVX);
			///------ End BATCH_PROCESS_FAILED_WHEN_COPY_COL_SOURCE_SAME_AS_DEST
			///------ End SPECIFY_ROW_FAILED_TO_WORK_IN_BATCH_PROCESS
		}
		else
		{
			vector		vX;
			///------ Folger 09/15/2011 ORG-3805-P1 SPECIFY_ROW_FAILED_TO_WORK_IN_BATCH_PROCESS
			//int			nUpperBound = colSrcY.GetUpperBound();
			///------ Folger 10/20/2011 ORG-3805-P2 BATCH_PROCESS_WITH_XYRANGE_FAILED_TO_GET_CORRECT_UPPER_BOUND
			//int			nUpperBound = nR2 > 0 ? nR2 - nR1 + 1 : colSrcY.GetUpperBound();
			int			nUpperBound = nR2 > 0 ? nR2 - nR1 : colSrcY.GetUpperBound();
			///------ End BATCH_PROCESS_WITH_XYRANGE_FAILED_TO_GET_CORRECT_UPPER_BOUND
			///------ End SPECIFY_ROW_FAILED_TO_WORK_IN_BATCH_PROCESS
			if ( colSrcY.IsEvenSampling(&dX0, &dXInc, &strUnits, &strLongName) )
			{
				if( dXInc == 0.0 )
				{
					vX.SetSize(nUpperBound + 1);
					vX = dX0;
				}
				else
				{
					vX.Data(dX0, dX0 + dXInc * nUpperBound, dXInc);
				}
				colDesX.SetLongName(strLongName);
				colDesX.SetUnits(strUnits);
			}
			else
			{
				vX.Data(1, nUpperBound + 1);
			}
			///------ Folger 09/15/2011 ORG-3805-P1 SPECIFY_ROW_FAILED_TO_WORK_IN_BATCH_PROCESS
			vectorbase&		vbDesX = colDesX.GetDataObject();
			///------ End SPECIFY_ROW_FAILED_TO_WORK_IN_BATCH_PROCESS
			vbDesX = vX;
		}
	}

	///------ Folger 07/04/2012 ORG-6071-S1 BATCH_PROCESS_SUPPORT_COPY_YERR
	if ( colSrcYErr && colDesYErr )
		CopyCol(colSrcYErr, colDesYErr, nR1, nR2, &m_pVYErr);
	///------ End BATCH_PROCESS_SUPPORT_COPY_YERR
}

///------ Folger 12/09/2011 ORG-4566-P1 BATCH_PROCESS_FAILED_WHEN_COPY_COL_SOURCE_SAME_AS_DEST
bool	BatchProcessingOnXYRange::CopyCol(Column& colSrc, Column& colDes, int nR1, int nR2, vectorbase** ppv)
{
	DWORD		dwOptions = COPY_COL_FORMAT
		| COPY_COL_LONGNAME
		| COPY_COL_UNITS
		| COPY_COL_COMMENTS
		| COPY_COL_PARAMETERS
		| COPY_COL_UDL
		| COPY_COL_DATA
		;

	if ( colSrc.GetDatasetName() == colDes.GetDatasetName() )
	{
		if ( nR1 == 0 && nR2 == -1 )
			return true;

		int		nDataType = colSrc.GetInternalData();
		*ppv = create_vector_from_data_type(nDataType);
		if ( NULL == *ppv )
		{
			O_A_FAIL;
			return false;
		}

		vectorbase*		pvSrc = create_vector_from_data_type(nDataType);
		vectorbase&		vvDes = colDes.GetDataObject();
		vectorbase&		vvBackup = **ppv;
		vvBackup = vvDes;		/// backup
		vvBackup.GetSubVector(*pvSrc, nR1, nR2 - 1);

		vvDes = *pvSrc;
		NICE_SAFE_REMOVAL(pvSrc);
		return true;
	}

	colDes.SetNumRows(0);
	return copy_col(colSrc, colDes, dwOptions, nR1, nR2);
}

void	BatchProcessingOnXYRange::CheckRestoreXYData(Column& colDesX, Column& colDesY)
{
	if ( m_pVX )
	{
		RestoreData(colDesX, *m_pVX);
		NICE_SAFE_REMOVAL(m_pVX);
	}
	if ( m_pVY )
	{
		RestoreData(colDesY, *m_pVY);
		NICE_SAFE_REMOVAL(m_pVY);
	}
	///------ Folger 07/04/2012 ORG-6071-S1 BATCH_PROCESS_SUPPORT_COPY_YERR
	if ( m_pVYErr )
	{
		RestoreData(colDesY, *m_pVYErr);
		NICE_SAFE_REMOVAL(m_pVYErr);
	}
	///------ End BATCH_PROCESS_SUPPORT_COPY_YERR
}

void	BatchProcessingOnXYRange::RestoreData(Column& col, vectorbase& vData)
{
	vectorbase&	vv = col.GetDataObject();
	vv = vData;
}
///------ End BATCH_PROCESS_FAILED_WHEN_COPY_COL_SOURCE_SAME_AS_DEST


void	TemplateSheetsInfo::Init(LPCSTR lpcszName, LPCSTR lpcszXFName)
{
	vsSheetNames.RemoveAll();
	vbHasSheetTheme.RemoveAll();
	vbExcluded.RemoveAll();
	vsLabels.RemoveAll();
	vsCols.RemoveAll();
	
	WorksheetPage	pgTempWks;
	bool			bSelfDeleted = true;
	if ( NULL == lpcszName )
	{
		pgTempWks = Project.Pages();
		bSelfDeleted = false;
	}
	else
	{
		string		strName = lpcszName;
		if ( strName.IsFile() )
			pgTempWks.Create(lpcszName, CREATE_KEEP_LAYER_NAMES | CREATE_HIDDEN);
	}
	
	if ( pgTempWks )
	{
		Tree		trTemp;
		vector<int>	vnIndices;
		foreach ( Layer lay in pgTempWks.Layers )
		{
			Worksheet		wks(lay);
			vector<uint>	vn;
			BOOL			bFilterThis = vnIndices.Find(vn, wks.GetIndex()) > 0;
			
			vsSheetNames.Add(wks.GetName());
			vbHasSheetTheme.Add(theme_active_book_sheet_storage(trTemp, lpcszXFName, PDS_SHEET, TRUE, NULL, &lay));
			
			/// vbExcluded
			if ( bFilterThis || UpdateSheetFilterIndices(vnIndices, wks) )
			{
				vbExcluded.Add(true);
			}
			else
			{
				Grid	grid;
				grid.Attach(wks);
				vbExcluded.Add((WP_SHEET_HIERARCHY & wks.GetSystemParam(0)));
			}
			
			vector<string>		vs;
			string				str;
			
			UpdateLabelInfo(wks);			
			UpdateColumnInfo(wks);
		}
		
		trFilter.Reset();
		fuLoadFilterFromPage(trFilter, pgTempWks, FILTER_TYPE_ASCII);
	}
	
	if ( bSelfDeleted )
		pgTempWks.Destroy();
}

BOOL	TemplateSheetsInfo::UpdateSheetFilterIndices(vector<int>& vnIndices, Worksheet& wks)
{
	vector<uint>	vnUIDs;
	wks.FindIncomingOperations(vnUIDs);
	if ( vnUIDs.GetSize() <= 0 )	
		return FALSE;
	
	vector<int>		vnFilterOutputIDs = { IDST_FIT_CURVES
		, IDST_FIT_RESIDUALS_SCATTER
		, IDST_FIT_RESIDUALS_DATA
		, IDST_FIT_RESIDUAL_ANALYSIS
		, IDST_FIT_RESIDUALS_VERSUS_FITS_DATA
		, IDST_FIT_RESIDUALS_VERSUS_ORDER_DATA
		, IDST_FIT_RESIDUALS_LAG_DATA
		, IDST_FIT_PARTIAL_REGRESSION_DATA
		, IDST_FIT_CONFIDENT_PLOT
		, IDST_PFM_BASELINE_DATA_TABLE
	};
	DWORD			dwMask = 0x0000ffff;
	int				ii = 0;
	for ( ii=0; ii<vnFilterOutputIDs.GetSize(); ++ii )
	{
		vnFilterOutputIDs[ii] &= dwMask;
	}
	
	BOOL			bFilterThis = FALSE;
	for ( ii=0; ii<vnUIDs.GetSize(); ++ii )
	{
		OperationBase& op = Project.GetOperationObject(vnUIDs[ii]);
		if ( op )
		{
			int		nOutputCount = op.GetOutputCount();
			for ( int jj=0; jj<nOutputCount; ++jj )
			{
				DataRange		dr;
				int				nNumRanges = 0;
				if ( op.GetOutput(dr, jj) && dr && (nNumRanges = dr.GetNumRanges()) > 0 )
				{
					for ( int kk=0; kk<nNumRanges; ++kk )
					{
						string		strName;
						int			r1, r2, c1, c2;
						Worksheet	wksTmp;
						dr.GetRange(kk, r1, c1, r2, c2, wksTmp, &strName);
						vector<uint>	vn;
						if ( vnFilterOutputIDs.Find(vn, atoi(strName) & dwMask) > 0 )
						{
							int		nIndex = wksTmp.GetIndex();
							vnIndices.Add(nIndex);
							if ( !bFilterThis && wks.GetIndex() == nIndex )
								bFilterThis = TRUE;
						}
					}
				}
			}
		}
	}
	
	return bFilterThis;
}

void	TemplateSheetsInfo::UpdateLabelInfo(Worksheet& wks)
{
	vector<string>		vs;
	string				str;
	
	WksCommonColLabelHelper		helper;
	helper.UpdateCommonColLabelInfo(wks);
	
	helper.GetLabelNames(vs);
	vs.InsertAt(0, wks.GetName());
	str.SetTokens(vs, ITEM_SEPARATOR);
	vsLabels.Add(str);
}

void	TemplateSheetsInfo::UpdateColumnInfo(Worksheet& wks)
{
	vector<string>		vs;
	string				str;
	
	vs.Add(wks.GetName());
	vector<uint>	vnUIDs;
	wks.FindOutgoingOperations(vnUIDs);
	if ( vnUIDs.GetSize() > 0 )
	{
		OperationBase&	op = Project.GetOperationObject(vnUIDs[0]);
		if ( op )
		{
			DataRange	drInput;
			op.GetInput(drInput);
			if ( drInput && drInput.GetNumRanges() > 0 )
			{
				vector<string>		vsCols;
				if ( CheckAddXYRangeInfo(vsCols, drInput) || CheckAddXYZRangeInfo(vsCols, drInput) || CheckAddNormalRangeInfo(vsCols, drInput) )
					;

				string	strTemp;
				strTemp.SetTokens(vsCols, ' ');
				vs.Add(strTemp);
			}
		}
	}

	str.SetTokens(vs, ITEM_SEPARATOR);	
	vsCols.Add(str);
}

BOOL	TemplateSheetsInfo::CheckAddXYRangeInfo(vector<string>& vsCols, DataRange& dr)
{
	XYRange		xy(dr);
	if ( !xy )
		return FALSE;

	int		nNumData = xy.GetNumData(DRR_GET_DEPENDENT | DRR_NO_FACTORS | DRR_NO_WEIGHTS);
	for ( int ii=0; ii<nNumData; ++ii )
	{
		Column	col;
		if ( xy.GetYColumn(col, ii) && col )
		{
			vsCols.Add(col.GetName());
		}
	}
	return TRUE;
}

BOOL	TemplateSheetsInfo::CheckAddXYZRangeInfo(vector<string>& vsCols, DataRange& dr)
{
	XYZRange		xyz(dr);
	if ( !xyz )
		return FALSE;

	int		nNumData = xyz.GetNumData(DRR_GET_Z_DEPENDENT | DRR_NO_FACTORS | DRR_NO_WEIGHTS);
	for ( int ii=0; ii<nNumData; ++ii )
	{
		Column	col;
		if ( xyz.GetZColumn(col, ii) && col )
		{
			vsCols.Add(col.GetName());
		}
	}
	return TRUE;
}

BOOL	TemplateSheetsInfo::CheckAddNormalRangeInfo(vector<string>& vsCols, DataRange& dr)
{
	int		nNumRanges = dr.GetNumRanges();
	for ( int ii=0; ii<nNumRanges; ++ii )
	{
		Worksheet	wksTemp;
		int			c1, c2;
		dr.GetRange(wksTemp, c1, c2, ii);
		for (; c1<=c2; ++c1 )
		{
			Column	col(wksTemp, c1);
			if ( col )
			{
				vsCols.Add(col.GetName());
			}
		}
	}
	return TRUE;
}

TemplateSheetsInfo&	get_template_sheets_info(LPCSTR lpcszName/* = NULL*/, LPCSTR lpcszXFName/* = NULL*/)
{
	static	TemplateSheetsInfo	l_stTemplateSheetsInfo;
	
	if ( NULL != lpcszXFName )		/// means update
	{
		l_stTemplateSheetsInfo.Init(lpcszName, lpcszXFName);
	}
	
	return l_stTemplateSheetsInfo;
}

static	vector<string>		s_vsFilterFiles;
static	vector<string>		s_vsFilterList;

static	string		_find_filter_file_by_display_name(LPCSTR lpcszFilter)
{
	int		nIndex = s_vsFilterList.Find(lpcszFilter);
	if ( nIndex >= 0 )
		return s_vsFilterFiles[nIndex];
	
	return NULL;
}

static	string		_get_default_filter()
{
	int		nSize = s_vsFilterList.GetSize();
	if ( 0 == nSize )
	{
		O_A_FAIL;		/// s_vsFilterList not init yet, should not come here
		return NULL;
	}
	
	return nSize > 1 ? s_vsFilterList[1] : s_vsFilterList[0];
}

static	bool	_has_sheet_theme_in_template(LPCSTR lpcszSheet)
{
	TemplateSheetsInfo&		stTemplateSheetsInfo = get_template_sheets_info();
	int						nIndex = stTemplateSheetsInfo.vsSheetNames.Find(lpcszSheet);
	
	return nIndex >= 0 && stTemplateSheetsInfo.vbHasSheetTheme[nIndex];
}

string	init_xf_theme_list(TreeNode& trGetN, LPCSTR lpcszMethod/* = NULL*/, LPCSTR lpcszFillSheet/* = NULL*/)
{
	vector<string>		vsList;
	vector<string>		vsFileName;
	
	if ( NULL == lpcszMethod )
		lpcszMethod = trGetN.method.strVal;
	if ( NULL == lpcszFillSheet )
		lpcszFillSheet = trGetN.fill.strVal;
	
	okutil_theme_get_class_settings(lpcszMethod, &vsFileName, &vsList, THTYPE_ANALYSIS, true, true, false
	, false
	, _has_sheet_theme_in_template(lpcszFillSheet)
	);
	
	int		nLastUsed = -1;
	if ( (nLastUsed = vsList.Find(STR_LAST_USED)) >= 0 )
		vsList.RemoveAt(nLastUsed);
	string		strCombo;
	strCombo.SetTokens(vsList, ITEM_SEPARATOR);
	check_append_pipe_char(strCombo);
	
	string	strDefault = strCombo.GetToken(0, ITEM_SEPARATOR);
	if ( trGetN )
	{
		trGetN.theme.SetAttribute(STR_COMBO_ATTRIB, strCombo);
		trGetN.theme.strVal = strDefault;
	}
	
	return strDefault;
}

bool	init_import_wizard_filter_list(TreeNode& trGetN, LPCSTR fname/* = NULL*/)
{
	fuGetFilterListForImportWizard(s_vsFilterList, FILTER_TYPE_ASCII, NULL == fname ? trGetN.fname.strVal : fname , get_template_sheets_info().trFilter.FirstNode ? true : false, &s_vsFilterFiles);
	
	string		strCombo;
	strCombo.SetTokens(s_vsFilterList, ITEM_SEPARATOR);
	check_append_pipe_char(strCombo);
	
	if ( trGetN )
	{
		trGetN.filter.SetAttribute(STR_COMBO_ATTRIB, strCombo);
		trGetN.filter.strVal = _get_default_filter();
	}
	
	return true;
}

void	check_append_pipe_char(string& str)
{
	if ( str.IsEmpty() )
		str = STR_EMPTY_COMBO;
	
	if ( str.Find(ITEM_SEPARATOR) < 0 )
		str += STR_ITEM_SEPARATOR;
}