#include "ocean_common.glslh"

in vec3 vertex_position_in;
in vec3 vertex_normal_in;
in vec3 vertex_uv_in;

uniform mat4 mat_view;
uniform mat4 mat_view_proj;
uniform mat4 mat_view_proj_prev;
uniform mat4 mat_world;

out vec3 world_position_out;
flat out vec3 normal_out;
out float foam_amount_out;
out vec3 view_position_out;
out vec4 vertex_position_out;
out vec4 vertex_position_prev_out;

uniform sampler2D texture_wave_height_scale;
uniform float world_to_texture_scale;
uniform vec3 world_offset; // offset for graphics, used in sampling height scale texture

uniform vec3 camera_position;

uniform float wave_timer;
uniform vec3 wind_direction;
uniform float wave_offset;
uniform float wave_magnitude;

void main()
{
	vec3 vertex_position = vertex_position_in.xyz;

	normal_out = vertex_normal_in;
	world_position_out = (mat_world * vec4(vertex_position_in.xyz, 1)).xyz;

	// Scale height according to texture
	// TODO: do we want to scale foam amount as well?
	float height_scale = get_height_scale_from_world(texture_wave_height_scale, world_position_out, world_offset, world_to_texture_scale);
	vertex_position.y *= height_scale;

	// Scale vertex height with distance from camera
	float dist_scale_factor = get_height_scale_from_camera_dist(world_position_out, camera_position);
	vertex_position.y *= dist_scale_factor;

	world_position_out = (mat_world * vec4(vertex_position.xyz, 1)).xyz;

	view_position_out = (mat_view * vec4(world_position_out, 1.0)).xyz;

	foam_amount_out = vertex_uv_in.x;

	if(wave_magnitude > 0.0)
	{
		// Gerstner wave
		vec3 gerstner_normal;
		vec3 gerstner_offset = get_gerstner_wave_offset(world_position_out - world_offset, wind_direction, wave_offset, wave_timer, wave_magnitude, gerstner_normal);
		world_position_out += gerstner_offset;
		foam_amount_out += gerstner_offset.y / 130.0;
	}

	vertex_position_out = mat_view_proj * vec4(world_position_out, 1);
	vertex_position_prev_out = mat_view_proj_prev * vec4(world_position_out, 1);

	gl_Position = vertex_position_out;
}
