#include "lighting_common.glslh"

in vec4 screen_pos_out;
in vec3 vertex_world_position_out;

out vec4 color_out;

uniform sampler2D gnormal;
uniform sampler2D texture_depth;
uniform sampler2D gcolor;
uniform sampler2D water_depth;

uniform mat4 mat_view_proj_inverse;

uniform vec4 light_pos_range;
uniform vec3 light_color;

uniform vec3 camera_position;
uniform float fog_density;

uniform int is_render_halo;

uniform float halo_factor;
uniform int mip_level;

void main()
{
    vec2 tex_coord = (screen_pos_out.xy / screen_pos_out.w) * 0.5 + 0.5;

    vec3 surface_color = vec3(0.0);

    vec3 camera_to_light_volume = vertex_world_position_out - camera_position;
	float eye_to_fragment_dist = -textureLod(texture_depth, tex_coord, mip_level).r;

	vec3 eye_to_fragment = normalize(camera_to_light_volume) * eye_to_fragment_dist;
    vec3 position = eye_to_fragment + camera_position;

	float fog_factor = 0.4 / exp((0.01 + length(light_pos_range.xyz - camera_position)) * fog_density);

    // TODO: This whole block should be moved into a separate shader, to be applied after
    // water rendering. For reference, look at the volumetric spotlight rendering to see
    // how this should be done. Essentially, it involves moving this code into a separate
    // pass that is rendered additively, using the same omni light geometry we use in
    // this pass.
	if(is_render_halo == 1)
	{
		float water_depth = texture(water_depth, tex_coord).r;
		vec3 water_position = world_pos_from_depth(mat_view_proj_inverse, tex_coord, water_depth);

		vec3 eye_to_water = water_position - camera_position;
		float eye_to_water_dist = length(eye_to_water);

		float fog_density_for_light = fog_density * 10.0;
	    vec3 eye_to_fragment_dir = eye_to_fragment / eye_to_fragment_dist;
		vec3 eye_to_light = light_pos_range.xyz - camera_position;
		float light_depth_dist = max(0.0, dot(eye_to_light, eye_to_fragment_dir));
		float light_dist = light_depth_dist;

	    float behind_water_depth_factor = (clamp((light_depth_dist - eye_to_water_dist) / light_pos_range.w, -1.0, 1.0) + 1.0) * 0.5;
	    behind_water_depth_factor = 1.0 - behind_water_depth_factor;

		if(eye_to_water_dist < eye_to_fragment_dist)
		{
			eye_to_fragment_dist = eye_to_water_dist;
		}
		if(eye_to_fragment_dist < light_depth_dist)
		{
			light_depth_dist = eye_to_fragment_dist;
		}

	    vec3 light_depth_position = camera_position + light_depth_dist * eye_to_fragment_dir;
	    vec3 light_depth_to_light_center = light_pos_range.xyz - light_depth_position;
	    float light_depth_dist_to_center = 0.01 + length(light_depth_to_light_center);

	    float light_volume_factor = max(0.0, (fog_density_for_light / light_depth_dist_to_center) - (fog_density_for_light / light_pos_range.w));

		surface_color += (light_color * halo_factor) * light_volume_factor * light_volume_factor;
	}

	surface_color *= fog_factor;

	// fix for temporal buffer bug where black spreads across screen
	surface_color = max(surface_color, 0.0);

	if(isnan(surface_color.r) || isinf(surface_color.r)) surface_color.r = 0.0;
	if(isnan(surface_color.g) || isinf(surface_color.g)) surface_color.g = 0.0;
	if(isnan(surface_color.b) || isinf(surface_color.b)) surface_color.b = 0.0;

    color_out = vec4(surface_color * 4, 1.0);
}
