//////////////////////////////////////////////////////////////////////////////
//  Copyright 2014 by Autodesk, Inc.  All rights reserved.
//
// This computer source code and related instructions and comments 
// are the unpublished confidential and proprietary information of 
// Autodesk, Inc. and are protected under applicable copyright and 
// trade secret law.  They may not be disclosed to, copied or used 
// by any third party without the prior written consent of Autodesk, Inc.
//////////////////////////////////////////////////////////////////////////////

/*globals $,jQuery,document, window, navigator*/

/*!
 * ntp.video - Implements video playback
 */
 
 // Namespace
var ntp = ntp || {};

ntp.video = ntp.video || {};

(function () {
    var videoPlayer = null; // The MediaElementPlayer instance.
    var $videoPlayerLightbox = $('#video_player_lightbox');
    
    var currentId = null;
    
    $.extend(ntp.video, {
        
        videoWidth: 800,
        videoHeight: 450,
                
        play: function () {
            if (videoPlayer) {
                videoPlayer.load();
                videoPlayer.play();
            }
        },
        stop: function () {
            if (videoPlayer) {
                videoPlayer.pause();
                
                // Restore the status of video controls including the close button
                // so that the lightbox will be in the same state if it is shown again.
                ntp.video.controls.show();
            }
        },

        /**
         * Stops video player in other new tab page instances. It works in this way:
         * When a new video starts playing, a new ID is generated to represent the
         * current video, and passed to AutoCAD. AutoCAD then invokes ntp.video.stopWithJSON(idJson)
         * in all new tab page instances with the current video ID. If the ID is different
         * from the currentId stored in a new tab page instance, the video in that instance
         * is stopped.
         */
        stopOthers: function () {
            this.currentId = new Date().getTime().toString();
            ntp.acad.stopVideoById(this.currentId);
        },
        
        stopWithJSON: function (idJSON) {
            var json = $.parseJSON(idJSON);
            if (json.id !== this.currentId) {
                this.stop();
            }
        },

        updateViewedState: function () {
            var $videoPlayer = $('#video_player');
            if (ntp.content.learnTab.urlIsHilightedContent($videoPlayer.data('url'))) {
                ntp.acad.updateLearnContentState($videoPlayer.data('url'), $videoPlayer.data('upload_time'));
            }
        },

        showPlayer: function ($item) {
            // Initialize video player if needed
            if (videoPlayer === null) {
                var $video = $('#video_player');        
                if ($video.size() == 0) {
                    // Create video element
                    $video = $('<video>').attr({
                        id: 'video_player',
                        preload: 'none',
                        type: 'video/webm'
                    });
                    $('#video_container').append($video);
                }
                
                $('#video_container').width(this.videoWidth);
            
                $video.attr({
                    src: $item.data('url'),
                    width: this.videoWidth,
                    height: this.videoHeight
                });
            
                // Enable/disable automatic hiding player controls
                $video.on('play', 
                    ntp.utils.makeCallback(this.controls.autohide, 'enable'));
                $video.on('pause error ended',
                    ntp.utils.makeCallback(this.controls.autohide, 'disable'));
                
                // Stop video player in other new tab page instances
                $video.on('play', ntp.utils.makeCallback(this, 'stopOthers'));

                //Once Played mark it as a viewed item
                $video.on('play', ntp.utils.makeCallback(this, 'updateViewedState'));

                // Start playing video to play the video in autoplay mode
                videoPlayer = new MediaElementPlayer('#video_player', {
                    pluginWidth: this.videoWidth,
                    pluginHeight: this.videoHeight,
                    alwaysShowControls: true,
                    hideVolumeOnTouchDevices: false
                });
                
                $videoPlayerLightbox.lightbox('bind', 'lightboxshown', 
                    ntp.utils.makeCallback(this, 'play'));
            
                $videoPlayerLightbox.lightbox('bind', 'lightboxhidden',
                    ntp.utils.makeCallback(this, 'stop'));
            
                // Disable activating video control buttons with tab key
                $('#video_container button').attr('tabindex', '-1');
                // Required for 'click-to-pause' to work inside AutoCAD
                $('.mejs-mediaelement').click(ntp.utils.makeCallback(this, 'stop'));
            } else {
                // Update existing video player
                videoPlayer.pause();
                videoPlayer.setSrc($item.data('url'));
            }
        
            //Add some data to query in updateViewedState callback
            $('#video_player').data({
                url: $item.data('url'),
                upload_time: $item.data('upload_time')
            });

            // Update video player title and prev/next links
            $('#video_container .title').text($item.data('title'));
            var $prevLink = $('#video_container .prev');
            $prevLink.off('click');
            var $prevItem = $item.prev();
            if ($prevItem.size() > 0) {
                $prevLink.children('span').text($prevItem.data('title'));
                $prevLink.show();
                $prevLink.on('click', 
                    ntp.utils.makeCallbackWithArgs(this, 'showPlayer', $prevItem));
            } else {
                $prevLink.hide();
            }
        
            var $nextLink = $('#video_container .next');
            $nextLink.off('click');
            var $nextItem = $item.next();
            if ($nextItem.size() > 0) {
                $nextLink.children('span').text($nextItem.data('title'));
                $nextLink.show();
                $nextLink.on('click',
                    ntp.utils.makeCallbackWithArgs(this, 'showPlayer', $nextItem));
            } else {
                $nextLink.hide();
            }
        
            // Play video
            if ($videoPlayerLightbox.lightbox('isShowing')) {
                videoPlayer.load();
                videoPlayer.play();
            } else {
                $videoPlayerLightbox.lightbox();
            }
        }
    });
    
})();

/**
 * Manipulation of video player controls
 */
ntp.video.controls = (function () {
    var isVisible = true;
    var updateVisibility = function (visible) {
        if (visible !== isVisible) {
            if (visible) {
                $('#video_container .mejs-controls')
                    .add('#video_container header')
                    .add('#lightbox_panel .close')
                    .stop(true, true).css('opacity', 1); // Use opacity to avoid layout issues.
            } else {
                $('#video_container .mejs-controls')
                        .add('#video_container header')
                        .add('#lightbox_panel .close')
                        .stop(true, true).delay(1000).fadeTo(400, 0);
            }
            isVisible = visible;
        }
    };
    
    return {
        show: function () {
            updateVisibility(true);
        },
        hoverArea: function () {
            return $('#video_container')
                .add($('#lightbox_panel .close'));
        },
        hovering: function () {
            var isHovering = false;
            this.hoverArea().each(function () {
                isHovering = $(this).is(':hover');
                return !isHovering; // Break when any element is :hover
            });
            return isHovering;
        },
        refresh: function () {
            updateVisibility(this.hovering());
        }
    };    
})();

/**
 * Automatically hide video player controls when user is not hovering over the player
 * area.
 */
ntp.video.controls.autohide = (function () {
    var isEnabled = false;    
    return {
        enable: function () {
            if (!isEnabled) {
                if ($('#video_container .me-cannotplay').size() > 0) {
                    // Disallow enabling auto hide if video cannot be played
                    return;
                }
                ntp.video.controls.refresh();
                ntp.video.controls.hoverArea()
                    .on('mouseenter mouseleave', 
                        ntp.utils.makeCallback(ntp.video.controls, 'refresh'));
                isEnabled = true;
            }
        },
        disable: function () {
            if (isEnabled) {
                ntp.video.controls.hoverArea()
                    .off('mouseenter mouseleave');            
                ntp.video.controls.show();
                isEnabled = false;
            }
        }
    };
})();

// SIG // Begin signature block
// SIG // MIIZjAYJKoZIhvcNAQcCoIIZfTCCGXkCAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // gSmWRtJg5qTgMwAnPK46dqGLWFLVV5CrZIUhN9/LOByg
// SIG // ggopMIIE2jCCA8KgAwIBAgIQfG5XIF7pXpl44AXkMQgG
// SIG // EDANBgkqhkiG9w0BAQsFADCBhDELMAkGA1UEBhMCVVMx
// SIG // HTAbBgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8w
// SIG // HQYDVQQLExZTeW1hbnRlYyBUcnVzdCBOZXR3b3JrMTUw
// SIG // MwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
// SIG // b2RlIFNpZ25pbmcgQ0EgLSBHMjAeFw0xODA3MDUwMDAw
// SIG // MDBaFw0xOTA4MDUyMzU5NTlaMIGKMQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECAwKQ2FsaWZvcm5pYTETMBEGA1UEBwwK
// SIG // U2FuIFJhZmFlbDEXMBUGA1UECgwOQXV0b2Rlc2ssIElu
// SIG // Yy4xHzAdBgNVBAsMFkRlc2lnbiBTb2x1dGlvbnMgR3Jv
// SIG // dXAxFzAVBgNVBAMMDkF1dG9kZXNrLCBJbmMuMIIBIjAN
// SIG // BgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA7mRyNXG+
// SIG // QgtKVZfb37ZZm2wDgkFnx9kZy5clwGxvrSFmo0RQb5gR
// SIG // 6Ez0lhlQRNsfBVcYZrp0YyNxKg25oCdLDgPnaqilCia0
// SIG // UkRTpMYIUdaD0acP8C7Cb7r4oeKl2c3qVOGpvyCSM7Ep
// SIG // x16v7HtO0xeiOtCGU+cQ2kussGyDxiuvg0nsCMR4zV7b
// SIG // cg0bjKS3q22JzkzNrJpxSJJPnemEif5NQVQExg8Dg8YO
// SIG // UVAEhPc5+TRCtWH0fT+c4asKEUo9Urz5TBRwqqg2h/Ti
// SIG // k2qUPok3YlHrI8JIOeKfItfgU4PYtwr6yt4PPW7Jie1g
// SIG // cYi6JNButwAUIwRfCI1brIyhwQIDAQABo4IBPjCCATow
// SIG // CQYDVR0TBAIwADAOBgNVHQ8BAf8EBAMCB4AwEwYDVR0l
// SIG // BAwwCgYIKwYBBQUHAwMwYQYDVR0gBFowWDBWBgZngQwB
// SIG // BAEwTDAjBggrBgEFBQcCARYXaHR0cHM6Ly9kLnN5bWNi
// SIG // LmNvbS9jcHMwJQYIKwYBBQUHAgIwGQwXaHR0cHM6Ly9k
// SIG // LnN5bWNiLmNvbS9ycGEwHwYDVR0jBBgwFoAU1MAGIknr
// SIG // OUvdk+JcobhHdglyA1gwKwYDVR0fBCQwIjAgoB6gHIYa
// SIG // aHR0cDovL3JiLnN5bWNiLmNvbS9yYi5jcmwwVwYIKwYB
// SIG // BQUHAQEESzBJMB8GCCsGAQUFBzABhhNodHRwOi8vcmIu
// SIG // c3ltY2QuY29tMCYGCCsGAQUFBzAChhpodHRwOi8vcmIu
// SIG // c3ltY2IuY29tL3JiLmNydDANBgkqhkiG9w0BAQsFAAOC
// SIG // AQEArEMBlWJ6bc5AKEqwyO4TWNdnsYX9FreWUauXniGK
// SIG // lAwGhAkH7N4Ma1sE3yKtQi6pE7n+1t0bmlPPLiAWfRgI
// SIG // oDZ/ZwiWkOoP7A+31Q7skKjdUN1M96QjcbNUQjDkFqxy
// SIG // bHUraVr0gPlGUpnMR1npXAFHgXCrw45F1NT+LZNHWt/l
// SIG // AmfQsquTmDNvs7o0f/AhQr45Ylo8/Gsr8neojm2Io8iF
// SIG // pcMHqY7jcNnVL4ethYXVVdY0JSBN+Vg/2G0j/Lq8ic+I
// SIG // f8LiWdBllb3DUrVrgdjmBIHYCxPPJTXgsTGv4hwlRifS
// SIG // QkMw83nBdgWsObnssKnDJEOrhm9Da8FRHL4/fzCCBUcw
// SIG // ggQvoAMCAQICEHwbNTVK59t050FfEWnKa6gwDQYJKoZI
// SIG // hvcNAQELBQAwgb0xCzAJBgNVBAYTAlVTMRcwFQYDVQQK
// SIG // Ew5WZXJpU2lnbiwgSW5jLjEfMB0GA1UECxMWVmVyaVNp
// SIG // Z24gVHJ1c3QgTmV0d29yazE6MDgGA1UECxMxKGMpIDIw
// SIG // MDggVmVyaVNpZ24sIEluYy4gLSBGb3IgYXV0aG9yaXpl
// SIG // ZCB1c2Ugb25seTE4MDYGA1UEAxMvVmVyaVNpZ24gVW5p
// SIG // dmVyc2FsIFJvb3QgQ2VydGlmaWNhdGlvbiBBdXRob3Jp
// SIG // dHkwHhcNMTQwNzIyMDAwMDAwWhcNMjQwNzIxMjM1OTU5
// SIG // WjCBhDELMAkGA1UEBhMCVVMxHTAbBgNVBAoTFFN5bWFu
// SIG // dGVjIENvcnBvcmF0aW9uMR8wHQYDVQQLExZTeW1hbnRl
// SIG // YyBUcnVzdCBOZXR3b3JrMTUwMwYDVQQDEyxTeW1hbnRl
// SIG // YyBDbGFzcyAzIFNIQTI1NiBDb2RlIFNpZ25pbmcgQ0Eg
// SIG // LSBHMjCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoC
// SIG // ggEBANeVQ9Tc32euOftSpLYmMQRw6beOWyq6N2k1lY+7
// SIG // wDDnhthzu9/r0XY/ilaO6y1L8FcYTrGNpTPTC3Uj1Wp5
// SIG // J92j0/cOh2W13q0c8fU1tCJRryKhwV1LkH/AWU6rnXmp
// SIG // AtceSbE7TYf+wnirv+9SrpyvCNk55ZpRPmlfMBBOcWNs
// SIG // WOHwIDMbD3S+W8sS4duMxICUcrv2RZqewSUL+6Mcntim
// SIG // CXBx7MBHTI99w94Zzj7uBHKOF9P/8LIFMhlM07Acn/6l
// SIG // eCBCcEGwJoxvAMg6ABFBekGwp4qRBKCZePR3tPNgKuZs
// SIG // UAS3FGD/DVH0qIuE/iHaXF599Sl5T7BEdG9tcv8CAwEA
// SIG // AaOCAXgwggF0MC4GCCsGAQUFBwEBBCIwIDAeBggrBgEF
// SIG // BQcwAYYSaHR0cDovL3Muc3ltY2QuY29tMBIGA1UdEwEB
// SIG // /wQIMAYBAf8CAQAwZgYDVR0gBF8wXTBbBgtghkgBhvhF
// SIG // AQcXAzBMMCMGCCsGAQUFBwIBFhdodHRwczovL2Quc3lt
// SIG // Y2IuY29tL2NwczAlBggrBgEFBQcCAjAZGhdodHRwczov
// SIG // L2Quc3ltY2IuY29tL3JwYTA2BgNVHR8ELzAtMCugKaAn
// SIG // hiVodHRwOi8vcy5zeW1jYi5jb20vdW5pdmVyc2FsLXJv
// SIG // b3QuY3JsMBMGA1UdJQQMMAoGCCsGAQUFBwMDMA4GA1Ud
// SIG // DwEB/wQEAwIBBjApBgNVHREEIjAgpB4wHDEaMBgGA1UE
// SIG // AxMRU3ltYW50ZWNQS0ktMS03MjQwHQYDVR0OBBYEFNTA
// SIG // BiJJ6zlL3ZPiXKG4R3YJcgNYMB8GA1UdIwQYMBaAFLZ3
// SIG // +mlIR59TEtXC6gcydgfRlwcZMA0GCSqGSIb3DQEBCwUA
// SIG // A4IBAQB/68qn6ot2Qus+jiBUMOO3udz6SD4Wxw9FlRDN
// SIG // J4ajZvMC7XH4qsJVl5Fwg/lSflJpPMnx4JRGgBi7odSk
// SIG // VqbzHQCR1YbzSIfgy8Q0aCBetMv5Be2cr3BTJ7noPn5R
// SIG // oGlxi9xR7YA6JTKfRK9uQyjTIXW7l9iLi4z+qQRGBIX3
// SIG // FZxLEY3ELBf+1W5/muJWkvGWs60t+fTf2omZzrI4RMD3
// SIG // R3vKJbn6Kmgzm1By3qif1M0sCzS9izB4QOCNjicbkG8a
// SIG // vggVgV3rL+JR51EeyXgp5x5lvzjvAUoBCSQOFsQUecFB
// SIG // NzTQPZFSlJ3haO8I8OJpnGdukAsak3HUJgLDwFojMYIO
// SIG // uzCCDrcCAQEwgZkwgYQxCzAJBgNVBAYTAlVTMR0wGwYD
// SIG // VQQKExRTeW1hbnRlYyBDb3Jwb3JhdGlvbjEfMB0GA1UE
// SIG // CxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29yazE1MDMGA1UE
// SIG // AxMsU3ltYW50ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBT
// SIG // aWduaW5nIENBIC0gRzICEHxuVyBe6V6ZeOAF5DEIBhAw
// SIG // DQYJYIZIAWUDBAIBBQCgfDAQBgorBgEEAYI3AgEMMQIw
// SIG // ADAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
// SIG // BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG
// SIG // 9w0BCQQxIgQgvlwByD73IhP6FLMmFUj1pkh9BmG5ci/u
// SIG // zaSLVOWmnaEwDQYJKoZIhvcNAQEBBQAEggEAZdsQi4KJ
// SIG // EDaRVOXx00zF4Iu6397nXT59tw7wnC6z8PgTZMzVg3ZW
// SIG // DQItoUEdMI40dmZj1RAsu/gV6KUZ6cTt/JBLuyC7rrzA
// SIG // Z9FFZWCQbjV+Di34EUUdfZxm21N2AzEVR5cU5xzOo672
// SIG // 7aOSpXmT42SHQTm5gYUBGYjlZ56pqLfh9IysZE12gNc/
// SIG // rky4wd76ZbSnmYzQAP4t5rZbY3WSUcAeLSJ6G15f6lvV
// SIG // 5Mmaefq30RGwljwPNXQaYXkgfNgYxnTSVSNALx6ZxD92
// SIG // pQzR0juDdry5L+zzXwvtDEm5R1AA3xULasnDsax+GwfN
// SIG // 66X63S628IyK4iwcmBZb/Pc2QKGCDHQwggxwBgorBgEE
// SIG // AYI3AwMBMYIMYDCCDFwGCSqGSIb3DQEHAqCCDE0wggxJ
// SIG // AgEDMQ8wDQYJYIZIAWUDBAIBBQAwga8GCyqGSIb3DQEJ
// SIG // EAEEoIGfBIGcMIGZAgEBBgkrBgEEAaAyAgMwMTANBglg
// SIG // hkgBZQMEAgEFAAQgKswyAa0jvTAoYK7+iqK0nlvBLgzf
// SIG // eDuoPsyuQivviQ0CFBadMgksOhpJJuhsGHDIXGfb9482
// SIG // GA8yMDE5MDEzMDA5NDc0N1qgL6QtMCsxKTAnBgNVBAMM
// SIG // IEdsb2JhbFNpZ24gVFNBIGZvciBBZHZhbmNlZCAtIEcy
// SIG // oIII0zCCBLYwggOeoAMCAQICDAynz10HBySsieeaOjAN
// SIG // BgkqhkiG9w0BAQsFADBbMQswCQYDVQQGEwJCRTEZMBcG
// SIG // A1UEChMQR2xvYmFsU2lnbiBudi1zYTExMC8GA1UEAxMo
// SIG // R2xvYmFsU2lnbiBUaW1lc3RhbXBpbmcgQ0EgLSBTSEEy
// SIG // NTYgLSBHMjAeFw0xODAyMTkwMDAwMDBaFw0yOTAzMTgx
// SIG // MDAwMDBaMCsxKTAnBgNVBAMMIEdsb2JhbFNpZ24gVFNB
// SIG // IGZvciBBZHZhbmNlZCAtIEcyMIIBIjANBgkqhkiG9w0B
// SIG // AQEFAAOCAQ8AMIIBCgKCAQEAt8eSiijY85HLUA1d6ylE
// SIG // wYJZpcpkzr2DtVHllMxugJoXbNz/YuY2Q14Bwgxkjcp3
// SIG // nvZdd0W5Xsgk60E3EWY3rMjNdJvMs+9M4n8bznO+yZad
// SIG // NR5891T3bOygEtaumZ1jxHhfWakuIvXDJ9iyW0VjJ5D/
// SIG // jLE9qbTzDsDxIGUGiZ0DHKOwUkGFNOu8y85U9hL+swGJ
// SIG // Sqb6VMUf+9McwOO/l+M/0Bv5QMcb1esmfEMwx3DyeBGq
// SIG // hU6lDg0ASjoRd6PKIwGAnsiTVl1GzIrY0qoKhIWyiVvV
// SIG // Ms5BexhOVMqr0VJeEtMknlbzDI/dvZKOedVBicOjoOWR
// SIG // /MdnnYczznEycwIDAQABo4IBqDCCAaQwDgYDVR0PAQH/
// SIG // BAQDAgeAMEwGA1UdIARFMEMwQQYJKwYBBAGgMgEeMDQw
// SIG // MgYIKwYBBQUHAgEWJmh0dHBzOi8vd3d3Lmdsb2JhbHNp
// SIG // Z24uY29tL3JlcG9zaXRvcnkvMAkGA1UdEwQCMAAwFgYD
// SIG // VR0lAQH/BAwwCgYIKwYBBQUHAwgwRgYDVR0fBD8wPTA7
// SIG // oDmgN4Y1aHR0cDovL2NybC5nbG9iYWxzaWduLmNvbS9n
// SIG // cy9nc3RpbWVzdGFtcGluZ3NoYTJnMi5jcmwwgZgGCCsG
// SIG // AQUFBwEBBIGLMIGIMEgGCCsGAQUFBzAChjxodHRwOi8v
// SIG // c2VjdXJlLmdsb2JhbHNpZ24uY29tL2NhY2VydC9nc3Rp
// SIG // bWVzdGFtcGluZ3NoYTJnMi5jcnQwPAYIKwYBBQUHMAGG
// SIG // MGh0dHA6Ly9vY3NwMi5nbG9iYWxzaWduLmNvbS9nc3Rp
// SIG // bWVzdGFtcGluZ3NoYTJnMjAdBgNVHQ4EFgQULW5u0Y3l
// SIG // A4Du52Ppp9naCzxXIj0wHwYDVR0jBBgwFoAUkiGnSpVd
// SIG // ZLCbtB7mADdH5p1BK0wwDQYJKoZIhvcNAQELBQADggEB
// SIG // AI39HR+PiNNl+LVcRumn07Y/SstDBcIHDcNeA4QjN6h6
// SIG // jJcj/yechH1h61xLVz6b9ETbgFj3tmXejxIKX6eHdLfa
// SIG // FECHUeyfSkdV8BfoGD5xIDHmKCKrTvaunGvC50T7aA0g
// SIG // f1M/lBuhhoJOMhYQWj+JWMHQtNu2kPuNMumhS0ZQN7bD
// SIG // OukpJBKSWk4rubYKejaMl8DviXwz3jHhoZfk50WtYVL9
// SIG // WonTVH6KTVuJew6Wd9HyPDgb2vmlYSBlArzFrdcMYhAo
// SIG // LN2246hhAKwIJqONhdBkyTaQHopSyYeNNaVdSJH8k6al
// SIG // ZfHs+TrcpSyeBG1iM/7e+p/Fzq+V81Go8s4wggQVMIIC
// SIG // /aADAgECAgsEAAAAAAExicZQBDANBgkqhkiG9w0BAQsF
// SIG // ADBMMSAwHgYDVQQLExdHbG9iYWxTaWduIFJvb3QgQ0Eg
// SIG // LSBSMzETMBEGA1UEChMKR2xvYmFsU2lnbjETMBEGA1UE
// SIG // AxMKR2xvYmFsU2lnbjAeFw0xMTA4MDIxMDAwMDBaFw0y
// SIG // OTAzMjkxMDAwMDBaMFsxCzAJBgNVBAYTAkJFMRkwFwYD
// SIG // VQQKExBHbG9iYWxTaWduIG52LXNhMTEwLwYDVQQDEyhH
// SIG // bG9iYWxTaWduIFRpbWVzdGFtcGluZyBDQSAtIFNIQTI1
// SIG // NiAtIEcyMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIB
// SIG // CgKCAQEAqpuOw6sRUSUBtpaU4k/YwQj2RiPZRcWVl1ur
// SIG // Gr/SbFfJMwYfoA/GPH5TSHq/nYeer+7DjEfhQuzj46FK
// SIG // bAwXxKbBuc1b8R5EiY7+C94hWBPuTcjFZwscsrPxNHaR
// SIG // ossHbTfFoEcmAhWkkJGpeZ7X61edK3wi2BTX8QceeCI2
// SIG // a3d5r6/5f45O4bUIMf3q7UtxYowj8QM5j0R5tnYDV56t
// SIG // LwhG3NKMvPSOdM7IaGlRdhGLD10kWxlUPSbMQI2CJxtZ
// SIG // IH1Z9pOAjvgqOP1roEBlH1d2zFuOBE8sqNuEUBNPxtyL
// SIG // ufjdaUyI65x7MCb8eli7WbwUcpKBV7d2ydiACoBuCQID
// SIG // AQABo4HoMIHlMA4GA1UdDwEB/wQEAwIBBjASBgNVHRMB
// SIG // Af8ECDAGAQH/AgEAMB0GA1UdDgQWBBSSIadKlV1ksJu0
// SIG // HuYAN0fmnUErTDBHBgNVHSAEQDA+MDwGBFUdIAAwNDAy
// SIG // BggrBgEFBQcCARYmaHR0cHM6Ly93d3cuZ2xvYmFsc2ln
// SIG // bi5jb20vcmVwb3NpdG9yeS8wNgYDVR0fBC8wLTAroCmg
// SIG // J4YlaHR0cDovL2NybC5nbG9iYWxzaWduLm5ldC9yb290
// SIG // LXIzLmNybDAfBgNVHSMEGDAWgBSP8Et/qC5FJK5NUPpj
// SIG // move4t0bvDANBgkqhkiG9w0BAQsFAAOCAQEABFaCSnzQ
// SIG // zsm/NmbRvjWek2yX6AbOMRhZ+WxBX4AuwEIluBjH/NSx
// SIG // N8RooM8oagN0S2OXhXdhO9cv4/W9M6KSfREfnops7yyw
// SIG // 9GKNNnPRFjbxvF7stICYePzSdnno4SGU4B/EouGqZ9uz
// SIG // nHPlQCLPOc7b5neVp7uyy/YZhp2fyNSYBbJxb051rvE9
// SIG // ZGo7Xk5GpipdCJLxo/MddL9iDSOMXCo4ldLA1c3PiNof
// SIG // KLW6gWlkKrWmotVzr9xG2wSukdduxZi61EfEVnSAR3hY
// SIG // jL7vK/3sbL/RlPe/UOB74JD9IBh4GCJdCC6MHKCX8x2Z
// SIG // faOdkdMGRE4EbnocIOM28LZQuTGCAqgwggKkAgEBMGsw
// SIG // WzELMAkGA1UEBhMCQkUxGTAXBgNVBAoTEEdsb2JhbFNp
// SIG // Z24gbnYtc2ExMTAvBgNVBAMTKEdsb2JhbFNpZ24gVGlt
// SIG // ZXN0YW1waW5nIENBIC0gU0hBMjU2IC0gRzICDAynz10H
// SIG // BySsieeaOjANBglghkgBZQMEAgEFAKCCAQ4wGgYJKoZI
// SIG // hvcNAQkDMQ0GCyqGSIb3DQEJEAEEMBwGCSqGSIb3DQEJ
// SIG // BTEPFw0xOTAxMzAwOTQ3NDdaMC8GCSqGSIb3DQEJBDEi
// SIG // BCDmHv4q/6e+vgJy2UWQbSWClTce2tRvFzSMVO7rr7pW
// SIG // QjCBoAYLKoZIhvcNAQkQAgwxgZAwgY0wgYowgYcEFJsS
// SIG // BXrnKq/21jdytJ9qI28mSc2pMG8wX6RdMFsxCzAJBgNV
// SIG // BAYTAkJFMRkwFwYDVQQKExBHbG9iYWxTaWduIG52LXNh
// SIG // MTEwLwYDVQQDEyhHbG9iYWxTaWduIFRpbWVzdGFtcGlu
// SIG // ZyBDQSAtIFNIQTI1NiAtIEcyAgwMp89dBwckrInnmjow
// SIG // DQYJKoZIhvcNAQEBBQAEggEArUojq1UHlxiin8dR6B6n
// SIG // nuZuGodwUvkOYTPj+lsqbPhLECg+/uFk0MZQMphEDtey
// SIG // tf1n2ase8TH3HED4vFyuSVAzUtr2f9ax1UmQv/UYANxP
// SIG // hbEwW5YQMqL9sLaVv1ghNynC6hhFVL3bQ3CIbDs4p/HC
// SIG // Yl6x29KMYPUskmeRs95UwiGQUmFmWVXq7HMlnNDixTDr
// SIG // 0hq9FoV0BM450U/l9Kfn3sbwCtgPAOZg/8QLGty5pKp6
// SIG // /JEgMl9fM8dehEZuAVq/A+vWvaRSZkpzb4kg6n2Kwj5j
// SIG // fsO80C0rIsW71GfABEmxFQJ3b0ZMJTX79Ja3BIHe/bR6
// SIG // kdpTtR6EzSbSKQ==
// SIG // End signature block
