 //////////////////////////////////////////////////////////////////////////////
//  Copyright 2014 by Autodesk, Inc.  All rights reserved.
//
// This computer source code and related instructions and comments 
// are the unpublished confidential and proprietary information of 
// Autodesk, Inc. and are protected under applicable copyright and 
// trade secret law.  They may not be disclosed to, copied or used 
// by any third party without the prior written consent of Autodesk, Inc.
//////////////////////////////////////////////////////////////////////////////
 
/*globals $,jQuery,document, window, navigator*/

/*!
 * ntp.utils - Utility functions
 */

// Namespace
var ntp = ntp || {};
ntp.utils = ntp.utils || {};
 
/**
 * Creates a callback function that calls the function 'name' on 'obj'.
 */
ntp.utils.makeCallback = function (obj, name) {
    return function () {
        var func = obj[name];
        if (func) {
            return func.apply(obj, arguments);
        }
    };
};

/**
 * Create a callback function that calls the function 'name' on 'obj' with fixed arguments
 */
ntp.utils.makeCallbackWithArgs = function (obj, name, args) {
    var funcArgs = Array.prototype.slice.call(arguments, 2);
    return function () {
        var func = obj[name];
        if (func) {
            return func.apply(obj, funcArgs);
        }
    };
};

/**
 * Gets a localized string with text-id, or apply localization on an element. 
 */
ntp.utils.localize = (function () {
    return function (param, param2) {
        if (typeof param === 'string') {
            if (ntp.strings) {
                return ntp.strings[param];
            }

            $.error('Localized resource has not been loaded.');
        } else if (param instanceof jQuery && typeof param2 === 'string') {
            if (ntp.strings) {
                param.html(ntp.strings[param2]);
            } else {
                param.attr('data-text-id', param2);
            }
        } else {
            $.error('Invalid parameter.');
        }
    };
})();

/**
 * Adds a CSS style rule to the page
 */
ntp.utils.addRule = (function () {
    var styleSheet = null;
    
    return function (rule) {
        if (typeof rule !== 'string') {
            $.error('Invalid parameter.');
            return;
        }
        if (styleSheet === null) {
            var style = document.createElement('style');
            document.getElementsByTagName('head')[0].appendChild(style);
            if (!window.createPopup) {
               style.appendChild(document.createTextNode(''));
            }
            styleSheet = document.styleSheets[document.styleSheets.length - 1];
        }
        styleSheet.insertRule(rule, styleSheet.cssRules.length);
    };
})();

/**
 * Resolves the absolute URL for a relative path
 */
ntp.utils.resolvePath = (function () {
    var aElement = null;
    
    return function (url) {
        if (typeof url !== 'string') {
            $.error('Invalid parameter.');
            return;
        }
        if (aElement === null) {
            aElement = document.createElement('a');
        }
        aElement.href = url;
        return aElement.href;
    };
})();

/**
 * A utility for executing function on timeout. If the same function
 * is scheduled again before the previous one is executed, the previous
 * timeout would be cleared.
 */
ntp.utils.timeout = (function () {
    // A map from function to timeout handle
    var timeoutHandles = {};
    
    return function (func, time) {
        if (typeof func !== 'function') {
            return; 
        }
        
        var timeoutHandle = timeoutHandles[func];
        if (timeoutHandle) {
            window.clearTimeout(timeoutHandle);
        };
        
        timeoutHandles[func] = window.setTimeout(function () {
            func();
            delete timeoutHandles[func];
        }, time);
    };
})();

/**
 * Utilities for extracting info from the query string.
 *
 * The product, locale, and automatic settings are passed in the URL, e.g. 
 *     index.html?lang=zh-CS&prod=ACAD&setting=%7B%0A%20%22openSheetSet%22%20%3A%20%22disabled%22%0A%7D
 */
ntp.utils.query = {
    
    locale: function () {
        var results = /[\\?&]lang=([^&#]*)/.exec(window.location.search);
        return results !== null && results !== '' ? results[1] : 'en-US';
    },
    
    product: function () {
        var results = /[\\?&]prod=([^&#]*)/.exec(window.location.search);
        return results !== null && results !== '' ? results[1] : null;
    },
    
    settings: function () {
        var results = /[\\?&]settings=([^&#]*)/.exec(window.location.search);
        if (results === null || results === '') {
            return null;
        }
        try {
            var settingsJSON = decodeURIComponent(results[1].replace(/\+/g, ' '));
            return $.parseJSON(settingsJSON);
        } catch (e) {
            // Failed to parse the JSON object
            return null;
        }
    }
};

/**
 * Shows the leaving Autodesk notice in a lightbox. Opens the URL in
 * external browser if user confirms.
 */
ntp.utils.openExternalUrlWithBrowser = (function () {
    
    var $leaveLightBox = null;
    var handleContinueButton = function (event) {
        if (event.keyCode == 13 && $leaveLightBox.lightbox('isShowing')) {
            if (!$('#button_leave_cancel').is(':focus')) {
                event.preventDefault();
                $('#button_leave_continue').click();
            }
        }
    };
    
    return function (url) {
        if ($leaveLightBox == null) {
            $leaveLightBox = $('#leave_lightbox');
            
            $leaveLightBox.lightbox('bind', 'lightboxshown', function () {
                $(document).on('keydown', handleContinueButton);
            });
            
            $leaveLightBox.lightbox('bind', 'lightboxhidden', function () {
                $(document).off('keydown', handleContinueButton);
            });
        }
        $('#button_leave_continue').off('click').click(function () {
            ntp.acad.openUrlWithBrowser(url);
            $leaveLightBox.lightbox('hide');
        });
        
        $('#button_leave_cancel').off('click').click(function () {
            $leaveLightBox.lightbox('hide');
        });
        
        $leaveLightBox.lightbox();
    };
})();

ntp.utils.string = {
        format : function() {
            if( arguments.length == 0 )
                return null;

            var str = arguments[0]; 
            for(var i=1;i<arguments.length;i++) {
                var re = new RegExp('\\{' + (i-1) + '\\}','gm');
                str = str.replace(re, arguments[i]);
            }
            return str;
        }
};
// SIG // Begin signature block
// SIG // MIIZjAYJKoZIhvcNAQcCoIIZfTCCGXkCAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // DlwcuCMvQ2iE2Hon98EFNx2Krp3TwKr2nKqwXhmdHiig
// SIG // ggopMIIE2jCCA8KgAwIBAgIQfG5XIF7pXpl44AXkMQgG
// SIG // EDANBgkqhkiG9w0BAQsFADCBhDELMAkGA1UEBhMCVVMx
// SIG // HTAbBgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8w
// SIG // HQYDVQQLExZTeW1hbnRlYyBUcnVzdCBOZXR3b3JrMTUw
// SIG // MwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
// SIG // b2RlIFNpZ25pbmcgQ0EgLSBHMjAeFw0xODA3MDUwMDAw
// SIG // MDBaFw0xOTA4MDUyMzU5NTlaMIGKMQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECAwKQ2FsaWZvcm5pYTETMBEGA1UEBwwK
// SIG // U2FuIFJhZmFlbDEXMBUGA1UECgwOQXV0b2Rlc2ssIElu
// SIG // Yy4xHzAdBgNVBAsMFkRlc2lnbiBTb2x1dGlvbnMgR3Jv
// SIG // dXAxFzAVBgNVBAMMDkF1dG9kZXNrLCBJbmMuMIIBIjAN
// SIG // BgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA7mRyNXG+
// SIG // QgtKVZfb37ZZm2wDgkFnx9kZy5clwGxvrSFmo0RQb5gR
// SIG // 6Ez0lhlQRNsfBVcYZrp0YyNxKg25oCdLDgPnaqilCia0
// SIG // UkRTpMYIUdaD0acP8C7Cb7r4oeKl2c3qVOGpvyCSM7Ep
// SIG // x16v7HtO0xeiOtCGU+cQ2kussGyDxiuvg0nsCMR4zV7b
// SIG // cg0bjKS3q22JzkzNrJpxSJJPnemEif5NQVQExg8Dg8YO
// SIG // UVAEhPc5+TRCtWH0fT+c4asKEUo9Urz5TBRwqqg2h/Ti
// SIG // k2qUPok3YlHrI8JIOeKfItfgU4PYtwr6yt4PPW7Jie1g
// SIG // cYi6JNButwAUIwRfCI1brIyhwQIDAQABo4IBPjCCATow
// SIG // CQYDVR0TBAIwADAOBgNVHQ8BAf8EBAMCB4AwEwYDVR0l
// SIG // BAwwCgYIKwYBBQUHAwMwYQYDVR0gBFowWDBWBgZngQwB
// SIG // BAEwTDAjBggrBgEFBQcCARYXaHR0cHM6Ly9kLnN5bWNi
// SIG // LmNvbS9jcHMwJQYIKwYBBQUHAgIwGQwXaHR0cHM6Ly9k
// SIG // LnN5bWNiLmNvbS9ycGEwHwYDVR0jBBgwFoAU1MAGIknr
// SIG // OUvdk+JcobhHdglyA1gwKwYDVR0fBCQwIjAgoB6gHIYa
// SIG // aHR0cDovL3JiLnN5bWNiLmNvbS9yYi5jcmwwVwYIKwYB
// SIG // BQUHAQEESzBJMB8GCCsGAQUFBzABhhNodHRwOi8vcmIu
// SIG // c3ltY2QuY29tMCYGCCsGAQUFBzAChhpodHRwOi8vcmIu
// SIG // c3ltY2IuY29tL3JiLmNydDANBgkqhkiG9w0BAQsFAAOC
// SIG // AQEArEMBlWJ6bc5AKEqwyO4TWNdnsYX9FreWUauXniGK
// SIG // lAwGhAkH7N4Ma1sE3yKtQi6pE7n+1t0bmlPPLiAWfRgI
// SIG // oDZ/ZwiWkOoP7A+31Q7skKjdUN1M96QjcbNUQjDkFqxy
// SIG // bHUraVr0gPlGUpnMR1npXAFHgXCrw45F1NT+LZNHWt/l
// SIG // AmfQsquTmDNvs7o0f/AhQr45Ylo8/Gsr8neojm2Io8iF
// SIG // pcMHqY7jcNnVL4ethYXVVdY0JSBN+Vg/2G0j/Lq8ic+I
// SIG // f8LiWdBllb3DUrVrgdjmBIHYCxPPJTXgsTGv4hwlRifS
// SIG // QkMw83nBdgWsObnssKnDJEOrhm9Da8FRHL4/fzCCBUcw
// SIG // ggQvoAMCAQICEHwbNTVK59t050FfEWnKa6gwDQYJKoZI
// SIG // hvcNAQELBQAwgb0xCzAJBgNVBAYTAlVTMRcwFQYDVQQK
// SIG // Ew5WZXJpU2lnbiwgSW5jLjEfMB0GA1UECxMWVmVyaVNp
// SIG // Z24gVHJ1c3QgTmV0d29yazE6MDgGA1UECxMxKGMpIDIw
// SIG // MDggVmVyaVNpZ24sIEluYy4gLSBGb3IgYXV0aG9yaXpl
// SIG // ZCB1c2Ugb25seTE4MDYGA1UEAxMvVmVyaVNpZ24gVW5p
// SIG // dmVyc2FsIFJvb3QgQ2VydGlmaWNhdGlvbiBBdXRob3Jp
// SIG // dHkwHhcNMTQwNzIyMDAwMDAwWhcNMjQwNzIxMjM1OTU5
// SIG // WjCBhDELMAkGA1UEBhMCVVMxHTAbBgNVBAoTFFN5bWFu
// SIG // dGVjIENvcnBvcmF0aW9uMR8wHQYDVQQLExZTeW1hbnRl
// SIG // YyBUcnVzdCBOZXR3b3JrMTUwMwYDVQQDEyxTeW1hbnRl
// SIG // YyBDbGFzcyAzIFNIQTI1NiBDb2RlIFNpZ25pbmcgQ0Eg
// SIG // LSBHMjCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoC
// SIG // ggEBANeVQ9Tc32euOftSpLYmMQRw6beOWyq6N2k1lY+7
// SIG // wDDnhthzu9/r0XY/ilaO6y1L8FcYTrGNpTPTC3Uj1Wp5
// SIG // J92j0/cOh2W13q0c8fU1tCJRryKhwV1LkH/AWU6rnXmp
// SIG // AtceSbE7TYf+wnirv+9SrpyvCNk55ZpRPmlfMBBOcWNs
// SIG // WOHwIDMbD3S+W8sS4duMxICUcrv2RZqewSUL+6Mcntim
// SIG // CXBx7MBHTI99w94Zzj7uBHKOF9P/8LIFMhlM07Acn/6l
// SIG // eCBCcEGwJoxvAMg6ABFBekGwp4qRBKCZePR3tPNgKuZs
// SIG // UAS3FGD/DVH0qIuE/iHaXF599Sl5T7BEdG9tcv8CAwEA
// SIG // AaOCAXgwggF0MC4GCCsGAQUFBwEBBCIwIDAeBggrBgEF
// SIG // BQcwAYYSaHR0cDovL3Muc3ltY2QuY29tMBIGA1UdEwEB
// SIG // /wQIMAYBAf8CAQAwZgYDVR0gBF8wXTBbBgtghkgBhvhF
// SIG // AQcXAzBMMCMGCCsGAQUFBwIBFhdodHRwczovL2Quc3lt
// SIG // Y2IuY29tL2NwczAlBggrBgEFBQcCAjAZGhdodHRwczov
// SIG // L2Quc3ltY2IuY29tL3JwYTA2BgNVHR8ELzAtMCugKaAn
// SIG // hiVodHRwOi8vcy5zeW1jYi5jb20vdW5pdmVyc2FsLXJv
// SIG // b3QuY3JsMBMGA1UdJQQMMAoGCCsGAQUFBwMDMA4GA1Ud
// SIG // DwEB/wQEAwIBBjApBgNVHREEIjAgpB4wHDEaMBgGA1UE
// SIG // AxMRU3ltYW50ZWNQS0ktMS03MjQwHQYDVR0OBBYEFNTA
// SIG // BiJJ6zlL3ZPiXKG4R3YJcgNYMB8GA1UdIwQYMBaAFLZ3
// SIG // +mlIR59TEtXC6gcydgfRlwcZMA0GCSqGSIb3DQEBCwUA
// SIG // A4IBAQB/68qn6ot2Qus+jiBUMOO3udz6SD4Wxw9FlRDN
// SIG // J4ajZvMC7XH4qsJVl5Fwg/lSflJpPMnx4JRGgBi7odSk
// SIG // VqbzHQCR1YbzSIfgy8Q0aCBetMv5Be2cr3BTJ7noPn5R
// SIG // oGlxi9xR7YA6JTKfRK9uQyjTIXW7l9iLi4z+qQRGBIX3
// SIG // FZxLEY3ELBf+1W5/muJWkvGWs60t+fTf2omZzrI4RMD3
// SIG // R3vKJbn6Kmgzm1By3qif1M0sCzS9izB4QOCNjicbkG8a
// SIG // vggVgV3rL+JR51EeyXgp5x5lvzjvAUoBCSQOFsQUecFB
// SIG // NzTQPZFSlJ3haO8I8OJpnGdukAsak3HUJgLDwFojMYIO
// SIG // uzCCDrcCAQEwgZkwgYQxCzAJBgNVBAYTAlVTMR0wGwYD
// SIG // VQQKExRTeW1hbnRlYyBDb3Jwb3JhdGlvbjEfMB0GA1UE
// SIG // CxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29yazE1MDMGA1UE
// SIG // AxMsU3ltYW50ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBT
// SIG // aWduaW5nIENBIC0gRzICEHxuVyBe6V6ZeOAF5DEIBhAw
// SIG // DQYJYIZIAWUDBAIBBQCgfDAQBgorBgEEAYI3AgEMMQIw
// SIG // ADAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
// SIG // BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG
// SIG // 9w0BCQQxIgQgYVXeC6vB/E25Yl7bxTi4m7CDNmbjLKwO
// SIG // 5fvnMhjabqUwDQYJKoZIhvcNAQEBBQAEggEARquIkb++
// SIG // yZvLVgNfIiiH8z19J96H+EGq/gVwPIaF1Gu0UNVWgBHA
// SIG // 06I3O8EgKTakAIdRto/rR0bY3n9VyO6X43e22Ka3KTso
// SIG // Z3ivt247MsX3IV6rG5wf06qRQfn/bId/l7VmXf9grjFg
// SIG // opu3OQW7uosEA7+7ASpKa196mnlqcMUoXEwkST7HSG6J
// SIG // oGYQ4pOOJMcaQ2dNzpAUMeaJqxmAiFIFVz2bkXz/Q+ep
// SIG // oc36O1jwp4r72oiaYyzywzayy2vxtwQRNhKLNSvn0Wc6
// SIG // ElaXo/G8Q78geu5xyokV3zBLQUusleFpS2fefCZWTzWm
// SIG // gwhLwY12ybvvcCPwose/lwq69KGCDHQwggxwBgorBgEE
// SIG // AYI3AwMBMYIMYDCCDFwGCSqGSIb3DQEHAqCCDE0wggxJ
// SIG // AgEDMQ8wDQYJYIZIAWUDBAIBBQAwga8GCyqGSIb3DQEJ
// SIG // EAEEoIGfBIGcMIGZAgEBBgkrBgEEAaAyAgMwMTANBglg
// SIG // hkgBZQMEAgEFAAQgDBZURdMXxX7hyMqkNcR5mrC8kJT9
// SIG // pCxLPNhopGZziTcCFEA7HAHsO8KglyS/x4pclsjUVcTH
// SIG // GA8yMDE5MDEzMDA5NDc0NlqgL6QtMCsxKTAnBgNVBAMM
// SIG // IEdsb2JhbFNpZ24gVFNBIGZvciBBZHZhbmNlZCAtIEcy
// SIG // oIII0zCCBLYwggOeoAMCAQICDAynz10HBySsieeaOjAN
// SIG // BgkqhkiG9w0BAQsFADBbMQswCQYDVQQGEwJCRTEZMBcG
// SIG // A1UEChMQR2xvYmFsU2lnbiBudi1zYTExMC8GA1UEAxMo
// SIG // R2xvYmFsU2lnbiBUaW1lc3RhbXBpbmcgQ0EgLSBTSEEy
// SIG // NTYgLSBHMjAeFw0xODAyMTkwMDAwMDBaFw0yOTAzMTgx
// SIG // MDAwMDBaMCsxKTAnBgNVBAMMIEdsb2JhbFNpZ24gVFNB
// SIG // IGZvciBBZHZhbmNlZCAtIEcyMIIBIjANBgkqhkiG9w0B
// SIG // AQEFAAOCAQ8AMIIBCgKCAQEAt8eSiijY85HLUA1d6ylE
// SIG // wYJZpcpkzr2DtVHllMxugJoXbNz/YuY2Q14Bwgxkjcp3
// SIG // nvZdd0W5Xsgk60E3EWY3rMjNdJvMs+9M4n8bznO+yZad
// SIG // NR5891T3bOygEtaumZ1jxHhfWakuIvXDJ9iyW0VjJ5D/
// SIG // jLE9qbTzDsDxIGUGiZ0DHKOwUkGFNOu8y85U9hL+swGJ
// SIG // Sqb6VMUf+9McwOO/l+M/0Bv5QMcb1esmfEMwx3DyeBGq
// SIG // hU6lDg0ASjoRd6PKIwGAnsiTVl1GzIrY0qoKhIWyiVvV
// SIG // Ms5BexhOVMqr0VJeEtMknlbzDI/dvZKOedVBicOjoOWR
// SIG // /MdnnYczznEycwIDAQABo4IBqDCCAaQwDgYDVR0PAQH/
// SIG // BAQDAgeAMEwGA1UdIARFMEMwQQYJKwYBBAGgMgEeMDQw
// SIG // MgYIKwYBBQUHAgEWJmh0dHBzOi8vd3d3Lmdsb2JhbHNp
// SIG // Z24uY29tL3JlcG9zaXRvcnkvMAkGA1UdEwQCMAAwFgYD
// SIG // VR0lAQH/BAwwCgYIKwYBBQUHAwgwRgYDVR0fBD8wPTA7
// SIG // oDmgN4Y1aHR0cDovL2NybC5nbG9iYWxzaWduLmNvbS9n
// SIG // cy9nc3RpbWVzdGFtcGluZ3NoYTJnMi5jcmwwgZgGCCsG
// SIG // AQUFBwEBBIGLMIGIMEgGCCsGAQUFBzAChjxodHRwOi8v
// SIG // c2VjdXJlLmdsb2JhbHNpZ24uY29tL2NhY2VydC9nc3Rp
// SIG // bWVzdGFtcGluZ3NoYTJnMi5jcnQwPAYIKwYBBQUHMAGG
// SIG // MGh0dHA6Ly9vY3NwMi5nbG9iYWxzaWduLmNvbS9nc3Rp
// SIG // bWVzdGFtcGluZ3NoYTJnMjAdBgNVHQ4EFgQULW5u0Y3l
// SIG // A4Du52Ppp9naCzxXIj0wHwYDVR0jBBgwFoAUkiGnSpVd
// SIG // ZLCbtB7mADdH5p1BK0wwDQYJKoZIhvcNAQELBQADggEB
// SIG // AI39HR+PiNNl+LVcRumn07Y/SstDBcIHDcNeA4QjN6h6
// SIG // jJcj/yechH1h61xLVz6b9ETbgFj3tmXejxIKX6eHdLfa
// SIG // FECHUeyfSkdV8BfoGD5xIDHmKCKrTvaunGvC50T7aA0g
// SIG // f1M/lBuhhoJOMhYQWj+JWMHQtNu2kPuNMumhS0ZQN7bD
// SIG // OukpJBKSWk4rubYKejaMl8DviXwz3jHhoZfk50WtYVL9
// SIG // WonTVH6KTVuJew6Wd9HyPDgb2vmlYSBlArzFrdcMYhAo
// SIG // LN2246hhAKwIJqONhdBkyTaQHopSyYeNNaVdSJH8k6al
// SIG // ZfHs+TrcpSyeBG1iM/7e+p/Fzq+V81Go8s4wggQVMIIC
// SIG // /aADAgECAgsEAAAAAAExicZQBDANBgkqhkiG9w0BAQsF
// SIG // ADBMMSAwHgYDVQQLExdHbG9iYWxTaWduIFJvb3QgQ0Eg
// SIG // LSBSMzETMBEGA1UEChMKR2xvYmFsU2lnbjETMBEGA1UE
// SIG // AxMKR2xvYmFsU2lnbjAeFw0xMTA4MDIxMDAwMDBaFw0y
// SIG // OTAzMjkxMDAwMDBaMFsxCzAJBgNVBAYTAkJFMRkwFwYD
// SIG // VQQKExBHbG9iYWxTaWduIG52LXNhMTEwLwYDVQQDEyhH
// SIG // bG9iYWxTaWduIFRpbWVzdGFtcGluZyBDQSAtIFNIQTI1
// SIG // NiAtIEcyMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIB
// SIG // CgKCAQEAqpuOw6sRUSUBtpaU4k/YwQj2RiPZRcWVl1ur
// SIG // Gr/SbFfJMwYfoA/GPH5TSHq/nYeer+7DjEfhQuzj46FK
// SIG // bAwXxKbBuc1b8R5EiY7+C94hWBPuTcjFZwscsrPxNHaR
// SIG // ossHbTfFoEcmAhWkkJGpeZ7X61edK3wi2BTX8QceeCI2
// SIG // a3d5r6/5f45O4bUIMf3q7UtxYowj8QM5j0R5tnYDV56t
// SIG // LwhG3NKMvPSOdM7IaGlRdhGLD10kWxlUPSbMQI2CJxtZ
// SIG // IH1Z9pOAjvgqOP1roEBlH1d2zFuOBE8sqNuEUBNPxtyL
// SIG // ufjdaUyI65x7MCb8eli7WbwUcpKBV7d2ydiACoBuCQID
// SIG // AQABo4HoMIHlMA4GA1UdDwEB/wQEAwIBBjASBgNVHRMB
// SIG // Af8ECDAGAQH/AgEAMB0GA1UdDgQWBBSSIadKlV1ksJu0
// SIG // HuYAN0fmnUErTDBHBgNVHSAEQDA+MDwGBFUdIAAwNDAy
// SIG // BggrBgEFBQcCARYmaHR0cHM6Ly93d3cuZ2xvYmFsc2ln
// SIG // bi5jb20vcmVwb3NpdG9yeS8wNgYDVR0fBC8wLTAroCmg
// SIG // J4YlaHR0cDovL2NybC5nbG9iYWxzaWduLm5ldC9yb290
// SIG // LXIzLmNybDAfBgNVHSMEGDAWgBSP8Et/qC5FJK5NUPpj
// SIG // move4t0bvDANBgkqhkiG9w0BAQsFAAOCAQEABFaCSnzQ
// SIG // zsm/NmbRvjWek2yX6AbOMRhZ+WxBX4AuwEIluBjH/NSx
// SIG // N8RooM8oagN0S2OXhXdhO9cv4/W9M6KSfREfnops7yyw
// SIG // 9GKNNnPRFjbxvF7stICYePzSdnno4SGU4B/EouGqZ9uz
// SIG // nHPlQCLPOc7b5neVp7uyy/YZhp2fyNSYBbJxb051rvE9
// SIG // ZGo7Xk5GpipdCJLxo/MddL9iDSOMXCo4ldLA1c3PiNof
// SIG // KLW6gWlkKrWmotVzr9xG2wSukdduxZi61EfEVnSAR3hY
// SIG // jL7vK/3sbL/RlPe/UOB74JD9IBh4GCJdCC6MHKCX8x2Z
// SIG // faOdkdMGRE4EbnocIOM28LZQuTGCAqgwggKkAgEBMGsw
// SIG // WzELMAkGA1UEBhMCQkUxGTAXBgNVBAoTEEdsb2JhbFNp
// SIG // Z24gbnYtc2ExMTAvBgNVBAMTKEdsb2JhbFNpZ24gVGlt
// SIG // ZXN0YW1waW5nIENBIC0gU0hBMjU2IC0gRzICDAynz10H
// SIG // BySsieeaOjANBglghkgBZQMEAgEFAKCCAQ4wGgYJKoZI
// SIG // hvcNAQkDMQ0GCyqGSIb3DQEJEAEEMBwGCSqGSIb3DQEJ
// SIG // BTEPFw0xOTAxMzAwOTQ3NDZaMC8GCSqGSIb3DQEJBDEi
// SIG // BCC997UP2Xbx6fH14dP3yHWtYcW0WBDgk/EmVhYZS5ox
// SIG // eDCBoAYLKoZIhvcNAQkQAgwxgZAwgY0wgYowgYcEFJsS
// SIG // BXrnKq/21jdytJ9qI28mSc2pMG8wX6RdMFsxCzAJBgNV
// SIG // BAYTAkJFMRkwFwYDVQQKExBHbG9iYWxTaWduIG52LXNh
// SIG // MTEwLwYDVQQDEyhHbG9iYWxTaWduIFRpbWVzdGFtcGlu
// SIG // ZyBDQSAtIFNIQTI1NiAtIEcyAgwMp89dBwckrInnmjow
// SIG // DQYJKoZIhvcNAQEBBQAEggEAMFaUh49rH+gA3r5fsw2K
// SIG // HLcj/QBwkW9eDO4PKu+QnrTK8FwpHf+V3ROtiE3/JlP1
// SIG // kN5vLf+HzKDIHFvCSmYU47BMPz74L3VGVzJsWt/9tu2g
// SIG // XRLzjIX9unSdjJur7RIOoeeWPD09Bvi2A2z5dMW53YPc
// SIG // s9ETgBZ6SAkXBcPlNY4Gg4xJtFkqPE9oLXP7xyEKJzEX
// SIG // qdN0dE8Tfs61X3LdtwWRLDJilsIsEPxFd5GxNahZBf36
// SIG // 8PYjiVxsN2tpe40YfuqpN85g7CBOCSQs8A7IPPUAEwKq
// SIG // b3+NcZ4bJ7nFPkKbjrGSJglPzlBMUNiIAgr2tDyXXbLP
// SIG // grciPGyJsoxIWg==
// SIG // End signature block
