package com.java110.store.listener.resourceStore;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.java110.core.annotation.Java110Listener;
import com.java110.core.context.DataFlowContext;
import com.java110.entity.center.Business;
import com.java110.po.purchase.ResourceStorePo;
import com.java110.store.dao.IResourceStoreServiceDao;
import com.java110.utils.constant.BusinessTypeConstant;
import com.java110.utils.constant.ResponseConstant;
import com.java110.utils.constant.StatusConstant;
import com.java110.utils.exception.ListenerExecuteException;
import com.java110.utils.util.Assert;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 删除资源信息 侦听
 * <p>
 * 处理节点
 * 1、businessResourceStore:{} 资源基本信息节点
 * 2、businessResourceStoreAttr:[{}] 资源属性信息节点
 * 3、businessResourceStorePhoto:[{}] 资源照片信息节点
 * 4、businessResourceStoreCerdentials:[{}] 资源证件信息节点
 * 协议地址 ：https://github.com/java110/MicroCommunity/wiki/%E5%88%A0%E9%99%A4%E5%95%86%E6%88%B7%E4%BF%A1%E6%81%AF-%E5%8D%8F%E8%AE%AE
 * Created by wuxw on 2018/5/18.
 */
@Java110Listener("deleteResourceStoreInfoListener")
@Transactional
public class DeleteResourceStoreInfoListener extends AbstractResourceStoreBusinessServiceDataFlowListener {

    private final static Logger logger = LoggerFactory.getLogger(DeleteResourceStoreInfoListener.class);
    @Autowired
    IResourceStoreServiceDao resourceResourceStoreServiceDaoImpl;

    @Override
    public int getOrder() {
        return 3;
    }

    @Override
    public String getBusinessTypeCd() {
        return BusinessTypeConstant.BUSINESS_TYPE_DELETE_RESOURCE_STORE;
    }

    /**
     * 根据删除信息 查出Instance表中数据 保存至business表 （状态写DEL） 方便撤单时直接更新回去
     *
     * @param dataFlowContext 数据对象
     * @param business        当前业务对象
     */
    @Override
    protected void doSaveBusiness(DataFlowContext dataFlowContext, Business business) {
        JSONObject data = business.getDatas();

        Assert.notEmpty(data, "没有datas 节点，或没有子节点需要处理");


        //处理 businessResourceStore 节点
        if (data.containsKey(ResourceStorePo.class.getSimpleName())) {
            Object _obj = data.get(ResourceStorePo.class.getSimpleName());
            JSONArray businessResourceStores = null;
            if (_obj instanceof JSONObject) {
                businessResourceStores = new JSONArray();
                businessResourceStores.add(_obj);
            } else {
                businessResourceStores = (JSONArray) _obj;
            }
            //JSONObject businessResourceStore = data.getJSONObject("businessResourceStore");
            for (int _resourceResourceStoreIndex = 0; _resourceResourceStoreIndex < businessResourceStores.size(); _resourceResourceStoreIndex++) {
                JSONObject businessResourceStore = businessResourceStores.getJSONObject(_resourceResourceStoreIndex);
                doBusinessResourceStore(business, businessResourceStore);
                if (_obj instanceof JSONObject) {
                    dataFlowContext.addParamOut("resId", businessResourceStore.getString("resId"));
                }
            }

        }


    }

    /**
     * 删除 instance数据
     *
     * @param dataFlowContext 数据对象
     * @param business        当前业务对象
     */
    @Override
    protected void doBusinessToInstance(DataFlowContext dataFlowContext, Business business) {
        String bId = business.getbId();
        //Assert.hasLength(bId,"请求报文中没有包含 bId");

        //资源信息
        Map info = new HashMap();
        info.put("bId", business.getbId());
        info.put("operate", StatusConstant.OPERATE_DEL);

        //资源信息
        List<Map> businessResourceStoreInfos = resourceResourceStoreServiceDaoImpl.getBusinessResourceStoreInfo(info);
        if (businessResourceStoreInfos != null && businessResourceStoreInfos.size() > 0) {
            for (int _resourceResourceStoreIndex = 0; _resourceResourceStoreIndex < businessResourceStoreInfos.size(); _resourceResourceStoreIndex++) {
                Map businessResourceStoreInfo = businessResourceStoreInfos.get(_resourceResourceStoreIndex);
                flushBusinessResourceStoreInfo(businessResourceStoreInfo, StatusConstant.STATUS_CD_INVALID);
                resourceResourceStoreServiceDaoImpl.updateResourceStoreInfoInstance(businessResourceStoreInfo);
                dataFlowContext.addParamOut("resId", businessResourceStoreInfo.get("res_id"));
            }
        }

    }

    /**
     * 撤单
     * 从business表中查询到DEL的数据 将instance中的数据更新回来
     *
     * @param dataFlowContext 数据对象
     * @param business        当前业务对象
     */
    @Override
    protected void doRecover(DataFlowContext dataFlowContext, Business business) {
        String bId = business.getbId();
        //Assert.hasLength(bId,"请求报文中没有包含 bId");
        Map info = new HashMap();
        info.put("bId", bId);
        info.put("statusCd", StatusConstant.STATUS_CD_INVALID);

        Map delInfo = new HashMap();
        delInfo.put("bId", business.getbId());
        delInfo.put("operate", StatusConstant.OPERATE_DEL);
        //资源信息
        List<Map> resourceResourceStoreInfo = resourceResourceStoreServiceDaoImpl.getResourceStoreInfo(info);
        if (resourceResourceStoreInfo != null && resourceResourceStoreInfo.size() > 0) {

            //资源信息
            List<Map> businessResourceStoreInfos = resourceResourceStoreServiceDaoImpl.getBusinessResourceStoreInfo(delInfo);
            //除非程序出错了，这里不会为空
            if (businessResourceStoreInfos == null || businessResourceStoreInfos.size() == 0) {
                throw new ListenerExecuteException(ResponseConstant.RESULT_CODE_INNER_ERROR, "撤单失败（resourceResourceStore），程序内部异常,请检查！ " + delInfo);
            }
            for (int _resourceResourceStoreIndex = 0; _resourceResourceStoreIndex < businessResourceStoreInfos.size(); _resourceResourceStoreIndex++) {
                Map businessResourceStoreInfo = businessResourceStoreInfos.get(_resourceResourceStoreIndex);
                flushBusinessResourceStoreInfo(businessResourceStoreInfo, StatusConstant.STATUS_CD_VALID);
                resourceResourceStoreServiceDaoImpl.updateResourceStoreInfoInstance(businessResourceStoreInfo);
            }
        }
    }


    /**
     * 处理 businessResourceStore 节点
     *
     * @param business              总的数据节点
     * @param businessResourceStore 资源节点
     */
    private void doBusinessResourceStore(Business business, JSONObject businessResourceStore) {

        Assert.jsonObjectHaveKey(businessResourceStore, "resId", "businessResourceStore 节点下没有包含 resId 节点");

        if (businessResourceStore.getString("resId").startsWith("-")) {
            throw new ListenerExecuteException(ResponseConstant.RESULT_PARAM_ERROR, "resId 错误，不能自动生成（必须已经存在的resId）" + businessResourceStore);
        }
        //自动插入DEL
        autoSaveDelBusinessResourceStore(business, businessResourceStore);
    }

    public IResourceStoreServiceDao getResourceStoreServiceDaoImpl() {
        return resourceResourceStoreServiceDaoImpl;
    }

    public void setResourceStoreServiceDaoImpl(IResourceStoreServiceDao resourceResourceStoreServiceDaoImpl) {
        this.resourceResourceStoreServiceDaoImpl = resourceResourceStoreServiceDaoImpl;
    }
}
