package com.java110.community.listener.community;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.java110.community.dao.ICommunityServiceDao;
import com.java110.core.annotation.Java110Listener;
import com.java110.core.context.DataFlowContext;
import com.java110.entity.center.Business;
import com.java110.po.community.CommunityPo;
import com.java110.utils.constant.BusinessTypeConstant;
import com.java110.utils.constant.ResponseConstant;
import com.java110.utils.constant.StatusConstant;
import com.java110.utils.exception.ListenerExecuteException;
import com.java110.utils.util.Assert;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;

import java.util.HashMap;
import java.util.Map;

/**
 * 删除小区信息 侦听
 * <p>
 * 处理节点
 * 1、businessCommunity:{} 小区基本信息节点
 * 2、businessCommunityAttr:[{}] 小区属性信息节点
 * 3、businessCommunityPhoto:[{}] 小区照片信息节点
 * 4、businessCommunityCerdentials:[{}] 小区证件信息节点
 * 协议地址 ：https://github.com/java110/MicroCommunity/wiki/%E5%88%A0%E9%99%A4%E5%95%86%E6%88%B7%E4%BF%A1%E6%81%AF-%E5%8D%8F%E8%AE%AE
 * Created by wuxw on 2018/5/18.
 */
@Java110Listener("deleteCommunityInfoListener")
@Transactional
public class DeleteCommunityInfoListener extends AbstractCommunityBusinessServiceDataFlowListener {

    private final static Logger logger = LoggerFactory.getLogger(DeleteCommunityInfoListener.class);
    @Autowired
    ICommunityServiceDao communityServiceDaoImpl;

    @Override
    public int getOrder() {
        return 3;
    }

    @Override
    public String getBusinessTypeCd() {
        return BusinessTypeConstant.BUSINESS_TYPE_DELETE_COMMUNITY_INFO;
    }

    /**
     * 根据删除信息 查出Instance表中数据 保存至business表 （状态写DEL） 方便撤单时直接更新回去
     *
     * @param dataFlowContext 数据对象
     * @param business        当前业务对象
     */
    @Override
    protected void doSaveBusiness(DataFlowContext dataFlowContext, Business business) {
        JSONObject data = business.getDatas();

        Assert.notEmpty(data, "没有datas 节点，或没有子节点需要处理");

        //处理 businessCommunity 节点
        if (data.containsKey(CommunityPo.class.getSimpleName())) {
            Object _obj = data.get(CommunityPo.class.getSimpleName());
            JSONArray businessCommunitys = null;
            if (_obj instanceof JSONObject) {
                businessCommunitys = new JSONArray();
                businessCommunitys.add(_obj);
            } else {
                businessCommunitys = (JSONArray) _obj;
            }
            //JSONObject businessCommunity = data.getJSONObject("businessCommunity");
            for (int _communityIndex = 0; _communityIndex < businessCommunitys.size(); _communityIndex++) {
                JSONObject businessCommunity = businessCommunitys.getJSONObject(_communityIndex);
                doBusinessCommunity(business, businessCommunity);
                if (_obj instanceof JSONObject) {
                    dataFlowContext.addParamOut("communityId", businessCommunity.getString("communityId"));
                }
            }

        }


    }

    /**
     * 删除 instance数据
     *
     * @param dataFlowContext 数据对象
     * @param business        当前业务对象
     */
    @Override
    protected void doBusinessToInstance(DataFlowContext dataFlowContext, Business business) {
        String bId = business.getbId();
        //Assert.hasLength(bId,"请求报文中没有包含 bId");

        //小区信息
        Map info = new HashMap();
        info.put("bId", business.getbId());
        info.put("operate", StatusConstant.OPERATE_DEL);

        //小区信息
        Map businessCommunityInfo = communityServiceDaoImpl.getBusinessCommunityInfo(info);
        if (businessCommunityInfo != null && businessCommunityInfo.size() > 0) {
            for (int _communityIndex = 0; _communityIndex < businessCommunityInfo.size(); _communityIndex++) {
                flushBusinessCommunityInfo(businessCommunityInfo, StatusConstant.STATUS_CD_INVALID);
                communityServiceDaoImpl.updateCommunityInfoInstance(businessCommunityInfo);
                dataFlowContext.addParamOut("communityId", businessCommunityInfo.get("community_id"));
            }
        }

    }

    /**
     * 撤单
     * 从business表中查询到DEL的数据 将instance中的数据更新回来
     *
     * @param dataFlowContext 数据对象
     * @param business        当前业务对象
     */
    @Override
    protected void doRecover(DataFlowContext dataFlowContext, Business business) {
        String bId = business.getbId();
        //Assert.hasLength(bId,"请求报文中没有包含 bId");
        Map info = new HashMap();
        info.put("bId", bId);
        info.put("statusCd", StatusConstant.STATUS_CD_INVALID);

        Map delInfo = new HashMap();
        delInfo.put("bId", business.getbId());
        delInfo.put("operate", StatusConstant.OPERATE_DEL);
        //小区信息
        Map communityInfo = communityServiceDaoImpl.getCommunityInfo(info);
        if (communityInfo != null && communityInfo.size() > 0) {

            //小区信息
            Map businessCommunityInfos = communityServiceDaoImpl.getBusinessCommunityInfo(delInfo);
            //除非程序出错了，这里不会为空
            if (businessCommunityInfos == null || businessCommunityInfos.size() == 0) {
                throw new ListenerExecuteException(ResponseConstant.RESULT_CODE_INNER_ERROR, "撤单失败（community），程序内部异常,请检查！ " + delInfo);
            }
            for (int _communityIndex = 0; _communityIndex < businessCommunityInfos.size(); _communityIndex++) {
                flushBusinessCommunityInfo(businessCommunityInfos, StatusConstant.STATUS_CD_VALID);
                communityServiceDaoImpl.updateCommunityInfoInstance(businessCommunityInfos);
            }
        }
    }


    /**
     * 处理 businessCommunity 节点
     *
     * @param business          总的数据节点
     * @param businessCommunity 小区节点
     */
    private void doBusinessCommunity(Business business, JSONObject businessCommunity) {

        Assert.jsonObjectHaveKey(businessCommunity, "communityId", "businessCommunity 节点下没有包含 communityId 节点");

        if (businessCommunity.getString("communityId").startsWith("-")) {
            throw new ListenerExecuteException(ResponseConstant.RESULT_PARAM_ERROR, "communityId 错误，不能自动生成（必须已经存在的communityId）" + businessCommunity);
        }
        //自动插入DEL
        autoSaveDelBusinessCommunity(business, businessCommunity);
    }

    public ICommunityServiceDao getCommunityServiceDaoImpl() {
        return communityServiceDaoImpl;
    }

    public void setCommunityServiceDaoImpl(ICommunityServiceDao communityServiceDaoImpl) {
        this.communityServiceDaoImpl = communityServiceDaoImpl;
    }
}
