#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# Copyright (C) 2011~2014 Deepin, Inc.
#               2011~2014 Kaisheng Ye
#
# Author:     Kaisheng Ye <kaisheng.ye@gmail.com>
# Maintainer: Kaisheng Ye <kaisheng.ye@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import signal
signal.signal(signal.SIGINT, signal.SIG_DFL)

import dbus
import dbus.service
from dbus.mainloop.glib import DBusGMainLoop
from gi.repository import Gtk, GLib
from gi.repository import AppIndicator3 as appindicator

DBUS_NAME = "com.youdao.indicator"
DBUS_PATH = "/com/youdao/indicator"
DBUS_INTERFACE = "com.youdao.indicator"

class YoudaoIndicator(dbus.service.Object):
    def __init__(self, bus_name, bus_path):
        dbus.service.Object.__init__(self, bus_name, bus_path)
        self.bus_name = bus_name

        self.ind = appindicator.Indicator.new(
                                "example-simple-client",
                                "youdao-dict",
                                appindicator.IndicatorCategory.APPLICATION_STATUS)
        self.ind.set_status(appindicator.IndicatorStatus.ACTIVE)
        self.ind.set_attention_icon ("indicator-messages-new")

        # create a menu
        menu = Gtk.Menu()

        # create some
        show_main_window_item = Gtk.MenuItem("显示主窗口")
        show_main_window_item.connect("activate", lambda w: \
            self.MenuItemClicked("main_window"))

        show_mini_window_item = Gtk.MenuItem("显示迷你窗口")
        show_mini_window_item.connect("activate", lambda w: \
            self.MenuItemClicked("mini_window"))

        show_setting_window_item = Gtk.MenuItem("软件设置")
        show_setting_window_item.connect("activate", lambda w: \
            self.MenuItemClicked("setting_window"))

        self.toggle_ocr_enable_item = Gtk.CheckMenuItem("启用屏幕取词   F8")
        self.toggle_ocr_enable_item.connect("activate", lambda w: \
            self.CheckMenuItemClicked("ocr_enable", \
            self.toggle_ocr_enable_item.get_active()))

        self.toggle_stroke_enable_item = Gtk.CheckMenuItem("启用划词翻译")
        self.toggle_stroke_enable_item.connect("activate", lambda w: \
            self.CheckMenuItemClicked("stroke_enable", \
            self.toggle_stroke_enable_item.get_active()))

        quit_item = Gtk.MenuItem("退出")
        quit_item.connect("activate", lambda w: self.Quit())

        menu.append(show_main_window_item)
        menu.append(show_mini_window_item)
        menu.append(show_setting_window_item)
        menu.append(Gtk.SeparatorMenuItem())
        menu.append(self.toggle_ocr_enable_item)
        menu.append(self.toggle_stroke_enable_item)
        menu.append(Gtk.SeparatorMenuItem())
        menu.append(quit_item)

        menu.show_all()
        self.ind.set_menu(menu)

    @dbus.service.method(DBUS_INTERFACE, in_signature="b", out_signature="")
    def SetOcrEnable(self, enable):
        self.toggle_ocr_enable_item.set_active(enable)

    @dbus.service.method(DBUS_INTERFACE, in_signature="b", out_signature="")
    def SetStrokeEnable(self, enable):
        self.toggle_stroke_enable_item.set_active(enable)

    @dbus.service.method(DBUS_INTERFACE, in_signature="", out_signature="")
    def Quit(self):
        Gtk.main_quit()
        self.MenuItemClicked("quit")

    @dbus.service.method(DBUS_INTERFACE, in_signature="", out_signature="")
    def Hello(self):
        print("hello")

    @dbus.service.signal(DBUS_INTERFACE, signature='s')
    def MenuItemClicked(self, menu_id):
        pass

    @dbus.service.signal(DBUS_INTERFACE, signature='sb')
    def CheckMenuItemClicked(self, menu_id, checked):
        pass

if __name__ == "__main__":
    DBusGMainLoop(set_as_default=True)
    session_bus = dbus.SessionBus()
    if session_bus.name_has_owner(DBUS_NAME):
        print(DBUS_NAME, "is running...")
    else:
        bus_name = dbus.service.BusName(DBUS_NAME, bus=session_bus)
        service = YoudaoIndicator(bus_name, DBUS_PATH)
        Gtk.main()
