/*************************************************************************

Filename:		ShowString.c
Purpose:		Shows all strings in a file
Author:			figugegl
Version:		2.0
Date:			20.7.2004

*************************************************************************/


/*------------------------------------------------------------------------
Include files
------------------------------------------------------------------------*/
#include <windows.h>
#include <windowsx.h>
#include <commctrl.h>
#include <string.h>
#include <process.h>
#include <shlobj.h>
#include "showstring.h"


/*------------------------------------------------------------------------
Prototypes
------------------------------------------------------------------------*/
static BOOL CALLBACK hMainWndProc (HWND, UINT, WPARAM, LPARAM);


/*------------------------------------------------------------------------
Type definitions
------------------------------------------------------------------------*/
#define MAX_LV_VIS_STR    260                 // FIXED max. number of VISIBLE chars in a listview column.
                                              // seems to be an internal windows listview limit
#define MAX_STREP         256                 // max. number of replace strings
#define MAX_STR_LEN       240                 // max. strlen
#define DEF_COL_WIDTH     100                 // default column width
#define COL_PADDING        20                 // text padding in column
#define STAT_BAR_HIGHT     15                 // height of statusbar
#define NUM_OF_COL          3                 // number of columns in listview

typedef struct
{
	HANDLE        hMainWnd;                   // handle of mainwindow
	HANDLE        hListView;                  // handle of listview
	HANDLE        hProgBar;                   // handle of progress bar
	HANDLE        hStatBar;                   // handle of status bar
	char          szIniFile[MAX_STR_LEN];     // name of ini-file
	char          szFileOpenDir[MAX_PATH];    // dir for open file
	char          szFileSaveDir[MAX_PATH];    // dir for save file
	int	          iStrMinLen;                 // min string length
	int	          iStrMaxLen;                 // max string length
	int           iStrType;                   // 1 = ascii; 2 = unicode; 3 = both
	int           iStrNullterm;               // only nullterminated strings allowed
	unsigned char cCharRangeLo;               // range start
	unsigned char cCharRangeHi;               // range end
	unsigned char cCharIncl[MAX_STR_LEN];     // include list
	unsigned char cCharExcl[MAX_STR_LEN];     // exclude list
	long          lFontHeight;                // font height, lfHeight member of LOGFONT structure
	long          lFontWeight;                // font weight, lfWeight member of LOGFONT structure
	BYTE          iFontItalic;                // font height, lfItalic member of LOGFONT structure
	char          szFontName[LF_FACESIZE];    // font name, lfFaceName member of LOGFONT structure
	unsigned long iLineNumMaxStr[NUM_OF_COL]; // line-number of longest string in each column
	char          szColNames[NUM_OF_COL][8];  // names of colum headers

}
OPTIONS, *POPTIONS;

typedef struct
{
	HANDLE        hMainWnd;                   // handle of maindialog
	HANDLE        hListView;                  // handle of listview
	HANDLE        hProgBar;                   // handle of progress bar
	HANDLE        hStatBar;                   // handle of status bar
	BOOL          bWorking;                   // TRUE if thread has started
	unsigned long lSearchTime;                // time in milliseconds
	unsigned long lNumStrings;                // number of strings found
	unsigned long lNumAsciiStr;               // number of ascii strings found
	unsigned long lNumUnicodeStr;             // number of unicode strings found
	int           iButStatus;                 // 0 = GetStrings, 1 = Stop
	char          szFileName[MAX_STR_LEN];    // filename for stringsearch
}
T_PARAMS, *PT_PARAMS;

typedef struct
{
	unsigned long iLVLineNum;                 // item number
	char          szOrigTxt[2*MAX_STR_LEN];   // original string
}
STR_REPLACE, *PSTR_REPLACE;

typedef struct
{
	char          szFileOffset[12];           // col 1: file offset
	char          szStrLen[4];                // col 2: string length
	char          szFileStr[2*MAX_STR_LEN];   // col 3: string
}
LVLINE, *PLVLINE;


/*------------------------------------------------------------------------
Global variables
------------------------------------------------------------------------*/
HINSTANCE		hInstance;
OPTIONS			options;
STR_REPLACE		strep[MAX_STREP];
unsigned long	iProgBarPos;
int				iNumOfReplace;
char			szAppName[] = "ShowString";


/*------------------------------------------------------------------------
Procedure:     WinMain
Purpose:       Application entry point. Registers a class of the same
               type than the dialog class
Input:         Standard
Output:        None
Errors:        None
------------------------------------------------------------------------*/
int APIENTRY WinMain (HINSTANCE hInst, HINSTANCE hInstPrev, LPSTR lpCmdLine, int nCmdShow)
{
	WNDCLASS				wc;
	HANDLE					hWnd;
	MSG						msg;
	INITCOMMONCONTROLSEX	iccex;

	// initialize and register window class
	memset (&wc, 0, sizeof (wc));
	wc.lpfnWndProc		= hMainWndProc;
	wc.cbWndExtra		= DLGWINDOWEXTRA;
	wc.hInstance		= hInst;
	hInstance = hInst;
	wc.hIcon			= LoadIcon (hInst, MAKEINTRESOURCE (ICO_SHOWSTRING));
	wc.hCursor			= LoadCursor(NULL, IDC_ARROW);
	wc.hbrBackground	= (HBRUSH) (COLOR_WINDOW + 1);
	wc.lpszMenuName		= MAKEINTRESOURCE (MEN_MAIN);
	wc.lpszClassName	= szAppName;
	if (!RegisterClass (&wc)) return 0;

	// initialize common controls
	iccex.dwSize = sizeof (INITCOMMONCONTROLSEX);
	iccex.dwICC  = ICC_LISTVIEW_CLASSES;
	InitCommonControlsEx (&iccex);

	// create mainwindow
	hWnd = CreateWindow (szAppName, szAppName, WS_OVERLAPPEDWINDOW | WS_CLIPCHILDREN,
		                 CW_USEDEFAULT, CW_USEDEFAULT, 600, 300, NULL, NULL, hInst, NULL);
	ShowWindow (hWnd, nCmdShow);
	UpdateWindow (hWnd);

	// message loop
	while (GetMessage (&msg, NULL, 0, 0))
	{
		TranslateMessage (&msg);
		DispatchMessage (&msg);
	}
	return msg.wParam;
}


/*------------------------------------------------------------------------
Procedure:     HexToInt
Purpose:       Convert hexstring to int
Input:         szHex: hexstring
Output:        int value of converted string
Errors:        None
------------------------------------------------------------------------*/
int HexToInt (unsigned char *szHex)
{
	int i, iValue, iStrLen, iMult;

	CharUpper (szHex);
	iStrLen = lstrlen (szHex);
	if (iStrLen > 0)
	{
		for (iMult = 1, iValue = 0, i = iStrLen-1; i >= 0; i--)
		{
			switch (szHex[i])
			{
			case '0':
			case '1':
			case '2':
			case '3':
			case '4':
			case '5':
			case '6':
			case '7':
			case '8':
			case '9':
				iValue += (szHex[i] - '0') * iMult;
				break;

			case 'A':
			case 'B':
			case 'C':
			case 'D':
			case 'E':
			case 'F':
				iValue += (szHex[i] - 55) * iMult;
				break;

			default:
				return -1;
			}
			iMult *= 16;
		}
	}
	return iValue;
}


/*------------------------------------------------------------------------
Procedure:     WriteIniFile
Purpose:       Write ini-file with options structure
Input:         poptions: pointer to options structure
Output:        TRUE if success
Errors:        None
------------------------------------------------------------------------*/
BOOL WriteIniFile (POPTIONS poptions)
{
	char szBuffer[MAX_STR_LEN];

	// section [file]
	WritePrivateProfileString ("File", "OpenDir", &poptions->szFileOpenDir, poptions->szIniFile);
	WritePrivateProfileString ("File", "SaveDir", &poptions->szFileSaveDir, poptions->szIniFile);

	// section [string]
	wsprintf (szBuffer, "%i", poptions->iStrMinLen);
	WritePrivateProfileString ("String", "MinLen",   szBuffer, poptions->szIniFile);
	wsprintf (szBuffer, "%i", poptions->iStrMaxLen);
	WritePrivateProfileString ("String", "MaxLen",   szBuffer, poptions->szIniFile);
	wsprintf (szBuffer, "%i", poptions->iStrType);
	WritePrivateProfileString ("String", "Sort",     szBuffer, poptions->szIniFile);
	wsprintf (szBuffer, "%i", poptions->iStrNullterm);
	WritePrivateProfileString ("String", "NullTerm", szBuffer, poptions->szIniFile);

	// section [char]
	wsprintf (szBuffer, "%0.2X", poptions->cCharRangeLo);
	WritePrivateProfileString ("Char", "RangeLo",  szBuffer, poptions->szIniFile);
	wsprintf (szBuffer, "%0.2X", poptions->cCharRangeHi);
	WritePrivateProfileString ("Char", "RangeHi",  szBuffer, poptions->szIniFile);
	WritePrivateProfileString ("Char", "CharIncl", &poptions->cCharIncl, poptions->szIniFile);
	WritePrivateProfileString ("Char", "CharExcl", &poptions->cCharExcl, poptions->szIniFile);

	// section [font]
	WritePrivateProfileString ("Font", "Name",   &poptions->szFontName, poptions->szIniFile);
	wsprintf (szBuffer, "%d", poptions->lFontHeight);
	WritePrivateProfileString ("Font", "Height", szBuffer, poptions->szIniFile);
	wsprintf (szBuffer, "%d", poptions->lFontWeight);
	WritePrivateProfileString ("Font", "Weight", szBuffer, poptions->szIniFile);
	wsprintf (szBuffer, "%d", poptions->iFontItalic);
	WritePrivateProfileString ("Font", "Italic", szBuffer, poptions->szIniFile);
	return TRUE;
}


/*------------------------------------------------------------------------
Procedure:     ReadIniFile
Purpose:       Read ini-file and fill options structure. Creates the
               file with default values if it doesn't exist
Input:         options:	pointer to options structure
Output:        None
Errors:        None
------------------------------------------------------------------------*/
void ReadIniFile (POPTIONS poptions)
{
	HWND hFile;
	int  iResult;
	char szBuffer[MAX_STR_LEN];
	char szDefault[] = "0";

	// read ini-file to fill options-structure
	hFile = CreateFileA (poptions->szIniFile, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
	if (hFile == INVALID_HANDLE_VALUE)
	{
		// ini-file doesn't exist, create it
		hFile = CreateFileA (poptions->szIniFile, GENERIC_READ, FILE_SHARE_READ, NULL, CREATE_NEW, FILE_ATTRIBUTE_NORMAL, NULL);
		CloseHandle (hFile);

		// default values
		lstrcpy (poptions->szFileOpenDir, "C:\\");
		lstrcpy (poptions->szFileSaveDir, "C:\\");
		poptions->iStrMaxLen   = MAX_STR_LEN;
		poptions->iStrMinLen   = 4;
		poptions->iStrType     = 3;                 // ascii & unicode
		poptions->iStrNullterm = 1;
		poptions->cCharRangeLo = 0x20;
		poptions->cCharRangeHi = 0x7F;
		lstrcpy (poptions->cCharIncl, "");   // include german "Umlaute"
		lstrcpy (poptions->szFontName, "Courier New");
		poptions->lFontHeight  = 15;
		poptions->lFontWeight  = 400;
		poptions->iFontItalic  = 0;
		WriteIniFile (poptions);
	}
	else
	    {
		CloseHandle (hFile);

		// read section [file]
		GetPrivateProfileString ("File", "OpenDir", szDefault, &poptions->szFileOpenDir, MAX_STR_LEN, poptions->szIniFile);
		GetPrivateProfileString ("File", "SaveDir", szDefault, &poptions->szFileSaveDir, MAX_STR_LEN, poptions->szIniFile);

		// read section [string]
		GetPrivateProfileString ("String", "MinLen",   szDefault, szBuffer, MAX_STR_LEN, poptions->szIniFile);
		poptions->iStrMinLen = atoi (szBuffer);
		GetPrivateProfileString ("String", "MaxLen",   szDefault, szBuffer, MAX_STR_LEN, poptions->szIniFile);
		poptions->iStrMaxLen = atoi (szBuffer);
		GetPrivateProfileString ("String", "Type",     szDefault, szBuffer, MAX_STR_LEN, poptions->szIniFile);
		poptions->iStrType = atoi (szBuffer);
		GetPrivateProfileString ("String", "NullTerm", szDefault, szBuffer, MAX_STR_LEN, poptions->szIniFile);
		poptions->iStrNullterm = atoi (szBuffer);

		// read section [char]
		GetPrivateProfileString ("Char", "RangeLo",  szDefault, szBuffer, MAX_STR_LEN, poptions->szIniFile);
		iResult = HexToInt (&szBuffer);
		poptions->cCharRangeLo = iResult < 0 ? 0x20 : iResult;
		GetPrivateProfileString ("Char", "RangeHi",  szDefault, szBuffer, MAX_STR_LEN, poptions->szIniFile);
		iResult = HexToInt (&szBuffer);
		poptions->cCharRangeHi = iResult < 0 ? 0x7F : iResult;
		GetPrivateProfileString ("Char", "CharIncl",  szDefault, &poptions->cCharIncl, MAX_STR_LEN, poptions->szIniFile);
		GetPrivateProfileString ("Char", "CharExcl",  szDefault, &poptions->cCharExcl, MAX_STR_LEN, poptions->szIniFile);

		// read section [font]
		GetPrivateProfileString ("Font", "Name",   szDefault, &poptions->szFontName, LF_FACESIZE, poptions->szIniFile);
		GetPrivateProfileString ("Font", "Height", szDefault, szBuffer, MAX_STR_LEN, poptions->szIniFile);
		poptions->lFontHeight = atol (szBuffer);
		GetPrivateProfileString ("Font", "Weight", szDefault, szBuffer, MAX_STR_LEN, poptions->szIniFile);
		poptions->lFontWeight = atol (szBuffer);
		GetPrivateProfileString ("Font", "Italic", szDefault, szBuffer, MAX_STR_LEN, poptions->szIniFile);
		poptions->iFontItalic = atoi (szBuffer);
	}
	return;
}


/*------------------------------------------------------------------------
Procedure:     GetListViewItems
Purpose:       Retrieves entire line from listview
Input:         hListView
               lLineNum
			   plvl
Output:        TRUE if line retrieved
Errors:        None
------------------------------------------------------------------------*/
BOOL GetListViewItems (HWND hListView, unsigned long lLineNum, PLVLINE plvl)
{
	LVITEM lvi;

	// get 1st column member (offset) from listview
	lvi.mask = LVIF_TEXT;
	lvi.iItem = lLineNum;
	lvi.iSubItem = 0;
	lvi.pszText = plvl->szFileOffset;
	lvi.cchTextMax = sizeof (plvl->szFileOffset);
	if (SendMessage (hListView, LVM_GETITEM, 0, (LPARAM) &lvi) == FALSE)
		return FALSE;

	// get 2nd column member (len)
	lvi.iSubItem = 1;
	lvi.pszText = plvl->szStrLen;
	lvi.cchTextMax = sizeof (plvl->szStrLen);
	if (SendMessage (hListView, LVM_GETITEM, 0, (LPARAM) &lvi) == FALSE)
		return FALSE;

	// get 3rd column member (string)
	lvi.iSubItem = 2;
	lvi.pszText = plvl->szFileStr;
	lvi.cchTextMax = sizeof (plvl->szFileStr);
	if (SendMessage (hListView, LVM_GETITEM, 0, (LPARAM) &lvi) == FALSE) return FALSE;
	else return TRUE;
}


/*------------------------------------------------------------------------
Procedure:     SetListViewItems
Purpose:       Stores entire line in listview
Input:         hListView:  Handle of listview
               lLineNum:   Number of line to be stored
			   plvl:       Pointer to strings
			   bInsert:    TRUE if new line is inserted, FALSE if overwrite
Output:        TRUE if line stored
Errors:        None
------------------------------------------------------------------------*/
BOOL SetListViewItems (HWND hListView, unsigned long lLineNum, PLVLINE plvl, BOOL bInsert)
{
	LVITEM lvi;

	// file offset (1st column)
	lvi.mask = LVIF_TEXT;
	lvi.iItem = lLineNum;
	lvi.iSubItem = 0;
	lvi.pszText = plvl->szFileOffset;
	if (bInsert) SendMessage (hListView, LVM_INSERTITEM, 0, (LPARAM) &lvi);
	else SendMessage (hListView, LVM_SETITEM, 0, (LPARAM) &lvi);

	// string length (2nd column)
	lvi.iSubItem = 1;
	lvi.pszText = plvl->szStrLen;
	SendMessage (hListView, LVM_SETITEM, 0, (LPARAM) &lvi);

	// string (3rd column)
	lvi.iSubItem = 2;
	lvi.pszText = plvl->szFileStr;
	SendMessage (hListView, LVM_SETITEM, 0, (LPARAM) &lvi);
	return TRUE;
}


/*------------------------------------------------------------------------
Procedure:     SetListViewFont
Purpose:       Change font in listview and adjust column width
Input:         poptions: pointer to options struct
Output:        None
Errors:        None
------------------------------------------------------------------------*/
void SetListViewFont (POPTIONS poptions)
{
	HFONT		hFontListView;
	HDC			hDCListView;
	LOGFONT		lf;
	LVITEM		lvi;
	RECT		rc;
	BOOL		bLVEmpty;
	long		lNumOfItems;
	int			i, iStrWidth, iAddColWidth;
	char        szBuf[2*MAX_STR_LEN+1];

	// fill struct, create and set font
	memset (&lf, 0, sizeof (lf));
	lstrcpy (lf.lfFaceName, poptions->szFontName);
	lf.lfHeight = poptions->lFontHeight;
	lf.lfWeight = poptions->lFontWeight;
	lf.lfItalic = poptions->iFontItalic;
	hFontListView = CreateFontIndirect (&lf);
	hDCListView = GetDC (poptions->hListView);
	if (hDCListView != NULL)
	{
		if (DeleteObject (SelectObject (hDCListView, hFontListView)))
		{
			// set listview font
			SendMessage (poptions->hListView, WM_SETFONT, (WPARAM) hFontListView, 0);
			ReleaseDC (poptions->hMainWnd, hDCListView);
		}
	}

	// get number of lines in listview
	lNumOfItems = SendMessage (poptions->hListView, LVM_GETITEMCOUNT, 0, 0);
	bLVEmpty = lNumOfItems == 0;
	GetClientRect (poptions->hMainWnd, &rc);
	lvi.mask = LVIF_TEXT;
	lvi.pszText = szBuf;
	lvi.cchTextMax = sizeof (szBuf);
	for (iAddColWidth = 0, i = 0; i < NUM_OF_COL; i++)
	{
		// get longest string of each column
		lvi.iItem = poptions->iLineNumMaxStr[i];
		lvi.iSubItem = i;
		SendMessage (poptions->hListView, LVM_GETITEMTEXT, (WPARAM) poptions->iLineNumMaxStr[i], (LPARAM) (LV_ITEM FAR *) &lvi);

		// limit buffer to max. visible listview strlen for determining the column width
		if (lstrlen (szBuf) > MAX_LV_VIS_STR) szBuf[MAX_LV_VIS_STR] = 0;
		iStrWidth = SendMessage (poptions->hListView, LVM_GETSTRINGWIDTH, 0, (LPARAM) (LPCSTR) szBuf);

		// set column width to max strlen in column or default if empty
		if (bLVEmpty) iStrWidth = (rc.right-rc.left)/NUM_OF_COL;
		else iStrWidth += COL_PADDING;

		// add up total columnwidth for all columns except string-column
		if (i != NUM_OF_COL-1) iAddColWidth += iStrWidth;

		// make sure that str-col covers the entire listview, because
		// with short strings an additional empty column might be visible
		else
		{
			if (iStrWidth + iAddColWidth < rc.right - rc.left)
				iStrWidth = rc.right - rc.left - iAddColWidth;

			// subtract width of scrollbar if necessary
			if (lNumOfItems > SendMessage (poptions->hListView, LVM_GETCOUNTPERPAGE, 0, 0))
				iStrWidth -= GetSystemMetrics (SM_CXHSCROLL);
		}

		// set column width
		SendMessage (poptions->hListView, LVM_SETCOLUMNWIDTH, i, MAKELPARAM (iStrWidth, 0));
	}
}


/*------------------------------------------------------------------------
Procedure:     OptionsDlgProc
Purpose:       Handles the messages of the options dialog
Input:         Standard
Output:        TRUE if the message was processed
Errors:        None
------------------------------------------------------------------------*/
LRESULT CALLBACK OptionsDlgProc (HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	BROWSEINFO		bi;
	LPITEMIDLIST	pidlBrowse;
	CHOOSEFONT		cf;
	LOGFONT			lf;
	char			szBuffer[MAX_STR_LEN];
	int				iResult;

	switch (uMsg)
	{
	case WM_INITDIALOG:

		// limit number of chars in editfields
		SendDlgItemMessage (hWnd, EDF_FILEOPENPATH, EM_SETLIMITTEXT, MAX_STR_LEN-1, 0);
		SendDlgItemMessage (hWnd, EDF_FILESAVEPATH, EM_SETLIMITTEXT, MAX_STR_LEN-1, 0);
		SendDlgItemMessage (hWnd, EDF_MINSTRLEN, EM_SETLIMITTEXT, 3, 0);
		SendDlgItemMessage (hWnd, EDF_MAXSTRLEN, EM_SETLIMITTEXT, 3, 0);
		SendDlgItemMessage (hWnd, EDF_FIRSTCHAR, EM_SETLIMITTEXT, 2, 0);
		SendDlgItemMessage (hWnd, EDF_LASTCHAR,  EM_SETLIMITTEXT, 2, 0);
		SendDlgItemMessage (hWnd, EDF_INCLUDECHAR, EM_SETLIMITTEXT, MAX_STR_LEN-1, 0);
		SendDlgItemMessage (hWnd, EDF_EXCLUDECHAR, EM_SETLIMITTEXT, MAX_STR_LEN-1, 0);

		// read ini-file into struct options
		ReadIniFile (&options);

		// write struct options to dialog
		SetDlgItemTextA (hWnd, EDF_FILEOPENPATH, options.szFileOpenDir);
		SetDlgItemTextA (hWnd, EDF_FILESAVEPATH, options.szFileSaveDir);
		SetDlgItemInt (hWnd, EDF_MINSTRLEN, options.iStrMinLen, FALSE);
		SetDlgItemInt (hWnd, EDF_MAXSTRLEN, options.iStrMaxLen, FALSE);

		switch (options.iStrType)
		{
		case 2:
			CheckDlgButton (hWnd, CHK_UNICODE, BST_CHECKED);
			break;

		case 3:
			CheckDlgButton (hWnd, CHK_ASCII,   BST_CHECKED);
			CheckDlgButton (hWnd, CHK_UNICODE, BST_CHECKED);
			break;

		default:
			CheckDlgButton (hWnd, CHK_ASCII, BST_CHECKED);
		}

		if (options.iStrNullterm == 1) CheckDlgButton (hWnd, CHK_NULLTERM_ONLY, BST_CHECKED);

		wsprintf (szBuffer, "%0.2X", options.cCharRangeLo);
		SetDlgItemTextA (hWnd, EDF_FIRSTCHAR, szBuffer);
		wsprintf (szBuffer, "%0.2X", options.cCharRangeHi);
		SetDlgItemTextA (hWnd, EDF_LASTCHAR,  szBuffer);
		SetDlgItemTextA (hWnd, EDF_INCLUDECHAR, options.cCharIncl);
		SetDlgItemTextA (hWnd, EDF_EXCLUDECHAR, options.cCharExcl);
		break;

	case WM_COMMAND:
		switch (LOWORD (wParam))
		{
		case BUT_FILEOPENPATH:
			// Fill in the BROWSEINFO structure
			memset (&bi, 0, sizeof (bi));
			bi.hwndOwner = hWnd;
			bi.pszDisplayName = szBuffer;
			bi.lpszTitle = "Choose folder for files to open";
			bi.ulFlags = BIF_RETURNONLYFSDIRS;

			// Browse for a folder and return its PIDL
			pidlBrowse = SHBrowseForFolder (&bi);
			if (pidlBrowse != NULL)
			{
				SHGetPathFromIDList (pidlBrowse, options.szFileOpenDir);
				SetDlgItemTextA (hWnd, EDF_FILEOPENPATH, options.szFileOpenDir);
				CoTaskMemFree (pidlBrowse);
			}
			break;

		case BUT_FILESAVEPATH:
			// Fill in the BROWSEINFO structure
			memset (&bi, 0, sizeof (bi));
			bi.hwndOwner = hWnd;
			bi.pszDisplayName = szBuffer;
			bi.lpszTitle = "Choose folder for files to save";
			bi.ulFlags = BIF_RETURNONLYFSDIRS;

			// Browse for a folder and return its PIDL
			pidlBrowse = SHBrowseForFolder (&bi);
			if (pidlBrowse != NULL)
			{
				SHGetPathFromIDList (pidlBrowse, options.szFileSaveDir);
				SetDlgItemTextA (hWnd, EDF_FILESAVEPATH, options.szFileSaveDir);
				CoTaskMemFree (pidlBrowse);
			}
			break;

		case BUT_FONT:
			memset (&cf, 0, sizeof (cf));
			cf.lStructSize = sizeof (cf);
			cf.hwndOwner = options.hListView;
			cf.lpLogFont = &lf;
			cf.Flags = CF_INITTOLOGFONTSTRUCT | CF_SCREENFONTS | CF_FORCEFONTEXIST;
			if (ChooseFont (&cf) > 0)
			{
				lstrcpy (options.szFontName, lf.lfFaceName);
				options.lFontHeight = lf.lfHeight;
				options.lFontWeight = lf.lfWeight;
				options.iFontItalic = lf.lfItalic;
			}
			break;

		case BUT_OK:
			// write dialog to struct options
			// open/save path
			GetDlgItemTextA (hWnd, EDF_FILEOPENPATH, options.szFileOpenDir, MAX_STR_LEN);
			GetDlgItemTextA (hWnd, EDF_FILESAVEPATH, options.szFileSaveDir, MAX_STR_LEN);

			// check stringlength
			options.iStrMinLen = GetDlgItemInt (hWnd, EDF_MINSTRLEN, &iResult, FALSE);
			if (options.iStrMinLen > MAX_STR_LEN) options.iStrMinLen = MAX_STR_LEN;
			if (options.iStrMinLen < 2) options.iStrMinLen = 2;
			options.iStrMaxLen = GetDlgItemInt (hWnd, EDF_MAXSTRLEN, &iResult, FALSE);
			if (options.iStrMaxLen > MAX_STR_LEN) options.iStrMaxLen = MAX_STR_LEN;
			if (options.iStrMaxLen < 2) options.iStrMaxLen = 2;

			// swap if min > max
			if (options.iStrMinLen > options.iStrMaxLen)
			{
				iResult = options.iStrMinLen;
				options.iStrMinLen = options.iStrMaxLen;
				options.iStrMaxLen = iResult;
			}

			// string type
			options.iStrType = (IsDlgButtonChecked (hWnd, CHK_ASCII) == BST_CHECKED);
			iResult = (IsDlgButtonChecked (hWnd, CHK_UNICODE) == BST_CHECKED);
			if (!(options.iStrType || iResult) || (options.iStrType && !iResult))
				options.iStrType = 1;
			else if (!options.iStrType && iResult)
				options.iStrType = 2;
			else
			    options.iStrType = 3;

			// nulltermination
			if (IsDlgButtonChecked (hWnd, CHK_NULLTERM_ONLY) == BST_CHECKED) options.iStrNullterm = 1;
			else options.iStrNullterm = 0;

			// check range
			GetDlgItemTextA (hWnd, EDF_FIRSTCHAR, szBuffer, MAX_STR_LEN);
			iResult = HexToInt (&szBuffer);
			options.cCharRangeLo = iResult < 0 ? 0x20 : iResult;
			GetDlgItemTextA (hWnd, EDF_LASTCHAR,  szBuffer, MAX_STR_LEN);
			iResult = HexToInt (&szBuffer);
			options.cCharRangeHi = iResult < 0 ? 0x7F : iResult;

			// swap if lo > hi
			if (options.cCharRangeLo > options.cCharRangeHi)
			{
				iResult = options.cCharRangeLo;
				options.cCharRangeLo = options.cCharRangeHi;
				options.cCharRangeHi = iResult;
			}

			// include/exclude list
			GetDlgItemTextA (hWnd, EDF_INCLUDECHAR, options.cCharIncl, MAX_STR_LEN);
			GetDlgItemTextA (hWnd, EDF_EXCLUDECHAR, options.cCharExcl, MAX_STR_LEN);

			// update ini-file and strings
			WriteIniFile (&options);
			SendMessage (options.hMainWnd, WM_COMMAND, MEN_UPDATE, 0);

			// set new font and adjust column header width
			SetListViewFont (&options);
			// fall through

			// close dialog
		case IDCANCEL:
			EndDialog (hWnd, 0);
			return 0;
		}
	}
	return FALSE;
}


/*------------------------------------------------------------------------
Procedure:     SearchStrings
Purpose:       Extract all strings from a file and put them in a listview
Input:         pvoid: pointer to thread parameters
Output:        None
Errors:        None
------------------------------------------------------------------------*/
void SearchStrings (PVOID pvoid)
{
	volatile PT_PARAMS  pTParams = (PT_PARAMS) pvoid;
	HANDLE 				hFile;
	LVLINE				lvl;
	BOOL				bWriteString = FALSE;
	DWORD				dwResult;
	unsigned long 		iAdr, iFileSize, iCount = 0, lStartTime;
	int					i, j, iFSizePercent = 0, iCountToPercent = 0, iUnicodeTermAdj = 0, iStrLen = 0, iMaxStrLen = 0;
	int					iStrState = 0;       // 0 = not valid;      1 = maybe str;    2 = maybe ascii;  3 = valid ascii
	         			                     // 4 = maybe uni char; 5 = maybe uni n1; 6 = maybe uni n2; 7 = valid unicode
	int					iCharState = 1;      // 0 = not valid;      1 = none yet,     2 = valid;        3 = NULL
	unsigned char		szBuf[2*MAX_STR_LEN] = "", *pszFileBuffer;

	// read source file into buffer
	hFile = CreateFileA (pTParams->szFileName, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
	if (hFile == INVALID_HANDLE_VALUE)
	{
		// quit when error
		wsprintf (szBuf, "Can't open file %s", pTParams->szFileName);
		MessageBox (pTParams->hMainWnd, szBuf, "Error", MB_OK | MB_ICONERROR);
		SendMessage (pTParams->hMainWnd, WM_COMMAND, MEN_UPDATE, 0);
		_endthread ();
		return;
	}

	// get file size and allocate memory
	iFileSize = (unsigned long long) GetFileSize (hFile, NULL);
	pszFileBuffer = (char *) malloc (iFileSize);
	if (pszFileBuffer == NULL)
	{
		// quit when error
		MessageBox (pTParams->hMainWnd, "Not enough memory!", "Error", MB_OK | MB_ICONERROR);
		SendMessage (pTParams->hMainWnd, WM_COMMAND, MEN_UPDATE, 0);
		_endthread ();
		return;
	}

	// read file into buffer
	if (ReadFile (hFile, pszFileBuffer, iFileSize, &dwResult, NULL) == 0)
	{
		// quit when error
		wsprintf (szBuf, "Can't read file %s", pTParams->szFileName);
		MessageBox (pTParams->hMainWnd, szBuf, "Error", MB_OK | MB_ICONERROR);
		SendMessage (pTParams->hMainWnd, WM_COMMAND, MEN_UPDATE, 0);
		_endthread ();
		return;
	}
	CloseHandle (hFile);

	// delete all listview entries and disable redraw listview
	SendMessage (pTParams->hListView, LVM_DELETEALLITEMS, 0, 0);
	SendMessage (pTParams->hListView, WM_SETREDRAW, FALSE, 0);

	// get time
	lStartTime = GetTickCount ();
	pTParams->lNumStrings = 0;
	pTParams->lNumAsciiStr = 0;
	pTParams->lNumUnicodeStr = 0;

	// set range [0-100] and increment [1] of progress bar
	SendMessage (pTParams->hProgBar, PBM_SETRANGE, 0, MAKELPARAM (0, 100));
	SendMessage (pTParams->hProgBar, PBM_SETSTEP, (WPARAM) 1, 0);
	iFSizePercent = iFileSize / 100;

	// parse file till eof or thread stopped
	iMaxStrLen = options.iStrMinLen;
	while ((iCount < iFileSize) && pTParams->bWorking)
	{
		// ----- 1st: determine char status
		iCharState = 1;
		if (pszFileBuffer[iCount] == '\0') iCharState = 3;
		else
		{
			// check whether char is part of exclude list
			j = lstrlen (options.cCharExcl);
			i = 0;
			while (i < j)
			{
				if (pszFileBuffer[iCount] == options.cCharExcl[i++])
				{
					iCharState = 0;
					break;
				}
			}

			// check whether char is in include list
			if (iCharState == 1)
			{
				j = lstrlen (options.cCharIncl);
				i = 0;
				while (i < j)
				{
					if (pszFileBuffer[iCount] == options.cCharIncl[i++])
					{
						iCharState = 2;
						break;
					}
				}
			}

			// check whether char is in range
			if (iCharState == 1)
			{
				if ((pszFileBuffer[iCount] >= options.cCharRangeLo) && (pszFileBuffer[iCount] <= options.cCharRangeHi)) iCharState = 2;
				else iCharState = 0;
			}
		}

		// ----- 2nd: determine string status
		switch (iCharState)
		{
		case 0:         // char not valid

			switch (iStrState)
			{
			case 2:     //   if "maybe ascii" --> "valid ascii" without nulltermination

				if (options.iStrNullterm == 0) iStrState = 3;
				else
				{
					iStrState = 0;
					iStrLen = 0;
				}
				break;

			case 4:     //   if "maybe unicode char" --> "valid unicode" without valid nullterm. (e.g. a b c* --> a b)

				if (options.iStrNullterm == 0)
				{
					iStrState = 7;
					iStrLen--;
					iUnicodeTermAdj = 1;
				}
				else
				{
					iStrState = 0;
					iStrLen = 0;
					iUnicodeTermAdj = 0;
				}
				break;

			case 5:     //   if "maybe unicode n1" --> "valid unicode" without valid nulltermination

				if (options.iStrNullterm == 0) iStrState = 7;
				else
				{
					iStrState = 0;
					iStrLen = 0;
					iUnicodeTermAdj = 0;
				}
				break;

			case 6:     //   if "maybe unicode n2" --> "valid unicode" without valid nulltermination

				if (options.iStrNullterm == 0)
				{
					iStrState = 7;
					iUnicodeTermAdj = 1;
				}
				else
				    {
					iStrState = 0;
					iStrLen = 0;
					iUnicodeTermAdj = 0;
				}
				break;

			default:    //   else --> invalid string

				iStrState = 0;
				iStrLen = 0;
				iUnicodeTermAdj = 0;
			}
			break;

		case 2:         // char valid (in range or in incl. list and not in excl. list)

			switch (iStrState)
			{
			case 0:     //   first valid char  --> "maybe string"
			case 1:     //   second valid char --> "maybe ascii"

				iStrState++;
				iStrLen++;
				break;

			case 2:     //   "maybe ascii"

				iStrLen++;
				break;

			case 4:     //   "maybe unicode" --> "maybe ascii" (e.g. a b c de --> a b c)

				if (options.iStrNullterm == 0)
				{
					iStrState = 7;
					iUnicodeTermAdj = -1;
					iStrLen--;
					iCount -= 2;
				}
				else
				{
					iStrState = 2;
					iStrLen = 2;
				}
				break;

			case 5:     //   "maybe unicode null1" --> "maybe unicode"

				iStrState = 4;
				iStrLen++;
				break;

			case 6:     //    "maybe unicode null2" --> "valid unicode" without valid nulltermination

				if (options.iStrNullterm == 0)
				{
					iStrState = 7;
					iCount--;
				}
				else
				    {
					iStrState = 1;
					iStrLen = 1;
				}
				break;

			default:    //   valid char after "valid ascii" or "valid unicode" or "maybe unicode null2" --> "maybe string"

				iStrState = 1;
				iStrLen = 1;
				iUnicodeTermAdj = 0;
				break;
			}
			break;

		case 3:         // nullchar

			switch (iStrState)
			{
			case 0:     //   no string

				iStrLen = 0;
				break;

			case 1:     //   first null after first valid char --> "maybe unicode"

				iStrState = 5;
				break;

			case 2:     //   valid nulltermination for ascii:    "maybe ascii"         --> "valid ascii"
			case 4:     //   unicode: null after valid char:     "maybe unicode"       --> "maybe unicode null1"
			case 5:     //   unicode: 2nd null after valid char: "maybe unicode null1" --> "maybe unicode null2"

				iStrState++;
				break;

			case 6:     //   valid nulltermination for unicode:  "maybe unicode null2" --> "valid unicode"

				iUnicodeTermAdj = 1;
				iStrState++;
				break;

			default:    //   null char after valid string --> invalid string

				iStrState = 0;
				iStrLen = 0;
				iUnicodeTermAdj = 0;
				break;
			}
			break;

		default:        // char undefined --> invalid string

			iStrState = 0;
			iStrLen = 0;
			iUnicodeTermAdj = 0;
			break;
		}

		// ----- 3rd: check strlen
		if ((iStrLen >= options.iStrMinLen) && (iStrLen <= options.iStrMaxLen))
		{
			switch (iStrState)
			{
			case 2:  // "maybe ascii"      --> invalid string if len >= max and invalid nulltermination
			case 4:  // "maybe unicode"    --> same
			case 5:  // "maybe unicode n1" --> same
			case 6:  // "maybe unicode n2" --> same

				if ((options.iStrNullterm == 0) && (iStrLen > options.iStrMaxLen))
				{
					iStrState = 0;
					iStrLen = 0;
					iUnicodeTermAdj = 0;
				}
				break;

			case 3:  // "valid ascii"

				if (options.iStrType != 2)
				{
					iAdr = iCount - iStrLen;
					wsprintfA (lvl.szFileOffset, "%.08X", iAdr);
					wsprintfA (lvl.szStrLen, "%.03d", iStrLen);
					lstrcpyn (lvl.szFileStr, &pszFileBuffer[iAdr], iStrLen+1);
					lvl.szFileStr[iStrLen] = 0;
					pTParams->lNumAsciiStr++;

					// save longest line for optimal column width in SetListViewFont
					if (iStrLen > iMaxStrLen)
					{
						options.iLineNumMaxStr[2] = pTParams->lNumStrings;
						iMaxStrLen = iStrLen;
					}
					bWriteString = TRUE;
				}
				break;

			case 7:  // "valid unicode"

				if (options.iStrType >= 2)
				{
					iStrLen *= 2;
					iAdr = iCount - iUnicodeTermAdj - iStrLen;
					wsprintfA (lvl.szFileOffset, "%.08X", iAdr);
					wsprintfA (lvl.szStrLen, "%.03d", iStrLen/2);
					for (i = 0; i < iStrLen; i += 2)
					{
						lvl.szFileStr[i] = pszFileBuffer[iAdr];
						lvl.szFileStr[i+1] = 0x20;
						iAdr += 2;
					}
					lvl.szFileStr[i] = 0;
					pTParams->lNumUnicodeStr++;

					// save longest line for optimal column width in SetListViewFont
					if (iStrLen > iMaxStrLen)
					{
						options.iLineNumMaxStr[2] = pTParams->lNumStrings;
						iMaxStrLen = iStrLen;
					}
					bWriteString = TRUE;
				}
				break;
			}

			if (bWriteString)
			{
				// insert string to listview
				SetListViewItems (pTParams->hListView, pTParams->lNumStrings, &lvl, TRUE);

				// update variables
				pTParams->lNumStrings++;
				bWriteString = FALSE;
				iUnicodeTermAdj = 0;
				iStrLen = 0;
			}
		}

		// next char
		iCount++;

		// update progressbar
		if (++iCountToPercent == iFSizePercent)
		{
			SendMessage (pTParams->hProgBar, PBM_STEPIT, 0, 0);
			iCountToPercent = 0;
		}
	}

	// enable redraw listview
	SendMessage (pTParams->hListView, WM_SETREDRAW, TRUE, 0);
	pTParams->lSearchTime = GetTickCount () - lStartTime;

	// cleanup and end thread
	SendMessage (pTParams->hProgBar, PBM_SETPOS, 0, 0);
	free (pszFileBuffer);
	if (pTParams->bWorking) SendMessage (pTParams->hMainWnd, WM_COMMAND, MEN_UPDATE, 0);
	_endthread ();
	return;
}


/*------------------------------------------------------------------------
Procedure:     SearchDlgProc
Purpose:       Handles the messages of the search dialog
Input:         Standard
Output:        TRUE if the message was processed
Errors:        None
------------------------------------------------------------------------*/
LRESULT CALLBACK SearchDlgProc (HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	LVITEM		lvi;
	LVLINE		lvl;
	BOOL		bStrFound;
	static long	i, iNumOfItems, iPrevSelItem;
	static char	szSearchTxt[MAX_STR_LEN];

	switch (uMsg)
	{
	case WM_INITDIALOG:

		// limit number of chars in search editfield
		SendDlgItemMessage (hWnd, EDF_SEARCH, EM_SETLIMITTEXT, MAX_STR_LEN-1, 0);
		EnableWindow (GetDlgItem (hWnd, BUT_SEARCH),      FALSE);
		EnableWindow (GetDlgItem (hWnd, BUT_SEARCHAGAIN), FALSE);
		memset (&lvi,  0, sizeof (lvi));
		iPrevSelItem = -1;
		SetFocus (GetDlgItem (hWnd, EDF_SEARCH));
		return 0;

	case WM_COMMAND:
		switch (LOWORD (wParam))
		{
		case EDF_SEARCH:
			// enable/disable buttons
			if (HIWORD (wParam) == EN_CHANGE)
			{
				if (SendDlgItemMessage (hWnd, EDF_SEARCH, WM_GETTEXTLENGTH, 0, 0) > 0)
				{
					EnableWindow (GetDlgItem (hWnd, BUT_SEARCH),      TRUE);
					EnableWindow (GetDlgItem (hWnd, BUT_SEARCHAGAIN), FALSE);
				}
				else EnableWindow (GetDlgItem (hWnd, BUT_SEARCH), FALSE);
			}
			break;

		case BUT_SEARCH:
			if (GetDlgItemText (hWnd, EDF_SEARCH, szSearchTxt, MAX_STR_LEN) == 0)
			{
				MessageBox (hWnd, "Can't search nothing!", "Error", MB_ICONERROR);
				break;
			}

			// deselect item
			if (iPrevSelItem >= 0)
			{
				ListView_SetItemState (options.hListView, iPrevSelItem, 0, LVIS_FOCUSED | LVIS_SELECTED);
				iPrevSelItem = -1;
			}

			// convert to lower case for non-case sensitive search
			if (IsDlgButtonChecked (hWnd, CHK_CASESENSITIVE) == BST_UNCHECKED) CharLower (szSearchTxt);
			iNumOfItems = SendMessage (options.hListView, LVM_GETITEMCOUNT, 0, 0);
			i = -1;
			// fall through

		case BUT_SEARCHAGAIN:
			bStrFound = FALSE;

			// parse entire listview
			while (!bStrFound && ++i < iNumOfItems)
			{
				// retrieve listview line
				if (GetListViewItems (options.hListView, i, &lvl) == FALSE)
				{
					// quit when error
					MessageBox (hWnd, "Can't retrieve listview items!", "Error", MB_OK | MB_ICONERROR);
					break;
				}

				// convert to lower case for case sensitive search
				if (IsDlgButtonChecked (hWnd, CHK_CASESENSITIVE) == BST_UNCHECKED) CharLower (lvl.szFileStr);

				// check if string contains search sequence
				if (IsDlgButtonChecked (hWnd, CHK_ENTIRESTRING) == BST_CHECKED)
					bStrFound = lstrcmp (lvl.szFileStr, szSearchTxt) == 0;

				// check if string equals search sequence
				else bStrFound = strstr (lvl.szFileStr, szSearchTxt) != NULL;
			}

			if (bStrFound)
			{
				// unselect previously selected item
				lvi.mask = LVIF_STATE;
				if (iPrevSelItem >= 0)
					ListView_SetItemState (options.hListView, iPrevSelItem, 0, LVIS_FOCUSED | LVIS_SELECTED);

				// make sure that found item is fully visible and highlit
				SendMessage (options.hListView, LVM_ENSUREVISIBLE, i, (LPARAM) (BOOL) FALSE);
				ListView_SetItemState (options.hListView, i, LVIS_FOCUSED | LVIS_SELECTED, LVIS_FOCUSED | LVIS_SELECTED);
				iPrevSelItem = i;
				EnableWindow (GetDlgItem (hWnd, BUT_SEARCHAGAIN), TRUE);
			}

			// if end of listview reached
			if (i >= iNumOfItems)
			{
				// if no item has been selected -> no match found in listview
				if (SendMessage (options.hListView, LVM_GETSELECTEDCOUNT, 0, 0) == 0)
					MessageBox (hWnd, "Nothing found!", "Error", MB_ICONERROR);

				// else beep
				else MessageBeep (0xffffffff);
			}
			return 0;

		case IDCANCEL:
			// unselect previously selected item
			lvi.mask = LVIF_STATE;
			if (iPrevSelItem >= 0)
				ListView_SetItemState (options.hListView, iPrevSelItem, 0, LVIS_FOCUSED | LVIS_SELECTED);

			// close dialog
			EndDialog (hWnd, 0);
			return 0;
		}
		break;
	}
	return FALSE;
}


/*------------------------------------------------------------------------
Procedure:     ReplaceDlgProc
Purpose:       Handles the messages of the replace dialog
Input:         Standard
Output:        TRUE if the message was processed
Errors:        None
------------------------------------------------------------------------*/
LRESULT CALLBACK ReplaceDlgProc (HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	LVITEM		lvi;
	LVLINE		lvl;
	BOOL		bStrFound;
	static long	i, j, iNumOfItems, iPrevSelItem;
	static char	szTmp[2*MAX_STR_LEN], szSearchTxt[MAX_STR_LEN], szReplaceTxt[MAX_STR_LEN], *pRepStart;
	static int	iSearchTxtLen, iReplaceTxtLen, iRepStrStart;

	switch (uMsg)
	{
	case WM_INITDIALOG:

		// limit number of chars in search editfield
		SendDlgItemMessage (hWnd, EDF_REP_SEARCH,  EM_SETLIMITTEXT, MAX_STR_LEN-1, 0);
		SendDlgItemMessage (hWnd, EDF_REP_REPLACE, EM_SETLIMITTEXT, MAX_STR_LEN-1, 0);
		EnableWindow (GetDlgItem (hWnd, BUT_REPLACE),      FALSE);
		EnableWindow (GetDlgItem (hWnd, BUT_REPLACEAGAIN), FALSE);
		CheckDlgButton (hWnd, CHK_REP_SPACES,   BST_CHECKED);
		CheckDlgButton (hWnd, CHK_REP_TRUNCATE, BST_CHECKED);
		CheckDlgButton (hWnd, CHK_REP_CONFIRM,  BST_CHECKED);
		memset (&lvi,  0, sizeof (lvi));
		iPrevSelItem = -1;
		SetFocus (GetDlgItem (hWnd, EDF_SEARCH));
		return 0;

	case WM_COMMAND:
		switch (LOWORD (wParam))
		{
		case CHK_REP_ENTIRESTRING:
			if (IsDlgButtonChecked (hWnd, CHK_REP_ENTIRESTRING) == BST_CHECKED)
			{
				CheckDlgButton (hWnd, CHK_REP_CASESENSITIVE, BST_CHECKED);
				EnableWindow (GetDlgItem (hWnd, CHK_REP_SPACES),   FALSE);
				EnableWindow (GetDlgItem (hWnd, CHK_REP_TRUNCATE), FALSE);
			}
			else
			{
				EnableWindow (GetDlgItem (hWnd, CHK_REP_SPACES),   TRUE);
				EnableWindow (GetDlgItem (hWnd, CHK_REP_TRUNCATE), TRUE);
			}
			break;

		case EDF_REP_SEARCH:
		case EDF_REP_REPLACE:
			// enable/disable buttons and deselect item in listview
			if (HIWORD (wParam) == EN_CHANGE)
			{
				if ((SendDlgItemMessage (hWnd, EDF_REP_SEARCH,  WM_GETTEXTLENGTH, 0, 0) > 0) &&
					(SendDlgItemMessage (hWnd, EDF_REP_REPLACE, WM_GETTEXTLENGTH, 0, 0) > 0))
				{
					EnableWindow (GetDlgItem (hWnd, BUT_REPLACE),      TRUE);
					EnableWindow (GetDlgItem (hWnd, BUT_REPLACEAGAIN), FALSE);
					ListView_SetItemState (options.hListView, iPrevSelItem, 0, LVIS_FOCUSED | LVIS_SELECTED);
				}
				else
				{
					EnableWindow (GetDlgItem (hWnd, BUT_REPLACE),      FALSE);
					EnableWindow (GetDlgItem (hWnd, BUT_REPLACEAGAIN), FALSE);
				}
			}
			break;

		case BUT_REPLACE:
			iSearchTxtLen  = GetDlgItemText (hWnd, EDF_REP_SEARCH,  szSearchTxt,  MAX_STR_LEN);
			iReplaceTxtLen = GetDlgItemText (hWnd, EDF_REP_REPLACE, szReplaceTxt, MAX_STR_LEN);
			if (iSearchTxtLen == 0 || iReplaceTxtLen == 0)
			{
				MessageBox (hWnd, "Fill in both fields!", "Error", MB_ICONERROR);
				return 1;
			}

			// truncate string if strlen too big
			if (iReplaceTxtLen > iSearchTxtLen)
			{
				if (IsDlgButtonChecked (hWnd, CHK_REP_TRUNCATE) == BST_CHECKED) szReplaceTxt[iSearchTxtLen] = '\0';
				else
				{
					MessageBox (hWnd, "Size of replace string can't be \nbigger than search string!", "Error", MB_ICONERROR);
					return 1;
				}
			}

			// fill missing chars with spaces
			else if (IsDlgButtonChecked (hWnd, CHK_REP_SPACES) == BST_CHECKED)
				for (i = iReplaceTxtLen; i < iSearchTxtLen; i++) szReplaceTxt[i] = ' ';

			// deselect item
			if (iPrevSelItem >= 0)
			{
				ListView_SetItemState (options.hListView, iPrevSelItem, 0, LVIS_FOCUSED | LVIS_SELECTED);
				iPrevSelItem = -1;
			}

			// convert to lower case for non-case sensitive search
			if (IsDlgButtonChecked (hWnd, CHK_REP_CASESENSITIVE) == BST_UNCHECKED) CharLower (szSearchTxt);
			iNumOfItems = SendMessage (options.hListView, LVM_GETITEMCOUNT, 0, 0);
			i = -1;
			// fall through

		case BUT_REPLACEAGAIN:
			bStrFound = FALSE;

			// parse entire listview
			while (!bStrFound && ++i < iNumOfItems)
			{
				// retrieve listview line
				if (GetListViewItems (options.hListView, i, &lvl) == FALSE)
				{
					// quit when error
					MessageBox (hWnd, "Can't retrieve listview items!", "Error", MB_OK | MB_ICONERROR);
					break;
				}

				// convert to lower case for non-case sensitive search. save original string first
				lstrcpy (szTmp, lvl.szFileStr);
				if (IsDlgButtonChecked (hWnd, CHK_REP_CASESENSITIVE) == BST_UNCHECKED) CharLower (szTmp);

				// check if string equals search sequence
				if (IsDlgButtonChecked (hWnd, CHK_REP_ENTIRESTRING) == BST_CHECKED)
					bStrFound = lstrcmp (szTmp, szSearchTxt) == 0;

				// check if string contains search sequence. save startposition
				else
				{
					pRepStart = strstr (szTmp, szSearchTxt);
					iRepStrStart = pRepStart - &szTmp[0];
					bStrFound = pRepStart != NULL;
				}
			}

			if (bStrFound)
			{
				// unselect previously selected item
				lvi.mask = LVIF_STATE;
				if (iPrevSelItem >= 0)
					ListView_SetItemState (options.hListView, iPrevSelItem, 0, LVIS_FOCUSED | LVIS_SELECTED);

				// make sure that found item is fully visible and highlit
				SendMessage (options.hListView, LVM_ENSUREVISIBLE, i, (LPARAM) (BOOL) FALSE);
				ListView_SetItemState (options.hListView, i, LVIS_FOCUSED | LVIS_SELECTED, LVIS_FOCUSED | LVIS_SELECTED);
				iPrevSelItem = i;
				EnableWindow (GetDlgItem (hWnd, BUT_REPLACEAGAIN), TRUE);

				// ask for confirmation
				if (IsDlgButtonChecked (hWnd, CHK_REP_CONFIRM) == BST_CHECKED)
				{
					if (MessageBox (hWnd, "Replace string?", "Confirm", MB_YESNO | MB_ICONQUESTION) == IDNO)
					{
						// deselect item if no change
						ListView_SetItemState (options.hListView, i, 0, LVIS_FOCUSED | LVIS_SELECTED);
						return 0;
					}
				}

				// update replacement buffer (used to update/undo exe-file with new strings)
				if (iNumOfReplace > MAX_STREP)
				{
					wsprintf (szTmp, "Can't replace more than %d strings!", MAX_STREP);
					MessageBox (hWnd, szTmp, "Error", MB_ICONERROR);
					return 0;
				}
				else
				{
					// save original string in replacement buffer
					strep[iNumOfReplace].iLVLineNum = i;
					lstrcpy (strep[iNumOfReplace].szOrigTxt, lvl.szFileStr);
					iNumOfReplace++;
				}

				// replace string in listview
				iPrevSelItem = i;
				if (IsDlgButtonChecked (hWnd, CHK_REP_ENTIRESTRING) == BST_CHECKED)
				{
					ListView_SetItemText (options.hListView, i, 2, szReplaceTxt);
				}
				else
				{
					for (j = 0; j < iReplaceTxtLen; j++)
						lvl.szFileStr[iRepStrStart++] = szReplaceTxt[j];

					// fill missing char with spaces or truncate string
					if (IsDlgButtonChecked (hWnd, CHK_REP_SPACES) == BST_CHECKED)
						for (j = iReplaceTxtLen; j < iSearchTxtLen; j++)
							lvl.szFileStr[iRepStrStart++] = ' ';
					else lvl.szFileStr[iRepStrStart] = '\0';
					ListView_SetItemText (options.hListView, i, 2, lvl.szFileStr);
				}
			}

			// if end of listview reached
			if (i >= iNumOfItems)
			{
				// if no item has been selected -> no match found in listview
				if (SendMessage (options.hListView, LVM_GETSELECTEDCOUNT, 0, 0) == 0)
					MessageBox (hWnd, "Nothing found!", "Error", MB_ICONERROR);

				// else beep
				else MessageBeep (0xffffffff);
			}
			return 0;

		case IDCANCEL:
			EndDialog (hWnd, 0);
			return 0;
		}
		break;
	}
	return FALSE;
}


/*------------------------------------------------------------------------
Procedure:     LVEditStringProc
Purpose:       Handles the messages of the LVEditString dialog
Input:         Standard. LineNum is passed as lParam in WM_INITDIALOG
Output:        TRUE if the message was processed
Errors:        None
------------------------------------------------------------------------*/
LRESULT CALLBACK LVEditStringProc (HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	static int	iOrigStrLen, iLineNum;
	int			iStrLen;
	char		szBuf[2*MAX_STR_LEN+20], szOrigTxt[2*MAX_STR_LEN+20];

	switch (uMsg)
	{
	case WM_INITDIALOG:
		// get selected item text and show it in editfield
		iLineNum = (long) lParam;
		ListView_GetItemText (options.hListView, iLineNum, 2, szOrigTxt, sizeof (szOrigTxt));
		iOrigStrLen = lstrlen (szOrigTxt);

		// limit strlen in editfield
		SendDlgItemMessage (hWnd, EDF_LVEDITSTRING, EM_SETLIMITTEXT, iOrigStrLen, 0);
		SetDlgItemText (hWnd, EDF_LVEDITSTRING, szOrigTxt);
		CheckDlgButton (hWnd, CHK_LVEDITSTRING_SPACES, BST_CHECKED);
		return 0;

	case WM_COMMAND:
		switch (LOWORD (wParam))
		{
		case IDOK:
		iStrLen = GetDlgItemText (hWnd, EDF_LVEDITSTRING, szBuf, sizeof (szBuf));

		// fill missing chars with spaces
		if (IsDlgButtonChecked (hWnd, CHK_LVEDITSTRING_SPACES) == BST_CHECKED)
		{
			while (iStrLen < iOrigStrLen)
				szBuf[iStrLen++] = ' ';
		}

		// replace string in listview
		ListView_SetItemText (options.hListView, iLineNum, 2, szBuf);

		// save original string in replacement buffer
		strep[iNumOfReplace].iLVLineNum = iLineNum;
		lstrcpy (strep[iNumOfReplace].szOrigTxt, szOrigTxt);
		iNumOfReplace++;

		// fall through

		case IDCANCEL:
			EndDialog (hWnd, 0);
			return 0;
		}
		break;
	}
	return FALSE;
}


/*------------------------------------------------------------------------
Procedure:     AboutDlgProc
Purpose:       Handles the messages of the about dialog
Input:         Standard
Output:        TRUE if the message was processed
Errors:        None
------------------------------------------------------------------------*/
LRESULT CALLBACK AboutDlgProc (HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg)
	{
	case WM_INITDIALOG:
		HICON hStatIcon = CreateWindowEx (WS_EX_LEFT, "STATIC", NULL, WS_CHILD | WS_VISIBLE | SS_ICON,
		                                  35, 25, 32, 32, hWnd, NULL, hInstance, NULL);
		HICON hIcon = LoadIcon (hInstance, MAKEINTRESOURCE (ICO_SHOWSTRING));
		SendMessage (hStatIcon, STM_SETIMAGE, (WPARAM) IMAGE_ICON, (LPARAM) (HANDLE) hIcon);
		SetFocus (GetDlgItem (hWnd, BUT_OK));
		return 0;

	case WM_COMMAND:
		if (LOWORD (wParam) == BUT_OK)
		{
			EndDialog (hWnd, 0);
			return 0;
		}
		break;
	}
	return FALSE;
}


/*------------------------------------------------------------------------
Procedure:     SortLVItems
Purpose:       Sort list in a listview based on column text
Input:         nCol:        column that contains text to be sorted
               bAscending:  sort direction, TRUE = ascending
			   low:         row to start scanning from, default = 0
			   high:        row to end scan, default = -1 (last row)
Output:        TRUE if the message was processed
Errors:        None
------------------------------------------------------------------------*/
BOOL SortLVItems (int nCol, BOOL bAscending, int iLow, int iHigh)
{
	LVLINE	lvlHi, lvlLo;
	int		iLo, iHi;
	char	szMidItem[2*MAX_STR_LEN], szBuf[2*MAX_STR_LEN], szTmp[2*MAX_STR_LEN];

	// check column number
	if (nCol >= NUM_OF_COL) return FALSE;

	// check hi and lo
	if (iHigh == -1) iHigh = ListView_GetItemCount (options.hListView) - 1;
	iLo = iLow;
	iHi = iHigh;
	if (iHi <= iLo) return FALSE;

	// get text of middle item
	ListView_GetItemText (options.hListView, (iLo+iHi)/2, nCol, szMidItem, sizeof (szMidItem));

	// loop through the list until indices cross
	while (iLo <= iHi)
	{
		// find 1st element that is >= partition element starting from left index
		if (bAscending)
		{
			while (iLo < iHigh)
			{
				ListView_GetItemText (options.hListView, iLo, nCol, szBuf, sizeof (szBuf));
				if (lstrcmp (szBuf, szMidItem) < 0) iLo++;
				else break;
			}
		}
		else
		{
			while (iLo < iHigh)
			{
				ListView_GetItemText (options.hListView, iLo, nCol, szBuf, sizeof (szBuf));
				if (lstrcmp (szBuf, szMidItem) > 0) iLo++;
				else break;
			}
		}

		// find element that is <= partition element starting from right index
		if (bAscending)
		{
			while (iHi > iLow)
			{
				ListView_GetItemText (options.hListView, iHi, nCol, szBuf, sizeof (szBuf));
				if (lstrcmp (szBuf, szMidItem) > 0) iHi--;
				else break;
			}
		}
		else
		{
			while (iHi > iLow)
			{
				ListView_GetItemText (options.hListView, iHi, nCol, szBuf, sizeof (szBuf));
				if (lstrcmp (szBuf, szMidItem) < 0) iHi--;
				else break;
			}
		}

		// swap if the indexes have not crossed and are not equal
		if (iLo <= iHi)
		{
			// swap only if the items are not equal
			ListView_GetItemText (options.hListView, iLo, nCol, szBuf, sizeof (szBuf));
			ListView_GetItemText (options.hListView, iHi, nCol, szTmp, sizeof (szTmp));
			if (lstrcmp (szBuf, szTmp) != 0)
			{
				// swap rows
				GetListViewItems (options.hListView, iLo, &lvlLo);
				GetListViewItems (options.hListView, iHi, &lvlHi);
				SetListViewItems (options.hListView, iLo, &lvlHi, FALSE);
				SetListViewItems (options.hListView, iHi, &lvlLo, FALSE);
			}
			++iLo;
			--iHi;
		}
	}

	// If the right index has not reached the left side of array -> sort the left partition
	if (iLow < iHi)
		SortLVItems (nCol, bAscending, iLow, iHi);

	// If the left index has not reached the right side of array -> sort the right partition
	if (iLo < iHigh)
	{
		// update progressbar (not very sophisticated but works fine for smaller files)
		if (iLo > iProgBarPos)
		{
			iProgBarPos = (iLo+iProgBarPos)/2;
			SendMessage (options.hProgBar, PBM_SETPOS, iProgBarPos, 0);
		}
		SortLVItems (nCol, bAscending, iLo, iHigh);
	}

	return TRUE;
}


/*------------------------------------------------------------------------
Procedure:     hMainWndProc
Purpose:       It handles all messages of the main dialog
Input:         Standard
Output:        TRUE if the message was processed
Errors:        None
------------------------------------------------------------------------*/
static BOOL CALLBACK hMainWndProc (HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	static HANDLE		hFile, hListView, hProgBar, hStatBar, hFontStatBar, hMenu, hPopMen;
	static HGLOBAL		hClip = NULL;
	static T_PARAMS		tparams;
	static OPENFILENAME	ofn;
	static LVCOLUMN		lvc[NUM_OF_COL];
	static BOOL         bFileOpened	= FALSE, bSortAscending = TRUE;
	static int			nSortedCol = -1;
	static char			szOpenFileName[MAX_PATH], szSaveFileName[MAX_PATH];
	static char			szOpenFilter[]	= "Executables (*.EXE, *.DLL)\0*.exe;*.dll\0"
	    				                  "All files (*.*)\0*.*\0\00";
	static char			szSaveFilter[]	= "Text files (*.TXT)\0*.txt\0\0";
	HDC					hDCStatBar;
	LVLINE				lvl;
	RECT				rc;
	LOGFONT				lf;
	HD_NOTIFY			*phdn;
	unsigned long		i, iNumOfItems, dwResult, lAdr, iStrLen, iLineNum;
	int					j, cxClient, cyClient;
	char				*szClipBuf, szBuf[2*MAX_STR_LEN+20], szButText[2][12] = {"Update", "Stop"};

	switch (uMsg)
	{
	// initialize
	case WM_CREATE:

		// save handle of maindialog (used for SendMessage in SearchString)
		memset (&options, 0, sizeof (options));
		options.hMainWnd = hDlg;
		memset (&tparams, 0, sizeof (tparams));
		tparams.hMainWnd = hDlg;
		SetWindowTextA (hDlg, szAppName);

		// create listview the size of the parent
		GetClientRect (hDlg, &rc);
		hListView = CreateWindow (WC_LISTVIEW, szAppName, WS_BORDER | WS_CHILD | WS_VISIBLE | LVS_REPORT | LVS_SORTASCENDING | LVS_SHOWSELALWAYS | LVS_SINGLESEL,
			                      0, 0, rc.right, rc.bottom-STAT_BAR_HIGHT, hDlg, NULL, hInstance, NULL );
		options.hListView = hListView;
		tparams.hListView  = hListView;

		// initialize listview
		SendMessage (hListView, LVM_SETEXTENDEDLISTVIEWSTYLE, 0, LVS_EX_FULLROWSELECT); // make selction the whole row
		lstrcpy (options.szColNames[0], "Offset");
		lstrcpy (options.szColNames[1], "Len");
		lstrcpy (options.szColNames[2], "String");

		// add listview columns and update listview font
		for (i = 0; i < NUM_OF_COL; i++)
		{
			lvc[i].mask		= LVCF_TEXT | LVCF_FMT;
			lvc[i].fmt		= LVCFMT_LEFT;
			lvc[i].pszText	= options.szColNames[i];
			if ((SendMessage (hListView, LVM_INSERTCOLUMN, i, (LPARAM) (LPLVCOLUMN) &lvc[i])) == -1)
				return 1;
		}
		SetListViewFont (&options);

		// create smooth progressbar
		hProgBar = CreateWindowEx (0, PROGRESS_CLASS, "", WS_CHILD | WS_VISIBLE | PBS_SMOOTH,
		                           0, rc.bottom-STAT_BAR_HIGHT, rc.right/5, STAT_BAR_HIGHT, hDlg, NULL, hInstance, NULL );
		options.hProgBar = hProgBar;
		tparams.hProgBar = hProgBar;

		// create statusbar
		hStatBar = CreateWindowEx (0, "EDIT", "", WS_CHILD | WS_VISIBLE | WS_DISABLED | ES_AUTOHSCROLL | ES_READONLY,
		                           rc.right/5, rc.bottom-STAT_BAR_HIGHT, rc.right-rc.right/5, STAT_BAR_HIGHT, hDlg, NULL, hInstance, NULL );
		options.hStatBar = hStatBar;
		tparams.hStatBar = hStatBar;

		// set statusbar font
		memset (&lf, 0, sizeof (lf));
		lstrcpy (lf.lfFaceName, "Arial");
		lf.lfHeight = STAT_BAR_HIGHT;
		lf.lfWeight = FW_NORMAL;
		hFontStatBar = CreateFontIndirect (&lf);
		hDCStatBar = GetDC (hStatBar);
		if (hDCStatBar != NULL)
		{
			if (DeleteObject (SelectObject (hDCStatBar, hFontStatBar)))
			{
				SendMessage (hStatBar, WM_SETFONT, (WPARAM) hFontStatBar, 0);
				ReleaseDC (hDlg, hDCStatBar);
			}
		}

		// disable menu items
		hMenu = GetMenu (hDlg);
		EnableMenuItem (hMenu, MEN_SAVE,     MF_GRAYED);
		EnableMenuItem (hMenu, MEN_SAVE_EXE, MF_GRAYED);
		EnableMenuItem (hMenu, MEN_UPDATE,  MF_GRAYED);
		EnableMenuItem (hMenu, MEN_SEARCH,   MF_GRAYED);
		EnableMenuItem (hMenu, MEN_REPLACE,  MF_GRAYED);

		// save handle of popup menu
		hPopMen = LoadMenu (hInstance, MAKEINTRESOURCE (MEN_POPMENU));
		hPopMen = GetSubMenu (hPopMen, 0);

		// read ini-file into options
		GetCurrentDirectory (MAX_STR_LEN, szBuf);
		wsprintf (options.szIniFile, "%s\\%s.ini", szBuf, szAppName);
		ReadIniFile (&options);

		// initialize structure ofn
		memset (&ofn, 0, sizeof (ofn));
		ofn.lStructSize     = sizeof (ofn);
		ofn.hwndOwner       = hDlg;
		ofn.nMaxFile        = MAX_PATH;
		ofn.nMaxFileTitle   = MAX_PATH;
		ofn.lpstrInitialDir = options.szFileOpenDir;

		// array to store originalstrings (replace)
		memset (&strep, 0, sizeof (strep));
		iNumOfReplace = 0;
		return 0;

	// resize childwindows when size of mainwindow has changed
	case WM_SIZE:
		cxClient = LOWORD (lParam);
		cyClient = HIWORD (lParam);
		MoveWindow (hListView, 0, 0, cxClient, cyClient-STAT_BAR_HIGHT, TRUE);
		MoveWindow (hProgBar, 0, cyClient-STAT_BAR_HIGHT, cxClient/5, STAT_BAR_HIGHT, TRUE);
		MoveWindow (hStatBar, cxClient/5, cyClient-STAT_BAR_HIGHT, cxClient-cxClient/5, STAT_BAR_HIGHT, TRUE);
		SetListViewFont (&options);
		return 0;

	// popup menu when mouse rightclick in listview window
	case WM_CONTEXTMENU:
		if ((HWND) wParam == hListView)
			if (SendMessage (hListView, LVM_GETSELECTEDCOUNT, 0, 0) == 1)
				TrackPopupMenuEx (hPopMen, TPM_RIGHTBUTTON, LOWORD (lParam), HIWORD (lParam), hDlg, NULL);
		return 0;

	case WM_COMMAND:
		switch (LOWORD (wParam))
		{
		// open file
		case MEN_OPEN:
			ofn.lpstrFilter	    = szOpenFilter;
			ofn.lpstrFile	    = szOpenFileName;
			ofn.lpstrTitle	    = TEXT ("Open file");
			ofn.Flags		    = OFN_FILEMUSTEXIST;
			ofn.lpstrInitialDir = options.szFileOpenDir;

			if (GetOpenFileName (&ofn))
			{
				// enable/disable buttons
				bFileOpened = TRUE;
				lstrcpy (tparams.szFileName, szOpenFileName);
				EnableMenuItem (hMenu, MEN_UPDATE, MF_ENABLED);
				SetWindowTextA (hStatBar, NULL);

				// set main window caption
				wsprintfA (szBuf, "%s   [%s]", szAppName, szOpenFileName);
				SetWindowTextA (hDlg, szBuf);

				// clear listview
				SendMessage (hListView, LVM_DELETEALLITEMS, 0, 0);
				EnableMenuItem (hMenu, MEN_SAVE,     MF_GRAYED);
				EnableMenuItem (hMenu, MEN_SAVE_EXE, MF_GRAYED);
				EnableMenuItem (hMenu, MEN_SEARCH,   MF_GRAYED);
				EnableMenuItem (hMenu, MEN_REPLACE,  MF_GRAYED);
			}
			// return 0;
			// fall through (get strings immediately after having chosen file)

		// update strings
		case MEN_UPDATE:
			// stop thread if active
			if (tparams.bWorking)
			{
				tparams.bWorking = FALSE;
				ModifyMenu (hMenu, MEN_UPDATE, MF_BYCOMMAND | MF_STRING, MEN_UPDATE, szButText[0]);
				EnableMenuItem (hMenu, MEN_OPEN,     MF_ENABLED);
				EnableMenuItem (hMenu, MEN_SAVE,     MF_ENABLED);
				EnableMenuItem (hMenu, MEN_SEARCH,   MF_ENABLED);
				EnableMenuItem (hMenu, MEN_REPLACE,  MF_ENABLED);
				EnableMenuItem (hMenu, MEN_OPTIONS,  MF_ENABLED);

				// update status field
				wsprintfA (szBuf, "Found %d strings (%d ASCII, %d Unicode) in %d.%0.3d seconds.",
				           tparams.lNumStrings, tparams.lNumAsciiStr, tparams.lNumUnicodeStr,
				           tparams.lSearchTime/1000, tparams.lSearchTime%1000);
				SetWindowTextA (hStatBar, szBuf);
				SetListViewFont (&options);
			}

			// else start thread (search strings)
			else if (bFileOpened)
			{
				tparams.bWorking = TRUE;
				_beginthread (SearchStrings, 0, &tparams);
				ModifyMenu (hMenu, MEN_UPDATE, MF_BYCOMMAND | MF_STRING, MEN_UPDATE, szButText[1]);
				EnableMenuItem (hMenu, MEN_OPEN,     MF_GRAYED);
				EnableMenuItem (hMenu, MEN_SAVE,     MF_GRAYED);
				EnableMenuItem (hMenu, MEN_SAVE_EXE, MF_GRAYED);
				EnableMenuItem (hMenu, MEN_SEARCH,   MF_GRAYED);
				EnableMenuItem (hMenu, MEN_REPLACE,  MF_GRAYED);
				EnableMenuItem (hMenu, MEN_OPTIONS,  MF_GRAYED);
				SetWindowTextA (hStatBar, NULL);
			}
			return 0;

		// save file
		case MEN_SAVE:
			// save only when file has been opened
			if (bFileOpened)
			{
				ofn.lpstrFilter	= szSaveFilter;
				ofn.lpstrFile	= szSaveFileName;
				ofn.lpstrTitle	= TEXT ("Save file");
				ofn.Flags		= OFN_EXPLORER;
				ofn.lpstrInitialDir = options.szFileSaveDir;
				ofn.lpstrDefExt = "txt";

				if (GetSaveFileName (&ofn))
				{
					// create file
					hFile = CreateFileA (szSaveFileName, GENERIC_WRITE, 0, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
					if (hFile == INVALID_HANDLE_VALUE)
					{
						// quit when error
						wsprintf (szBuf, "Can't open file %s", szSaveFileName);
						MessageBox (hDlg, szBuf, "Error", MB_OK | MB_ICONERROR);
					}
					else
					{
						iNumOfItems = SendMessage (hListView, LVM_GETITEMCOUNT, 0, 0);
						i = 0;
						while (i < iNumOfItems)
						{
							// retrieve listview line
							if (GetListViewItems (hListView, i, &lvl) == FALSE)
							{
								// quit when error
								MessageBox (hDlg, "Can't retrieve listview items!", "Error", MB_OK | MB_ICONERROR);
								break;
							}

							// write string to file
							if (WriteFile (hFile, szBuf, wsprintfA (szBuf, "%s  %s  %s%c%c", lvl.szFileOffset, lvl.szStrLen, lvl.szFileStr, 0xD, 0xA), &dwResult, NULL) == FALSE)
							{
								// quit when error
								wsprintf (szBuf, "Can't write file %s", szSaveFileName);
								MessageBox (hDlg, szBuf, "Error", MB_OK | MB_ICONERROR);
								break;
							}
							i++;
						}
						CloseHandle (hFile);
					}
				}
			}
			return 0;

		// save changes in exe-file
		case MEN_SAVE_EXE:
			// ask to backup original file
			wsprintf (szBuf, "Back-up file %s first?", szOpenFileName);
			if (MessageBox (hDlg, szBuf, "Backup", MB_YESNO | MB_ICONQUESTION) == IDYES)
			{
				wsprintf (szBuf, "%s.bak", szOpenFileName);
				CopyFile (szOpenFileName, szBuf, 0);
			}

			// open file
			hFile = CreateFileA (szOpenFileName, GENERIC_WRITE, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
			if (hFile == INVALID_HANDLE_VALUE)
			{
				// quit when error
				wsprintf (szBuf, "Can't open file %s", szOpenFileName);
				MessageBox (hDlg, szBuf, "Error", MB_OK | MB_ICONERROR);
				return 0;
			}

			// replace strings in file
			for (i = 0; i < iNumOfReplace; i++)
			{
				// retrieve listview line
				if (GetListViewItems (hListView, strep[i].iLVLineNum, &lvl) == FALSE)
				{
					// quit when error
					MessageBox (hDlg, "Can't retrieve listview items!", "Error", MB_OK | MB_ICONERROR);
					break;
				}
				lAdr    = strtol (lvl.szFileOffset, (char **) NULL, 16);
				iStrLen = strtol (lvl.szStrLen,     (char **) NULL, 10);

				// check if string is unicode: iStrLen = 1/2 lstrlen (lvl.szFileStr)
				if (iStrLen < lstrlen (lvl.szFileStr))
				{
					iStrLen = lstrlen (lvl.szFileStr);
					j = 0;

					// replace every second char with '\0' (in LV they're ' ')
					while (j < iStrLen)
					{
						lvl.szFileStr[++j] = '\0';
						j++;
					}
				}

				// set filepointer
				if (SetFilePointer (hFile, lAdr, NULL, FILE_BEGIN) == 0xFFFFFFFF)
				{
					MessageBox (hDlg, "Can't set filepointer!", "Error", MB_OK | MB_ICONERROR);
					break;
				}

				// replace string
				if (WriteFile (hFile, lvl.szFileStr, iStrLen, &dwResult, NULL) == 0)
				{
					wsprintf (szBuf, "Can't write file %s", szOpenFileName);
					MessageBox (hDlg, szBuf, "Error", MB_OK | MB_ICONERROR);
					break;
				}
			}

			// file successfully updated
			CloseHandle (hFile);
			MessageBoxA (hDlg, "File successfully updated!", "Success", MB_OK);
			iNumOfReplace = 0;
			return 0;

		// search dialog
		case MEN_SEARCH:
			DialogBox (NULL, MAKEINTRESOURCE (DLG_SEARCH), hDlg, (DLGPROC) SearchDlgProc);
			return 0;

		// replace dialog
		case MEN_REPLACE:
			DialogBox (NULL, MAKEINTRESOURCE (DLG_REPLACE), hDlg, (DLGPROC) ReplaceDlgProc);
			if (iNumOfReplace >= 0) EnableMenuItem (hMenu, MEN_SAVE_EXE, MF_ENABLED);
			return 0;

		// options dialog
		case MEN_OPTIONS:
			DialogBox (NULL, MAKEINTRESOURCE (DLG_OPTIONS), hDlg, (DLGPROC) OptionsDlgProc);
			return 0;

		// about box
		case MEN_ABOUT:
			DialogBox (NULL, MAKEINTRESOURCE (DLG_ABOUT), hDlg, (DLGPROC) AboutDlgProc);
			return 0;

		// exit
		case MEN_EXIT:
			SendMessage (hDlg, WM_CLOSE, 0, 0);
			return 0;

		// popup menu: edit
		case MEN_POP_EDIT:
			iLineNum = ListView_GetNextItem (hListView, -1, LVNI_SELECTED);
			DialogBoxParam (NULL, MAKEINTRESOURCE (DLG_LVEDITSTRING), hDlg, (DLGPROC) LVEditStringProc, iLineNum);
			if (iNumOfReplace >= 0) EnableMenuItem (hMenu, MEN_SAVE_EXE, MF_ENABLED);
			return 0;

		// popup menu: copy
		case MEN_POP_COPY:
			if (GetOpenClipboardWindow () != NULL) CloseClipboard ();
			if (OpenClipboard (NULL))
			{
				// get selected string from listview
				iLineNum = ListView_GetNextItem (hListView, -1, LVNI_SELECTED);
				GetListViewItems (hListView, iLineNum, &lvl);

				// allocate memory
				if (hClip != NULL) GlobalFree (hClip);
				hClip = GlobalAlloc (GMEM_MOVEABLE | GMEM_DDESHARE, sizeof (lvl.szFileStr)+1);

				// get pointer to allocated memory and copy string into it
				szClipBuf = (char *) GlobalLock (hClip);
				lstrcpy (szClipBuf, lvl.szFileStr);
				GlobalUnlock (hClip);

				// copy string to clipboard
				EmptyClipboard ();
				SetClipboardData (CF_TEXT, hClip);
				CloseClipboard ();
			}
			return 0;

		// popup menu: paste
		case MEN_POP_PASTE:
			if (GetOpenClipboardWindow () != NULL) CloseClipboard ();
			if (OpenClipboard (NULL))
			{
				// get data from clipboard
				szClipBuf = (char *) GetClipboardData (CF_TEXT);
				CloseClipboard ();

				// get string from listview
				iLineNum = ListView_GetNextItem (hListView, -1, LVNI_SELECTED);
				GetListViewItems (hListView, iLineNum, &lvl);

				// truncate string
				i = lstrlen (lvl.szFileStr);
				if (lstrlen (szClipBuf) > i)
					if (MessageBox (hDlg, "String too big!\nTruncate?", "Confirm", MB_YESNO | MB_ICONQUESTION) == IDYES)
						szClipBuf[i] = '\0';
					else return 0;

				// update listview with new string
				lstrcpy (lvl.szFileStr, szClipBuf);
				SetListViewItems (options.hListView, iLineNum, &lvl, FALSE);

				// save original string in replacement buffer
				if (iNumOfReplace > MAX_STREP)
				{
					wsprintf (szBuf, "Can't replace more than %d strings!", MAX_STREP);
					MessageBox (hDlg, szBuf, "Error", MB_ICONERROR);
				}
				else
				{
					strep[iNumOfReplace].iLVLineNum = iLineNum;
					lstrcpy (strep[iNumOfReplace].szOrigTxt, szClipBuf);
					iNumOfReplace++;
				}
			}
			if (iNumOfReplace >= 0) EnableMenuItem (hMenu, MEN_SAVE_EXE, MF_ENABLED);
			return 0;
		}
		break;

	// sort listview
	case WM_NOTIFY:
		// check if user clicked on header using left mouse button
		phdn = (HD_NOTIFY *) (LPNMHDR) lParam;
        if ((phdn->iButton == 0) && (phdn->hdr.code == HDN_ITEMCLICK))
        {
			if (phdn->iItem == nSortedCol) bSortAscending = !bSortAscending;
			else bSortAscending = TRUE;
			nSortedCol = phdn->iItem;

			// set range of progress bar
			iNumOfItems = SendMessage (hListView, LVM_GETITEMCOUNT, 0, 0);
			SendMessage (hProgBar, PBM_SETRANGE, 0, MAKELPARAM (0, iNumOfItems));
			iProgBarPos = 0;

			// sort listview
			SortLVItems (nSortedCol, bSortAscending, 0, -1);

			// reset progress bar
			SendMessage (hProgBar, PBM_SETPOS, 0, 0);
        }
		return 0;

	// close program
	case WM_DESTROY:
	case WM_CLOSE:
		GlobalFree (hClip);
		PostQuitMessage (0);
		return 0;
	}
	return DefWindowProc (hDlg, uMsg, wParam, lParam);
}
