--[[
	Peephole
	Attempt to remember and display what lies through your housing
	portal.

TODO
	Some sort of whimsical asset.
	Clean up the localization, fill out translations.

ISSUES
	Return Teleporter unit is not always seen/labelled, until
	clicked.

	Should unitcreate events get disabled while not rendering indicators?
	I am unsure if a unit to render could fire a create event before we see an
	event which enables rendering units, in which case they might get lost...
]]

require "Apollo"
require "ChatSystemLib"
require "GameLib"
require "HousingLib"
require "XmlDoc"

local function _dump(item) -- for debugging, remove before production
	local t = type(item)
	if     t == "table" then
		local out = {}
		for k, v in pairs(item) do
			table.insert(out, _dump(k) .. " = " .. _dump(v))
		end
		return "{ " .. table.concat(out, ", ") .. " }"
	elseif t == "string" then
		return "'" .. item .. "'"
	end
	return "(" .. tostring(item) .. ")"
end

local function _debug(strMessage, ...) -- extra-noisy print
	local trace = { "Peephole" }
	local index = 2
	while true do
		local cf = debug.getinfo(index)
		if cf == nil or cf.name == nil then
			break
		end
		table.insert(trace, 2, cf.name)
		index = index + 1
	end
	Print("[" .. table.concat(trace, "/") .. "] " .. strMessage:format(...))
end


local ktDefaultsAccount = { -- every setting must have a default
	strVersion = "",
	strColorFore = "FFFFFFFF",
	strFont = "Default",
	nMaxRange = 22.4, -- this is roughly the range to the end of the ramp
	kbOnlyInOwn = true
}

local knMaxRangeLimit = 2000 -- only allow setting draw range below this

local L = Apollo.GetPackage("Gemini:Locale-1.0").tPackage:GetLocale("Peephole", true)

local Peephole = {}

function Peephole:new(o)
	o = o or {}
	self.__index = self
	setmetatable(o, self)

	-- state persisted per-character
	o.state = {
		strLocation = "(" .. L["Mystery Awaits"] .. ")",
		tTimestamp = nil
	}

	-- options persisted per-account
	o.options = {}
	for k, v in pairs(ktDefaultsAccount) do
		o.options[k] = v
	end

	o.arPreloadUnits = {}
	o.nPreloadTally = 0
	o.bAddonRestoredOrLoaded = false

	o.strUnitNameMatch = L["Return Teleporter"]

	o.unitPlayer =  nil
	o.xmlDoc =  nil
	o.arUnits = {}
	o.bRendering = false

	o.wndCfg = nil
	o.xmlCfg = nil
	o.wndColorPicker = nil
	o.xmlTOC = nil

	return o
end

function Peephole:Init()
	Apollo.RegisterAddon(self, true, "Peephole", {})
end

function Peephole:OnLoad()
	Apollo.RegisterEventHandler("UnitCreated", "OnPreloadUnitCreated")

	self.xmlTOC = XmlDoc.CreateFromFile("toc.xml")
	self.xmlTOC:RegisterCallback("OnTOCLoaded", self)

	self.xmlDoc = XmlDoc.CreateFromFile("Peephole.xml")
	self.xmlDoc:RegisterCallback("OnDocLoaded", self)

	self.xmlCfg = XmlDoc.CreateFromFile("PeepholeConfigure.xml")
	self.xmlCfg:RegisterCallback("OnCfgLoaded", self)

	self.nMaxRange2 = self.options.nMaxRange * self.options.nMaxRange
end

function Peephole:OnPreloadUnitCreated(unitNew)
	self.arPreloadUnits[unitNew:GetId()] = unitNew
	self.nPreloadTally = self.nPreloadTally + 1
end

function Peephole:OnSave(eLevel)
	local tSaved = {}

	-- save all state
	if eLevel == GameLib.CodeEnumAddonSaveLevel.Character then
		for k, v in pairs(self.state) do
			tSaved[k] = v
		end
	end

	-- only save options which differ from defaults
	if eLevel == GameLib.CodeEnumAddonSaveLevel.Account then
		for k, v in pairs(ktDefaultsAccount) do
			if self.options[k] ~= v then
				tSaved[k] = self.options[k]
			end
		end
	end

	return (next(tSaved) ~= nil) and tSaved or nil
end

function Peephole:OnRestore(eLevel, tData)
	if eLevel == GameLib.CodeEnumAddonSaveLevel.Character then
		self.state.strLocation = tData.strLocation or self.state.strLocation
		self.state.tTimestamp = tData.tTimestamp or self.state.tTimestamp

		-- update any existing indicators
		for unitId, tInfo in pairs(self.arUnits) do
			if tInfo.wnd then
				tInfo.wnd:SetText(self.state.strLocation)
			end
		end

		self:CreateUnitsFromPreload()
	end

	if eLevel == GameLib.CodeEnumAddonSaveLevel.Account then
		-- only restore settings we know about
		for k, v in pairs(ktDefaultsAccount) do
			if tData[k] ~= nil then
				self.options[k] = tData[k]
			end
		end

--[[	Never allow zero transparency for the indicator color.
		This is ostensibly for the users' own good, but mostly because
		of a bug in release 1.0.3 which might have set that as the
		default... ]]
		local crFore = ApolloColor.new(self.options.strColorFore)
		if crFore:ToTable().a == 0 then
			ChatSystemLib.PostOnChannel(ChatSystemLib.ChatChannel_Command, "[Peephole] " .. L["Indicator transparency is being reset, because it was invisible."], "")
			-- this could be better
			self.options.strColorFore = "FF" .. string.sub(self.options.strColorFore, 3)
		end

		-- refresh distance squared
		self.nMaxRange2 = self.options.nMaxRange * self.options.nMaxRange

		-- update any existing indicators
		self:OnNextFrame()

		-- update option window if it exists
		self:ConfigSet()
	end
end

function Peephole:OnConfigure()
	if self.xmlCfg and self.xmlCfg:IsLoaded() and self.wndCfg then
		self:ConfigSet() -- ensure settings match reality
		self:ConfigureSizeReset() -- ensure window isn't odd size
		self.wndCfg:Show(true)
	end
end

function Peephole:OnWindowManagementReady()
	local wnd = self.wndCfg
	local strName = wnd:FindChild("PeepholeConfiguration:Title"):GetText()
	Event_FireGenericEvent("WindowManagementAdd", {wnd = wnd, strName = strName})
end

-----------------------------------------------------------------------------------------------

function Peephole:OnTOCLoaded()
	if self.xmlTOC ~= nil and self.xmlTOC:IsLoaded() then
		self.strVersion = self.xmlTOC:ToTable().Version
		self.xmlTOC = nil
		self:ConfigureVersion()
	end
end

function Peephole:OnDocLoaded()
	if self.xmlDoc ~= nil and self.xmlDoc:IsLoaded() then
		Apollo.RegisterEventHandler("ChangeWorld", "On_NewLocation", self)
		Apollo.RegisterEventHandler("SubZoneChanged", "On_NewLocation", self)
		Apollo.RegisterEventHandler("VarChange_ZoneName", "On_NewLocation", self)

		Apollo.RemoveEventHandler("UnitCreated", self)
		Apollo.RegisterEventHandler("UnitCreated", "OnUnitCreated", self)
		Apollo.RegisterEventHandler("UnitDestroyed", "OnUnitDestroyed", self)

		self:SetRenderIndicators()
		self:CreateUnitsFromPreload()
		self:CommensalUnitHarvest()

		Apollo.RegisterSlashCommand("peephole", "OnSlashCommand", self)
	end
end

function Peephole:OnCfgLoaded()
	if self.xmlCfg and self.xmlCfg:IsLoaded() then
		self.wndCfg = Apollo.LoadForm(self.xmlCfg, "PeepholeConfigure", nil, self)
		if self.wndCfg == nil then
			Apollo.AddAddonErrorText(self, "unexpected nil from LoadForm")
			return
		end
		self.wndCfg:Show(false, true)

		-- store original window size, before windowmanager resizes it
		-- see ConfigureSizeReset
		local nLeft, nTop, nRight, nBottom = self.wndCfg:GetAnchorOffsets()
		self.tCfgSize = {nWidth = (nRight - nLeft), nHeight = (nBottom - nTop)}

		Apollo.RegisterEventHandler("WindowManagementReady", "OnWindowManagementReady", self)

		self.wndCfg:FindChild("OnlyInOwnHouseButton"):SetText(L["Only show while in own house."])
		self.wndCfg:FindChild("RangeWindow"):SetText(L["Display indicator when closer than this to a Return Teleporter."])
		self.wndCfg:FindChild("IndicatorFontLabel"):SetText(L["Indicator Font"])
		self:ConfigSet()
		self:ConfigureAvailableFontList()
		self:ConfigureVersion()
	end
end

--[[ ConfigureVersion
	Embeds the version from the TOC into the appropriate string in the configuration window.
	This needs both the TOC and Cfg doc-ready callbacks to have fired, so we invoke it from
	both but soak the first call.
]]
function Peephole:ConfigureVersion()
	self.ConfigureVersion = function(self)
		self.wndCfg:FindChild("PeepholeConfigure:Version"):SetText(String_GetWeaselString(L["version $1n"], self.strVersion))
	end
end

--[[ ConfigureSizeReset
	The configuration window is not currently user-resizable, but WindowManagement will
	merrilly restore a prior version's dimensions.  Things look rather stupid if the
	window has actually changed sizes between versions.
]]
function Peephole:ConfigureSizeReset()
	if not self.strVersion
	or self.options.strVersion == self.strVersion then
		return
	end

	local nLeft, nTop, nRight, nBottom = self.wndCfg:GetAnchorOffsets()
	nRight = nLeft + self.tCfgSize.nWidth
	nBottom = nTop + self.tCfgSize.nHeight

	self.wndCfg:SetAnchorOffsets(nLeft, nTop, nLeft + self.tCfgSize.nWidth, nTop + self.tCfgSize.nHeight)

	local OptionsInterface = Apollo.GetAddon("OptionsInterface")
	if OptionsInterface ~= nil then
		OptionsInterface:UpdateTrackedWindow(self.wndCfg)
		self.options.strVersion = self.strVersion
	end
end

function Peephole:CreateUnitsFromPreload()
	if self.bAddonRestoredOrLoaded then
		if self.arPreloadUnits then
			for idUnit, unitNew in pairs(self.arPreloadUnits) do
				self:OnUnitCreated(unitNew)
			end
			self.arPreloadUnits = nil
		end
	end
	self.bAddonRestoredOrLoaded = true
end

--[[ CommensalUnitHarvest
	Attempt to scrape other addons for extant units.
	If we load late, we wouldn't have seen the unit creation events we
	need.
	This is not foolproof for locating the teleporter on load, but it
	certainly improves our chances.
]]
function Peephole:CommensalUnitHarvest()
	local function _res_bare(v) return v end
	local function _res_carbine(v) return v.unitOwner end
	local function _res_nyan(v) return v.unit end

	local arAddonLookupMap = {
		{
			strAddon = "Nameplates",
			strUnitArrayField = "arPreloadUnits",
			fnUnitResolve = _res_bare
		},
		{
			strAddon = "Nameplates",
			strUnitArrayField = "arUnit2Nameplate",
			fnUnitResolve = _res_carbine
		},
		{
			strAddon = "BijiPlates",
			strUnitArrayField = "arUnit2Nameplate",
			fnUnitResolve = _res_carbine
			},
		{
			strAddon = "BijiPlates",
			strUnitArrayField = "arUnit2Nameplate",
			fnUnitResolve = _res_carbine
		},
		{
			strAddon = "NPrimeNameplates",
			strUnitArrayField = "buffer",
			fnUnitResolve = _res_bare
		},
		{
			strAddon = "NPrimeNameplates",
			strUnitArrayField = "nameplates",
			fnUnitResolve = _res_nyan
		},
		{
			strAddon = "RazorPlates",
			strUnitArrayField = "arPreloadUnits",
			fnUnitResolve = _res_bare
		},
		{
			strAddon = "RazorPlates",
			strUnitArrayField = "arUnit2Nameplate",
			fnUnitResolve = _res_carbine
		}
	}

	for idx, map in pairs(arAddonLookupMap) do
		local NP = Apollo.GetAddon(map.strAddon)
		if NP ~= nil then
			for k,v in pairs(NP[map.strUnitArrayField]) do
				unit = map.fnUnitResolve(v)
				if unit ~= nil then
					self:OnUnitCreated(unit)
				end
			end
		end
	end
end

-- repopulate font dropdown list
function Peephole:ConfigureAvailableFontList()
	local wndFontList = self.wndCfg:FindChild("FontPickerList")
	local strFont = self.options.strFont

	wndFontList:DestroyChildren()

	local btnSelected = nil
	local nSelection = 0
	local index = 0
	local tFontDup = {} -- GetGameFonts seems to mention some fonts multiple times
	for i, tFontInfo in pairs(Apollo.GetGameFonts()) do
		if tFontDup[tFontInfo.name] == nil then -- so only list each one once
			tFontDup[tFontInfo.name] = true
			local bCurrentSelection = (strFont == tFontInfo.name)
			if string.sub(tFontInfo.name, -1) == "O" -- outlined fonts only
			or bCurrentSelection -- unless something special is already set
			or tFontInfo.name == ktDefaultsAccount.strFont then -- or the default
				local wndEntry = Apollo.LoadForm(self.xmlCfg, "FontSelectorButton", wndFontList, self)
				if wndEntry == nil then
					Apollo.AddAddonErrorText(self, "unexpected nil from LoadForm")
					return
				end
				index = index + 1
				if bCurrentSelection then
					nSelection = index
					btnSelected = wndEntry
				end
				wndEntry:FindChild("SelectedIcon"):Show(bCurrentSelection)
				wndEntry:SetData(tFontInfo.name)
				wndEntry:SetText(tFontInfo.face .. ":" .. tFontInfo.name)
				wndEntry:SetFont(tFontInfo.name)
				wndEntry:SetStyleEx("UseWindowTextColor", bCurrentSelection)
			end
		end
	end

	wndFontList:ArrangeChildrenVert(0)
	-- scroll to selected
	local nScrollPos = ((nSelection - 1) * (btnSelected and (btnSelected:GetHeight() or 0))) + ((btnSelected:GetHeight() - wndFontList:GetHeight()) / 2)
	wndFontList:SetVScrollPos(nScrollPos)
end

function Peephole:SetColorFore(strColor)
	local color = ApolloColor.new(strColor)

	-- update option
	self.options.strColorFore = strColor

--[[ Chaining the FindChildren here because for some reason a FindChild on
	'ColorPickerForeButton:ColorChip' returns nil, even though the full-path
	(PeepholeConfigure:...:ColorChip) will find it fine.  'ColorChip' alone
	will also find it, but I intend to someday have a background color
	option as well, so am being more explicit here. ]]
	-- update color chip
	wndColorChipFore = self.wndCfg:FindChild("ColorPickerForeButton")
	wndColorChipFore:FindChild("ColorChip"):SetBGColor(color)

	-- update indicators
	for unitId, tInfo in pairs(self.arUnits) do
		tInfo.wnd:SetTextColor(color)
	end
end

function Peephole:SetFont(strFont)
	local wndFontPicker = self.wndCfg:FindChild("FontPickerButton")
	local wndFontList = self.wndCfg:FindChild("FontPickerList")

	-- deselect the old font in the list
	local wndFontOld = wndFontList:FindChildByUserData(self.options.strFont)
	if wndFontOld then
		wndFontOld:FindChild("SelectedIcon"):Show(false)
		wndFontOld:SetStyleEx("UseWindowTextColor", false)
	end

	self.options.strFont = strFont

	-- select the new font in the list
	local wndFontNew = wndFontList:FindChildByUserData(strFont)
	if wndFontNew then
		wndFontNew:FindChild("SelectedIcon"):Show(true)
		wndFontNew:SetStyleEx("UseWindowTextColor", true)
	end

	-- update the dropdown indicator
	wndFontPicker:SetData(strFont)
	wndFontPicker:SetText(strFont)

	-- update indicators
	for unitId, tInfo in pairs(self.arUnits) do
		tInfo.wnd:SetFont(strFont)
	end

	-- if there were no matches in the list for old or new font,
	-- try recreating it
	if not wndFontOld and not wndFontNew then
		self:ConfigureAvailableFontList()
	end
end

function Peephole:ConfigSet(tOptions)
	if self.wndCfg == nil then
		return
	end

	tOptions = tOptions or self.options
	self.wndCfg:FindChild("RevertButton"):Show(self.tOptionsCheckpoint ~= nil)

	if tOptions.bOnlyInOwn then
		self.wndCfg:FindChild("OnlyInOwnHouseButton"):SetCheck(tOptions.bOnlyInOwn)
	end

	if tOptions.nMaxRange then
		self.wndCfg:FindChild("RangeEditBox"):SetText(tOptions.nMaxRange)
	end

	if tOptions.strFont then
		self:SetFont(tOptions.strFont)
	end

	if tOptions.strColorFore then
		self:SetColorFore(tOptions.strColorFore)
	end
end

function Peephole:OnCfgRevert(wndHandler, wndControl, eMouseButton)
	self:CfgOptionsRevert()
end

function Peephole:CfgOptionsRevert()
	if self.tOptionsCheckpoint then
		for k, v in pairs(self.tOptionsCheckpoint) do
			self.options[k] = v
		end
		self:CfgOptionsClear()
		self:ConfigSet()
	end
end

function Peephole:CfgOptionsClear()
	self.tOptionsCheckpoint = nil
	self.wndCfg:FindChild("RevertButton"):Show(false, true)
end

function Peephole:CfgOptionsCheckpoint()
	if self.tOptionsCheckpoint == nil then
		self.tOptionsCheckpoint = {}
		for k, v in pairs(self.options) do
			self.tOptionsCheckpoint[k] = v
		end
		self.wndCfg:FindChild("RevertButton"):Show(true, true)
	end
end

function Peephole:ConfigureHide()
	self.wndCfg:FindChild("FontPickerListFrame"):Show(false)
	self.wndCfg:FindChild("FontPickerButton"):SetCheck(false)

	if self.wndColorPicker ~= nil then
		self.wndColorPicker:Show(false)
		self.wndColorPicker:Destroy()
		self.wndColorPicker = nil
	end

	self.wndCfg:Show(false)
end

function Peephole:OnCfgOK(wndHandler, wndControl, eMouseButton)
	self:ConfigureHide()
	self:CfgOptionsClear()
	self:ConfigSet()
end

function Peephole:OnCfgCancel(wndHandler, wndControl, eMouseButton)
	self:ConfigureHide()
	self:CfgOptionsRevert()
end

function Peephole:OnCfgOnlyInOwnHouseCheck(wndHandler, wndControl, eMouseButton)
	self:CfgOptionsCheckpoint()
	self.options.bOnlyInOwn = wndControl:IsChecked()
end

function Peephole:OnCfgRangeChanged(wndHandler, wndControl, strText)
	self:CfgOptionsCheckpoint()
	local nRangeNew = tonumber(strText)
	local bOutOfRange = false
	if nRangeNew == nil then
		bOutOfRange = true
		nRangeNew = self.options.nMaxRange
	elseif nRangeNew < 0 then
		bOutOfRange = true
		nRangeNew = 1
	elseif nRangeNew > knMaxRangeLimit then
		bOutOfRange = true
		nRangeNew = knMaxRangeLimit
	end
	if bOutOfRange then
		wndControl:SetText(nRangeNew)
	end

	self.options.nMaxRange = nRangeNew
	self.nMaxRange2 = nRangeNew * nRangeNew
end

function Peephole:OnCfgRangeEnter(wndHandler, wndControl, strText)
	self:OnCfgRangeChanged(wndHandler, wndControl, wndControl:GetText())
end

function Peephole:OnCfgFontPickerButton(wndHandler, wndControl, eMouseButton)
	self.wndCfg:FindChild("FontPickerListFrame"):Show(wndControl:IsChecked())
end

function Peephole:OnCfgFontSelectorButton(wndHandler, wndControl, eMouseButton)
	self:CfgOptionsCheckpoint()
	self:SetFont(wndControl:GetData())
	self.wndCfg:FindChild("FontPickerButton"):SetCheck(false)
	self.wndCfg:FindChild("FontPickerListFrame"):Show(false)
end

function Peephole:OnCfgColorSelected(strColor)
	-- colorpicker returns a value when cancelled...
	-- so ignore if value has not changed
	if strColor == self.options.strColorFore then
		return
	end
	self:CfgOptionsCheckpoint()
	self:SetColorFore(strColor)
end

function Peephole:OnCfgColorForeButton(wndHandler, wndControl, eMouseButton)
	if self.wndColorPicker == nil then
		local GeminiColor = Apollo.GetPackage("GeminiColor").tPackage
		self.wndColorPicker = GeminiColor:CreateColorPicker(self, "OnCfgColorSelected", true, self.options.strColorFore)
	end
	self.wndColorPicker:Show(true)
end

function Peephole:OnSlashCommand(strCmd, strArg)
	local fnIterWord = string.gmatch(strArg, "%S+")
	local strAction = fnIterWord()
	if strAction == "debug" then
		Print("[Peephole] Debug")
		Print("state: " .. _dump(self.state))
		Print("options: " .. _dump(self.options))
		Print("arUnits: " .. _dump(self.arUnits))
		Print("nPreloadTally: " .. _dump(self.nPreloadTally))
		Print("bRendering: " .. _dump(self.bRendering))
	elseif strAction == "config" then
		self:OnConfigure()
	else
		ChatSystemLib.PostOnChannel(ChatSystemLib.ChatChannel_Command, "[Peephole] Commands: help, config, debug", "")
	end
end

function Peephole:On_NewLocation(idZone, strZoneName)
	-- idZone and strZoneName are only present for some events

	local strLocation = strZoneName or GetCurrentZoneName()
	if strLocation == nil then
		Apollo.AddAddonErrorText(self, "nil zone name")
		-- XXX: maybe defer this, queue and poll until not nil?
	end

	self.unitPlayer = GameLib.GetPlayerUnit()

	if not HousingLib.IsHousingWorld() then
		self:RecordLocation()
	end

	self:SetRenderIndicators()
end

function Peephole:OnUnitCreated(unit)
	if unit == nil or not unit:IsValid() then
		return
	end

	-- XXX: is there a better way to match these?
	local strUnitName = unit:GetName()
	if strUnitName == self.strUnitNameMatch then
		self:IndicateUnit(unit)
	end
end

function Peephole:OnUnitDestroyed(unit)
	self:UnindicateUnit(unit)
end

function Peephole:OnNextFrame()
	if self.unitPlayer == nil or not self.unitPlayer:IsValid() then
		self.unitPlayer = GameLib.GetPlayerUnit()
		if self.unitPlayer == nil then
			return
		end
	end

	local tPosPlayer = self.unitPlayer:GetPosition()
	if tPosPlayer == nil then
		return
	end

	for unitId, tInfo in pairs(self.arUnits) do
		local unit = tInfo.unit
		local wnd = tInfo.wnd

		if not unit or not unit:IsValid()
		or not wnd or not wnd:IsValid() then
			if wnd then
				wnd:Show(false, true)
				wnd:Destroy()
			end
			self.arUnits[unitId] = nil
		else
			local tPosUnit = unit:GetPosition()
			if tPosUnit then
				local nDeltaX = tPosUnit.x - tPosPlayer.x
				local nDeltaY = tPosUnit.y - tPosPlayer.y
				local nDeltaZ = tPosUnit.z - tPosPlayer.z
				nDeltaX = nDeltaX * nDeltaX
				nDeltaY = nDeltaY * nDeltaY
				nDeltaZ = nDeltaZ * nDeltaZ
				local bInRange = ((nDeltaX + nDeltaY + nDeltaZ) <= self.nMaxRange2)

				local bShow = (self.bRendering
				               and bInRange
				               and GameLib.GetUnitScreenPosition(unit).bOnScreen
				               and not unit:IsOccluded()
				               and not wnd:IsOccluded())

				wnd:Show(bShow)
			end
		end
	end
end

-----------------------------------------------------------------------------------------------

function Peephole:IndicateUnit(unit)
	if unit == nil or not unit:IsValid() then
		return
	end

	local unitId = unit:GetId()
	if not unitId then
		return
	end

	local tInfo = self.arUnits[unitId]
	if tInfo == nil then
		tInfo = {}
		self.arUnits[unitId] = tInfo
	end
	tInfo.unit = unit
	if tInfo.wnd and not tInfo.wnd:IsValid() then
		tInfo.wnd:Destroy()
		tInfo.wnd = nil
	end
	if tInfo.wnd == nil then
		tInfo.wnd = Apollo.LoadForm(self.xmlDoc, "DestinationIndicator", "InWorldHudStratum", self)
		if tInfo.wnd == nil then
			Apollo.AddAddonErrorText(self, "unexpected nil from LoadForm")
			return
		end
	end
	tInfo.wnd:SetUnit(tInfo.unit, 1)
	tInfo.wnd:SetText(self.state.strLocation)
	tInfo.wnd:SetTextColor(self.options.strColorFore)
	tInfo.wnd:SetFont(self.options.strFont)

--[[ One time, the indicator window wasn't being displayed, despite everything
	appearing to be set correctly.  It showed up when I invoked it, so I guess
	I'll just call that once, preventatively, even though it doesn't seem to
	fit the model of what it's supposed to do. ]]
	tInfo.wnd:Invoke()

	self:OnNextFrame()
end

function Peephole:UnindicateUnit(unit)
	if unit == nil then
		return
	end

	local unitId = unit:GetId()
	local tInfo = self.arUnits[unitId]
	if tInfo == nil then
		return
	end
	self.arUnits[unitId] = nil
	tInfo.wnd:Show(false, true)
	tInfo.wnd:Destroy()
end

function Peephole:RecordLocation()
	local strLocation = GetCurrentZoneName()

	if strLocation == nil then
		Apollo.AddAddonErrorText("nil zone name")
		return
	end

	local strSubzone = GetCurrentSubZoneName()
	if strSubzone and strSubzone ~= strLocation then
		strLocation = strLocation .. " - " .. strSubzone
	end

	if self.state.strLocation ~= strLocation then
		self.state.strLocation = strLocation
		for unitId, tInfo in pairs(self.arUnits) do
			if tInfo.wnd then
				tInfo.wnd:SetText(self.state.strLocation)
			end
		end
	end
	self.state.tTimestamp = GameLib.GetServerTime()
end

function Peephole:SetRenderIndicators(bDisplay)
	if bDisplay == nil then
		bDisplay = HousingLib.IsHousingWorld()
		if bDisplay and self.options.bOnlyInOwn then
			bDisplay = HousingLib.IsOnMyResidence()
		end
	end

	local eFrameEvent = "VarChange_FrameCount" -- "NextFrame"
	if bDisplay then
		if not self.bRendering then
			Apollo.RegisterEventHandler(eFrameEvent, "OnNextFrame", self)
			self.bRendering = true
		end
	else
		if self.bRendering then
			Apollo.RemoveEventHandler(eFrameEvent, self)
			self.bRendering = false
			-- but invoke one more time to update any indicators
			self:OnNextFrame()
		end
	end
end

-----------------------------------------------------------------------------------------------

local PeepholeInst = Peephole:new()
PeepholeInst:Init()
