﻿using System;
using System.Text;
using System.IO;
using System.Collections.Generic;
using System.Xml;

namespace CaoNiMaCoolPad
{
    class Program
    {
        const int messageBlockLength = 1624;
        readonly static byte[] startFlag = { 0xff, 00, 00, 00 };
        readonly static byte[] receiveBox = { 01, 00, 00, 00, 01, 00, 00, 00, 00, 00, 00, 00 };
        readonly static byte[] sentBox = { 07, 00, 00, 00, 04, 00, 00, 00, 00, 00, 00, 00 };
        readonly static byte[] darftBox = { 08, 00, 00, 00, 06, 00, 00, 00, 00, 00, 00, 00 };

        const int timeFieldLength = 32;
        const int phoneNumberFieldLength = 68;
        const int nameFieldLength = 82;
        const int contentFieldLength = 1343;


        static void Main(string[] args)
        {
            Console.WriteLine("指定酷派导出的短消息的TBL文件。例如E:\\smsinfo.tbl");

            List<Message> messages = ReadCoolPadMessageTblFile(Console.ReadLine()); //@"E:\my\桌面\研究\完整备份\smsinfo.tbl");

            Console.WriteLine("指定转换后的文件的位置，例如D:\\message.xml");
            string xmlFile = Console.ReadLine(); //"D:\\message.xml";
            SaveToXml(xmlFile, messages);
            string xslFile = Path.GetDirectoryName(xmlFile) + "style.xsl";
            WriteXslFile(xslFile);

            Console.WriteLine("\r\n本程序已导出您的短消息，生成了以下文件：\r\n"
                               + "  {0}： 您的短消息的原始副本，可二次开发使用\r\n"
                               + "  {1}： 用于修饰上一文件的样式信息\r\n"
                               + "{0}的阅读方法是右击此文件，打开方式，选择IE(7)浏览器。高版本的火狐浏览器、Chrome、Opera或许也可以。\r\n\r\n"
                               + "如果您需要导出多个xml文件，可以将它们导出到同一目录下，而一个xsl文件可以被共用。于是节省了您的硬盘空间。"
                               , xmlFile, xslFile);

            Console.WriteLine("  完成");
            Console.ReadKey();

        }

        private static void WriteXslFile(string fileName)
        {
            if (File.Exists(fileName) == false)
            {
                TextWriter sw = new StreamWriter(fileName);
                sw.Write(CaoNiMaCoolPad.Properties.Resources.style);
                sw.Close();
            }
        }

        private static void SaveToXml(string fileName, List<Message> messages)
        {

            XmlTextWriter writer = new XmlTextWriter(fileName, Encoding.UTF8);

            writer.WriteStartDocument();
            writer.WriteProcessingInstruction("xml-stylesheet", "type='text/xsl' href='style.xsl'");


            writer.Formatting = Formatting.Indented;

            writer.WriteStartElement("Messages");

            writer.WriteStartElement("count");
            writer.WriteValue(messages.Count);
            writer.WriteEndElement();

            foreach (Message m in messages)
            {
                writer.WriteStartElement("Message");

                writer.WriteStartElement("location");
                writer.WriteString(m.Location);
                writer.WriteEndElement();

                writer.WriteStartElement("phoneNumber");
                writer.WriteString(m.PhoneNumber);
                writer.WriteEndElement();

                writer.WriteStartElement("name");
                writer.WriteString(m.Name);
                writer.WriteEndElement();

                writer.WriteStartElement("time");
                writer.WriteString(m.time.ToString());
                writer.WriteEndElement();

                writer.WriteStartElement("time2");
                writer.WriteString(m.time2.ToString());
                writer.WriteEndElement();

                writer.WriteStartElement("content");
                writer.WriteString(m.Content);
                writer.WriteEndElement();

                writer.WriteEndElement();
            }
            writer.WriteEndElement();

            writer.WriteEndDocument();

            writer.Close();
        }

        private static List<Message> ReadCoolPadMessageTblFile(string fileName)
        {
            byte[] message = new byte[messageBlockLength];
            FileStream fs = new FileStream(fileName, FileMode.Open);

            ReadBeginning(fs, message, 4);
            List<Message> messages = new List<Message>();
            while (ReadMessage(fs, message, messages)) ;
            fs.Close();

            return messages;
        }


        private static void ReadBeginning(FileStream fs, byte[] message, int length)
        {
            if (length != fs.Read(message, 0, length))
                throw new FormatException();
        }

        private static bool ReadMessage(FileStream fs, byte[] message, IList<Message> list)
        {
            if (fs.Read(message, 0, messageBlockLength) >= 1620)
            {
                int position = 0;
                if (Match(message, position, startFlag) == false)
                    throw new FormatException();

                position += startFlag.Length;
                string location = GetLocation(message, position);
                position += receiveBox.Length;

                DateTime time1 = GetTime(message, position);
                position += timeFieldLength;

                DateTime time2 = GetTime(message, position);
                position += timeFieldLength;

                string phoneNumber = PhraseUnicode(message, position, phoneNumberFieldLength);
                position += phoneNumberFieldLength;

                string name = PhraseUnicode(message, position, nameFieldLength);
                position += nameFieldLength;

                string content = PhraseUnicode(message, position, contentFieldLength);

                list.Add(new Message { Location = location, Content = content, Name = name, PhoneNumber = phoneNumber, time = time1, time2 = time2 });

                return true;
            }
            else
                return false;
        }

        private static string PhraseUnicode(byte[] message, int position, int length)
        {
            int p = position;
            while (p + 1 < message.Length && (message[p] != 0 || message[p + 1] != 0) && p < position + length)
                p += 2;

            char[] cs = Encoding.Unicode.GetChars(message, position, p - position);
            return new string(cs);
        }

        private static DateTime GetTime(byte[] message, int position)
        {
            int year = message[position + 1] * 16 * 16 + message[position];
            position += 4;

            int month = message[position];
            position += 4;

            int day = message[position];
            position += 4;

            int hour = message[position];
            position += 4;

            int minute = message[position];
            position += 4;

            int second = message[position];
            position += 12;

            return new DateTime(year, month, day, hour, minute, second);
        }

        private static string GetLocation(byte[] message, int position)
        {
            if (Match(message, position, receiveBox))
                return "收件箱";
            else if (Match(message, position, sentBox))
                return "已发送";
            else if (Match(message, position, darftBox))
                return "草稿箱";
            else
            {
                string location = "未知：";
                for (int i = 0; i < receiveBox.Length; i++)
                    location += message[position + i].ToString("X2") + " ";
                return location;
            }
        }

        private static bool Match(byte[] message, int position, byte[] pattern)
        {
            for (int i = 0; i < pattern.Length; i++)
                if (message[position + i] != pattern[i])
                    return false;
            return true;
        }
    }

    struct Message
    {
        public static Message Empty = new Message();

        public string Location { get; set; }
        public string Content { set; get; }
        public string PhoneNumber { set; get; }
        public string Name { set; get; }
        public DateTime time { set; get; }
        public DateTime time2 { set; get; }
    }
}
